/*******************************************************************************
 * The information contained in this file is confidential and proprietary to
 * QLogic Corporation.  No part of this file may be reproduced or
 * distributed, in any form or by any means for any purpose, without the
 * express written permission of QLogic Corporation.
 *
 * (c) COPYRIGHT 2015 QLogic Corporation, ALL RIGHTS RESERVED.
 *******************************************************************************/
/* **********************************************************
 * Copyright 2015 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * qfle3_cnic.c --
 *
 *      CNIC interface for storage drivers.
 */
#ifdef QFLE3_CNIC
#include "qfle3.h"
#include "qfle3_sm.h"
#include "qfle3_cnic.h"

static void qfle3_get_iscsi_info(struct qfle3_adapter *adapter);
static void qfle3_get_fcoe_info(struct qfle3_adapter *adapter);
extern vmk_uint32 netpoll_count;

int qfle3_set_fcoe_eth_rx_mode(struct qfle3_adapter *adapter, vmk_Bool start);
/*
 * Global Data
 */
vmk_DeviceOps qfle3_cnic_device_ops = {
   .removeDevice = qfle3_remove_cnic_device,
};
VMK_ReturnStatus qfle3_remove_cnic_device(vmk_Device device)
{
   //TODO: what else todo other than Unregistering the device?
   
   vmk_LogMessage("Removing CNIC devie");
   return vmk_DeviceUnregister(device);
}

struct cnic_eth_dev *qfle3_cnic_probe(vmk_AddrCookie driverData)
{
   qfle3_adapter *adapter = (qfle3_adapter*) driverData.ptr;
   vmk_uint32 port = QFLE3_PORT(adapter);
   vmk_uint32 func = QFLE3_FUNC(adapter);
//   int i,j,k;
   
   vmk_LogMessage("nicOps.cnicProbeNic");
   if (adapter == NULL) {
      vmk_LogMessage("%s: Invalid cookie provided.\n", __func__);
      return NULL;
   }
   
   struct cnic_eth_dev *cp = (struct cnic_eth_dev *)adapter->cnicEthDev;

   /* If both iSCSI and FCoE are disabled - return NULL in
    * order to indicate CNIC that it should not try to work
    * with this device.
    */
   if (NO_ISCSI_OOO(adapter) && NO_FCOE(adapter))
      return NULL;
   
   QFLE3_DBG(QFLE3_DBG_CNIC, "nicOps.cnicProbeNic");
   cp->apiRevision = CNIC_ETH_DEV_VER;
   cp->moduleID = vmk_ModuleCurrentID;
   cp->driverData.ptr = adapter;

   cp->driverState = 0;
   if (adapter->intr.intrType == QFLE3_IT_MSIX)
      cp->driverState = CNIC_DRV_STATE_USING_MSIX;

   cp->chip_id = adapter->hw_info.chip_id;
   cp->link_supported_speeds = adapter->port.supported[port];
   cp->mf_mode = adapter->mf_mode;
   cp->max_kwqe_pending = 8;
   cp->pdev = adapter->pdev;
   vmk_PCIQueryDeviceID(cp->pdev, &cp->pdev_id);
   vmk_PCIQueryDeviceAddr(cp->pdev, &cp->pdev_addr);
   vmk_NameCopy(&cp->pdev_name, &adapter->pdev_name);

   cp->device = adapter->device;
   cp->reg_base = adapter->reg_base;
   cp->db_base = adapter->db_base;
   cp->db_size = adapter->db_size;
   cp->mtu = adapter->mtu;

   cp->ctx_blk_size = CDU_ILT_PAGE_SZ;
   cp->ctx_tbl_offset = FUNC_ILT_BASE(QFLE3_FUNC(adapter)) +
         qfle3_cid_ilt_lines(adapter);
   cp->ctx_tbl_len = CNIC_ILT_LINES;
   cp->starting_cid = qfle3_cid_ilt_lines(adapter) * ILT_PAGE_CIDS;


   cp->mf_sub_mode = adapter->mf_sub_mode;
   cp->e1hov_tag = MF_CFG_RD(adapter, func_mf_config[func].e1hov_tag) &
              FUNC_MF_CFG_E1HOV_TAG_MASK;
   //cp->cna_vlgrp;

   cp->max_iscsi_conn = CNIC_ISCSI_CID_MAX;
   cp->max_fcoe_conn = CNIC_FCOE_CID_MAX;
   cp->max_fcoe_exchanges = MAX_NUM_FCOE_TASKS_PER_ENGINE;

   cp->fcoe_init_cid = QFLE3_FCOE_ETH_CID(adapter);
   
   cp->fcoe_wwn_port_name_hi = 
      adapter->cnicEthDev->fcoe_wwn_port_name_hi;
   cp->fcoe_wwn_port_name_lo = 
      adapter->cnicEthDev->fcoe_wwn_port_name_lo;
   
   cp->fcoe_wwn_node_name_hi = 
      adapter->cnicEthDev->fcoe_wwn_node_name_hi;
   cp->fcoe_wwn_node_name_lo = 
      adapter->cnicEthDev->fcoe_wwn_node_name_lo;
   
   cp->iscsi_l2_client_id = qfle3_cnic_eth_cl_id(adapter, QFLE3_ISCSI_ETH_CL_ID_IDX);
   cp->iscsi_l2_cid = QFLE3_ISCSI_ETH_CID(adapter);

   cp->cnicSlowPathCmd = &qfle3_cnic_sp_queue;
   cp->cnicDriverCtl = &qfle3_cnic_drv_ctl;
   cp->cnicRegister = &qfle3_register_cnic;
   cp->cnicUnregister = &qfle3_unregister_cnic;
   cp->drv_get_sfp_diagnostic = &qfle3_get_sfp_diagnostic;
   cp->cnicGetFcNpivTbl = &qfle3_get_fc_npiv_tbl;

   if (NO_ISCSI_OOO(adapter)){
      cp->driverState |= CNIC_DRV_STATE_NO_ISCSI_OOO;
      QFLE3_DBG(QFLE3_DBG_CNIC, "CNIC_PROBE, NO_ISCSI_OOO supported\n");
   }

   if (NO_ISCSI(adapter)){
      cp->driverState |= CNIC_DRV_STATE_NO_ISCSI;
      QFLE3_DBG(QFLE3_DBG_CNIC, "CNIC_PROBE, NO_ISCSI supported\n");
   }
   if (NO_FCOE(adapter)){
      cp->driverState |= CNIC_DRV_STATE_NO_FCOE;
      QFLE3_DBG(QFLE3_DBG_CNIC, "CNIC_PROBE, NO_FCOE supported\n");
   }
   cp->addr_drv_info_to_mcp = &adapter->sp->drv_info_to_mcp;

   QFLE3_DBG(QFLE3_DBG_CNIC, 
         "page_size %d, tbl_offset %d, tbl_lines %d, starting cid %d\n",
         cp->ctx_blk_size,
         cp->ctx_tbl_offset,
         cp->ctx_tbl_len,
         cp->starting_cid);
   return cp;
}

VMK_ReturnStatus qfle3_register_cnic(vmk_Device nicDev, cnic_ops *ops, void *data)
{
   qfle3_adapter *adapter;
   struct cnic_eth_dev *cp;
   vmk_AddrCookie driverData;
   VMK_ReturnStatus rc;

   vmk_DeviceGetAttachedDriverData(nicDev, &driverData);
   adapter = (qfle3_adapter*) driverData.ptr;
   cp = (struct cnic_eth_dev *)adapter->cnicEthDev;
   QFLE3_DBG(QFLE3_DBG_CNIC, "nicOps.cnicRegister");

   if (ops == NULL) {
      QFLE3_ERR("NULL ops received\n");
      return VMK_FAILURE;
   }

//   adapter->cnic_enabled = VMK_TRUE;

   adapter->cnic_kwq = qfle3_heap_alloc(QFLE3_PAGE_SIZE);
   if (!adapter->cnic_kwq)
      return VMK_FAILURE;

   adapter->cnic_kwq_cons = adapter->cnic_kwq;
   adapter->cnic_kwq_prod = adapter->cnic_kwq;
   adapter->cnic_kwq_last = adapter->cnic_kwq + MAX_SP_DESC_CNT;

   adapter->cnic_spq_pending = 0;
   adapter->cnic_kwq_pending = 0;

   adapter->cnic_data = data;
   vmk_SemaLock(&adapter->cnic_mutex);
   cp->driverState |= CNIC_DRV_STATE_REGD;
   adapter->cnicOps = ops;

   qfle3_register_cnic_irq_handlr(adapter, ops->cnicIRQHandler);

   vmk_SemaUnlock(&adapter->cnic_mutex);

//   qfle3_start_cnicqs(adapter);
   if (!NO_ISCSI_OOO(adapter)) {
      
      if (adapter->flags & OWN_CNIC_IRQ) {// if we own it, release it.
         vmk_IntrDisable(adapter->intr.cookies[1]);
         //    qfle3_unregister_interrupt(adapter, 1);

         rc = vmk_IntrUnregister(vmk_ModuleCurrentID, adapter->intr.cookies[1], adapter->intr.handlerData[1]);
         if (rc != VMK_OK)
         QFLE3_ERR("Failed to give up CNIC interrupt vector 0x%x\n", adapter->intr.cookies[1]);

         adapter->flags &= ~OWN_CNIC_IRQ;
         
         QFLE3_ERR("Releasing CNIC IRQ to qcnic");
      }
   }
   /* now give up the interrupt handler */
	qfle3_get_iscsi_info(adapter);
   qfle3_setup_cnic_irq_info(adapter);
	qfle3_get_fcoe_info(adapter);
   adapter->cnic_loaded = VMK_TRUE;

   /* Schedule driver to read CNIC driver versions */
   //qfle3_schedule_sp_rtnl(bp, QFLE3_SP_RTNL_GET_DRV_VERSION, 0);
   return VMK_OK;
}
VMK_ReturnStatus qfle3_unregister_cnic(vmk_Device nicDev)
{
   qfle3_adapter *adapter;
   struct cnic_eth_dev *cp;
   vmk_AddrCookie driverData;
   struct qfle3_fastpath *fp;
   VMK_ReturnStatus status;
   int tries = 0;

   vmk_DeviceGetAttachedDriverData(nicDev, &driverData);
   adapter = (qfle3_adapter*) driverData.ptr;
   
   adapter->cnic_loaded = VMK_FALSE;
   cp = (struct cnic_eth_dev *)adapter->cnicEthDev;
   QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_unregister_cnic called\n");
//   qfle3_stop_cnicqs(adapter);
   // regain cnic irq
   if (!NO_ISCSI_OOO(adapter)) {
      fp = qfle3_ooo_fp(adapter);
try_register_again:
      status = qfle3_register_interrupt(adapter, (1),
              fp->name, fp,
              qfle3_interrupt_ack, qfle3_msix_rx);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to retake intrCookie #%d 0x%x (%x)",1,
              adapter->intr.cookies[1], status);
         if (tries < 1) {
            tries++;
//            qfle3_unregister_interrupt(adapter,1);
            goto try_register_again;
         }
      }else {
      
         adapter->flags |= OWN_CNIC_IRQ;
         
         QFLE3_DBG(QFLE3_DBG_CNIC, "Retook intrCookie #%d 0x%x (%x) for CNIC qfle3_msix_rx", 1,
              adapter->intr.cookies[1], status);
         status = vmk_IntrEnable(adapter->intr.cookies[1]);
         if (status != VMK_OK) {
            QFLE3_ERR("Failed to enable intrCookie[%d] 0x%x (%x)", 1, adapter->intr.cookies[1], status);
   //         goto fail_intr_enable;
         }
      }
   }
   vmk_SemaLock(&adapter->cnic_mutex);
   cp->driverState = 0;
   adapter->cnicOps = NULL;
   adapter->cnicIRQHandler = NULL;
   adapter->cnic_data = NULL;
   vmk_SemaUnlock(&adapter->cnic_mutex);
//   adapter->cnic_enabled = VMK_FALSE;
   qfle3_heap_free(adapter->cnic_kwq);
   adapter->cnic_kwq = NULL;
	return 0;
}

VMK_ReturnStatus qfle3_get_sfp_diagnostic(vmk_Device dev, struct sfp_diagnostic *data)
{
   qfle3_adapter *adapter;
   vmk_AddrCookie driverData;
   VMK_ReturnStatus rc;
   int phy_idx;
   __be16 field;

   vmk_DeviceGetAttachedDriverData(dev, &driverData);
   adapter = (qfle3_adapter*) driverData.ptr;


   if (!qfle3_is_nvm_accessible(adapter)) {
      QFLE3_DBG(QFLE3_DBG_NVM,
         "Cannot access eeprom when interface is down\n");
      return VMK_FAILURE;
   }

   phy_idx = qfle3_get_cur_phy_idx(adapter);
   qfle3_acquire_phy_lock(adapter);

   /* Data read from SFP is going to be ordered in big-endian
    * and we'll need to convert it to host.
    */
   rc = elink_read_sfp_module_eeprom(&adapter->link_params.phy[phy_idx],
            &adapter->link_params,
            ELINK_I2C_DEV_ADDR_A2,
            ELINK_SFP_EEPROM_A2_TEMPERATURE_ADDR,
            ELINK_SFP_EEPROM_A2_TEMPERATURE_SIZE,
            (u8 *)&field);
   if (rc)
      goto out;
   data->temperature = vmk_BE16ToCPU(field);

   rc = elink_read_sfp_module_eeprom(&adapter->link_params.phy[phy_idx],
            &adapter->link_params,
            ELINK_I2C_DEV_ADDR_A2,
            ELINK_SFP_EEPROM_A2_VCC_ADDR,
            ELINK_SFP_EEPROM_A2_VCC_SIZE,
            (u8 *)&field);
   if (rc)
      goto out;
   data->vcc = vmk_BE16ToCPU(field);

   rc = elink_read_sfp_module_eeprom(&adapter->link_params.phy[phy_idx],
            &adapter->link_params,
            ELINK_I2C_DEV_ADDR_A2,
            ELINK_SFP_EEPROM_A2_TX_BIAS_ADDR,
            ELINK_SFP_EEPROM_A2_TX_BIAS_SIZE,
            (u8 *)&field);
   if (rc)
      goto out;
   data->tx_bias = vmk_BE16ToCPU(field);

   rc = elink_read_sfp_module_eeprom(&adapter->link_params.phy[phy_idx],
            &adapter->link_params,
            ELINK_I2C_DEV_ADDR_A2,
            ELINK_SFP_EEPROM_A2_TX_POWER_ADDR,
            ELINK_SFP_EEPROM_A2_TX_POWER_SIZE,
            (u8 *)&field);
   if (rc)
      goto out;
   data->tx_power = vmk_BE16ToCPU(field);

   rc = elink_read_sfp_module_eeprom(&adapter->link_params.phy[phy_idx],
            &adapter->link_params,
            ELINK_I2C_DEV_ADDR_A2,
            ELINK_SFP_EEPROM_A2_RX_POWER_ADDR,
            ELINK_SFP_EEPROM_A2_RX_POWER_SIZE,
            (u8 *)&field);
   if (rc)
      goto out;
   data->rx_power = vmk_BE16ToCPU(field);

   qfle3_release_phy_lock(adapter);
   QFLE3_DBG(QFLE3_DBG_NVM,
      "Temp: %08x\nVCC: %08x\nTx Bias: %08x\nTx Power: %08x\nRx Power: %08x\n",
      data->temperature, data->vcc, data->tx_bias, data->tx_power,
      data->rx_power);

   /* TODO - fill-in the speed-cap [not sure of the right format] */
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_LOADED) && adapter->link_vars.link_up &&
       !(adapter->flags & MF_FUNC_DIS)) {
      if (IS_MF(adapter) && !QFLE3_NOMCP(adapter))
      data->opr_speed = qfle3_get_mf_speed(adapter);
      else
      data->opr_speed = adapter->link_vars.line_speed;
   } else {
      data->opr_speed = 0;
   }

   return VMK_OK;
out:
   qfle3_release_phy_lock(adapter);
   if (rc)
      return VMK_FAILURE;
   else
      return VMK_OK;
}
VMK_ReturnStatus qfle3_get_fc_npiv_tbl(vmk_Device dev, struct qcnic_fc_npiv_tbl *cnic_tbl)
{
   
   struct bdn_fc_npiv_tbl *tbl = NULL;
   u32 offset, entries;
   int i;
   vmk_uint32 len;
   
   qfle3_adapter *adapter;
   vmk_AddrCookie driverData;
   VMK_ReturnStatus rc = VMK_FAILURE;

   vmk_DeviceGetAttachedDriverData(dev, &driverData);
   adapter = (qfle3_adapter*) driverData.ptr;

   if (!SHMEM2_HAS(adapter, fc_npiv_nvram_tbl_addr[0]))
      goto out;

   QFLE3_DBG(QFLE3_DBG_NVM, "About to read the FC-NPIV table\n");

   tbl = qfle3_heap_alloc(sizeof(*tbl));
   if (!tbl) {
      QFLE3_ERR("Failed to allocate fc_npiv table\n");
      goto out;
   }

   offset = SHMEM2_RD(adapter, fc_npiv_nvram_tbl_addr[QFLE3_PORT(adapter)]);
   if (!offset) {
      QFLE3_DBG(QFLE3_DBG_NVM, "No FC-NPIV in NVRAM\n");
      goto out;
   }
   QFLE3_DBG(QFLE3_DBG_NVM, "Offset of FC-NPIV in NVRAM: %08x\n", offset);

   /* Read the table contents from nvram */
   if (qfle3_nvram_read(adapter, offset, (u8 *)tbl, sizeof(*tbl), &len)) {
      QFLE3_ERR("Failed to read FC-NPIV table\n");
      goto out;
   }

   /* Since bnx2x_nvram_read() returns data in be32, we need to convert
    * the number of entries back to cpu endianess.
    */
   entries = tbl->fc_npiv_cfg.num_of_npiv;
   entries = (u32)vmk_BE32ToCPU(( __be32)entries);
   tbl->fc_npiv_cfg.num_of_npiv = entries;

   if (!tbl->fc_npiv_cfg.num_of_npiv) {
      QFLE3_DBG(QFLE3_DBG_NVM,
         "No FC-NPIV table [valid, simply not present]\n");
      goto out;
   } else if (tbl->fc_npiv_cfg.num_of_npiv > MAX_NUMBER_NPIV) {
      QFLE3_ERR("FC-NPIV table with bad length 0x%08x\n",
           tbl->fc_npiv_cfg.num_of_npiv);
      goto out;
   } else {
      QFLE3_DBG(QFLE3_DBG_NVM, "Read 0x%08x entries from NVRAM\n",
         tbl->fc_npiv_cfg.num_of_npiv);
   }

   /* Copy the data into cnic-provided struct */
   cnic_tbl->count = tbl->fc_npiv_cfg.num_of_npiv;
   for (i = 0; i < cnic_tbl->count; i++) {
      vmk_Memcpy(cnic_tbl->wwpn[i], tbl->settings[i].npiv_wwpn, 8);
      vmk_Memcpy(cnic_tbl->wwnn[i], tbl->settings[i].npiv_wwnn, 8);
   }

   rc = VMK_OK;
out:
   qfle3_heap_free(tbl);
   return rc;
}

VMK_ReturnStatus qfle3_cnic_sp_post(struct qfle3_adapter *adapter, int count)
{
   
   struct eth_spe *spe;
//   int cxt_index, cxt_offset;
   struct cnic_eth_dev *cp = (struct cnic_eth_dev *)adapter->cnicEthDev;

   vmk_SpinlockLock(adapter->spq_lock);
   adapter->cnic_spq_pending -= count;

   for (; adapter->cnic_kwq_pending; adapter->cnic_kwq_pending--) {
      vmk_uint16 type =  (le16toh(adapter->cnic_kwq_cons->hdr.type)
            & SPE_HDR_CONN_TYPE) >> SPE_HDR_CONN_TYPE_SHIFT;
//      vmk_uint8 cmd = (le32toh(adapter->cnic_kwq_cons->hdr.conn_and_cmd_data)
//            >> SPE_HDR_CMD_ID_SHIFT) & 0xff;


      /*
      * There may be not more than 8 L2, not more than 8 L5 SPEs
      * and in the air. We also check that number of outstanding
      * COMMON ramrods is not more than the EQ and SPQ can
      * accommodate.
      */
      if (type == ETH_CONNECTION_TYPE) {
         if (!vmk_AtomicRead64(&adapter->cq_spq_left))
            break;
         else
            vmk_AtomicDec64(&adapter->cq_spq_left);
      } else if (type == NONE_CONNECTION_TYPE) {
         if (!vmk_AtomicRead64(&adapter->eq_spq_left))
            break;
         else
            vmk_AtomicDec64(&adapter->eq_spq_left);
      } else if ((type == ISCSI_CONNECTION_TYPE) || (type == FCOE_CONNECTION_TYPE)) {
         if (adapter->cnic_spq_pending >= cp->max_kwqe_pending)
            break;
         else
            adapter->cnic_spq_pending++;
      } else {
         QFLE3_ERR("Unknown SPE type: %d\n", type);
         vmk_PanicWithModuleID(vmk_ModuleCurrentID, "Unknown SPE type: %d\n", type);
         break;
      }

      spe = qfle3_sp_get_next(adapter);
      *spe = *adapter->cnic_kwq_cons;

      QFLE3_DBG(QFLE3_DBG_CNIC, "pending on SPQ %d, on KWQ %d count %d\n",
      adapter->cnic_spq_pending, adapter->cnic_kwq_pending, count);

      if (adapter->cnic_kwq_cons == adapter->cnic_kwq_last)
         adapter->cnic_kwq_cons = adapter->cnic_kwq;
      else
         adapter->cnic_kwq_cons++;
   }
   qfle3_sp_prod_update(adapter);
   vmk_SpinlockUnlock(adapter->spq_lock);
   return VMK_OK;
}

VMK_ReturnStatus qfle3_cnic_sp_queue(vmk_Device nicDev,
      struct kwqe_16 *kwqes[], vmk_uint32 count)
{
   qfle3_adapter *adapter;
   int i;
   vmk_AddrCookie driverData;
   struct cnic_eth_dev *cp;

   vmk_DeviceGetAttachedDriverData(nicDev, &driverData);
   adapter = (qfle3_adapter*) driverData.ptr;
   cp = (struct cnic_eth_dev *)adapter->cnicEthDev;
   QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_cnic_sp_queue called\n");
   if ((adapter->recovery_state != QFLE3_RECOVERY_DONE) &&
       (adapter->recovery_state != QFLE3_RECOVERY_NIC_LOADING)) {
      QFLE3_INFO("Handling parity error recovery. Try again later\n");
      return VMK_FAILURE;
   }
   if (adapter->cnic_kwq_pending >= MAX_SP_DESC_CNT - count) {
      QFLE3_INFO("CNIC KWQ full. Try again later\n");
      return VMK_FAILURE;
   }

   vmk_SpinlockLock(adapter->spq_lock);

   for (i = 0; i < count; i++) {
      struct eth_spe *spe = (struct eth_spe *)kwqes[i];

      *adapter->cnic_kwq_prod = *spe;

      adapter->cnic_kwq_pending++;

      QFLE3_DBG(QFLE3_DBG_CNIC, "L5 SPQE %x %x %x:%x pos %d\n",
         spe->hdr.conn_and_cmd_data, spe->hdr.type,
         spe->data.update_data_addr.hi,
         spe->data.update_data_addr.lo,
         adapter->cnic_kwq_pending);

      if (adapter->cnic_kwq_prod == adapter->cnic_kwq_last)
         adapter->cnic_kwq_prod = adapter->cnic_kwq;
      else
         adapter->cnic_kwq_prod++;
   }

   vmk_SpinlockUnlock(adapter->spq_lock);

   if (adapter->cnic_spq_pending < cp->max_kwqe_pending)
      qfle3_cnic_sp_post(adapter, 0);

   return i;
}
/**
 * qfle3_wait_sp_comp - wait for the outstanding SP commands.
 *
 * @bp:		driver handle
 * @mask:	bits that need to be cleared
 */
static inline vmk_Bool qfle3_wait_sp_comp(struct qfle3_adapter *adapter, unsigned long mask)
{
	int tout = 5000; /* Wait for 5 secs tops */

	while (tout--) {
//		smp_mb();
//		netif_addr_lock_bh(bp->dev);
		if (!(adapter->sp_state & mask)) {
//			netif_addr_unlock_bh(bp->dev);
			return VMK_TRUE;
		}
//		netif_addr_unlock_bh(bp->dev);

//		vmk_DelayUsecs(1000);
      vmk_WorldSleep(1000);
	}

//	smp_mb();

//	netif_addr_lock_bh(bp->dev);
	if (adapter->sp_state & mask) {
		QFLE3_ERR("Filtering completion timed out. sp_state 0x%lx, mask 0x%lx\n",
			  adapter->sp_state, mask);
//		netif_addr_unlock_bh(bp->dev);
		return VMK_FALSE;
	}
//	netif_addr_unlock_bh(bp->dev);

	return VMK_TRUE;
}

static int qfle3_set_iscsi_eth_mac_addr(struct qfle3_adapter *adapter)
{
	unsigned long ramrod_flags = 0;

	ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
	return qfle3_set_mac_one(adapter, adapter->iscsi_mac,
				 &adapter->iscsi_l2_mac_obj, VMK_TRUE,
				 ECORE_ISCSI_ETH_MAC, &ramrod_flags);
}

VMK_ReturnStatus qfle3_cnic_drv_ctl(vmk_Device nicDev, struct drv_ctl_info *ctl)
{
   qfle3_adapter *adapter;
   VMK_ReturnStatus rc = VMK_OK;
   vmk_AddrCookie driverData;

   vmk_DeviceGetAttachedDriverData(nicDev, &driverData);
   adapter = (qfle3_adapter*) driverData.ptr;
   QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_cnic_drv_ctl called cmd 0x%x \n", ctl->cmd);
   
   switch (ctl->cmd) {
   case DRV_CTL_CTXTBL_WR_CMD: {
      vmk_uint32 index = ctl->data.io.offset;
      vmk_IOA addr = ctl->data.io.dma_addr;

      qfle3_ilt_wr(adapter, index, addr);
      break;
   }

   case DRV_CTL_RET_L5_SPQ_CREDIT_CMD: {
      int count = ctl->data.credit.credit_count;

      qfle3_cnic_sp_post(adapter, count);
      break;
   }

   /* rtnl_lock is held.  */
   case DRV_CTL_START_L2_CMD: {
      struct cnic_eth_dev *cp = adapter->cnicEthDev;
      unsigned long sp_bits = 0;

      /* Configure the iSCSI classification object */
      ecore_init_mac_obj(adapter, &adapter->iscsi_l2_mac_obj,
      		   cp->iscsi_l2_client_id,
      		   cp->iscsi_l2_cid, QFLE3_FUNC(adapter),
      		   QFLE3_SP(adapter, mac_rdata),
      		   QFLE3_SP_MAPPING(adapter, mac_rdata),
      		   ECORE_FILTER_MAC_PENDING,
      		   &adapter->sp_state, ECORE_OBJ_TYPE_RX,
      		   &adapter->macs_pool);

   	/* Set iSCSI MAC address */
   	rc = qfle3_set_iscsi_eth_mac_addr(adapter);
   	if (rc)
   		break;

      vmk_CPUMemFenceReadWrite();

   	/* Start accepting on iSCSI L2 ring */

//   	netif_addr_lock_bh(dev);
   	qfle3_set_iscsi_eth_rx_mode(adapter, VMK_TRUE);
//   	netif_addr_unlock_bh(dev);

   	/* bits to wait on */
   	ECORE_SET_BIT(ECORE_FILTER_RX_MODE_PENDING, &sp_bits);
   	ECORE_SET_BIT(ECORE_FILTER_ISCSI_ETH_START_SCHED, &sp_bits);

   	if (!qfle3_wait_sp_comp(adapter, sp_bits))
   		QFLE3_ERR("rx_mode completion timed out!\n");

      break;
   }

   /* rtnl_lock is held.  */
   case DRV_CTL_STOP_L2_CMD: {
   	unsigned long sp_bits = 0;

   	/* Stop accepting on iSCSI L2 ring */
//   	netif_addr_lock_bh(dev);
   	qfle3_set_iscsi_eth_rx_mode(adapter, VMK_FALSE);
//   	netif_addr_unlock_bh(dev);

   	/* bits to wait on */
   	ECORE_SET_BIT(ECORE_FILTER_RX_MODE_PENDING, &sp_bits);
   	ECORE_SET_BIT(ECORE_FILTER_ISCSI_ETH_STOP_SCHED, &sp_bits);

   	if (!qfle3_wait_sp_comp(adapter, sp_bits))
   		QFLE3_ERR("rx_mode completion timed out!\n");

      vmk_CPUMemFenceReadWrite();

   	/* Unset iSCSI L2 MAC */
   	rc = qfle3_del_all_macs(adapter, &adapter->iscsi_l2_mac_obj,
   				ECORE_ISCSI_ETH_MAC, VMK_TRUE);
      break;
   }
   case DRV_CTL_GET_OOO_CQE:
      rc = qfle3_cnic_get_ooo_cqe(adapter, ctl->data.ooo_cqe.cqe);
      break;

   case DRV_CTL_SEND_OOO_PKT:
      rc = qfle3_cnic_send_ooo_pkt(adapter, ctl->data.pkt_desc.skb);
      break;

   case DRV_CTL_COMP_OOO_TX_PKTS:
      rc = qfle3_cnic_comp_ooo_tx_pkts(adapter);
      break;

   case DRV_CTL_REUSE_OOO_PKT:
      rc = qfle3_cnic_reuse_ooo_pkt(adapter, ctl->data.pkt_desc.skb);
      break;
   case DRV_CTL_RET_L2_SPQ_CREDIT_CMD: {
      int count = ctl->data.credit.credit_count;

      vmk_CPUMemFenceReadWrite();
      vmk_AtomicAdd64(&adapter->cq_spq_left,count);
      vmk_CPUMemFenceReadWrite();
      break;
   }
   //SV: Try enabling NPAR ctls
   case DRV_CTL_START_NPAR_CMD:
      if (IS_MF_SI(adapter) || IS_MF_AFEX(adapter)) {
         ecore_set_mac_in_nig(adapter, VMK_TRUE, adapter->iscsi_mac,
             ECORE_LLH_CAM_ISCSI_ETH_LINE);
      } else {
         QFLE3_ERR("Received START_NPAR_CMD in SD mode\n");
         rc = VMK_FAILURE;
      }
      break;

   case DRV_CTL_STOP_NPAR_CMD:
      if (IS_MF_SI(adapter) || IS_MF_AFEX(adapter)) {
         ecore_set_mac_in_nig(adapter, VMK_FALSE, adapter->iscsi_mac,
            ECORE_LLH_CAM_ISCSI_ETH_LINE);
      } else {
         QFLE3_ERR("Received STOP_NPAR_CMD in SD mode\n");
         rc = VMK_FAILURE;
      }
      break;

   case DRV_CTL_ULP_REGISTER_CMD: {
   	int ulp_type = ctl->data.register_data.ulp_type;

   	if (CHIP_IS_E3(adapter)) {
   		int idx = QFLE3_FW_MB_IDX(adapter);
   		vmk_uint32 cap = SHMEM2_RD(adapter, drv_capabilities_flag[idx]);
   		int path = QFLE3_PATH(adapter);
   		int port = QFLE3_PORT(adapter);
   		int i;
   		vmk_uint32 scratch_offset;
   		vmk_uint32 *host_addr;

   		/* first write capability to shmem2 */
   		if (ulp_type == CNIC_ULP_ISCSI)
   			cap |= DRV_FLAGS_CAPABILITIES_LOADED_ISCSI;
   		else if (ulp_type == CNIC_ULP_FCOE)
   			cap |= DRV_FLAGS_CAPABILITIES_LOADED_FCOE;
   		SHMEM2_WR(adapter, drv_capabilities_flag[idx], cap);

      	if ((ulp_type != CNIC_ULP_FCOE) ||
      	    (!SHMEM2_HAS(adapter, ncsi_oem_data_addr)) ||
      	    (!(adapter->flags &  BC_SUPPORTS_FCOE_FEATURES)))
      		break;

      	/* if reached here - should write fcoe capabilities */
      	scratch_offset = SHMEM2_RD(adapter, ncsi_oem_data_addr);
      	if (!scratch_offset)
      		break;
      	scratch_offset += vmk_offsetof(struct glob_ncsi_oem_data,
      				   fcoe_features[path][port]);
      	host_addr = (vmk_uint32 *) &(ctl->data.register_data.
      			      fcoe_features);
      	for (i = 0; i < sizeof(struct fcoe_capabilities);
      	     i += 4)
      		REG_WR(adapter, scratch_offset + i,
      		       *(host_addr + i/4));
      }
//   	qfle3_schedule_sp_rtnl(adapter, QFLE3_SP_RTNL_GET_DRV_VERSION, 0);
   	break;
   }

   case DRV_CTL_ULP_UNREGISTER_CMD: {
   	int ulp_type = ctl->data.ulp_type;

   	if (CHIP_IS_E3(adapter)) {
   		int idx = QFLE3_FW_MB_IDX(adapter);
   		vmk_uint32 cap;

   		cap = SHMEM2_RD(adapter, drv_capabilities_flag[idx]);
   		if (ulp_type == CNIC_ULP_ISCSI)
   			cap &= ~DRV_FLAGS_CAPABILITIES_LOADED_ISCSI;
   		else if (ulp_type == CNIC_ULP_FCOE)
   			cap &= ~DRV_FLAGS_CAPABILITIES_LOADED_FCOE;
   		SHMEM2_WR(adapter, drv_capabilities_flag[idx], cap);
   	}
//   	qfle3_schedule_sp_rtnl(bp, QFLE3_SP_RTNL_GET_DRV_VERSION, 0);
   	break;
   }

   default:
      QFLE3_ERR("unknown command %x\n", ctl->cmd);
      rc = VMK_FAILURE;
   }

   /* For storage-only interfaces, change driver state */
   if (IS_MF_SD_STORAGE_PERSONALITY_ONLY(adapter)) {
      switch (ctl->drv_state) {
      case DRV_NOP:
         break;
      case DRV_ACTIVE:
         qfle3_set_os_driver_state(adapter,
            OS_DRIVER_STATE_ACTIVE);
         break;
      case DRV_INACTIVE:
         qfle3_set_os_driver_state(adapter,
            OS_DRIVER_STATE_DISABLED);
         break;
      case DRV_UNLOADED:
         qfle3_set_os_driver_state(adapter,
            OS_DRIVER_STATE_NOT_LOADED);
         break;
      default:
         QFLE3_ERR("Unknown cnic driver state: %d\n", ctl->drv_state);
      }
   }

   return rc;
}

void qfle3_setup_cnic_irq_info(struct qfle3_adapter *adapter)
{
	struct cnic_eth_dev *cp = adapter->cnicEthDev;

	if (adapter->flags & USING_MSIX_FLAG) {
		cp->driverState |= CNIC_DRV_STATE_USING_MSIX;
		cp->irq_arr[0].irq_flags |= CNIC_IRQ_FL_MSIX;
		cp->irq_arr[0].cnic_intr_cookie = adapter->intr.cookies[1];
      
      cp->irq_arr[0].status_blk = adapter->cnic_sb.e2_sb;
      cp->irq_arr[0].status_block_ioa = adapter->cnic_sb_mapping;
      
      cp->irq_arr[0].status_blk_num = qfle3_cnic_fw_sb_id(adapter);
      cp->irq_arr[0].status_blk_num2 = qfle3_cnic_igu_sb_id(adapter);
	} else {
		cp->driverState &= ~CNIC_DRV_STATE_USING_MSIX;
		cp->irq_arr[0].irq_flags &= ~CNIC_IRQ_FL_MSIX;
	}
	cp->irq_arr[1].status_blk = adapter->def_status_blk;
	cp->irq_arr[1].status_blk_num = DEF_SB_ID;
	cp->irq_arr[1].status_blk_num2 = DEF_SB_IGU_ID;

}


void qfle3_register_cnic_irq_handlr(struct qfle3_adapter *adapter, void *cnicIRQHandler)
{
   adapter->cnicIRQHandler = cnicIRQHandler;
}
void qfle3_set_rx_mode_inner(struct qfle3_adapter *adapter)
{
	vmk_uint32 rx_mode = QFLE3_RX_MODE_NORMAL;

	QFLE3_DBG(QFLE3_DBG_CNIC, "dev->flags = %llx\n", (long long unsigned int)adapter->flags);

//	netif_addr_lock_bh(bp->dev);

//#if (VMWARE_ESX_DDK_VERSION == 55000) /* ! QFLE3_UPSTREAM */
//	/* IFF_PROMISC mode is not set in SR-IOV mode, so set here */
//	if (IS_SRIOV(adapter))
//		adapter->dev->flags |= IFF_PROMISC;
//#endif

//	if (bp->dev->flags & IFF_PROMISC) {
//	#if defined(__VMKLNX__) /* not QFLE3_UPSTREAM */
//		rx_mode = QFLE3_RX_MODE_ALLMULTI;
//	#else /* QFLE3_UPSTREAM */
//		rx_mode = QFLE3_RX_MODE_PROMISC;
//	#endif
//	} else if ((bp->dev->flags & IFF_ALLMULTI) ||
//		   ((netdev_mc_count(bp->dev) > QFLE3_MAX_MULTICAST) &&
//		    CHIP_IS_E1(bp))) {
//		rx_mode = QFLE3_RX_MODE_ALLMULTI;
//	} else {
//		if (IS_PF(bp)) {
//			/* some multicasts */
//			if (QFLE3_set_mc_list(bp) < 0)
//				rx_mode = QFLE3_RX_MODE_ALLMULTI;

//#if (LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 22)) /* QFLE3_UPSTREAM */
//			/* release bh lock, as qfle3_set_uc_list might sleep */
//			netif_addr_unlock_bh(bp->dev);
//			if (qfle3_set_uc_list(bp) < 0)
//				rx_mode = QFLE3_RX_MODE_PROMISC;
//			netif_addr_lock_bh(bp->dev);
//#endif
//		} else {
//			/* configuring mcast to a vf involves sleeping (when we
//			 * wait for the pf's response).
//			 */
//			qfle3_schedule_sp_rtnl(bp,
//					       QFLE3_SP_RTNL_VFPF_MCAST, 0);
//		}
//	}

	adapter->rx_mode = rx_mode;
//#ifndef __VMKLNX__ /* QFLE3_UPSTREAM */
//	/* handle ISCSI SD mode */
//	if (IS_MF_ISCSI_ONLY(bp))
//		bp->rx_mode = QFLE3_RX_MODE_NONE;
//#endif

//#ifdef QFLE3_ESX_CNA /* ! _UPSTREAM */
	if (!NO_FCOE(adapter) && (adapter->rx_mode != QFLE3_RX_MODE_NONE)) {
//		bool start = QFLE3_IS_NETQ_RX_QUEUE_ACTIVE(qfle3_fcoe_fp(bp));

      QFLE3_DBG(QFLE3_DBG_CNIC, "SV:Set rx-mode for fcoe rx. faking: queue-active\n");
		/*
		 * Set a 'scheduled' bit for an ETH rx_mode if we have sent a
		 * ramrod for an FCoE.
		 */
		if (qfle3_set_fcoe_eth_rx_mode(adapter, VMK_TRUE)) {
			ECORE_SET_BIT(ECORE_FILTER_RX_MODE_SCHED, &adapter->sp_state);
//			netif_addr_unlock_bh(bp->dev);
         QFLE3_DBG(QFLE3_DBG_CNIC, "SV:Set rx-mode: schedule.\n");
			return;
		}
      QFLE3_DBG(QFLE3_DBG_CNIC, "SV: Set rx-mode: Done\n");
	}
//#endif
	/* Schedule the rx_mode command */
	if (ECORE_TEST_BIT(ECORE_FILTER_RX_MODE_PENDING, &adapter->sp_state)) {
		ECORE_SET_BIT(ECORE_FILTER_RX_MODE_SCHED, &adapter->sp_state);
//		netif_addr_unlock_bh(adapter->dev);
		return;
	}

	qfle3_set_storm_rx_mode(adapter);
//	netif_addr_unlock_bh(adapter->dev);
}
int qfle3_cnic_notify(struct qfle3_adapter * adapter, int cmd)
{
	struct cnic_ctl_info ctl = {0};

	ctl.cmd = cmd;

	return qfle3_cnic_ctl_send(adapter, &ctl);
}

VMK_ReturnStatus qfle3_start_cnicqs(struct qfle3_adapter * adapter)
{
   int status;
   unsigned long ramrod_flags = 0;
   int rc;
   qfle3_intr *intr = &adapter->intr;
   struct cnic_ctl_info ctl;
   vmk_IntrCookie cookie;
   struct ecore_vlan_mac_obj *obj;
   QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_start_cnicqs\n");
   struct qfle3_fastpath *fpq;

   /* Enable Timer scan */
   REG_WR(adapter, TM_REG_EN_LINEAR0_TIMER + QFLE3_PORT(adapter) * 4, 1);
   // register cnic interrupt

   
      
   if (!NO_ISCSI_OOO(adapter)) {
      fpq = qfle3_ooo_fp(adapter);
      fpq->qid = OOO_IDX(adapter);
      vmk_NameFormat(&fpq->name, "%s-fp-ooo",
           QFLE3_NAME_TO_STRING(adapter->pdev_name));
        if (!(adapter->flags & OWN_CNIC_IRQ)) { // we don't have a interrupt handler registered
           status = qfle3_register_interrupt(adapter, (1),
                    fpq->name, fpq,
                    qfle3_interrupt_ack, qfle3_msix_rx);
           if (status != VMK_OK) {
              QFLE3_ERR("Failed to initiate intrCookie #%d 0x%x (%x)",1,
                 intr->cookies[1], status);
              
            }
           adapter->flags |= OWN_CNIC_IRQ;

           QFLE3_DBG(QFLE3_DBG_CNIC, "initiate intrCookie #%d 0x%x (%x) for CNIC qfle3_msix_rx", 1,
                intr->cookies[1], status);
        }

      fpq = qfle3_ooo_fp(adapter);
      if (fpq->netpoll) {
         if (intr->intrType != QFLE3_IT_MSIX)
            cookie = intr->cookies[0];
         else
            cookie = intr->cookies[1];
         status = vmk_NetPollInterruptSet(fpq->netpoll, cookie);
         if (status != VMK_OK) {
            QFLE3_ERR("Failed to set associated interrupt cookie #%d 0x%x"               
               "with fp[%d] netpoll", 1, cookie, fpq->qid);         
         }

         QFLE3_DBG(QFLE3_DBG_CNIC, "CNIC poll interrupt set");
      }

      fpq = qfle3_ooo_fp(adapter);
      if (fpq->netpoll && fpq->pollDisabled) {
         vmk_NetPollEnable(fpq->netpoll);
         netpoll_count++;
         fpq->pollDisabled = VMK_FALSE;
         QFLE3_DBG(QFLE3_DBG_CNIC, "enabled netpoll for ooo fp, count %d\n", netpoll_count);
      }
   
      cookie = intr->cookies[1];
      status = vmk_IntrEnable(cookie);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to enable intrCookie[%d] 0x%x (%x)", 1, cookie, status);
      }
      
      QFLE3_DBG(QFLE3_DBG_CNIC, "Interrupt enabled on CNIC");

      
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_start_cnicqs FWD setup started \n");
      fpq = qfle3_fwd_fp(adapter);
      status = qfle3_setup_queue(adapter,fpq,0);
      
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_start_cnicqs FWD setup completed with status %d\n", status);
      
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_start_cnicqs OOO setup started \n");

      fpq = qfle3_ooo_fp(adapter);
      status = qfle3_setup_queue(adapter,fpq,0);
      
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_start_cnicqs OOO setup completed with status %d\n", status);
      
   }
   if (!NO_FCOE(adapter)) {
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_start_cnicqs FCOE setup started \n");
      fpq = qfle3_fcoe_fp(adapter);
      status = qfle3_setup_queue(adapter,fpq,0);
      fpq->fp_state = QFLE3_SM_Q_STARTED;
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_start_cnicqs FCOE setup completed with status %d\n", status);
   }
   /* Initialize Rx filter. */
   qfle3_set_rx_mode_inner(adapter);
   // apply fcoe mac filter
   if (!NO_FCOE(adapter)) {
      fpq = qfle3_fcoe_fp(adapter);
      
         /* add MAC */
         ramrod_flags = 0;
         ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
         obj = &QFLE3_SP_OBJ(adapter, fpq).mac_obj;
      
         rc = qfle3_set_mac_one(adapter, adapter->fip_mac, obj, VMK_TRUE, ECORE_NETQ_ETH_MAC, &ramrod_flags);
         if (rc)
            QFLE3_ERR("qfle3_start_cnicqs applying FCOE FIP MAC filter failed %d\n", rc);
         else
            QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_start_cnicqs applying FCOE FIP MAC filter success\n");
   }
   
   qfle3_get_iscsi_info(adapter);
   qfle3_setup_cnic_irq_info(adapter);
   qfle3_get_fcoe_info(adapter);
   /* send CNIC_CTL_START_CMD */
   // send stop command to cnic
   ctl.cmd = CNIC_CTL_START_CMD;

   if (adapter->cnicOps) {
      if (!NO_ISCSI_OOO(adapter)) {
        
        QFLE3_ERR("Releasing CNIC IRQ to qcnic");
         vmk_IntrDisable(adapter->intr.cookies[1]);
         qfle3_unregister_interrupt(adapter, 1);
      
         adapter->flags &= ~OWN_CNIC_IRQ;
      }
      adapter->cnicOps->cnicNotify(adapter->cnic_data, &ctl);
   }
      return VMK_OK;
}
/**
 * qfle3_update_ooo_prod - updates rx OOO producers
 *
 * @bp:			driver handle
 * @fp:			poinet to the fastpath
 * @bd_prod		current BD producer
 * @rx_comp_prod	current RX completion producer
 * @rx_sge_prod		cuurent RX SGE producer
 *
 * updates rx producers according to
 * iSCSI OOO ETH ring spec.
 */
static inline void qfle3_update_ooo_prod(struct qfle3_adapter * adapter,
			struct qfle3_fastpath *fp, vmk_uint16 bd_prod,
			vmk_uint16 rx_comp_prod, vmk_uint16 rx_sge_prod)
{
	struct ustorm_eth_rx_producers rx_prods = {0};
	vmk_uint32 i;
	vmk_uint32 start = fp->ustorm_rx_prods_offset;

	/* Update producers */
	rx_prods.bd_prod = bd_prod + 0x4000;
	rx_prods.cqe_prod = rx_comp_prod + 0x4000;
	rx_prods.sge_prod = rx_sge_prod;

	/*
	 * Make sure that the BD and SGE data is updated before updating the
	 * producers since FW might read the BD/SGE right after the producer
	 * is updated.
	 * This is only applicable for weak-ordered memory model archs such
	 * as IA-64. The following barrier is also mandatory since FW will
	 * assumes BDs must have buffers.
	 */
   vmk_CPUMemFenceReadWrite();

	for (i = 0; i < sizeof(rx_prods)/4; i++)
		REG_WR(adapter, start + i*4, ((vmk_uint32 *)&rx_prods)[i]);

   vmk_CPUMemFenceReadWrite();

//	REG_WR16(adapter, BAR_TSTRORM_INTMEM +
//		 TSTORM_ISCSI_L2_ISCSI_OOO_PROD_OFFSET(QFLE3_FUNC(adapter)),
//		 fp->rx_pkts_avail);
    REG_WR16(adapter, BAR_TSTRORM_INTMEM +
        TSTORM_ISCSI_L2_ISCSI_OOO_PROD_OFFSET(QFLE3_FUNC(adapter)),
        adapter->rx_ring_size);

   vmk_CPUMemFenceReadWrite();

	QFLE3_DBG(QFLE3_DBG_CNIC, 
	   "queue[%d]:  wrote  bd_prod %u  cqe_prod %u  sge_prod %u\n",
	   fp->qid, bd_prod, rx_comp_prod, rx_sge_prod);
}

void qfle3_init_rx_rings_cnic(struct qfle3_adapter * adapter)
{
   vmk_uint32 qid ;
   int i;
   
   struct qfle3_fastpath *fp;

   FOREACH_RX_CNIC_QUEUE(i,qid,fp) 
      
   	QFLE3_DBG(QFLE3_DBG_CNIC, 
   	   "foreach rx cnic queue [%d]\n",qid);
//      fp = qfle3_ooo_fp(adapter);
		fp->rx_bd_cons = 0;

		/* Activate BD ring */
		/* Warning!
		 * this will generate an interrupt (to the TSTORM)
		 * must only be done after chip is initialized
		 */
		 
		if (IS_OOO_FP(fp))
			qfle3_update_ooo_prod(adapter, fp, fp->rx_bd_prod,
					      fp->rx_cq_prod,
					      fp->rx_sge_prod);
		else
		qfle3_update_rx_prod(adapter, fp, fp->rx_bd_prod, fp->rx_cq_prod,
				     fp->rx_sge_prod);
	}
}
static void qfle3_init_tx_rings_cnic(struct qfle3_adapter * adapter)
{
   vmk_uint32 i, qid ;
   struct qfle3_fastpath *fpq;
   FOREACH_TX_CNIC_QUEUE(i,qid,fpq) 
         
      QFLE3_DBG(QFLE3_DBG_CNIC, 
         "foreach tx cnic queue [%d]\n",qid);
		qfle3_tq_init(&adapter->fp[qid]);
   }
//   fpq = qfle3_fwd_fp(adapter);
//		qfle3_tq_init(fpq);


}
VMK_ReturnStatus qfle3_create_cnicqs(struct qfle3_adapter * adapter)
{
   struct qfle3_fastpath *fp;
   QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_create_cnicqs\n");
//   QFLE3_SMCMD_STATUS status = QFLE3_SMCMD_STATUS_COMPLETED;
   if (!NO_FCOE(adapter)) {
      
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_create_cnicqs fcoe rx \n");
      fp = qfle3_fcoe_fp(adapter);
      qfle3_alloc_fp_buffers(fp);
      qfle3_init_fcoe_fp(adapter);
      vmk_NetPollEnable(fp->netpoll);
      netpoll_count++;
      fp->pollDisabled = VMK_FALSE;
      fp->fp_state = QFLE3_SM_Q_CREATED;
      
   }
   if (!NO_ISCSI_OOO(adapter)) {
      
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_create_cnicqs fwd \n");
      fp = qfle3_fwd_fp(adapter);
//      qfle3_alloc_fp_buffers(fp);
      qfle3_init_fwd_fp(adapter);
//      vmk_NetPollEnable (fp->netpoll);
      fp->pollDisabled = VMK_FALSE;
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_create_cnicqs ooo \n");
      fp = qfle3_ooo_fp(adapter);
      qfle3_alloc_fp_buffers(fp);
//      qfle3_alloc_ooo_rx_bd_ring(fp);
      qfle3_init_ooo_fp(adapter);
      
//      vmk_NetPollEnable(fp->netpoll);
//      fp->pollDisabled = VMK_FALSE;
      
   }
   qfle3_init_sb(adapter, adapter->cnic_sb_mapping, 0xFF, VMK_FALSE,
   qfle3_cnic_fw_sb_id(adapter), qfle3_cnic_igu_sb_id(adapter));
   
   vmk_CPUMemFenceReadWrite();
	qfle3_init_rx_rings_cnic(adapter);
	qfle3_init_tx_rings_cnic(adapter);
   vmk_CPUMemFenceReadWrite();
   return VMK_OK;
}


VMK_ReturnStatus qfle3_stop_cnicqs(struct qfle3_adapter * adapter)
{
   unsigned long ramrod_flags = 0;
   struct qfle3_fastpath *fp;
   struct cnic_ctl_info ctl;
   struct ecore_vlan_mac_obj *obj;
   int rc;
   vmk_IntrCookie cookie;
   qfle3_intr *intr = &adapter->intr;
   struct ecore_queue_state_params qstate = {NULL};
	unsigned long sp_bits = 0;
   VMK_ReturnStatus status = VMK_FAILURE;
   
   // send stop command to cnic
   ctl.cmd = CNIC_CTL_STOP_CMD;

   if (adapter->cnicOps)
      adapter->cnicOps->cnicNotify(adapter->cnic_data, &ctl);

   
   if (!NO_ISCSI_OOO(adapter)) {
      
      fp = qfle3_ooo_fp(adapter);
      if (!(adapter->flags & OWN_CNIC_IRQ)) { // we don't have a interrupt handler registered
      
      QFLE3_ERR(" OWN_CNIC_IRQ flag is not set, we  have to register an interrupt vector\n");
         status = qfle3_register_interrupt(adapter, (1),
                  fp->name, fp,
                  qfle3_interrupt_ack, qfle3_msix_rx);
         if (status != VMK_OK) {
            QFLE3_ERR("Failed to initiate intrCookie #%d 0x%x (%x)",1,
               intr->cookies[1], status);
            
          }
         
         cookie = intr->cookies[1];
         status = vmk_IntrEnable(cookie);
         if (status != VMK_OK) {
            QFLE3_ERR("Failed to enable intrCookie[%d] 0x%x (%x)", 1, cookie, status);
         }
         adapter->flags |= OWN_CNIC_IRQ;
      }else{
         QFLE3_ERR(" OWN_CNIC_IRQ flag is set, we already have an interrupt vector\n");
      }
   }
   // remove FCOE mac filter
   if (!NO_FCOE(adapter)) {
      
      fp = qfle3_fcoe_fp(adapter);
      /* remove  MAC */
      ramrod_flags = 0;
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
      if (adapter->recovery_state != QFLE3_RECOVERY_DONE){
         ECORE_SET_BIT(RAMROD_DRV_CLR_ONLY, &ramrod_flags);
         ECORE_SET_BIT(RAMROD_DRV_CLR_ONLY, &qstate.ramrod_flags);
      }
      obj = &QFLE3_SP_OBJ(adapter, fp).mac_obj;
   
      rc = qfle3_set_mac_one(adapter, adapter->fip_mac, obj, VMK_FALSE, ECORE_NETQ_ETH_MAC, &ramrod_flags);
      if (VMK_UNLIKELY(rc)) {
         QFLE3_ERR("FCOE could not remove mac filter");
         status =  VMK_FAILURE;
      }
      // stop receiving 
      qfle3_set_fcoe_eth_rx_mode(adapter, VMK_FALSE);

      /* bits to wait on */
      ECORE_SET_BIT(ECORE_FILTER_RX_MODE_PENDING, &sp_bits);
      ECORE_SET_BIT(ECORE_FILTER_FCOE_ETH_STOP_SCHED, &sp_bits);
      QFLE3_DBG(QFLE3_DBG_CNIC, "rx_mode wait for completion state, 0x%lu!, sp_bits 0x%lu\n", adapter->sp_state, sp_bits);
      if (adapter->recovery_state == QFLE3_RECOVERY_DONE) {
         if (!qfle3_wait_sp_comp(adapter, sp_bits)) {
            QFLE3_ERR("rx_mode completion timed out! 0x%lu!\n", adapter->sp_state);
            status =  VMK_FAILURE;
         } else {
         		/* send empty-ramrod to flush packets lurking in the HW */
         		qstate.q_obj = &QFLE3_SP_OBJ(adapter, fp).q_obj;
         		qstate.cmd = ECORE_Q_CMD_EMPTY;
         		/* wait for completion */
         		ECORE_SET_BIT(RAMROD_COMP_WAIT, &qstate.ramrod_flags);
         		if (ecore_queue_state_change(adapter, &qstate)) {
                  
                  QFLE3_ERR("Error Flushing FCoE queue with Empy Ramrod\n");
                  status =  VMK_FAILURE;
         		}
         }
         QFLE3_DBG(QFLE3_DBG_CNIC, "Waiting for FCoE queue Tx complete\n");
         qfle3_clean_tx_queue(adapter,fp->txdata_ptr[0]);
         if  (qfle3_has_tx_work_unload(fp->txdata_ptr[0])) {
            status = VMK_FAILURE;
            QFLE3_ERR("FCoE queue not drained\n");
         }
         
         qfle3_stop_queue(adapter, fp->qid);
      }
      adapter->fcoe_init = VMK_FALSE;
      fp->fp_state = QFLE3_SM_Q_CREATED;
   }
   if (!NO_ISCSI_OOO(adapter)) {

      adapter->ooo_init = VMK_FALSE;
      fp = qfle3_ooo_fp(adapter);
      if (adapter->recovery_state == QFLE3_RECOVERY_DONE){
         qfle3_stop_queue(adapter, fp->qid);
      }
      vmk_NetPollInterruptUnSet(fp->netpoll);
      vmk_NetPollDisable(fp->netpoll);
      vmk_NetPollFlushRx(fp->netpoll);
      netpoll_count--;
      fp->pollDisabled = VMK_FALSE;
      QFLE3_DBG(QFLE3_DBG_CNIC, "disabled netpoll for ooo fp count %d\n", netpoll_count);
      
      fp = qfle3_fwd_fp(adapter);
      
      if (adapter->recovery_state == QFLE3_RECOVERY_DONE){
         qfle3_stop_queue(adapter, fp->qid);
      }

      // if we are in RESET mode, then keep the interrupt handler, 
      // otherwise, it means that we are going down, release the interrupt handler
         
      QFLE3_ERR( "Release CNIC IRQ because we are going down");
      vmk_IntrDisable(adapter->intr.cookies[1]);
      qfle3_unregister_interrupt(adapter, 1);
      adapter->flags &= ~OWN_CNIC_IRQ;
      
   }
   /* Disable Timer scan */
   REG_WR(adapter, TM_REG_EN_LINEAR0_TIMER + QFLE3_PORT(adapter)*4, 0);
   
   return status;
}
VMK_ReturnStatus qfle3_remove_cnicqs(struct qfle3_adapter * adapter)
{
   struct qfle3_fastpath *fp;
   
   struct ecore_vlan_mac_obj *mac_vlan_obj;
   if (!NO_FCOE(adapter)) {
      
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_remove_cnicqs fcoe rx \n");
      fp = qfle3_fcoe_fp(adapter);
      vmk_NetPollDisable(fp->netpoll);
      
      vmk_NetPollFlushRx(fp->netpoll);
      netpoll_count--;
      
      QFLE3_DBG(QFLE3_DBG_CNIC, "disable network for FCOE count %d \n", netpoll_count);
      fp->pollDisabled = VMK_TRUE;
      qfle3_free_rx_bd_chain(fp);
      qfle3_free_tpa_pool(fp);
      qfle3_free_sge_chain(fp);
      mac_vlan_obj = &QFLE3_SP_OBJ(adapter, fp).mac_obj;
      qfle3_destroy_spinlock(mac_vlan_obj->exe_queue.lock);
      mac_vlan_obj->exe_queue.lock = NULL;
      fp->fp_state = 0;
      fp->tpa_enable = 0;
      
   }
   if (!NO_ISCSI_OOO(adapter)) {
      
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_remove_cnicqs fwd \n");
      fp = qfle3_fwd_fp(adapter);
//      vmk_NetPollDisable(fp->netpoll);
      fp->pollDisabled = VMK_TRUE;
      fp = qfle3_ooo_fp(adapter);
      QFLE3_DBG(QFLE3_DBG_CNIC, "qfle3_remove_cnicqs ooo \n");
//      vmk_NetPollDisable(fp->netpoll);
//      fp->pollDisabled = VMK_TRUE;
      mac_vlan_obj = &QFLE3_SP_OBJ(adapter, fp).mac_obj;
      qfle3_destroy_spinlock(mac_vlan_obj->exe_queue.lock);
      mac_vlan_obj->exe_queue.lock = NULL;
      mac_vlan_obj = &QFLE3_SP_OBJ(adapter, fp).vlan_obj;
      qfle3_destroy_spinlock(mac_vlan_obj->exe_queue.lock);
      mac_vlan_obj->exe_queue.lock = NULL;
      if (adapter->vxlan_filters_en) {
         mac_vlan_obj = &QFLE3_SP_OBJ(adapter, fp).vxlan_filter_obj;
         qfle3_destroy_spinlock(mac_vlan_obj->exe_queue.lock);
         mac_vlan_obj->exe_queue.lock = NULL;
      }
      qfle3_free_rx_bd_chain(fp);
      qfle3_free_tpa_pool(fp);
      qfle3_free_sge_chain(fp);
      
//      fp->state = 0;
      fp->tpa_enable = 0;
   }
   return VMK_OK;
}

static void qfle3_get_iscsi_info(struct qfle3_adapter *adapter)
{
	vmk_uint32 no_flags = NO_ISCSI_FLAG | NO_ISCSI_OOO_FLAG;
	int port = QFLE3_PORT(adapter);
	vmk_uint32 max_iscsi_conn = FW_ENCODE_32BIT_PATTERN ^ SHMEM_RD(adapter,
				drv_lic_key[port].max_iscsi_conn);

	if (!CNIC_SUPPORT(adapter)) {
		adapter->flags |= no_flags;
		return;
	}

	/* Get the number of maximum allowed iSCSI connections */
	adapter->cnicEthDev->max_iscsi_conn =
		(max_iscsi_conn & QFLE3_MAX_ISCSI_INIT_CONN_MASK) >>
		QFLE3_MAX_ISCSI_INIT_CONN_SHIFT;

	/*
	 * If maximum allowed number of connections is zero -
	 * disable the feature.
	 */
//TODO: work around for no storage configuration in the flash

	if (!adapter->cnicEthDev->max_iscsi_conn)
		adapter->flags |= no_flags;
	if (adapter->cnicEthDev->max_iscsi_conn > CNIC_ISCSI_CID_MAX){
		adapter->cnicEthDev->max_iscsi_conn = CNIC_ISCSI_CID_MAX;
	}

	QFLE3_DBG(QFLE3_DBG_CNIC, "max_iscsi_conn 0x%x\n",
		       adapter->cnicEthDev->max_iscsi_conn);
//#ifdef BCM_OOO /* ! QFLE3_UPSTREAM */
//	if (disable_iscsi_ooo)
//		adapter->flags |= NO_ISCSI_OOO_FLAG;
//#endif
}

static void qfle3_get_ext_wwn_info(struct qfle3_adapter *adapter, int func)
{
	/* Port info */
	adapter->cnicEthDev->fcoe_wwn_port_name_hi =
		MF_CFG_RD(adapter, func_ext_config[func].fcoe_wwn_port_name_upper);
	adapter->cnicEthDev->fcoe_wwn_port_name_lo =
		MF_CFG_RD(adapter, func_ext_config[func].fcoe_wwn_port_name_lower);

	/* Node info */
	adapter->cnicEthDev->fcoe_wwn_node_name_hi =
		MF_CFG_RD(adapter, func_ext_config[func].fcoe_wwn_node_name_upper);
	adapter->cnicEthDev->fcoe_wwn_node_name_lo =
		MF_CFG_RD(adapter, func_ext_config[func].fcoe_wwn_node_name_lower);
}

static int qfle3_shared_fcoe_funcs(struct qfle3_adapter *adapter)
{
	vmk_uint8 count = 0;

	if (IS_MF(adapter)) {
		vmk_uint8 fid;

		/* iterate over absolute function ids for this path: */
		for (fid = QFLE3_PATH(adapter); fid < E2_FUNC_MAX * 2; fid += 2) {
			if (IS_MF_SD(adapter)) {
				vmk_uint32 cfg = MF_CFG_RD(adapter,
						    func_mf_config[fid].config);

				if (!(cfg & FUNC_MF_CFG_FUNC_HIDE) &&
				    ((cfg & FUNC_MF_CFG_PROTOCOL_MASK) ==
					    FUNC_MF_CFG_PROTOCOL_FCOE))
					count++;
			} else {
				vmk_uint32 cfg = MF_CFG_RD(adapter,
						    func_ext_config[fid].
						                     func_cfg);

				if ((cfg & MACP_FUNC_CFG_FLAGS_ENABLED) &&
				    (cfg & MACP_FUNC_CFG_FLAGS_FCOE_OFFLOAD))
					count++;
			}
		}
	} else { /* SF */
		int port, port_cnt = CHIP_MODE_IS_4_PORT(adapter) ? 2 : 1;

		for (port = 0; port < port_cnt; port++) {
			vmk_uint32 lic = SHMEM_RD(adapter,
					   drv_lic_key[port].max_fcoe_conn) ^
				  FW_ENCODE_32BIT_PATTERN;
			if (lic)
				count++;
		}
	}

	return count;
}

static void qfle3_get_fcoe_info(struct qfle3_adapter *adapter)
{
	int port = QFLE3_PORT(adapter);
	int func = QFLE3_ABS_FUNC(adapter);
	vmk_uint32 max_fcoe_conn = FW_ENCODE_32BIT_PATTERN ^ SHMEM_RD(adapter,
				drv_lic_key[port].max_fcoe_conn);
	vmk_uint8 num_fcoe_func = qfle3_shared_fcoe_funcs(adapter);

	if (!CNIC_SUPPORT(adapter)) {
		adapter->flags |= NO_FCOE_FLAG;
		return;
	}

	/* Get the number of maximum allowed FCoE connections */
	adapter->cnicEthDev->max_fcoe_conn =
		(max_fcoe_conn & QFLE3_MAX_FCOE_INIT_CONN_MASK) >>
		QFLE3_MAX_FCOE_INIT_CONN_SHIFT;

	/* Calculate the number of maximum allowed FCoE tasks */
	adapter->cnicEthDev->max_fcoe_exchanges = MAX_NUM_FCOE_TASKS_PER_ENGINE;

	/* check if FCoE resources must be shared between different functions */
	if (num_fcoe_func)
		adapter->cnicEthDev->max_fcoe_exchanges /= num_fcoe_func;

	/* Read the WWN: */
	if (!IS_MF(adapter)) { /* SF */
		/* Port info */
		adapter->cnicEthDev->fcoe_wwn_port_name_hi =
			SHMEM_RD(adapter,
				 dev_info.port_hw_config[port].
				 fcoe_wwn_port_name_upper);
		adapter->cnicEthDev->fcoe_wwn_port_name_lo =
			SHMEM_RD(adapter,
				 dev_info.port_hw_config[port].
				 fcoe_wwn_port_name_lower);

		/* Node info */
		adapter->cnicEthDev->fcoe_wwn_node_name_hi =
			SHMEM_RD(adapter,
				 dev_info.port_hw_config[port].
				 fcoe_wwn_node_name_upper);
		adapter->cnicEthDev->fcoe_wwn_node_name_lo =
			SHMEM_RD(adapter,
				 dev_info.port_hw_config[port].
				 fcoe_wwn_node_name_lower);
	} else if (!IS_MF_SD(adapter)) { /* SI + AFEX */
		if (QFLE3_HAS_MF_EXT_PROTOCOL_FCOE(adapter))
			qfle3_get_ext_wwn_info(adapter, func);
	} else /* SD */ {
		if (QFLE3_IS_MF_SD_PROTOCOL_FCOE(adapter) && !CHIP_IS_E1x(adapter))
			qfle3_get_ext_wwn_info(adapter, func);
	}
   if (adapter->cnicEthDev->max_fcoe_conn > CNIC_FCOE_CID_MAX){
      adapter->cnicEthDev->max_fcoe_conn = CNIC_FCOE_CID_MAX;
   }
	QFLE3_DBG(QFLE3_DBG_CNIC, "max_fcoe_conn 0x%x\n", adapter->cnicEthDev->max_fcoe_conn);

	/*
	 * If maximum allowed number of connections is zero -
	 * disable the feature.
	 */
      if (!adapter->cnicEthDev->max_fcoe_conn)
         adapter->flags |= NO_FCOE_FLAG;
}
static inline int is_invalid_ether_addr(vmk_uint8 *addr)
{
    return((addr[0] & 0x1) || !(addr[0]|addr[1]|addr[2]|addr[3]|addr[4]|addr[5]));
        
}

void qfle3_get_cnic_info(struct qfle3_adapter *adapter)
{
	/*
	 * iSCSI may be dynamically disabled but reading
	 * info here we will decrease memory usage by driver
	 * if the feature is disabled for good
	 */
	qfle3_get_iscsi_info(adapter);
	qfle3_get_fcoe_info(adapter);
}

void  qfle3_get_cnic_mac_hwinfo(struct qfle3_adapter *adapter)
{
	vmk_uint32 val, val2;
	int func = QFLE3_ABS_FUNC(adapter);
	int port = QFLE3_PORT(adapter);
   
//   struct cnic_eth_dev *cp = (struct cnic_eth_dev *)adapter->cnicEthDev;
	vmk_uint8 *iscsi_mac = adapter->iscsi_mac;
	vmk_uint8 *fip_mac = adapter->fip_mac;

	if (IS_MF(adapter)) {
		/* iSCSI and FCoE NPAR MACs: if there is no either iSCSI or
		 * FCoE MAC then the appropriate feature should be disabled.
		 * In non SD mode features configuration comes from struct
		 * func_ext_config.
		 */
		if (!IS_MF_SD(adapter)) {
			vmk_uint32 cfg = MF_CFG_RD(adapter, func_ext_config[func].func_cfg);
			if (cfg & MACP_FUNC_CFG_FLAGS_ISCSI_OFFLOAD) {
				val2 = MF_CFG_RD(adapter, func_ext_config[func].
						 iscsi_mac_addr_upper);
				val = MF_CFG_RD(adapter, func_ext_config[func].
						iscsi_mac_addr_lower);
				qfle3_set_mac_buf(iscsi_mac, val, val2);
				QFLE3_DBG(QFLE3_DBG_CNIC, "Read iSCSI MAC: %02x:%02x:%02x:%02x:%02x:%02x\n", 
					(iscsi_mac)[0], (iscsi_mac)[1], (iscsi_mac)[2], (iscsi_mac)[3], \
				(iscsi_mac)[4], (iscsi_mac)[5]);
			} else {
				adapter->flags |= NO_ISCSI_OOO_FLAG | NO_ISCSI_FLAG;
				QFLE3_DBG(QFLE3_DBG_CNIC, "ISCSI not enabled on Function %d\n", func);
			}

			if (cfg & MACP_FUNC_CFG_FLAGS_FCOE_OFFLOAD) {
				val2 = MF_CFG_RD(adapter, func_ext_config[func].
						 fcoe_mac_addr_upper);
				val = MF_CFG_RD(adapter, func_ext_config[func].
						fcoe_mac_addr_lower);
				qfle3_set_mac_buf(fip_mac, val, val2);
				QFLE3_DBG(QFLE3_DBG_CNIC, "Read FCoE L2 MAC: %02x:%02x:%02x:%02x:%02x:%02x\n", 
					(fip_mac)[0], (fip_mac)[1], (fip_mac)[2], (fip_mac)[3], \
				(fip_mac)[4], (fip_mac)[5]);
			} else {
				adapter->flags |= NO_FCOE_FLAG;
				QFLE3_DBG(QFLE3_DBG_CNIC, "FCOE not enabled on Function %d\n", func);
			}

			adapter->mf_ext_config = cfg;

		} else { /* SD MODE */
			if (QFLE3_IS_MF_SD_PROTOCOL_ISCSI(adapter)) {
				/* use primary mac as iscsi mac */
				vmk_Memcpy(iscsi_mac, adapter->hwMacAddr, ETH_ALEN);

				QFLE3_DBG(QFLE3_DBG_CNIC, "SD ISCSI MODE\n");
				QFLE3_DBG(QFLE3_DBG_CNIC, "Read iSCSI MAC: %02x:%02x:%02x:%02x:%02x:%02x\n", 
                  (iscsi_mac)[0], (iscsi_mac)[1], (iscsi_mac)[2], (iscsi_mac)[3], \
               (iscsi_mac)[4], (iscsi_mac)[5]);
			} else if (QFLE3_IS_MF_SD_PROTOCOL_FCOE(adapter)) {
				/* use primary mac as fip mac */
				vmk_Memcpy(fip_mac, adapter->hwMacAddr, ETH_ALEN);
				QFLE3_DBG(QFLE3_DBG_CNIC, "SD FCoE MODE\n");
				QFLE3_DBG(QFLE3_DBG_CNIC, "Read FIP MAC: %02x:%02x:%02x:%02x:%02x:%02x\n", 
                  (fip_mac)[0], (fip_mac)[1], (fip_mac)[2], (fip_mac)[3], \
               (fip_mac)[4], (fip_mac)[5]);
			}
		}

		/* If this is a storage-only interface, use SAN mac as
		 * primary MAC. Notice that for SD this is already the case,
		 * as the SAN mac was copied from the primary MAC.
		 */
		if (IS_MF_FCOE_AFEX(adapter))
			vmk_Memcpy(adapter->hwMacAddr, fip_mac, ETH_ALEN);
	} else {
		val2 = SHMEM_RD(adapter, dev_info.port_hw_config[port].
				iscsi_mac_upper);
		val = SHMEM_RD(adapter, dev_info.port_hw_config[port].
			       iscsi_mac_lower);
		qfle3_set_mac_buf(iscsi_mac, val, val2);
                QFLE3_DBG(QFLE3_DBG_CNIC, "SF-mode: Read iSCSI MAC: %02x:%02x:%02x:%02x:%02x:%02x\n",
                         (iscsi_mac)[0], (iscsi_mac)[1], (iscsi_mac)[2], (iscsi_mac)[3], 
                         (iscsi_mac)[4], (iscsi_mac)[5]);
      
		val2 = SHMEM_RD(adapter, dev_info.port_hw_config[port].
				fcoe_fip_mac_upper);
		val = SHMEM_RD(adapter, dev_info.port_hw_config[port].
			       fcoe_fip_mac_lower);
		qfle3_set_mac_buf(fip_mac, val, val2);
                QFLE3_DBG(QFLE3_DBG_CNIC, "SF-mode: Read FIP MAC: %02x:%02x:%02x:%02x:%02x:%02x\n",
                         (fip_mac)[0], (fip_mac)[1], (fip_mac)[2], (fip_mac)[3], 
                         (fip_mac)[4], (fip_mac)[5]);
	}

	/* Disable iSCSI OOO if MAC configuration is invalid. */
	if (is_invalid_ether_addr(iscsi_mac)) {
		adapter->flags |= NO_ISCSI_OOO_FLAG | NO_ISCSI_FLAG;
		vmk_Memset(iscsi_mac, 0, 6);
	}

	/* Disable FCoE if MAC configuration is invalid. */
	if (is_invalid_ether_addr(fip_mac)) {
		adapter->flags |= NO_FCOE_FLAG;
		vmk_Memset(fip_mac, 0, 6);
	}
}
VMK_ReturnStatus qfle3_alloc_ooo_rx_bd_ring(struct qfle3_fastpath * fp)
{
   VMK_ReturnStatus status = VMK_FAILURE;
	struct qfle3_adapter *adapter = fp->adapter;
	int i;
//   int rx_ring_size;

	/* - OOO BD ring size should be less or equal to half FWD
	 *   Tx ring size.
	 * - OOO BD ring size must be less than than CQE ring size
	 *   minus maximum number of outstanding ramrods.
	 */
//#if 1
//	/* Delete me! For integration only! */
//	rx_ring_size = min_t(int, bp->tx_ring_size / 2, 500);
//#else
//	rx_ring_size = bp->rx_ring_size ? bp->rx_ring_size :
//					  MAX_RX_AVAIL/QFLE3_NUM_RX_QUEUES(bp);

//	rx_ring_size = min_t(int, bp->tx_ring_size / 2,
//				  rx_ring_size);
//#endif

//	rx_ring_size = min_t(int, rx_ring_size, INIT_OOO_RING_SIZE);

//	fp->rx_pkts_avail = qfle3_alloc_rx_bds(fp, rx_ring_size);
	status = qfle3_alloc_fp_buffers(fp);
        fp->rx_pkts_avail = RX_BD_USABLE;
	/* Add more CQEs for ramrods to ensure the demand above */
	for (i = 0; i < MAX_SPQ_PENDING; i++)
		fp->rx_cq_prod = RCQ_NEXT(fp->rx_cq_prod);

	return status;
}
void qfle3_init_fcoe_fp(struct qfle3_adapter * adapter)
{
   
   struct qfle3_fastpath *fp;
   unsigned long q_type = 0;
//   vmk_uint32 cids = 0;

   fp = qfle3_fcoe_fp(adapter);
   fp->adapter = adapter;
   fp->num_rss = 1;
   fp->is_leading_rss = 0;
   fp->cl_id = qfle3_cnic_eth_cl_id(adapter,QFLE3_FCOE_ETH_CL_ID_IDX);
   fp->cid = QFLE3_FCOE_ETH_CID(adapter);
//   fp->cid = fp->qid;
//   fp->cl_id = fp->adapter->igu_base_sb + fp->qid;
   fp->fw_sb_id = DEF_SB_ID;
   fp->igu_sb_id = adapter->igu_dsb_id;
   fp->rx_cq_cons_sb = QFLE3_FCOE_L2_RX_INDEX;
   fp->txdata_ptr[0]->tx_cons_sb = QFLE3_FCOE_L2_TX_INDEX;
   fp->txdata_ptr[0]->tx_ring_size = TX_BD_USABLE_PER_PAGE * FCOE_NUM_TX_PAGES_DEF;
   fp->txdata_ptr[0]->txq_index = FCOE_TXQ_IDX(adapter);
   fp->txdata_ptr[0]->parent_fp = fp;
   fp->txdata_ptr[0]->cid = fp->cid;
   fp->rx_bd_cons = 0;
   fp->rx_cq_cons = 0;

	/* qZone id equals to FW (per path) client id */
   fp->cl_qzone_id  = fp->cl_id;

   /* init shortcut */
   fp->ustorm_rx_prods_offset = qfle3_rx_ustorm_prods_offset(fp);
   
   QFLE3_DBG(QFLE3_DBG_START, "created FCOE rx data cid %d, txq %d cl_id %d, fw sb id %d, "
      "igu sb id %d rx_cq_cons_sb %p tx_cons_sb %p\n", 
         fp->cid, fp->qid,fp->cl_id, fp->fw_sb_id, fp->igu_sb_id, fp->rx_cq_cons_sb, fp->txdata_ptr[0]->tx_cons_sb);
   
   /* Configure Queue State object */
   SET_BIT(ECORE_Q_TYPE_HAS_RX, &q_type);
   SET_BIT(ECORE_Q_TYPE_HAS_TX, &q_type);
   
   fp->max_cos = 1;
   
   
   //	/* No multi-CoS for FCoE L2 client */
   //	BUG_ON(fp->max_cos != 1);

   ecore_init_queue_obj(adapter, &QFLE3_SP_OBJ(adapter, fp).q_obj, fp->cl_id, &fp->cid,
         fp->max_cos, QFLE3_FUNC(adapter), QFLE3_SP(adapter, q_rdata),
         QFLE3_SP_MAPPING(adapter, q_rdata),q_type);


   QFLE3_DBG(QFLE3_DBG_START,
        "queue[%d]:  qfle3_init_sb(%p,%p)  cl_id %d  fw_sb %d  igu_sb %d\n",
        fp->qid, adapter, fp->status_block.e2_sb, fp->cl_id, fp->fw_sb_id,
        fp->igu_sb_id);

   
   /* Configure classification DBs */
   ecore_init_mac_obj(adapter, &QFLE3_SP_OBJ(adapter, fp).mac_obj, fp->cl_id,
		      fp->cid, QFLE3_FUNC(adapter), QFLE3_SP(adapter, mac_rdata),
		      QFLE3_SP_MAPPING(adapter, mac_rdata),
		      ECORE_FILTER_MAC_PENDING,
		      &adapter->sp_state, ECORE_OBJ_TYPE_RX_TX,
		      &adapter->macs_pool);
   if (QFLE3_SP_OBJ(adapter, fp).mac_obj.exe_queue.lock == NULL){
      
      QFLE3_DBG(QFLE3_DBG_START,
           "failed in initializing mac obj for fcoe q\n");
      return;
   }
   return ;
}
#if 0
void qfle3_init_fcoe_tx_fp(struct qfle3_adapter * adapter)
{
   
//   VMK_ReturnStatus status;
   struct qfle3_fastpath *fp;
   unsigned long q_type = 0;
   vmk_uint32 cids = 0;

   fp = qfle3_fcoe_tx_fp(adapter);

//	fp->rx_queue = FCOE_TX_IDX(adapter);
//	fp->cl_id = qfle3_cnic_eth_cl_id(adapter,
//						     QFLE3_FCOE_ETH_CL_ID_IDX);
//	fp->cid = QFLE3_FCOE_ETH_CID(adapter);
   
   fp->cid = fp->qid;
   fp->cl_id = fp->adapter->igu_base_sb + fp->qid;
	fp->fw_sb_id = DEF_SB_ID;
	fp->igu_sb_id = adapter->igu_dsb_id;
//	fp->rx_cq_cons_sb = QFLE3_FCOE_L2_RX_INDEX;
	fp->rx_cq_cons_sb = &adapter->def_status_blk->sp_sb.index_values[HC_SP_INDEX_ETH_FCOE_RX_CQ_CONS];
   fp->rx_bd_cons = 0;
   fp->rx_cq_cons = 0;

   /* Configure Queue State object */
   SET_BIT(ECORE_Q_TYPE_HAS_RX, &q_type);
   SET_BIT(ECORE_Q_TYPE_HAS_TX, &q_type);
   
   fp->max_cos = 1;
   
   /* init tx data */

   //fp->tx_cid = cid;
   //txdata->txq_index = txq_index;
//   fp->tx_cons_sb = QFLE3_FCOE_L2_TX_INDEX;
   fp->tx_cons_sb = &adapter->def_status_blk->sp_sb.index_values[HC_SP_INDEX_ETH_FCOE_TX_CQ_CONS];
   fp->tx_ring_size = TX_BD_USABLE;
   cids = fp->cid;

   QFLE3_DBG(QFLE3_DBG_START, "created FCOE tx data cid %d, txq %d cl_id %d, fw sb id %d, "
      "igu sb id %d rx_cq_cons_sb %p tx_cons_sb %p\n", 
         cids, fp->qid,fp->cl_id, fp->fw_sb_id, fp->igu_sb_id, fp->rx_cq_cons_sb, fp->tx_cons_sb);

	/* qZone id equals to FW (per path) client id */
   fp->cl_qzone_id  = fp->cl_id;
   /* init shortcut */
   fp->ustorm_rx_prods_offset = qfle3_rx_ustorm_prods_offset(fp);
   
//	/* No multi-CoS for FCoE L2 client */
//	BUG_ON(fp->max_cos != 1);

   ecore_init_queue_obj(adapter, &QFLE3_SP_OBJ(adapter, fp).q_obj, fp->cl_id, &fp->cid,
         fp->max_cos, QFLE3_FUNC(adapter), QFLE3_SP(adapter, q_rdata),
         QFLE3_SP_MAPPING(adapter, q_rdata),q_type);


   QFLE3_DBG(QFLE3_DBG_START,
        "queue[%d]:  qfle3_init_sb(%p,%p)  cl_id %d  fw_sb %d  igu_sb %d\n",
        fp->qid, adapter, fp->status_block.e2_sb, fp->cl_id, fp->fw_sb_id,
        fp->igu_sb_id);
   return ;
}
#endif
static inline vmk_uint8 qfle3_fp_qzone_id(struct qfle3_fastpath *fp)
{
   return fp->cl_id;
}

void qfle3_init_ooo_fp(struct qfle3_adapter * adapter)
{
   
//   VMK_ReturnStatus status;
   struct qfle3_fastpath *fp;
   unsigned long q_type = 0;
//   vmk_uint32 cids = 0;

   fp = qfle3_ooo_fp(adapter);
   
   fp->adapter = adapter;
   fp->num_rss = 1;
   fp->is_leading_rss = 0;
	/*
	 * OOO ring should not pass any packet to the stack,
	 * so rx_queue 0 is good for us
	 * we don't need set it explicitly - assumed zero_fp did this
	 * qfle3_ooo(bp, rx_queue) = 0;
	 */

//	fp->rx_queue = OOO_IDX(adapter);
	fp->cl_id = qfle3_cnic_eth_cl_id(adapter,
						     QFLE3_OOO_ETH_CL_ID_IDX);
	fp->cid = QFLE3_OOO_ETH_CID(adapter);
   
//   fp->cid = fp->qid;
//   fp->cl_id = fp->adapter->igu_base_sb + fp->qid;
	fp->fw_sb_id = qfle3_cnic_fw_sb_id(adapter);
	fp->igu_sb_id = qfle3_cnic_igu_sb_id(adapter);
   
   fp->cl_qzone_id  = qfle3_fp_qzone_id(fp);
	fp->rx_cq_cons_sb = QFLE3_RX_OOO_INDEX;
   fp->rx_bd_cons = 0;
   fp->rx_cq_cons = 0;

   
   fp->max_cos = 1;
   
   /* init tx data */

   //fp->tx_cid = cid;
   //txdata->txq_index = txq_index;
   fp->txdata_ptr[0]->tx_cons_sb = QFLE3_TX_OOO_INDEX;
   fp->txdata_ptr[0]->tx_ring_size = TX_BD_USABLE_PER_PAGE * FCOE_NUM_TX_PAGES_DEF;
   
   fp->txdata_ptr[0]->txq_index = -1;
   fp->txdata_ptr[0]->parent_fp = fp;
   fp->txdata_ptr[0]->cid = fp->cid;
   QFLE3_DBG(QFLE3_DBG_START, "created OOO tx data cid %d, txq %d cl_id %d, fw sb id %d, "
      "igu sb id %d rx_cq_cons_sb %p tx_cons_sb %p\n", 
         fp->cid, fp->qid,fp->cl_id, fp->fw_sb_id, fp->igu_sb_id, fp->rx_cq_cons_sb, fp->txdata_ptr[0]->tx_cons_sb);
	/* qZone id equals to FW (per path) client id */
   fp->cl_qzone_id  = fp->cl_id;// qfle3_fp_qzone_id(fp);
   
   /* init shortcut */
   fp->ustorm_rx_prods_offset = qfle3_rx_ustorm_prods_offset(fp);
   
	/* No multi-CoS for FCoE L2 client */
//	VMK_ASSERT(fp->max_cos == 1);

   
	/* Init OOO related internal memory */
	/* Set OOO ring CID */
	REG_WR(adapter, BAR_TSTRORM_INTMEM +
	       TSTORM_ISCSI_L2_ISCSI_OOO_CID_TABLE_OFFSET(QFLE3_FUNC(adapter)),
		HW_CID(adapter, fp->cid));

	/* Set OOO ring Client ID */
	REG_WR8(adapter, BAR_TSTRORM_INTMEM +
		TSTORM_ISCSI_L2_ISCSI_OOO_CLIENT_ID_TABLE_OFFSET(QFLE3_FUNC(adapter)),
		fp->cl_id);
   ecore_init_mac_obj(adapter, &QFLE3_SP_OBJ(adapter, fp).mac_obj, fp->cl_id,
		      fp->cid, QFLE3_FUNC(adapter), QFLE3_SP(adapter, mac_rdata),
		      QFLE3_SP_MAPPING(adapter, mac_rdata),
		      ECORE_FILTER_MAC_PENDING,
		      &adapter->sp_state, ECORE_OBJ_TYPE_RX,
		      &adapter->macs_pool);
   /* Configure classification DBs */
   ecore_init_vlan_obj(adapter, &QFLE3_SP_OBJ(adapter, fp).vlan_obj,
		       fp->cl_id, fp->cid, QFLE3_FUNC(adapter),
		       QFLE3_SP(adapter, vlan_rdata),
		       QFLE3_SP_MAPPING(adapter, vlan_rdata),
		       ECORE_FILTER_VLAN_PENDING,
		       &adapter->sp_state, ECORE_OBJ_TYPE_RX,
		       &adapter->vlans_pool);
   
	/* Configure VXLAN classification DBs */
   if (adapter->vxlan_filters_en) {
      ecore_init_vxlan_fltr_obj
         (adapter, &QFLE3_SP_OBJ(adapter, fp).vxlan_filter_obj,
         fp->cl_id,
         fp->cid, QFLE3_FUNC(adapter), QFLE3_SP(adapter, mac_rdata),
         QFLE3_SP_MAPPING(adapter, mac_rdata),
         ECORE_FILTER_VXLAN_PENDING,
         &adapter->sp_state, ECORE_OBJ_TYPE_RX,
         &adapter->macs_pool, &adapter->vlans_pool);
   }
   /* Configure Queue State object */
   SET_BIT(ECORE_Q_TYPE_HAS_RX, &q_type);
   
	/* No multi-CoS for OOO queue */
//	VMK_ASSERT(fp->max_cos != 1);
   ecore_init_queue_obj(adapter, &QFLE3_SP_OBJ(adapter, fp).q_obj, fp->cl_id, &fp->cid,
         1, QFLE3_FUNC(adapter), QFLE3_SP(adapter, q_rdata),
         QFLE3_SP_MAPPING(adapter, q_rdata),q_type);

   QFLE3_DBG(QFLE3_DBG_START,
        "queue[%d]:  qfle3_init_ooo_fp(%p,%p)  cl_id %d  fw_sb %d  igu_sb %d\n",
        fp->qid, adapter, fp->status_block.e2_sb, fp->cl_id, fp->fw_sb_id,
        fp->igu_sb_id);
   return ;
}

void qfle3_init_fwd_fp(struct qfle3_adapter * adapter)
{
   
//   VMK_ReturnStatus status;
   struct qfle3_fastpath *fp;
   unsigned long q_type = 0;
//   vmk_uint32 cids = 0;

   fp = qfle3_fwd_fp(adapter);
   fp->adapter = adapter;
   fp->num_rss = 1;
   fp->is_leading_rss = 0;
	/*
	 * FWD ring should not pass any packet to the stack,
	 * so rx_queue 0 is good for us
	 * we don't need set it explicitly - assumed zero_fp did this
	 * qfle3_fwd(bp, rx_queue) = 0;
	 */

//	fp->rx_queue = OOO_IDX(adapter);
	fp->cl_id = -1;/* No CL_ID for forwarding */
	fp->cid = QFLE3_FWD_ETH_CID(adapter);
   
//   fp->cid = fp->qid;
//   fp->cl_id = fp->adapter->igu_base_sb + fp->qid;
	fp->fw_sb_id = qfle3_cnic_fw_sb_id(adapter);  /* Connect to CNIC SB */
	fp->igu_sb_id = qfle3_cnic_igu_sb_id(adapter);
	fp->rx_cq_cons_sb = NULL;
   fp->rx_bd_cons = 0;
   fp->rx_cq_cons = 0;

   
   fp->max_cos = 1;
   
   /* init tx data */

   //fp->tx_cid = cid;
   fp->txdata_ptr[0]->tx_cons_sb = QFLE3_TX_FWD_INDEX;
   fp->txdata_ptr[0]->tx_ring_size = TX_BD_USABLE_PER_PAGE * FCOE_NUM_TX_PAGES_DEF;
   fp->txdata_ptr[0]->txq_index = -1;
   fp->txdata_ptr[0]->parent_fp = fp;
   fp->txdata_ptr[0]->cid = fp->cid;

   QFLE3_DBG(QFLE3_DBG_START, "created FWD tx data cid %d, txq %d cl_id %d, fw sb id %d, "
      "igu sb id %d rx_cq_cons_sb %p tx_cons_sb %p\n", 
         fp->cid, fp->qid,fp->cl_id, fp->fw_sb_id, fp->igu_sb_id, fp->rx_cq_cons_sb, fp->txdata_ptr[0]->tx_cons_sb);
   /* Configure Queue State object */
   SET_BIT(ECORE_Q_TYPE_FWD, &q_type);
   SET_BIT(ECORE_Q_TYPE_HAS_TX, &q_type);
      
//	/* No multi-CoS for FCoE L2 client */

   ecore_init_queue_obj(adapter, &QFLE3_SP_OBJ(adapter, fp).q_obj, fp->cl_id, &fp->cid,
         1, QFLE3_FUNC(adapter), QFLE3_SP(adapter, q_rdata),
         QFLE3_SP_MAPPING(adapter, q_rdata),q_type);

   QFLE3_DBG(QFLE3_DBG_START,
        "queue[%d]:  qfle3_init_fwd_fp(%p,%p)  cl_id %d  fw_sb %d  igu_sb %d\n",
        fp->qid, adapter, fp->status_block.e2_sb, fp->cl_id, fp->fw_sb_id,
        fp->igu_sb_id);
   return ;
}

int qfle3_cnic_ctl_send(struct qfle3_adapter * adapter, struct cnic_ctl_info *ctl)
{
	struct cnicOps *c_ops;
	int rc = 0;

   c_ops = adapter->cnicOps;
		rc = c_ops->cnicNotify(adapter->cnic_data, ctl);

	return rc;
}

static void qfle3_cnic_cfc_comp(struct qfle3_adapter * adapter, int cid, vmk_uint8 err)
{
	struct cnic_ctl_info ctl = {0};

	/* first we tell CNIC and only then we count this as a completion */
	ctl.cmd = CNIC_CTL_COMPLETION_CMD;
	ctl.data.comp.cid = cid;
	ctl.data.comp.error = err;

	qfle3_cnic_ctl_send(adapter, &ctl);
	qfle3_cnic_sp_post(adapter, 0);
}

int  qfle3_cnic_handle_cfc_del(struct qfle3_adapter * adapter, vmk_uint32 cid,
				      union event_ring_elem *elem)
{
	vmk_uint8 err = elem->message.error;

	if (!adapter->cnicEthDev->starting_cid  ||
	    (cid < adapter->cnicEthDev->starting_cid &&
	    cid != adapter->cnicEthDev->iscsi_l2_cid))
		return 1;

	QFLE3_DBG(QFLE3_DBG_START, "got delete ramrod for CNIC CID %d\n", cid);

	if (VMK_UNLIKELY(err)) {

		QFLE3_ERR("got delete ramrod for CNIC CID %d with error!\n",
			  cid);
//		qfle3_panic_dump(bp, false);
	}
	qfle3_cnic_cfc_comp(adapter, cid, err);
	return 0;
}
#if 0
VMK_ReturnStatus qfle3_alloc_cnicqs_mem(struct qfle3_adapter * adapter)
{
//   VMK_ReturnStatus status;
   struct qfle3_fastpath *fp;
   adapter->cnic_sb.e2_sb = qfle3_alloc_dma_mapping(adapter, 
                              adapter->dmaEngineCoherent, 
                              sizeof(struct host_hc_status_block_e2),
                              &adapter->cnic_sb_mapping);
   if (!adapter->cnic_sb.e2_sb)
      goto alloc_mem_err;

//   if (CONFIGURE_NIC_MODE(bp) && !bp->t2) {
//      /* allocate searcher T2 table, as it wasn't allocated before */
//      bp->t2 = QFLE3_PCI_ALLOC(&bp->t2_mapping, SRC_T2_SZ);
//      if (!bp->t2)
//         goto alloc_mem_err;
//   }

   /* write address to which L5 should insert its values */
   adapter->cnicEthDev->addr_drv_info_to_mcp =
      &adapter->sp->drv_info_to_mcp;

   if (ecore_ilt_mem_op_cnic(adapter, ILT_MEMOP_ALLOC))
      goto alloc_mem_err;



   if (!NO_FCOE(adapter)){
      /* FCoE */
      fp = qfle3_fcoe_rx_fp(adapter);
      fp->qid = FCOE_RX_IDX(adapter);
      fp->adapter = adapter;
      qfle3_set_fp_rx_buf_size(adapter, fp);
      if (qfle3_alloc_queue_mem(adapter, ALLOC_RXQ, fp) != VMK_OK){
         /* we will fail load process instead of mark
          * NO_FCOE_FLAG
          */
          
         qfle3_free_cnicqs_mem(adapter);
         goto alloc_mem_err;
      }

      
      fp = qfle3_fcoe_tx_fp(adapter);
      fp->qid = FCOE_TX_IDX(adapter);
      fp->adapter = adapter;
      if (qfle3_alloc_queue_mem(adapter, ALLOC_TXQ, fp) != VMK_OK){
         /* we will fail load process instead of mark
          * NO_FCOE_FLAG
          */
          
         qfle3_free_cnicqs_mem(adapter);
         goto alloc_mem_err;
      }
   }
   if (!NO_ISCSI_OOO(adapter)) {
      
      /* ISCSI */
      fp = qfle3_ooo_fp(adapter);
      fp->qid = OOO_IDX(adapter);
      fp->adapter = adapter;
      qfle3_set_fp_rx_buf_size(adapter, fp);
      if (qfle3_alloc_queue_mem(adapter, ALLOC_RXQ, fp) != VMK_OK){
         /* we will fail load process instead of mark
          * NO_FCOE_FLAG
          */
          
         qfle3_free_cnicqs_mem(adapter);
         goto alloc_mem_err;
      }
      
      fp = qfle3_fwd_fp(adapter);
      fp->qid = FWD_IDX(adapter);
      fp->adapter = adapter;
      if (qfle3_alloc_queue_mem(adapter, ALLOC_TXQ, fp) != VMK_OK){
         /* we will fail load process instead of mark
          * NO_FCOE_FLAG
          */
          
         qfle3_free_cnicqs_mem(adapter);
         goto alloc_mem_err;
      }
   }

   return 0;
   

   return VMK_OK;

alloc_mem_err:
   qfle3_free_cnicqs_mem(adapter);
   QFLE3_ERR("Can't allocate memory\n");
   return VMK_NO_MEMORY;
}
#endif
VMK_ReturnStatus cnicLL2Tx (vmk_AddrCookie data, vmk_PktHandle * pkt)
{
   qfle3_adapter *adapter = (qfle3_adapter *) data.ptr;
   struct qfle3_fastpath *fp;
   int qid;
   // let's use default qid for now
//   qid = QFLE3_NUM_TOTAL_RX_QUEUES(adapter);
   fp = qfle3_fcoe_fp(adapter);
   qid = fp->qid - QFLE3_DEFAULT_TX_QID(adapter);
//   vmk_LogMessage("cnicLL2Tx, sending on qid %d\n", fp->qid);
   if (!NO_FCOE(adapter) && CNIC_LOADED(adapter)) {
      return qfle3_xmit_pkt(adapter,qid, pkt);
   } else {
      QFLE3_ERR("cnicLL2Tx called when NO_FCOE or CNIC not loaded\n");
      return VMK_FAILURE;
   }
}
static inline int qfle3_func_switch_update(qfle3_adapter *adapter, int suspend)
{
	int rc;
	struct ecore_func_state_params func_params = {NULL};
	struct ecore_func_switch_update_params *switch_update_params =
		&func_params.params.switch_update;

	/* Prepare parameters for function state transitions */
	ECORE_SET_BIT(RAMROD_COMP_WAIT, &func_params.ramrod_flags);
	ECORE_SET_BIT(RAMROD_RETRY, &func_params.ramrod_flags);

	func_params.f_obj = &adapter->func_obj;
	func_params.cmd = ECORE_F_CMD_SWITCH_UPDATE;

	/* Function parameters */
	ECORE_SET_BIT(ECORE_F_UPDATE_TX_SWITCH_SUSPEND_CHNG,
		  &switch_update_params->changes);
	if (suspend)
		ECORE_SET_BIT(ECORE_F_UPDATE_TX_SWITCH_SUSPEND,
			  &switch_update_params->changes);

	rc = ecore_func_state_change(adapter, &func_params);

	return rc;
}

int qfle3_reset_nic_mode(qfle3_adapter *adapter)
{
	int rc, i, port = QFLE3_PORT(adapter);
	int vlan_en = 0, mac_en[NUM_MACS];

	/* Close input from network */
	if (adapter->mf_mode == SINGLE_FUNCTION) {
		elink_set_rx_filter(&adapter->link_params, 0);
	} else {
		vlan_en = REG_RD(adapter, port ? NIG_REG_LLH1_FUNC_EN :
				   NIG_REG_LLH0_FUNC_EN);
		REG_WR(adapter, port ? NIG_REG_LLH1_FUNC_EN :
			  NIG_REG_LLH0_FUNC_EN, 0);
		for (i = 0; i < NUM_MACS; i++) {
			mac_en[i] = REG_RD(adapter, port ?
					     (NIG_REG_LLH1_FUNC_MEM_ENABLE +
					      4 * i) :
					     (NIG_REG_LLH0_FUNC_MEM_ENABLE +
					      4 * i));
			REG_WR(adapter, port ? (NIG_REG_LLH1_FUNC_MEM_ENABLE +
					      4 * i) :
				  (NIG_REG_LLH0_FUNC_MEM_ENABLE + 4 * i), 0);
		}
	}

	/* Close BMC to host */
	REG_WR(adapter, port ? NIG_REG_P0_TX_MNG_HOST_ENABLE :
	       NIG_REG_P1_TX_MNG_HOST_ENABLE, 0);

	/* Suspend Tx switching to the PF. Completion of this ramrod
	 * further guarantees that all the packets of that PF / child
	 * VFs in BRB were processed by the Parser, so it is safe to
	 * change the NIC_MODE register.
	 */
	rc = qfle3_func_switch_update(adapter, 1);
	if (rc) {
		QFLE3_ERR("Can't suspend tx-switching!\n");
		return rc;
	}

	/* Change NIC_MODE register */
	REG_WR(adapter, PRS_REG_NIC_MODE, 0);
        QFLE3_DBG(QFLE3_DBG_START, "NIC MODE Disabled\n");

	/* Open input from network */
	if (adapter->mf_mode == SINGLE_FUNCTION) {
		elink_set_rx_filter(&adapter->link_params, 1);
	} else {
		REG_WR(adapter, port ? NIG_REG_LLH1_FUNC_EN :
			  NIG_REG_LLH0_FUNC_EN, vlan_en);
		for (i = 0; i < NUM_MACS; i++) {
			REG_WR(adapter, port ? (NIG_REG_LLH1_FUNC_MEM_ENABLE +
					      4 * i) :
				  (NIG_REG_LLH0_FUNC_MEM_ENABLE + 4 * i),
				  mac_en[i]);
		}
	}

	/* Enable BMC to host */
	REG_WR(adapter, port ? NIG_REG_P0_TX_MNG_HOST_ENABLE :
	       NIG_REG_P1_TX_MNG_HOST_ENABLE, 1);

	/* Resume Tx switching to the PF */
	rc = qfle3_func_switch_update(adapter, 0);
	if (rc) {
		QFLE3_ERR("Can't resume tx-switching!\n");
		return rc;
	}

	QFLE3_DBG(QFLE3_DBG_START, "NIC MODE disabled\n");
	return 0;
}

/**
 * qfle3_cnic_get_ooo_cqe - get OOO CQE info.
 *
 *
 * Returns	positive value if there are more BDs available
 *		0 if no more available BDs
 *		negative value if there was an error
 */
int qfle3_cnic_get_ooo_cqe(struct qfle3_adapter *adapter,
				struct cnic_ooo_cqe *ooo_cqe)
{
   struct qfle3_fastpath *fp = qfle3_ooo_fp(adapter);
   vmk_uint16 bd_cons, bd_prod, bd_prod_fw, comp_ring_cons;
   vmk_uint16 hw_cq_cons, sw_cq_cons, sw_cq_prod;
   int rx_pkt = 0, done;
   int rc;
//   vmk_Bool csum_failure = VMK_FALSE;

   vmk_SpinlockLock(fp->fp_lock);

   /* CQ "next element" is of the size of the regular element,
      that's why it's ok here */
   hw_cq_cons = le16toh(*fp->rx_cq_cons_sb);
   if ((hw_cq_cons & RCQ_USABLE_PER_PAGE) == RCQ_USABLE_PER_PAGE)
      hw_cq_cons++;

   bd_cons = fp->rx_bd_cons;
   bd_prod = fp->rx_bd_prod;
   bd_prod_fw = bd_prod;
   sw_cq_cons = fp->rx_cq_cons;
   sw_cq_prod = fp->rx_cq_prod;

   /* Memory barrier necessary as speculative reads of the rx
    * buffer can be ahead of the index in the status block
    */
   vmk_CPUMemFenceReadWrite();

   QFLE3_DBG(QFLE3_DBG_CNIC,
      "queue[%d]:  hw_cq_cons %u  sw_cq_cons %u\n",
      fp->qid, hw_cq_cons, sw_cq_cons);

   do {
      done = 0;

      if (sw_cq_cons != hw_cq_cons) {
      	union eth_rx_cqe *cqe;
      	vmk_uint8 cqe_fp_flags, cqe_fp_type;

      	comp_ring_cons = RCQ(sw_cq_cons);
      	bd_prod = RX_BD(bd_prod);
      	bd_cons = RX_BD(bd_cons);

      	cqe = &fp->rcq_chain[comp_ring_cons];
      	cqe_fp_flags = cqe->fast_path_cqe.type_error_flags;
      	cqe_fp_type = CQE_TYPE(cqe_fp_flags);

      	QFLE3_DBG(QFLE3_DBG_CNIC, "CQE type %x  err %x  status %x  queue %x  vlan %x  len %u\n",
      	   cqe_fp_type, cqe_fp_flags,
      	   cqe->fast_path_cqe.status_flags,
      	   le32toh(cqe->fast_path_cqe.rss_hash_result),
      	   le16toh(cqe->fast_path_cqe.vlan_tag),
      	   le16toh(cqe->fast_path_cqe.pkt_len_or_gro_seg_len));

      	/* is this a slowpath msg? */
      	if (VMK_UNLIKELY(CQE_TYPE_SLOW(cqe_fp_type))) {
            ooo_cqe->cqe_type = OOO_RAMROD_CQE;
            ooo_cqe->u.ramrod_data.data.lo =
            	cqe->ramrod_cqe.protocol_data.data_lo;
            ooo_cqe->u.ramrod_data.data.hi =
            	cqe->ramrod_cqe.protocol_data.data_hi;

            done = 1;
      	/* this is an rx packet */
      	} else if (VMK_LIKELY(CQE_TYPE_FAST(cqe_fp_type))){// CQE_TYPE_FAST
            qfle3_rxbuf_info *rx_buf = &fp->rxbuf_chain[bd_cons];
            vmk_PktHandle *pkt = NULL;
            vmk_uint8 pad = cqe->fast_path_cqe.placement_offset;
            vmk_uint16 len = le16toh(cqe->fast_path_cqe.pkt_len_or_gro_seg_len);
            vmk_uint8 tryagain_alloc = 1;

            /* is this an error packet? */
            if (VMK_UNLIKELY(cqe_fp_flags &
            	     ETH_FAST_PATH_RX_CQE_PHY_DECODE_ERR_FLG)) {
               QFLE3_ERR("Error packet! flags 0x%x rx packet %u\n", cqe_fp_flags, sw_cq_cons);
               //fp->eth_q_stats.rx_soft_errors++;
               fp->drv_stats.rx_Errors++;
               adapter->drv_stats.rx_Errors++;
               goto next_rx;
            }
            pkt = rx_buf->pkt;
            if (VMK_UNLIKELY(pkt == NULL)) {
               QFLE3_ERR("No mbuf in rx chain descriptor %d for fp[%02d]\n",
                    bd_cons, fp->qid);
               adapter->drv_stats.rx_Errors++;
               goto next_rx;
            }
            
            /*
            * If all the buffer descriptors are filled with mbufs then fill in
            * the current consumer index with a new BD. Else if a maximum Rx
            * buffer limit is imposed then fill in the next producer index.
            */
         try_once_again:
            rc = qfle3_alloc_rx_bd_mbuf(fp, bd_cons);
            if (rc != 0) {
               if (tryagain_alloc--)
                  goto try_once_again;
               QFLE3_ERR("mbuf alloc fail for fp[%02d] rx chain[%d]\n",
                 fp->qid, (int)bd_cons);
               //fp->eth_q_stats.rx_soft_errors++;
               pkt = NULL;
               fp->drv_stats.rx_Drops++;
               adapter->drv_stats.rx_Drops++;
               goto next_rx;
            }
            vmk_PktFrameLenSet(pkt, len + pad);
            vmk_PktPushHeadroom(pkt, pad);
            
            ooo_cqe->cqe_type = OOO_BD_CQE;
            vmk_Memcpy(ooo_cqe->u.cqe.raw_data,
                   cqe->fast_path_cqe.sgl_or_raw_data.raw_data,
                   sizeof(ooo_cqe->u.cqe.raw_data));
            ooo_cqe->u.cqe.pkt_desc = pkt;

            done = 1;
         next_rx:
            //				rx_buf->data = NULL;
            bd_cons = RX_BD_NEXT(bd_cons);
            bd_prod = RX_BD_NEXT(bd_prod);
            bd_prod_fw = RX_BD_NEXT(bd_prod_fw);
            rx_pkt++;
      	}else {
      	   VMK_ASSERT(0,"Unexpected CQE_TYPE %d\n", cqe_fp_type);
      	}

      	sw_cq_cons = RCQ_NEXT(sw_cq_cons);
      	sw_cq_prod = RCQ_NEXT(sw_cq_prod);
      } else {

         vmk_SpinlockUnlock(fp->fp_lock);
      	QFLE3_ERR("Called but no new BDs\n");
      	return 0;
      }
   } while (!done);

   fp->rx_bd_cons = bd_cons;
   /** If 'reuse' case is going to be legal then producers update
    *  should be protected with mutex/spinlock against concurrent
    *  update here and in qfle3_cnic_reuse_ooo_pkt().
    */
   fp->rx_bd_prod = bd_prod_fw;
   fp->rx_cq_cons = sw_cq_cons;
   fp->rx_cq_prod = sw_cq_prod;
   
   qfle3_update_rx_prod(adapter, fp, bd_prod_fw, sw_cq_prod, fp->rx_sge_prod);
   vmk_SpinlockUnlock(fp->fp_lock);
   return (hw_cq_cons != sw_cq_cons) ? 1 : 0;
}
/**
 * qfle3_cnic_send_ooo_pkt - send OOO packet
 *
 */
int qfle3_cnic_send_ooo_pkt(struct qfle3_adapter *adapter, vmk_PktHandle *pkt)
{
   struct qfle3_fastpath *fp = qfle3_fwd_fp(adapter);
   qfle3_txbuf_info *tx_buf;
   qfle3_pkt_tx_info_t tx_info;
   struct eth_tx_start_bd *tx_start_bd;
   struct eth_tx_bd *tx_data_bd;
   struct eth_tx_parse_bd_e2 *pbd_e2 = NULL;
   vmk_uint16 pkt_prod, bd_prod;
   int nbd;
   vmk_uint16 pkt_len = vmk_PktFrameLenGet(pkt);
   vmk_uint8 mac_type = UNICAST_ADDRESS;
   int nsegs;
   const vmk_SgElem *sgex;
   VMK_ReturnStatus status;
   int i;
   
   vmk_SpinlockLock(fp->fp_lock);

   nsegs = vmk_PktSgArrayGet(pkt)->numElems;
   vmk_Memset(&tx_info, 0, sizeof(qfle3_pkt_tx_info_t));
   
	/**
	 * Please read carefully!
	 * First we have a "start BD", then there is a parsing BD and
	 * then there are optional BDs for fragmented skb (this will
	 * never happen in OOO flow).
	 *
	 * And above all, all pdb sizes are in words - NOT DWORDS!
	 */
	pkt_prod = fp->txdata_ptr[0]->tx_pkt_prod++;

	bd_prod = TX_BD(fp->txdata_ptr[0]->tx_bd_prod);

	/* get a tx_buf and first BD */
	tx_buf = &fp->txdata_ptr[0]->txbuf_chain[TX_BD(pkt_prod)];
   
   for (i = 0; i < nsegs; i++) {
      sgex = vmk_PktSgElemGet(pkt, i);
      status = qfle3_dma_map_ma(adapter, sgex->addr, sgex->length,
				VMK_DMA_DIRECTION_FROM_MEMORY, adapter->dmaEngine,
				&tx_buf->tx_segs[i].ioa);
      if (status != VMK_OK)
	 goto unmap_drop_pkt;
      tx_buf->tx_segs[i].len = sgex->length;
   }
   tx_buf->nsegs = nsegs;
   tx_buf->mapType = QFLE3_MAP_ELEM;

	tx_start_bd = &fp->txdata_ptr[0]->tx_chain[bd_prod].start_bd;

	tx_start_bd->bd_flags.as_bitfield = ETH_TX_BD_FLAGS_START_BD;
	/* header nbd */
	SET_FLAG(tx_start_bd->general_data,
		  ETH_TX_START_BD_HDR_NBDS,
		  1);
	SET_FLAG(tx_start_bd->general_data,
		 ETH_TX_START_BD_PARSE_NBDS,
		 0);

	/* remember the first BD of the packet */
	tx_buf->first_bd = fp->txdata_ptr[0]->tx_bd_prod;
	tx_buf->pkt = pkt;
	tx_buf->flags = 0;

	QFLE3_DBG(QFLE3_DBG_CNIC,
	   "sending pkt %u @%p  next_idx %u  bd %u @%p\n",
	   pkt_prod, tx_buf, fp->txdata_ptr[0]->tx_pkt_prod, bd_prod, tx_start_bd);

	/* iSCSI OOO L2 Queue is never configured to do HW VLAN
	 * acceleration.
	 */
	tx_start_bd->vlan_or_ethertype = htole16(pkt_prod);

	/* turn on parsing and get a BD */
	bd_prod = TX_BD_NEXT(bd_prod);

	vmk_uint32 parsing_data = 0;
	pbd_e2 = &fp->txdata_ptr[0]->tx_chain[bd_prod].parse_bd_e2;
	vmk_Memset(pbd_e2, 0, sizeof(struct eth_tx_parse_bd_e2));
	SET_FLAG(parsing_data, ETH_TX_PARSE_BD_E2_ETH_ADDR_TYPE, mac_type);
	pbd_e2->parsing_data = htole32(parsing_data);

	/* Setup the data pointer of the first BD of the packet */
	tx_start_bd->addr_hi = htole32(U64_HI(tx_buf->tx_segs[0].ioa));
	tx_start_bd->addr_lo = htole32(U64_LO(tx_buf->tx_segs[0].ioa));
	nbd = 2; /* start_bd + pbd */
	tx_start_bd->nbd = htole16(nbd);
	tx_start_bd->nbytes = htole16(pkt_len);

	QFLE3_DBG(QFLE3_DBG_CNIC, "first bd @%p  addr (%x:%x)  nbd %d  nbytes %d  flags %x  vlan %x\n",
	   tx_start_bd, tx_start_bd->addr_hi, tx_start_bd->addr_lo,
	   le16toh(tx_start_bd->nbd), le16toh(tx_start_bd->nbytes),
	   tx_start_bd->bd_flags.as_bitfield, le16toh(tx_start_bd->vlan_or_ethertype));

	tx_data_bd = (struct eth_tx_bd *)tx_start_bd;

	QFLE3_DBG(QFLE3_DBG_CNIC, "last bd @%p\n", tx_data_bd);

	bd_prod = TX_BD_NEXT(bd_prod);

	/* now send a tx doorbell, counting the next BD
	 * if the packet contains or ends with it
	 */
	if (TX_BD_IDX(bd_prod) < nbd)
		nbd++;

	if (pbd_e2)
		QFLE3_DBG(QFLE3_DBG_CNIC,
		   "PBD (E2) @%p  dst %x %x %x src %x %x %x parsing_data %x\n",
		   pbd_e2,
		   pbd_e2->data.mac_addr.dst_hi,
		   pbd_e2->data.mac_addr.dst_mid,
		   pbd_e2->data.mac_addr.dst_lo,
		   pbd_e2->data.mac_addr.src_hi,
		   pbd_e2->data.mac_addr.src_mid,
		   pbd_e2->data.mac_addr.src_lo,
		   pbd_e2->parsing_data);
	QFLE3_DBG(QFLE3_DBG_CNIC, "doorbell: nbd %d  bd %u\n", nbd, bd_prod);

	/*
	 * Make sure that the BD data is updated before updating the producer
	 * since FW might read the BD right after the producer is updated.
	 * This is only applicable for weak-ordered memory model archs such
	 * as IA-64. The following barrier is also mandatory since FW will
	 * assumes packets must have BDs.
	 */
	vmk_CPUMemFenceReadWrite();

	fp->txdata_ptr[0]->tx_db.data.prod += nbd;
	vmk_CPUMemFenceReadWrite();

	DOORBELL(adapter, fp->cid, fp->txdata_ptr[0]->tx_db.raw);

	vmk_CPUMemFenceReadWrite();

	fp->txdata_ptr[0]->tx_bd_prod += nbd;
   
	vmk_CPUMemFenceReadWrite();

	fp->drv_stats.tx_pkts++;
   vmk_SpinlockUnlock(fp->fp_lock);
	return VMK_OK;
   
   unmap_drop_pkt:
    qfle3_unmap_txbuf(adapter, tx_buf, TX_BD(pkt_prod), fp->qid);
//   drop_pkt:
    vmk_PktRelease(pkt);
    fp->drv_stats.tx_Drops++;
    adapter->drv_stats.tx_Drops++;
    vmk_SpinlockUnlock(fp->fp_lock);
    return (VMK_OK);
}
/* unmap pkt in the FWD (OOO Tx) ring at pos idx
 * return idx of last bd freed
 */
static vmk_uint16 qfle3_free_ooo_tx_pkt(struct qfle3_adapter *adapter, 
      struct qfle3_fastpath *fp, vmk_uint16 idx)
{
	qfle3_txbuf_info *tx_buf = &fp->txdata_ptr[0]->txbuf_chain[idx];
	struct eth_tx_start_bd *tx_start_bd;
	vmk_PktHandle *pkt = tx_buf->pkt;
	vmk_uint16 bd_idx = TX_BD(tx_buf->first_bd), new_cons;
	int nbd, nsegs, i;

	/* unmap bd */
	tx_start_bd = &fp->txdata_ptr[0]->tx_chain[bd_idx].start_bd;
   
   nsegs = vmk_PktSgArrayGet(pkt)->numElems;
   for (i = 0; i < nsegs; i++) {
      qfle3_dma_unmap(adapter, tx_buf->tx_segs[i].ioa, tx_buf->tx_segs[i].len,
         VMK_DMA_DIRECTION_FROM_MEMORY, adapter->dmaEngine);
      tx_buf->tx_segs[i].len = 0;
      tx_buf->tx_segs[i].ioa = 0;
   }

	nbd = le16toh(tx_start_bd->nbd) - 1;
	new_cons = nbd + tx_buf->first_bd;
	tx_buf->first_bd = 0;
	tx_buf->pkt = NULL;

	if (qfle3_cnic_reuse_ooo_pkt(adapter, pkt))
		QFLE3_DBG(QFLE3_DBG_CNIC,
		   "Failed to reuse a completed skb at index %d", idx);

	return new_cons;
}

/**
 * qfle3_cnic_comp_ooo_tx_pkts - get number of transmitted packets.
 *
 * Returns negative value if there was an error.
 */
int qfle3_cnic_comp_ooo_tx_pkts(struct qfle3_adapter *adapter)
{
	struct qfle3_fastpath *fp = qfle3_fwd_fp(adapter);
	vmk_uint16 hw_cons, sw_cons, bd_cons = fp->txdata_ptr[0]->tx_bd_cons;
	int comp_pkts = 0;

	/* Ignore the first increment as it was due to ramrod completion */
	hw_cons = le16toh(*fp->txdata_ptr[0]->tx_cons_sb);
	sw_cons = fp->txdata_ptr[0]->tx_pkt_cons;

	while (sw_cons != hw_cons) {
		vmk_uint16 pkt_cons;

		pkt_cons = TX_BD(sw_cons);

		QFLE3_DBG(QFLE3_DBG_CNIC, "queue[%d]: hw_cons %u  sw_cons %u  pkt_cons %u\n",
		   fp->qid, hw_cons, sw_cons, pkt_cons);

		bd_cons = qfle3_free_ooo_tx_pkt(adapter, fp, pkt_cons);
		sw_cons++;
		comp_pkts++;
	}

	fp->txdata_ptr[0]->tx_pkt_cons = sw_cons;
	fp->txdata_ptr[0]->tx_bd_cons = bd_cons;

	vmk_CPUMemFenceReadWrite();
	return comp_pkts;
}
/**
 * qfle3_cnic_reuse_ooo_pkt - reuse given packet.
 * 
 * Returns zero if the operation was successful.
 */
int qfle3_cnic_reuse_ooo_pkt(struct qfle3_adapter *adapter, vmk_PktHandle *pkt)
{
   vmk_PktRelease(pkt);
	return 0;
}

#endif
