/*
 * Copyright Qlogic Corportation.
 */
#include "qfle3.h"
#include "qfle3_stats.h"
#include "qfle3_sm.h"

#ifdef __i386__
#define BITS_PER_LONG 32
#else
#define BITS_PER_LONG 64
#endif
#define STATS_TIMEOUT 3

static inline long
bxe_hilo(vmk_uint32 *hiref)
{
   vmk_uint32 lo = *(hiref + 1);
#if (BITS_PER_LONG == 64)
   vmk_uint32 hi = *hiref;
   return (HILO_U64(hi, lo));
#else
   return (lo);
#endif
}


/*
 * Statistics ID are global per chip/path, while Client IDs for E1x
 * are per port.
 */
vmk_uint8
qfle3_stats_id(struct qfle3_fastpath *fp)
{
   struct qfle3_adapter *adapter = fp->adapter;
   QFLE3_DBG(QFLE3_DBG_STATS, "enter\n");
   if (!CHIP_IS_E1x(adapter)) {
#if 0
      /* there are special statistics counters for FCoE 136..140 */
		if (IS_FCOE_FP(fp)) {
							   return (adapter->cnic_base_cl_id + (sc->pf_num >> 1));
		}
#endif
      QFLE3_DBG(QFLE3_DBG_STATS, "!CHIP_IS_E1\n ");
      QFLE3_DBG(QFLE3_DBG_STATS, "qfle3 stats id fastpath %p, cl_id %d port %d\n", fp, fp->cl_id, SC_PORT(adapter));
      return (fp->cl_id);
   }

   return (fp->cl_id + SC_PORT(adapter) * FP_SB_MAX_E1x);
}

static inline vmk_uint16
qfle3_get_port_stats_dma_len(struct qfle3_adapter *adapter)
{
   vmk_uint16 res = 0;
   vmk_uint32 size;

   /* 'newest' convention - shmem2 contains the size of the port stats */
   if (SHMEM2_HAS(adapter, sizeof_port_stats)) {
      size = SHMEM2_RD(adapter, sizeof_port_stats);
      if (size) {
	 res = size;
      }

      /* prevent newer BC from causing buffer overflow */
      if (res > sizeof(struct host_port_stats)) {
	 res = sizeof(struct host_port_stats);
      }
   }

   /*
    * Older convention - all BCs support the port stats fields up until
    * the 'not_used' field
    */
   if (!res) {
      res = (vmk_offsetof(struct host_port_stats, not_used) + 4);

      /* if PFC stats are supported by the MFW, DMA them as well */
      if (adapter->hw_info.bc_ver >= REQ_BC_VER_4_PFC_STATS_SUPPORTED) {
	 res += (vmk_offsetof(struct host_port_stats, pfc_frames_rx_lo) -
		 vmk_offsetof(struct host_port_stats, pfc_frames_tx_hi) + 4);
      }
   }

   res >>= 2;

//    DBASSERT(sc, !(res > 2 * DMAE_LEN32_RD_MAX), ("big stats dmae length\n"));
   VMK_ASSERT(!(res > 2 * DMAE_LEN32_RD_MAX));
   return (res);
}

/*
 * Init service functions
 */

static void
qfle3_dp_stats(struct qfle3_adapter *adapter)
{
   int i;

   QFLE3_DBG(QFLE3_DBG_STATS,
	     "dumping stats:\n"
	     "  fw_stats_req\n"
	     "    hdr\n"
	     "      cmd_num %d\n"
	     "      reserved0 %d\n"
	     "      drv_stats_counter %d\n"
	     "      reserved1 %d\n"
	     "      stats_counters_addrs %x %x\n",
	     adapter->fw_stats_req->hdr.cmd_num,
	     adapter->fw_stats_req->hdr.reserved0,
	     adapter->fw_stats_req->hdr.drv_stats_counter,
	     adapter->fw_stats_req->hdr.reserved1,
	     adapter->fw_stats_req->hdr.stats_counters_addrs.hi,
	     adapter->fw_stats_req->hdr.stats_counters_addrs.lo);

   for (i = 0; i < adapter->fw_stats_req->hdr.cmd_num; i++) {
      QFLE3_DBG(QFLE3_DBG_STATS,
		"query[%d]\n"
		"  kind %d\n"
		"  index %d\n"
		"  funcID %d\n"
		"  reserved %d\n"
		"  address %x %x\n",
		i,
		adapter->fw_stats_req->query[i].kind,
		adapter->fw_stats_req->query[i].index,
		adapter->fw_stats_req->query[i].funcID,
		adapter->fw_stats_req->query[i].reserved,
		adapter->fw_stats_req->query[i].address.hi,
		adapter->fw_stats_req->query[i].address.lo);
   }
}

#ifdef QFLE3_SRIOV
static int
qfle3_esx_iov_adjust_stats_req_to_pf(struct qfle3_adapter *adapter,
                 struct qfle3_virtf *vf,
                 int vfq_idx,
                 struct stats_query_entry **cur_query_entry,
                 vmk_uint8 *stats_count)
{
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];
   struct qfle3_vf_queue *rxq = vfq_get(vf, vfq_idx);
   vmk_IOA q_stats_addr;

   /* collect stats for active queues only */
   if (ecore_get_q_logical_state(adapter, &rxq->sp_obj) ==
       ECORE_Q_LOGICAL_STATE_STOPPED)
      return 0;

   q_stats_addr = adapter_esx_vf->vf_fw_stats_mapping +
         (vfq_idx * QFLE3_ESX_VF_FW_STATS_SIZE);

   /* create stats query entry for this queue */
   (*cur_query_entry)->kind = STATS_TYPE_QUEUE;
   (*cur_query_entry)->index = vfq_stat_id(vf, rxq);
   (*cur_query_entry)->funcID = htole16(QFLE3_FUNC(adapter));
   (*cur_query_entry)->address.hi = htole32(U64_HI(q_stats_addr));
   (*cur_query_entry)->address.lo = htole32(U64_LO(q_stats_addr));
   QFLE3_DBG(QFLE3_DBG_IOV,
      "SRIOV: To PF: added address %x %x for vf %d funcID %d queue %d "
      "q->index %d client %d\n",
      (*cur_query_entry)->address.hi,
      (*cur_query_entry)->address.lo,
      vf->index,
      (*cur_query_entry)->funcID,
      vfq_idx, rxq->index, (*cur_query_entry)->index);
   *cur_query_entry = *cur_query_entry + 1;
   *stats_count = *stats_count + 1;
   /* all stats are coalesced to the leading queue */
   if (vf->cfg_flags & VF_CFG_STATS_COALESCE)
      return 0;

   return 0;
}


/*
 * Post the next statistics ramrod. Protect it with the lock in
 * order to ensure the strict order between statistics ramrods
 * (each ramrod has a sequence number passed in a
 * adapter->fw_stats_req->hdr.drv_stats_counter and ramrods must be
 * sent in order).
 */
void qfle3_adjust_stats_req(struct qfle3_adapter *adapter)
{
   int i;
   int first_queue_query_index;
   vmk_IOA cur_data_offset;
   struct stats_query_entry *cur_query_entry;
   vmk_uint8 stats_count = 0;
   vmk_Bool is_fcoe = VMK_FALSE;

   cur_data_offset = (adapter->fw_stats_data_mapping +
		   vmk_offsetof(struct qfle3_fw_stats_data, queue_stats));

   /*
    * First queue query index depends whether FCoE offloaded request will
    * be included in the ramrod
    */

   first_queue_query_index = QFLE3_FIRST_QUEUE_QUERY_IDX -
      (is_fcoe ? 0 : 1);

   cur_query_entry = &adapter->fw_stats_req->query[first_queue_query_index];

   for (i = 0; i < QFLE3_NUM_ETH_QUEUES(adapter); i++) {
      if (adapter->fp[i].collect_stats == 0) 
                  continue;

      //cur_query_entry = &adapter->fw_stats_req->query[first_queue_query_index + i];
      cur_query_entry->kind = STATS_TYPE_QUEUE;
      cur_query_entry->index = qfle3_stats_id(&(adapter->fp[i]));
      cur_query_entry->funcID = htole16(QFLE3_FUNC(adapter));
      cur_query_entry->address.hi = htole32(U64_HI(cur_data_offset));
      cur_query_entry->address.lo = htole32(U64_LO(cur_data_offset));
      cur_query_entry++;

      cur_data_offset += sizeof(struct per_queue_stats);
      stats_count++;

   }

   if (adapter->vfdb) {
    for (i = 0; i< adapter->vfdb->sriov.nr_virtfn; i++) {
      int j;
      struct qfle3_virtf *vf = QFLE3_VF(adapter, i);

      if (vf->state != VF_ENABLED) {
     //    QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: vf %d not enabled so no stats for it\n",
     //           vf->abs_vfid);
         continue;
      }

      for(j=0; j < vf->alloc_resc.num_rxqs; j++) {
         struct qfle3_vf_queue *rxq = vfq_get(vf, j);

         vmk_IOA q_stats_addr =
            vf->fw_stat_map + j * vf->stats_stride;

         /* collect stats fro active queues only */
         if (ecore_get_q_logical_state(adapter, &rxq->sp_obj) ==
             ECORE_Q_LOGICAL_STATE_STOPPED)
            continue;

         /* create stats query entry for this queue */
         cur_query_entry->kind = STATS_TYPE_QUEUE;
         cur_query_entry->index = vfq_stat_id(vf, rxq);
         cur_query_entry->funcID =
            htole16(FW_VF_HANDLE(vf->abs_vfid));
         cur_query_entry->address.hi =
            htole32(U64_HI(q_stats_addr));
         cur_query_entry->address.lo =
            htole32(U64_LO(q_stats_addr));
         /*QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: added address %x %x for vf %d queue %d client %d\n",
            cur_query_entry->address.hi,
            cur_query_entry->address.lo, cur_query_entry->funcID,
            j, cur_query_entry->index);*/
         cur_query_entry++;
         cur_data_offset += sizeof(struct per_queue_stats);
         stats_count++;

         qfle3_esx_iov_adjust_stats_req_to_pf(adapter, vf, j,
                          &cur_query_entry,&stats_count);


         /* all stats are coalesced to the leading queue */
         if (vf->cfg_flags & VF_CFG_STATS_COALESCE)
            break;
      }
    }
   }
   adapter->fw_stats_req->hdr.cmd_num = 2 + stats_count;
}

#endif //QFLE3_SRIOV

/*
 * Post the next statistics ramrod. Protect it with the lock in
 * order to ensure the strict order between statistics ramrods
 * (each ramrod has a sequence number passed in a
 * adapter->fw_stats_req->hdr.drv_stats_counter and ramrods must be
 * sent in order).
 */
static void
qfle3_storm_stats_post(qfle3_adapter *adapter)
{
   int rc;

      if (adapter->stats_pending) {
	 return;
      }

      adapter->fw_stats_req->hdr.drv_stats_counter =
	 htole16(adapter->stats_counter++);

      QFLE3_DBG(QFLE3_DBG_STATS,
		"sending statistics ramrod %d\n",
		le16toh(adapter->fw_stats_req->hdr.drv_stats_counter));

      /* adjust the ramrod to include VF queues statistics */
      qfle3_adjust_stats_req(adapter);

      qfle3_dp_stats(adapter);

      /*fw_stats_req_mapping allocates space for stats requests as well as 
      stats data i.e. adapter->fw_stats_req_size + adapter->fw_stats_data_size */
      /* send FW stats ramrod */
      rc = ecore_sp_post(adapter, RAMROD_CMD_ID_COMMON_STAT_QUERY, 0,
			 adapter->fw_stats_req_mapping,
			 NONE_CONNECTION_TYPE);
      if (rc == 0) {
	 adapter->stats_pending = 1;
      }

}

static void
qfle3_hw_stats_post(struct qfle3_adapter *adapter)
{
   // struct dmae_command *dmae = &adapter->stats_dmae;
   struct dmae_command *dmae = &adapter->stats_dmae;
   vmk_uint32 *stats_comp = QFLE3_SP(adapter, stats_comp);
   int loader_idx;
   vmk_uint32 opcode;

   *stats_comp = DMAE_COMP_VAL;
   if (CHIP_REV_IS_SLOW(adapter)) {
      return;
   }

   /* Update MCP's statistics if possible */
   if (adapter->func_stx) {
      vmk_Memcpy(QFLE3_SP(adapter, func_stats), &adapter->func_stats,
		 sizeof(adapter->func_stats));
   }

   /* loader */
   if (adapter->executer_idx) {
      loader_idx = PMF_DMAE_C(adapter);
      opcode =  qfle3_dmae_opcode(adapter, DMAE_SRC_PCI, DMAE_DST_GRC,
				  VMK_TRUE, DMAE_COMP_GRC);
      QFLE3_DBG(QFLE3_DBG_STATS, "hw_stats_post executer_idx %d loader_idx %d opcode %d \n", adapter->executer_idx, loader_idx, opcode);
      // opcode = qfle3_dmae_opcode_clr_src_reset(opcode);
      opcode = opcode & ~DMAE_COMMAND_SRC_RESET;

      vmk_Memset(dmae, 0, sizeof(struct dmae_command));
      dmae->opcode = opcode;
      dmae->src_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, dmae[0]));
      dmae->src_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, dmae[0]));
      dmae->dst_addr_lo = ((DMAE_REG_CMD_MEM +
			    sizeof(struct dmae_command) *
			    (loader_idx + 1)) >> 2);
      dmae->dst_addr_hi = 0;
      dmae->len = sizeof(struct dmae_command) >> 2;
      if (CHIP_IS_E1(adapter)) {
	 dmae->len--;
      }
      dmae->comp_addr_lo = (dmae_reg_go_c[loader_idx + 1] >> 2);
      dmae->comp_addr_hi = 0;
      dmae->comp_val = 1;

      *stats_comp = 0;
      qfle3_post_dmae(adapter, dmae, loader_idx);
      QFLE3_DBG(QFLE3_DBG_STATS, "hw_stats_post post dmae loader_idx %d opcode %d \n", loader_idx, opcode);
   } else if (adapter->func_stx) {
      *stats_comp = 0;
      qfle3_post_dmae(adapter, dmae, INIT_DMAE_C(adapter));
      QFLE3_DBG(QFLE3_DBG_STATS, "hw_stats_post post dmae adapter->func_stx %d \n", adapter->func_stx);
   }
}

static int
qfle3_stats_comp(struct qfle3_adapter *adapter)
{
   vmk_uint32 *stats_comp = QFLE3_SP(adapter, stats_comp);
   int cnt = 10;

   while (*stats_comp != DMAE_COMP_VAL) {
      if (!cnt) {
	 QFLE3_DBG(QFLE3_DBG_STATS, "Timeout waiting for stats finished\n");
	 break;
      }

      cnt--;
      vmk_DelayUsecs(1000);//To be decided
   }
   QFLE3_DBG(QFLE3_DBG_STATS, "stats_comp %d \n", *stats_comp);
   return (1);
}

/*
 * Statistics service functions
 */

static void
qfle3_stats_pmf_update(struct qfle3_adapter *adapter)
{
   struct dmae_command *dmae;
   vmk_uint32 opcode;
   int loader_idx = PMF_DMAE_C(adapter);
   vmk_uint32 *stats_comp = QFLE3_SP(adapter, stats_comp);

   if (adapter->hw_info.bc_ver <= 0x06001400) {
      /*
       * Bootcode v6.0.21 fixed a GRC timeout that occurs when accessing
       * BRB registers while the BRB block is in reset. The DMA transfer
       * below triggers this issue resulting in the DMAE to stop
       * functioning. Skip this initial stats transfer for old bootcode
       * versions <= 6.0.20.
       */
      return;
   }

   /* sanity */
   if (!adapter->port.pmf || !adapter->port.port_stx) {
      QFLE3_DBG(QFLE3_DBG_STATS, "BUG!\n");
      return;
   }

   adapter->executer_idx = 0;

   opcode = qfle3_dmae_opcode(adapter, DMAE_SRC_GRC, DMAE_DST_PCI, VMK_FALSE, 0);

   dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
   dmae->opcode = qfle3_dmae_opcode_add_comp(opcode, DMAE_COMP_GRC);
   dmae->src_addr_lo = (adapter->port.port_stx >> 2);
   dmae->src_addr_hi = 0;
   dmae->dst_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, port_stats));
   dmae->dst_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, port_stats));
   dmae->len = DMAE_LEN32_RD_MAX;
   dmae->comp_addr_lo = (dmae_reg_go_c[loader_idx] >> 2);
   dmae->comp_addr_hi = 0;
   dmae->comp_val = 1;

   dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
   dmae->opcode = qfle3_dmae_opcode_add_comp(opcode, DMAE_COMP_PCI);
   dmae->src_addr_lo = ((adapter->port.port_stx >> 2) + DMAE_LEN32_RD_MAX);
   dmae->src_addr_hi = 0;
   dmae->dst_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, port_stats) +
			      DMAE_LEN32_RD_MAX * 4);
   dmae->dst_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, port_stats) +
			      DMAE_LEN32_RD_MAX * 4);
   dmae->len = (qfle3_get_port_stats_dma_len(adapter) - DMAE_LEN32_RD_MAX);

   dmae->comp_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, stats_comp));
   dmae->comp_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, stats_comp));
   dmae->comp_val = DMAE_COMP_VAL;

   *stats_comp = 0;
   qfle3_hw_stats_post(adapter);
   qfle3_stats_comp(adapter);
}

static void
qfle3_port_stats_init(struct qfle3_adapter *adapter)
{
   struct dmae_command *dmae;
   int port = SC_PORT(adapter);
   vmk_uint32 opcode;
   int loader_idx = PMF_DMAE_C(adapter);
   vmk_uint32 mac_addr;
   vmk_uint32 *stats_comp = QFLE3_SP(adapter, stats_comp);

   /* sanity */
   if (!adapter->link_vars.link_up || !adapter->port.pmf) {
      QFLE3_DBG(QFLE3_DBG_STATS, "BUG!\n");
      return;
   }

   adapter->executer_idx = 0;

   /* MCP */
   opcode = qfle3_dmae_opcode(adapter, DMAE_SRC_PCI, DMAE_DST_GRC,
			      VMK_TRUE, DMAE_COMP_GRC);

   if (adapter->port.port_stx) {
      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
      dmae->opcode = opcode;
      dmae->src_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, port_stats));
      dmae->src_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, port_stats));
      dmae->dst_addr_lo = adapter->port.port_stx >> 2;
      dmae->dst_addr_hi = 0;
      dmae->len = qfle3_get_port_stats_dma_len(adapter);
      dmae->comp_addr_lo = dmae_reg_go_c[loader_idx] >> 2;
      dmae->comp_addr_hi = 0;
      dmae->comp_val = 1;
   }

   if (adapter->func_stx) {
      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
      dmae->opcode = opcode;
      dmae->src_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, func_stats));
      dmae->src_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, func_stats));
      dmae->dst_addr_lo = (adapter->func_stx >> 2);
      dmae->dst_addr_hi = 0;
      dmae->len = (sizeof(struct host_func_stats) >> 2);
      dmae->comp_addr_lo = (dmae_reg_go_c[loader_idx] >> 2);
      dmae->comp_addr_hi = 0;
      dmae->comp_val = 1;
   }

   /* MAC */
   opcode = qfle3_dmae_opcode(adapter, DMAE_SRC_GRC, DMAE_DST_PCI,
			      VMK_TRUE, DMAE_COMP_GRC);

   /* EMAC is special */
   if (adapter->link_vars.mac_type == ELINK_MAC_TYPE_EMAC) {
      mac_addr = (port ? GRCBASE_EMAC1 : GRCBASE_EMAC0);

      /* EMAC_REG_EMAC_RX_STAT_AC (EMAC_REG_EMAC_RX_STAT_AC_COUNT)*/
      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
      dmae->opcode = opcode;
      dmae->src_addr_lo = (mac_addr + EMAC_REG_EMAC_RX_STAT_AC) >> 2;
      dmae->src_addr_hi = 0;
      dmae->dst_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, mac_stats));
      dmae->dst_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, mac_stats));
      dmae->len = EMAC_REG_EMAC_RX_STAT_AC_COUNT;
      dmae->comp_addr_lo = (dmae_reg_go_c[loader_idx] >> 2);
      dmae->comp_addr_hi = 0;
      dmae->comp_val = 1;

      /* EMAC_REG_EMAC_RX_STAT_AC_28 */
      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
      dmae->opcode = opcode;
      dmae->src_addr_lo = ((mac_addr + EMAC_REG_EMAC_RX_STAT_AC_28) >> 2);
      dmae->src_addr_hi = 0;
      dmae->dst_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, mac_stats) +
				 vmk_offsetof(struct emac_stats,
					      rx_stat_falsecarriererrors));
      dmae->dst_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, mac_stats) +
				 vmk_offsetof(struct emac_stats,
					      rx_stat_falsecarriererrors));
      dmae->len = 1;
      dmae->comp_addr_lo = (dmae_reg_go_c[loader_idx] >> 2);
      dmae->comp_addr_hi = 0;
      dmae->comp_val = 1;

      /* EMAC_REG_EMAC_TX_STAT_AC (EMAC_REG_EMAC_TX_STAT_AC_COUNT)*/
      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
      dmae->opcode = opcode;
      dmae->src_addr_lo = ((mac_addr + EMAC_REG_EMAC_TX_STAT_AC) >> 2);
      dmae->src_addr_hi = 0;
      dmae->dst_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, mac_stats) +
				 vmk_offsetof(struct emac_stats,
					      tx_stat_ifhcoutoctets));
      dmae->dst_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, mac_stats) +
				 vmk_offsetof(struct emac_stats,
					      tx_stat_ifhcoutoctets));
      dmae->len = EMAC_REG_EMAC_TX_STAT_AC_COUNT;
      dmae->comp_addr_lo = (dmae_reg_go_c[loader_idx] >> 2);
      dmae->comp_addr_hi = 0;
      dmae->comp_val = 1;
   } else {
      vmk_uint32 tx_src_addr_lo, rx_src_addr_lo;
      vmk_uint16 rx_len, tx_len;

      /* configure the params according to MAC type */
      switch (adapter->link_vars.mac_type) {
      case ELINK_MAC_TYPE_BMAC:
	 mac_addr = (port) ? NIG_REG_INGRESS_BMAC1_MEM :
	 NIG_REG_INGRESS_BMAC0_MEM;

	 /* BIGMAC_REGISTER_TX_STAT_GTPKT ..
	    BIGMAC_REGISTER_TX_STAT_GTBYT */
	 if (CHIP_IS_E1x(adapter)) {
	    tx_src_addr_lo =
	       ((mac_addr + BIGMAC_REGISTER_TX_STAT_GTPKT) >> 2);
	    tx_len = ((8 + BIGMAC_REGISTER_TX_STAT_GTBYT -
		       BIGMAC_REGISTER_TX_STAT_GTPKT) >> 2);
	    rx_src_addr_lo =
	       ((mac_addr + BIGMAC_REGISTER_RX_STAT_GR64) >> 2);
	    rx_len = ((8 + BIGMAC_REGISTER_RX_STAT_GRIPJ -
		       BIGMAC_REGISTER_RX_STAT_GR64) >> 2);
	 } else {
	    tx_src_addr_lo =
	       ((mac_addr + BIGMAC2_REGISTER_TX_STAT_GTPOK) >> 2);
	    tx_len = ((8 + BIGMAC2_REGISTER_TX_STAT_GTBYT -
		       BIGMAC2_REGISTER_TX_STAT_GTPOK) >> 2);
	    rx_src_addr_lo =
	       ((mac_addr + BIGMAC2_REGISTER_RX_STAT_GR64) >> 2);
	    rx_len = ((8 + BIGMAC2_REGISTER_RX_STAT_GRIPJ -
		       BIGMAC2_REGISTER_RX_STAT_GR64) >> 2);
	 }

	 break;

      case ELINK_MAC_TYPE_UMAC: /* handled by MSTAT */
      case ELINK_MAC_TYPE_XMAC: /* handled by MSTAT */
      default:
	 mac_addr = (port) ? GRCBASE_MSTAT1 : GRCBASE_MSTAT0;
	 tx_src_addr_lo = ((mac_addr + MSTAT_REG_TX_STAT_GTXPOK_LO) >> 2);
	 rx_src_addr_lo = ((mac_addr + MSTAT_REG_RX_STAT_GR64_LO) >> 2);
	 tx_len =
	    (sizeof(adapter->sp->mac_stats.mstat_stats.stats_tx) >> 2);
	 rx_len =
	    (sizeof(adapter->sp->mac_stats.mstat_stats.stats_rx) >> 2);
	 break;
      }

      /* TX stats */
      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
      dmae->opcode = opcode;
      dmae->src_addr_lo = tx_src_addr_lo;
      dmae->src_addr_hi = 0;
      dmae->len = tx_len;
      dmae->dst_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, mac_stats));
      dmae->dst_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, mac_stats));
      dmae->comp_addr_lo = dmae_reg_go_c[loader_idx] >> 2;
      dmae->comp_addr_hi = 0;
      dmae->comp_val = 1;

      /* RX stats */
      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
      dmae->opcode = opcode;
      dmae->src_addr_hi = 0;
      dmae->src_addr_lo = rx_src_addr_lo;
      dmae->dst_addr_lo =
	 U64_LO(QFLE3_SP_MAPPING(adapter, mac_stats) + (tx_len << 2));
      dmae->dst_addr_hi =
	 U64_HI(QFLE3_SP_MAPPING(adapter, mac_stats) + (tx_len << 2));
      dmae->len = rx_len;
      dmae->comp_addr_lo = dmae_reg_go_c[loader_idx] >> 2;
      dmae->comp_addr_hi = 0;
      dmae->comp_val = 1;
   }

   /* NIG */
   if (!CHIP_IS_E3(adapter)) {
      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
      dmae->opcode = opcode;
      dmae->src_addr_lo =
	 (port ? NIG_REG_STAT1_EGRESS_MAC_PKT0 :
	  NIG_REG_STAT0_EGRESS_MAC_PKT0) >> 2;
      dmae->src_addr_hi = 0;
      dmae->dst_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, nig_stats) +
				 vmk_offsetof(struct nig_stats,
					      egress_mac_pkt0_lo));
      dmae->dst_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, nig_stats) +
				 vmk_offsetof(struct nig_stats,
					      egress_mac_pkt0_lo));
      dmae->len = ((2 * sizeof(vmk_uint32)) >> 2);
      dmae->comp_addr_lo = (dmae_reg_go_c[loader_idx] >> 2);
      dmae->comp_addr_hi = 0;
      dmae->comp_val = 1;

      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
      dmae->opcode = opcode;
      dmae->src_addr_lo =
	 (port ? NIG_REG_STAT1_EGRESS_MAC_PKT1 :
	  NIG_REG_STAT0_EGRESS_MAC_PKT1) >> 2;
      dmae->src_addr_hi = 0;
      dmae->dst_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, nig_stats) +
				 vmk_offsetof(struct nig_stats,
					      egress_mac_pkt1_lo));
      dmae->dst_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, nig_stats) +
				 vmk_offsetof(struct nig_stats,
					      egress_mac_pkt1_lo));
      dmae->len = ((2 * sizeof(vmk_uint32)) >> 2);
      dmae->comp_addr_lo = (dmae_reg_go_c[loader_idx] >> 2);
      dmae->comp_addr_hi = 0;
      dmae->comp_val = 1;
   }

   dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
   dmae->opcode = qfle3_dmae_opcode(adapter, DMAE_SRC_GRC, DMAE_DST_PCI,
				    VMK_TRUE, DMAE_COMP_PCI);
   dmae->src_addr_lo =
      (port ? NIG_REG_STAT1_BRB_DISCARD :
       NIG_REG_STAT0_BRB_DISCARD) >> 2;
   dmae->src_addr_hi = 0;
   dmae->dst_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, nig_stats));
   dmae->dst_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, nig_stats));
   dmae->len = (sizeof(struct nig_stats) - 4*sizeof(vmk_uint32)) >> 2;

   dmae->comp_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, stats_comp));
   dmae->comp_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, stats_comp));
   dmae->comp_val = DMAE_COMP_VAL;

   *stats_comp = 0;
}

static void
qfle3_func_stats_init(struct qfle3_adapter *adapter)
{
   struct dmae_command *dmae = &adapter->stats_dmae;
   vmk_uint32 *stats_comp = QFLE3_SP(adapter, stats_comp);

   /* sanity */
   if (!adapter->func_stx) {
      QFLE3_DBG(QFLE3_DBG_STATS, "BUG!\n");
      return;
   }

   adapter->executer_idx = 0;
   vmk_Memset(dmae, 0, sizeof(struct dmae_command));

   dmae->opcode = qfle3_dmae_opcode(adapter, DMAE_SRC_PCI, DMAE_DST_GRC,
				    VMK_TRUE, DMAE_COMP_PCI);
   dmae->src_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, func_stats));
   dmae->src_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, func_stats));
   dmae->dst_addr_lo = (adapter->func_stx >> 2);
   dmae->dst_addr_hi = 0;
   dmae->len = (sizeof(struct host_func_stats) >> 2);
   dmae->comp_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, stats_comp));
   dmae->comp_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, stats_comp));
   dmae->comp_val = DMAE_COMP_VAL;

   *stats_comp = 0;
}

static void
qfle3_stats_start(struct qfle3_adapter *adapter)
{
   /*
    * VFs travel through here as part of the statistics FSM, but no action
    * is required
    */
//Todo, IS_VF not defined now

   /* if (IS_VF(adapter)) { */
   /*     return; */
   /* } */

   if (adapter->port.pmf) {
      QFLE3_DBG(QFLE3_DBG_STATS,"adapter->port.pmf %d\n", adapter->port.pmf);
      qfle3_port_stats_init(adapter);
   }

   else if (adapter->func_stx) {
      qfle3_func_stats_init(adapter);
      QFLE3_DBG(QFLE3_DBG_STATS,"adapter->func_stx %d\n", adapter->func_stx);
   }

   qfle3_hw_stats_post(adapter);
   qfle3_storm_stats_post(adapter);
}

static void
qfle3_stats_pmf_start(struct qfle3_adapter *adapter)
{
   qfle3_stats_comp(adapter);
   qfle3_stats_pmf_update(adapter);
   qfle3_stats_start(adapter);
}

static void
qfle3_stats_restart(struct qfle3_adapter *adapter)
{
   /*
    * VFs travel through here as part of the statistics FSM, but no action
    * is required
    */
   /* if (IS_VF(adapter)) { */
   /*     return; */
   /* } */

   qfle3_stats_comp(adapter);
   qfle3_stats_start(adapter);
}

static void
qfle3_bmac_stats_update(struct qfle3_adapter *adapter)
{
   struct host_port_stats *pstats = QFLE3_SP(adapter, port_stats);
   struct qfle3_eth_stats *estats = &adapter->eth_stats;
   struct {
      vmk_uint32 lo;
      vmk_uint32 hi;
   } diff;

   if (CHIP_IS_E1x(adapter)) {
      struct bmac1_stats *new = QFLE3_SP(adapter, mac_stats.bmac1_stats);

      /* the macros below will use "bmac1_stats" type */
      UPDATE_STAT64(rx_stat_grerb, rx_stat_ifhcinbadoctets);
      UPDATE_STAT64(rx_stat_grfcs, rx_stat_dot3statsfcserrors);
      UPDATE_STAT64(rx_stat_grund, rx_stat_etherstatsundersizepkts);
      UPDATE_STAT64(rx_stat_grovr, rx_stat_dot3statsframestoolong);
      UPDATE_STAT64(rx_stat_grfrg, rx_stat_etherstatsfragments);
      UPDATE_STAT64(rx_stat_grjbr, rx_stat_etherstatsjabbers);
      UPDATE_STAT64(rx_stat_grxcf, rx_stat_maccontrolframesreceived);
      UPDATE_STAT64(rx_stat_grxpf, rx_stat_xoffstateentered);
      UPDATE_STAT64(rx_stat_grxpf, rx_stat_mac_xpf);

      UPDATE_STAT64(tx_stat_gtxpf, tx_stat_outxoffsent);
      UPDATE_STAT64(tx_stat_gtxpf, tx_stat_flowcontroldone);
      UPDATE_STAT64(tx_stat_gt64, tx_stat_etherstatspkts64octets);
      UPDATE_STAT64(tx_stat_gt127,
		    tx_stat_etherstatspkts65octetsto127octets);
      UPDATE_STAT64(tx_stat_gt255,
		    tx_stat_etherstatspkts128octetsto255octets);
      UPDATE_STAT64(tx_stat_gt511,
		    tx_stat_etherstatspkts256octetsto511octets);
      UPDATE_STAT64(tx_stat_gt1023,
		    tx_stat_etherstatspkts512octetsto1023octets);
      UPDATE_STAT64(tx_stat_gt1518,
		    tx_stat_etherstatspkts1024octetsto1522octets);
      UPDATE_STAT64(tx_stat_gt2047, tx_stat_mac_2047);
      UPDATE_STAT64(tx_stat_gt4095, tx_stat_mac_4095);
      UPDATE_STAT64(tx_stat_gt9216, tx_stat_mac_9216);
      UPDATE_STAT64(tx_stat_gt16383, tx_stat_mac_16383);
      UPDATE_STAT64(tx_stat_gterr,
		    tx_stat_dot3statsinternalmactransmiterrors);
      UPDATE_STAT64(tx_stat_gtufl, tx_stat_mac_ufl);
   } else {
      struct bmac2_stats *new = QFLE3_SP(adapter, mac_stats.bmac2_stats);
      struct qfle3_fw_port_stats_old *fwstats = &adapter->fw_stats_old;

      /* the macros below will use "bmac2_stats" type */
      QFLE3_DBG(QFLE3_DBG_STATS,"bmac_stats_update using bmac2_stats\n");
      UPDATE_STAT64(rx_stat_grerb, rx_stat_ifhcinbadoctets);
      UPDATE_STAT64(rx_stat_grfcs, rx_stat_dot3statsfcserrors);
      UPDATE_STAT64(rx_stat_grund, rx_stat_etherstatsundersizepkts);
      UPDATE_STAT64(rx_stat_grovr, rx_stat_dot3statsframestoolong);
      UPDATE_STAT64(rx_stat_grfrg, rx_stat_etherstatsfragments);
      UPDATE_STAT64(rx_stat_grjbr, rx_stat_etherstatsjabbers);
      UPDATE_STAT64(rx_stat_grxcf, rx_stat_maccontrolframesreceived);
      UPDATE_STAT64(rx_stat_grxpf, rx_stat_xoffstateentered);
      UPDATE_STAT64(rx_stat_grxpf, rx_stat_mac_xpf);
      UPDATE_STAT64(tx_stat_gtxpf, tx_stat_outxoffsent);
      UPDATE_STAT64(tx_stat_gtxpf, tx_stat_flowcontroldone);
      UPDATE_STAT64(tx_stat_gt64, tx_stat_etherstatspkts64octets);
      UPDATE_STAT64(tx_stat_gt127,
		    tx_stat_etherstatspkts65octetsto127octets);
      UPDATE_STAT64(tx_stat_gt255,
		    tx_stat_etherstatspkts128octetsto255octets);
      UPDATE_STAT64(tx_stat_gt511,
		    tx_stat_etherstatspkts256octetsto511octets);
      UPDATE_STAT64(tx_stat_gt1023,
		    tx_stat_etherstatspkts512octetsto1023octets);
      UPDATE_STAT64(tx_stat_gt1518,
		    tx_stat_etherstatspkts1024octetsto1522octets);
      UPDATE_STAT64(tx_stat_gt2047, tx_stat_mac_2047);
      UPDATE_STAT64(tx_stat_gt4095, tx_stat_mac_4095);
      UPDATE_STAT64(tx_stat_gt9216, tx_stat_mac_9216);
      UPDATE_STAT64(tx_stat_gt16383, tx_stat_mac_16383);
      UPDATE_STAT64(tx_stat_gterr,
		    tx_stat_dot3statsinternalmactransmiterrors);
      UPDATE_STAT64(tx_stat_gtufl, tx_stat_mac_ufl);

      /* collect PFC stats */
      pstats->pfc_frames_tx_hi = new->tx_stat_gtpp_hi;
      pstats->pfc_frames_tx_lo = new->tx_stat_gtpp_lo;
      ADD_64(pstats->pfc_frames_tx_hi, fwstats->pfc_frames_tx_hi,
	     pstats->pfc_frames_tx_lo, fwstats->pfc_frames_tx_lo);

      pstats->pfc_frames_rx_hi = new->rx_stat_grpp_hi;
      pstats->pfc_frames_rx_lo = new->rx_stat_grpp_lo;
      ADD_64(pstats->pfc_frames_rx_hi, fwstats->pfc_frames_rx_hi,
	     pstats->pfc_frames_rx_lo, fwstats->pfc_frames_rx_lo);
   }

   estats->pause_frames_received_hi = pstats->mac_stx[1].rx_stat_mac_xpf_hi;
   estats->pause_frames_received_lo = pstats->mac_stx[1].rx_stat_mac_xpf_lo;

   estats->pause_frames_sent_hi = pstats->mac_stx[1].tx_stat_outxoffsent_hi;
   estats->pause_frames_sent_lo = pstats->mac_stx[1].tx_stat_outxoffsent_lo;

   estats->pfc_frames_received_hi = pstats->pfc_frames_rx_hi;
   estats->pfc_frames_received_lo = pstats->pfc_frames_rx_lo;
   estats->pfc_frames_sent_hi = pstats->pfc_frames_tx_hi;
   estats->pfc_frames_sent_lo = pstats->pfc_frames_tx_lo;
}

static void
qfle3_print_adapter_eth_stats(struct qfle3_adapter *adapter)
{
   vmk_uint32 offset;
   struct qfle3_eth_stats *estats = &adapter->eth_stats;
   QFLE3_DBG(QFLE3_DBG_STATS, "print the adpater level qfle3_eth_stats::\n");
   for (offset = 0; offset < sizeof(struct qfle3_eth_stats); offset += 4){
      QFLE3_DBG(QFLE3_DBG_STATS,"%d",*(vmk_uint32*)((void *)estats + offset));
   }

}
static void
qfle3_mstat_stats_update(struct qfle3_adapter *adapter)
{
   struct host_port_stats *pstats = QFLE3_SP(adapter, port_stats);
   struct qfle3_eth_stats *estats = &adapter->eth_stats;
   struct mstat_stats *new = QFLE3_SP(adapter, mac_stats.mstat_stats);
   QFLE3_DBG(QFLE3_DBG_STATS,"mstat_stats_update\n");
   ADD_STAT64(stats_rx.rx_grerb, rx_stat_ifhcinbadoctets);
   ADD_STAT64(stats_rx.rx_grfcs, rx_stat_dot3statsfcserrors);
   ADD_STAT64(stats_rx.rx_grund, rx_stat_etherstatsundersizepkts);
   ADD_STAT64(stats_rx.rx_grovr, rx_stat_dot3statsframestoolong);
   ADD_STAT64(stats_rx.rx_grfrg, rx_stat_etherstatsfragments);
   ADD_STAT64(stats_rx.rx_grxcf, rx_stat_maccontrolframesreceived);
   ADD_STAT64(stats_rx.rx_grxpf, rx_stat_xoffstateentered);
   ADD_STAT64(stats_rx.rx_grxpf, rx_stat_mac_xpf);
   ADD_STAT64(stats_tx.tx_gtxpf, tx_stat_outxoffsent);
   ADD_STAT64(stats_tx.tx_gtxpf, tx_stat_flowcontroldone);

   /* collect pfc stats */
   ADD_64(pstats->pfc_frames_tx_hi, new->stats_tx.tx_gtxpp_hi,
	  pstats->pfc_frames_tx_lo, new->stats_tx.tx_gtxpp_lo);
   ADD_64(pstats->pfc_frames_rx_hi, new->stats_rx.rx_grxpp_hi,
	  pstats->pfc_frames_rx_lo, new->stats_rx.rx_grxpp_lo);

   ADD_STAT64(stats_tx.tx_gt64, tx_stat_etherstatspkts64octets);
   ADD_STAT64(stats_tx.tx_gt127, tx_stat_etherstatspkts65octetsto127octets);
   ADD_STAT64(stats_tx.tx_gt255, tx_stat_etherstatspkts128octetsto255octets);
   ADD_STAT64(stats_tx.tx_gt511, tx_stat_etherstatspkts256octetsto511octets);
   ADD_STAT64(stats_tx.tx_gt1023,
	      tx_stat_etherstatspkts512octetsto1023octets);
   ADD_STAT64(stats_tx.tx_gt1518,
	      tx_stat_etherstatspkts1024octetsto1522octets);
   ADD_STAT64(stats_tx.tx_gt2047, tx_stat_mac_2047);

   ADD_STAT64(stats_tx.tx_gt4095, tx_stat_mac_4095);
   ADD_STAT64(stats_tx.tx_gt9216, tx_stat_mac_9216);
   ADD_STAT64(stats_tx.tx_gt16383, tx_stat_mac_16383);

   ADD_STAT64(stats_tx.tx_gterr, tx_stat_dot3statsinternalmactransmiterrors);
   ADD_STAT64(stats_tx.tx_gtufl, tx_stat_mac_ufl);

   estats->etherstatspkts1024octetsto1522octets_hi =
      pstats->mac_stx[1].tx_stat_etherstatspkts1024octetsto1522octets_hi;
   estats->etherstatspkts1024octetsto1522octets_lo =
      pstats->mac_stx[1].tx_stat_etherstatspkts1024octetsto1522octets_lo;

   estats->etherstatspktsover1522octets_hi =
      pstats->mac_stx[1].tx_stat_mac_2047_hi;
   estats->etherstatspktsover1522octets_lo =
      pstats->mac_stx[1].tx_stat_mac_2047_lo;

   ADD_64(estats->etherstatspktsover1522octets_hi,
	  pstats->mac_stx[1].tx_stat_mac_4095_hi,
	  estats->etherstatspktsover1522octets_lo,
	  pstats->mac_stx[1].tx_stat_mac_4095_lo);

   ADD_64(estats->etherstatspktsover1522octets_hi,
	  pstats->mac_stx[1].tx_stat_mac_9216_hi,
	  estats->etherstatspktsover1522octets_lo,
	  pstats->mac_stx[1].tx_stat_mac_9216_lo);

   ADD_64(estats->etherstatspktsover1522octets_hi,
	  pstats->mac_stx[1].tx_stat_mac_16383_hi,
	  estats->etherstatspktsover1522octets_lo,
	  pstats->mac_stx[1].tx_stat_mac_16383_lo);

   estats->pause_frames_received_hi = pstats->mac_stx[1].rx_stat_mac_xpf_hi;
   estats->pause_frames_received_lo = pstats->mac_stx[1].rx_stat_mac_xpf_lo;

   estats->pause_frames_sent_hi = pstats->mac_stx[1].tx_stat_outxoffsent_hi;
   estats->pause_frames_sent_lo = pstats->mac_stx[1].tx_stat_outxoffsent_lo;

   estats->pfc_frames_received_hi = pstats->pfc_frames_rx_hi;
   estats->pfc_frames_received_lo = pstats->pfc_frames_rx_lo;
   estats->pfc_frames_sent_hi = pstats->pfc_frames_tx_hi;
   estats->pfc_frames_sent_lo = pstats->pfc_frames_tx_lo;

   qfle3_print_adapter_eth_stats(adapter);
}

static void
qfle3_emac_stats_update(struct qfle3_adapter *adapter)
{
   struct emac_stats *new = QFLE3_SP(adapter, mac_stats.emac_stats);
   struct host_port_stats *pstats = QFLE3_SP(adapter, port_stats);
   struct qfle3_eth_stats *estats = &adapter->eth_stats;
   QFLE3_DBG(QFLE3_DBG_STATS,"emac_stats_update\n");

   UPDATE_EXTEND_STAT(rx_stat_ifhcinbadoctets);
   UPDATE_EXTEND_STAT(tx_stat_ifhcoutbadoctets);
   UPDATE_EXTEND_STAT(rx_stat_dot3statsfcserrors);
   UPDATE_EXTEND_STAT(rx_stat_dot3statsalignmenterrors);
   UPDATE_EXTEND_STAT(rx_stat_dot3statscarriersenseerrors);
   UPDATE_EXTEND_STAT(rx_stat_falsecarriererrors);
   UPDATE_EXTEND_STAT(rx_stat_etherstatsundersizepkts);
   UPDATE_EXTEND_STAT(rx_stat_dot3statsframestoolong);
   UPDATE_EXTEND_STAT(rx_stat_etherstatsfragments);
   UPDATE_EXTEND_STAT(rx_stat_etherstatsjabbers);
   UPDATE_EXTEND_STAT(rx_stat_maccontrolframesreceived);
   UPDATE_EXTEND_STAT(rx_stat_xoffstateentered);
   UPDATE_EXTEND_STAT(rx_stat_xonpauseframesreceived);
   UPDATE_EXTEND_STAT(rx_stat_xoffpauseframesreceived);
   UPDATE_EXTEND_STAT(tx_stat_outxonsent);
   UPDATE_EXTEND_STAT(tx_stat_outxoffsent);
   UPDATE_EXTEND_STAT(tx_stat_flowcontroldone);
   UPDATE_EXTEND_STAT(tx_stat_etherstatscollisions);
   UPDATE_EXTEND_STAT(tx_stat_dot3statssinglecollisionframes);
   UPDATE_EXTEND_STAT(tx_stat_dot3statsmultiplecollisionframes);
   UPDATE_EXTEND_STAT(tx_stat_dot3statsdeferredtransmissions);
   UPDATE_EXTEND_STAT(tx_stat_dot3statsexcessivecollisions);
   UPDATE_EXTEND_STAT(tx_stat_dot3statslatecollisions);
   UPDATE_EXTEND_STAT(tx_stat_etherstatspkts64octets);
   UPDATE_EXTEND_STAT(tx_stat_etherstatspkts65octetsto127octets);
   UPDATE_EXTEND_STAT(tx_stat_etherstatspkts128octetsto255octets);
   UPDATE_EXTEND_STAT(tx_stat_etherstatspkts256octetsto511octets);
   UPDATE_EXTEND_STAT(tx_stat_etherstatspkts512octetsto1023octets);
   UPDATE_EXTEND_STAT(tx_stat_etherstatspkts1024octetsto1522octets);
   UPDATE_EXTEND_STAT(tx_stat_etherstatspktsover1522octets);
   UPDATE_EXTEND_STAT(tx_stat_dot3statsinternalmactransmiterrors);

   estats->pause_frames_received_hi =
      pstats->mac_stx[1].rx_stat_xonpauseframesreceived_hi;
   estats->pause_frames_received_lo =
      pstats->mac_stx[1].rx_stat_xonpauseframesreceived_lo;
   ADD_64(estats->pause_frames_received_hi,
	  pstats->mac_stx[1].rx_stat_xoffpauseframesreceived_hi,
	  estats->pause_frames_received_lo,
	  pstats->mac_stx[1].rx_stat_xoffpauseframesreceived_lo);

   estats->pause_frames_sent_hi =
      pstats->mac_stx[1].tx_stat_outxonsent_hi;
   estats->pause_frames_sent_lo =
      pstats->mac_stx[1].tx_stat_outxonsent_lo;
   ADD_64(estats->pause_frames_sent_hi,
	  pstats->mac_stx[1].tx_stat_outxoffsent_hi,
	  estats->pause_frames_sent_lo,
	  pstats->mac_stx[1].tx_stat_outxoffsent_lo);
}

static int
qfle3_hw_stats_update(struct qfle3_adapter *adapter)
{
   struct nig_stats *new = QFLE3_SP(adapter, nig_stats);
   struct nig_stats *old = &(adapter->port.old_nig_stats);
   struct host_port_stats *pstats = QFLE3_SP(adapter, port_stats);
   struct qfle3_eth_stats *estats = &adapter->eth_stats;
   vmk_uint32 lpi_reg, nig_timer_max;
   struct {
      vmk_uint32 lo;
      vmk_uint32 hi;
   } diff;
   QFLE3_DBG(QFLE3_DBG_STATS, "nig_stats new %p old %p host_port_stats pstats %p qfle3_eth_stats %p \n", new, old, pstats, estats);
   switch (adapter->link_vars.mac_type) {
   case ELINK_MAC_TYPE_BMAC:
      qfle3_bmac_stats_update(adapter);
      break;

   case ELINK_MAC_TYPE_EMAC:
      qfle3_emac_stats_update(adapter);
      break;

   case ELINK_MAC_TYPE_UMAC:
   case ELINK_MAC_TYPE_XMAC:
      qfle3_mstat_stats_update(adapter);
      break;

   case ELINK_MAC_TYPE_NONE: /* unreached */
      QFLE3_DBG(QFLE3_DBG_STATS,"stats updated by DMAE but no MAC active\n");
      return (-1);

   default: /* unreached */
      QFLE3_DBG(QFLE3_DBG_STATS, "stats update failed, unknown MAC type\n");
   }

   ADD_EXTEND_64(pstats->brb_drop_hi, pstats->brb_drop_lo,
		 new->brb_discard - old->brb_discard);
   ADD_EXTEND_64(estats->brb_truncate_hi, estats->brb_truncate_lo,
		 new->brb_truncate - old->brb_truncate);

   if (!CHIP_IS_E3(adapter)) {
      UPDATE_STAT64_NIG(egress_mac_pkt0,
			etherstatspkts1024octetsto1522octets);
      UPDATE_STAT64_NIG(egress_mac_pkt1,
			etherstatspktsover1522octets);
   }

   vmk_Memcpy(old, new, sizeof(struct nig_stats));

   vmk_Memcpy(&(estats->rx_stat_ifhcinbadoctets_hi), &(pstats->mac_stx[1]),
	      sizeof(struct mac_stx));
   estats->brb_drop_hi = pstats->brb_drop_hi;
   estats->brb_drop_lo = pstats->brb_drop_lo;

   pstats->host_port_stats_counter++;

   if (CHIP_IS_E3(adapter)) {
      lpi_reg = (SC_PORT(adapter)) ?
	 MISC_REG_CPMU_LP_SM_ENT_CNT_P1 :
	 MISC_REG_CPMU_LP_SM_ENT_CNT_P0;
      estats->eee_tx_lpi += REG_RD(adapter, lpi_reg);
   }

   if (!QFLE3_NOMCP(adapter)) {
      nig_timer_max = SHMEM_RD(adapter, port_mb[SC_PORT(adapter)].stat_nig_timer);
      if (nig_timer_max != estats->nig_timer_max) {
	 estats->nig_timer_max = nig_timer_max;
	 QFLE3_DBG(QFLE3_DBG_STATS, "invalid NIG timer max (%u)\n",
		   estats->nig_timer_max);
      }
   }

   return (0);
}

static int
qfle3_storm_stats_validate_counters(struct qfle3_adapter *adapter)
{
   struct stats_counter *counters = &adapter->fw_stats_data->storm_counters;
   vmk_uint16 cur_stats_counter;

   /*
    * Make sure we use the value of the counter
    * used for sending the last stats ramrod.
    */
   cur_stats_counter = (adapter->stats_counter - 1);

   QFLE3_DBG(QFLE3_DBG_STATS,"xstats_counter %d\n", le16toh(counters->xstats_counter));
   /* are storm stats valid? */
   if (le16toh(counters->xstats_counter) != cur_stats_counter) {
      QFLE3_DBG(QFLE3_DBG_STATS,
		"stats not updated by xstorm, "
		"counter 0x%x != stats_counter 0x%x\n",
		le16toh(counters->xstats_counter), adapter->stats_counter);
      return VMK_BUSY;
   }

   if (le16toh(counters->ustats_counter) != cur_stats_counter) {
      QFLE3_DBG(QFLE3_DBG_STATS,
		"stats not updated by ustorm, "
		"counter 0x%x != stats_counter 0x%x\n",
		le16toh(counters->ustats_counter), adapter->stats_counter);
      return VMK_BUSY;
   }

   if (le16toh(counters->cstats_counter) != cur_stats_counter) {
      QFLE3_DBG(QFLE3_DBG_STATS,
		"stats not updated by cstorm, "
		"counter 0x%x != stats_counter 0x%x\n",
		le16toh(counters->cstats_counter), adapter->stats_counter);
      return VMK_BUSY;
   }

   if (le16toh(counters->tstats_counter) != cur_stats_counter) {
      QFLE3_DBG(QFLE3_DBG_STATS,
		"stats not updated by tstorm, "
		"counter 0x%x != stats_counter 0x%x\n",
		le16toh(counters->tstats_counter), adapter->stats_counter);
      return VMK_BUSY;
   }

   return (0);
}

static int
qfle3_storm_stats_update(struct qfle3_adapter *adapter)
{
   struct tstorm_per_port_stats *tport =
      &adapter->fw_stats_data->port.tstorm_port_statistics;
   struct tstorm_per_pf_stats *tfunc =
      &adapter->fw_stats_data->pf.tstorm_pf_statistics;
   struct host_func_stats *fstats = &adapter->func_stats;
   struct qfle3_eth_stats *estats = &adapter->eth_stats;
   struct qfle3_eth_stats_old *estats_old = &adapter->eth_stats_old;
   int i;

   /* vfs stat counter is managed by pf */
   /* if (IS_PF(adapter) && qfle3_storm_stats_validate_counters(adapter)) { */
   /*     return VMK_BUSY; */
   /* } */
   if (qfle3_storm_stats_validate_counters(adapter)) {
      return VMK_BUSY;
   }
   estats->error_bytes_received_hi = 0;
   estats->error_bytes_received_lo = 0;
// num_queues Todo
   for (i = 0; i < QFLE3_NUM_ETH_QUEUES(adapter); i++) {
      struct qfle3_fastpath *fp = &adapter->fp[i];
      struct tstorm_per_queue_stats *tclient =
	 &adapter->fw_stats_data->queue_stats[i].tstorm_queue_statistics;
      struct tstorm_per_queue_stats *old_tclient = &fp->old_tclient;
      struct ustorm_per_queue_stats *uclient =
	 &adapter->fw_stats_data->queue_stats[i].ustorm_queue_statistics;
      struct ustorm_per_queue_stats *old_uclient = &fp->old_uclient;
      struct xstorm_per_queue_stats *xclient =
	 &adapter->fw_stats_data->queue_stats[i].xstorm_queue_statistics;
      struct xstorm_per_queue_stats *old_xclient = &fp->old_xclient;
      struct qfle3_eth_q_stats *qstats = &fp->eth_q_stats;
      struct qfle3_eth_q_stats_old *qstats_old = &fp->eth_q_stats_old;

      vmk_uint32 diff;

      QFLE3_DBG(QFLE3_DBG_STATS,
		"queue[%d]: ucast_sent 0x%x bcast_sent 0x%x mcast_sent 0x%x\n",
		i, xclient->ucast_pkts_sent, xclient->bcast_pkts_sent,
		xclient->mcast_pkts_sent);

      QFLE3_DBG(QFLE3_DBG_STATS,"---------------\n");

      QFLE3_DBG(QFLE3_DBG_STATS,
		"queue[%d]:rcv_bcast_bytes %d rcv_mcast_bytes %d rcv_ucast_bytes %d\n",
		i, le32toh(tclient->rcv_bcast_bytes.lo), le32toh(tclient->rcv_mcast_bytes.lo),
		le32toh(tclient->rcv_ucast_bytes.lo));

      QFLE3_DBG(QFLE3_DBG_STATS,"---------------\n");

      UPDATE_QSTAT(tclient->rcv_bcast_bytes,
		   total_broadcast_bytes_received);
      UPDATE_QSTAT(tclient->rcv_mcast_bytes,
		   total_multicast_bytes_received);
      UPDATE_QSTAT(tclient->rcv_ucast_bytes,
		   total_unicast_bytes_received);

      /*
       * sum to total_bytes_received all
       * unicast/multicast/broadcast
       */
      qstats->total_bytes_received_hi =
	 qstats->total_broadcast_bytes_received_hi;
      qstats->total_bytes_received_lo =
	 qstats->total_broadcast_bytes_received_lo;

      ADD_64(qstats->total_bytes_received_hi,
	     qstats->total_multicast_bytes_received_hi,
	     qstats->total_bytes_received_lo,
	     qstats->total_multicast_bytes_received_lo);

      ADD_64(qstats->total_bytes_received_hi,
	     qstats->total_unicast_bytes_received_hi,
	     qstats->total_bytes_received_lo,
	     qstats->total_unicast_bytes_received_lo);

      qstats->valid_bytes_received_hi = qstats->total_bytes_received_hi;
      qstats->valid_bytes_received_lo = qstats->total_bytes_received_lo;

      UPDATE_EXTEND_TSTAT(rcv_ucast_pkts, total_unicast_packets_received);
      UPDATE_EXTEND_TSTAT(rcv_mcast_pkts, total_multicast_packets_received);
      UPDATE_EXTEND_TSTAT(rcv_bcast_pkts, total_broadcast_packets_received);
      UPDATE_EXTEND_E_TSTAT(pkts_too_big_discard,
			    etherstatsoverrsizepkts, 32);
      UPDATE_EXTEND_E_TSTAT(no_buff_discard, no_buff_discard, 16);

      SUB_EXTEND_USTAT(ucast_no_buff_pkts, total_unicast_packets_received);
      SUB_EXTEND_USTAT(mcast_no_buff_pkts,
		       total_multicast_packets_received);
      SUB_EXTEND_USTAT(bcast_no_buff_pkts,
		       total_broadcast_packets_received);
      UPDATE_EXTEND_E_USTAT(ucast_no_buff_pkts, no_buff_discard);
      UPDATE_EXTEND_E_USTAT(mcast_no_buff_pkts, no_buff_discard);
      UPDATE_EXTEND_E_USTAT(bcast_no_buff_pkts, no_buff_discard);

      UPDATE_QSTAT(xclient->bcast_bytes_sent,
		   total_broadcast_bytes_transmitted);
      UPDATE_QSTAT(xclient->mcast_bytes_sent,
		   total_multicast_bytes_transmitted);
      UPDATE_QSTAT(xclient->ucast_bytes_sent,
		   total_unicast_bytes_transmitted);

      /*
       * sum to total_bytes_transmitted all
       * unicast/multicast/broadcast
       */
      qstats->total_bytes_transmitted_hi =
	 qstats->total_unicast_bytes_transmitted_hi;
      qstats->total_bytes_transmitted_lo =
	 qstats->total_unicast_bytes_transmitted_lo;

      ADD_64(qstats->total_bytes_transmitted_hi,
	     qstats->total_broadcast_bytes_transmitted_hi,
	     qstats->total_bytes_transmitted_lo,
	     qstats->total_broadcast_bytes_transmitted_lo);

      ADD_64(qstats->total_bytes_transmitted_hi,
	     qstats->total_multicast_bytes_transmitted_hi,
	     qstats->total_bytes_transmitted_lo,
	     qstats->total_multicast_bytes_transmitted_lo);

      UPDATE_EXTEND_XSTAT(ucast_pkts_sent,
			  total_unicast_packets_transmitted);
      UPDATE_EXTEND_XSTAT(mcast_pkts_sent,
			  total_multicast_packets_transmitted);
      UPDATE_EXTEND_XSTAT(bcast_pkts_sent,
			  total_broadcast_packets_transmitted);

      UPDATE_EXTEND_TSTAT(checksum_discard,
			  total_packets_received_checksum_discarded);
      UPDATE_EXTEND_TSTAT(ttl0_discard,
			  total_packets_received_ttl0_discarded);

      UPDATE_EXTEND_XSTAT(error_drop_pkts,
			  total_transmitted_dropped_packets_error);

      /* TPA aggregations completed */
      UPDATE_EXTEND_E_USTAT(coalesced_events, total_tpa_aggregations);
      /* Number of network frames aggregated by TPA */
      UPDATE_EXTEND_E_USTAT(coalesced_pkts, total_tpa_aggregated_frames);
      /* Total number of bytes in completed TPA aggregations */
      UPDATE_QSTAT(uclient->coalesced_bytes, total_tpa_bytes);

      UPDATE_ESTAT_QSTAT_64(total_tpa_bytes);

      UPDATE_FSTAT_QSTAT(total_bytes_received);
      UPDATE_FSTAT_QSTAT(total_bytes_transmitted);
      UPDATE_FSTAT_QSTAT(total_unicast_packets_received);
      UPDATE_FSTAT_QSTAT(total_multicast_packets_received);
      UPDATE_FSTAT_QSTAT(total_broadcast_packets_received);
      UPDATE_FSTAT_QSTAT(total_unicast_packets_transmitted);
      UPDATE_FSTAT_QSTAT(total_multicast_packets_transmitted);
      UPDATE_FSTAT_QSTAT(total_broadcast_packets_transmitted);
      UPDATE_FSTAT_QSTAT(valid_bytes_received);
   }

   ADD_64(estats->total_bytes_received_hi,
	  estats->rx_stat_ifhcinbadoctets_hi,
	  estats->total_bytes_received_lo,
	  estats->rx_stat_ifhcinbadoctets_lo);

   ADD_64_LE(estats->total_bytes_received_hi,
	     tfunc->rcv_error_bytes.hi,
	     estats->total_bytes_received_lo,
	     tfunc->rcv_error_bytes.lo);

   ADD_64_LE(estats->error_bytes_received_hi,
	     tfunc->rcv_error_bytes.hi,
	     estats->error_bytes_received_lo,
	     tfunc->rcv_error_bytes.lo);

   UPDATE_ESTAT(etherstatsoverrsizepkts, rx_stat_dot3statsframestoolong);

   ADD_64(estats->error_bytes_received_hi,
	  estats->rx_stat_ifhcinbadoctets_hi,
	  estats->error_bytes_received_lo,
	  estats->rx_stat_ifhcinbadoctets_lo);

   if (adapter->port.pmf) {
      struct qfle3_fw_port_stats_old *fwstats = &adapter->fw_stats_old;
      UPDATE_FW_STAT(mac_filter_discard);
      UPDATE_FW_STAT(mf_tag_discard);
      UPDATE_FW_STAT(brb_truncate_discard);
      UPDATE_FW_STAT(mac_discard);
   }

   fstats->host_func_stats_start = ++fstats->host_func_stats_end;

   adapter->stats_pending = 0;

   return (0);
}
// Todo uplayer interface
static void
qfle3_net_stats_update(struct qfle3_adapter *adapter)
{
//num_queues ?
   /* for (int i = 0; i < adapter->num_ethqs; i++) */
   /*     if_inc_counter(adapter->ifp, IFCOUNTER_IQDROPS, */
   /*         le32toh(adapter->fp[i].old_tclient.checksum_discard)); */
}
// Todo uplayer interface
vmk_uint64
qfle3_get_counter(struct qfle3_adapter *adapter/*if_t ifp, ift_counter cnt*/)
{
   /* struct qfle3_adapter *adapter; */
   /* struct qfle3_eth_stats *estats; */

   /* sc = if_getsoftc(ifp); */
   /* estats = &adapter->eth_stats; */

   /* switch (cnt) { */
   /* case IFCOUNTER_IPACKETS: */
   /*	return (qfle3_hilo(&estats->total_unicast_packets_received_hi) + */
   /*	    qfle3_hilo(&estats->total_multicast_packets_received_hi) + */
   /*	    qfle3_hilo(&estats->total_broadcast_packets_received_hi)); */
   /* case IFCOUNTER_OPACKETS: */
   /*	return (qfle3_hilo(&estats->total_unicast_packets_transmitted_hi) + */
   /*	    qfle3_hilo(&estats->total_multicast_packets_transmitted_hi) + */
   /*	    qfle3_hilo(&estats->total_broadcast_packets_transmitted_hi)); */
   /* case IFCOUNTER_IBYTES: */
   /*	return (qfle3_hilo(&estats->total_bytes_received_hi)); */
   /* case IFCOUNTER_OBYTES: */
   /*	return (qfle3_hilo(&estats->total_bytes_transmitted_hi)); */
   /* case IFCOUNTER_IERRORS: */
   /*	return (qfle3_hilo(&estats->rx_stat_etherstatsundersizepkts_hi) + */
   /*	    qfle3_hilo(&estats->etherstatsoverrsizepkts_hi) + */
   /*	    qfle3_hilo(&estats->brb_drop_hi) + */
   /*	    qfle3_hilo(&estats->brb_truncate_hi) + */
   /*	    qfle3_hilo(&estats->rx_stat_dot3statsfcserrors_hi) + */
   /*	    qfle3_hilo(&estats->rx_stat_dot3statsalignmenterrors_hi) + */
   /*	    qfle3_hilo(&estats->no_buff_discard_hi)); */
   /* case IFCOUNTER_OERRORS: */
   /*	return (qfle3_hilo(&estats->rx_stat_dot3statscarriersenseerrors_hi) + */
   /*	    qfle3_hilo(&estats->tx_stat_dot3statsinternalmactransmiterrors_hi)); */
   /* case IFCOUNTER_IMCASTS: */
   /*	return (qfle3_hilo(&estats->total_multicast_packets_received_hi)); */
   /* case IFCOUNTER_COLLISIONS: */
   /*	return (qfle3_hilo(&estats->tx_stat_etherstatscollisions_hi) + */
   /*	    qfle3_hilo(&estats->tx_stat_dot3statslatecollisions_hi) + */
   /*	    qfle3_hilo(&estats->tx_stat_dot3statsexcessivecollisions_hi)); */
   /* default: */
   /*	return (if_get_counter_default(ifp, cnt)); */
   /* } */
   return 1;
}

static void
qfle3_drv_stats_update(struct qfle3_adapter *adapter)
{
   struct qfle3_eth_stats *estats = &adapter->eth_stats;
   int i;
//num_queues Todo
   for (i = 0; i < QFLE3_NUM_ETH_QUEUES(adapter); i++) {
      struct qfle3_eth_q_stats *qstats = &adapter->fp[i].eth_q_stats;
      struct qfle3_eth_q_stats_old *qstats_old = &adapter->fp[i].eth_q_stats_old;

      UPDATE_ESTAT_QSTAT(rx_calls);
      UPDATE_ESTAT_QSTAT(rx_pkts);
      UPDATE_ESTAT_QSTAT(rx_tpa_pkts);
      UPDATE_ESTAT_QSTAT(rx_soft_errors);
      UPDATE_ESTAT_QSTAT(rx_hw_csum_errors);
      UPDATE_ESTAT_QSTAT(rx_ofld_frames_csum_ip);
      UPDATE_ESTAT_QSTAT(rx_ofld_frames_csum_tcp_udp);
      UPDATE_ESTAT_QSTAT(rx_budget_reached);
      UPDATE_ESTAT_QSTAT(tx_pkts);
      UPDATE_ESTAT_QSTAT(tx_soft_errors);
      UPDATE_ESTAT_QSTAT(tx_ofld_frames_csum_ip);
      UPDATE_ESTAT_QSTAT(tx_ofld_frames_csum_tcp);
      UPDATE_ESTAT_QSTAT(tx_ofld_frames_csum_udp);
      UPDATE_ESTAT_QSTAT(tx_ofld_frames_lso);
      UPDATE_ESTAT_QSTAT(tx_ofld_frames_lso_hdr_splits);
      UPDATE_ESTAT_QSTAT(tx_encap_failures);
      UPDATE_ESTAT_QSTAT(tx_hw_queue_full);
      UPDATE_ESTAT_QSTAT(tx_hw_max_queue_depth);
      UPDATE_ESTAT_QSTAT(tx_dma_mapping_failure);
      UPDATE_ESTAT_QSTAT(tx_max_drbr_queue_depth);
      UPDATE_ESTAT_QSTAT(tx_window_violation_std);
      UPDATE_ESTAT_QSTAT(tx_window_violation_tso);
      //UPDATE_ESTAT_QSTAT(tx_unsupported_tso_request_ipv6);
      //UPDATE_ESTAT_QSTAT(tx_unsupported_tso_request_not_tcp);
      UPDATE_ESTAT_QSTAT(tx_chain_lost_mbuf);
      UPDATE_ESTAT_QSTAT(tx_frames_deferred);
      UPDATE_ESTAT_QSTAT(tx_queue_xoff);

      /* mbuf driver statistics */
      UPDATE_ESTAT_QSTAT(mbuf_defrag_attempts);
      UPDATE_ESTAT_QSTAT(mbuf_defrag_failures);
      UPDATE_ESTAT_QSTAT(mbuf_rx_bd_alloc_failed);
      UPDATE_ESTAT_QSTAT(mbuf_rx_bd_mapping_failed);
      UPDATE_ESTAT_QSTAT(mbuf_rx_tpa_alloc_failed);
      UPDATE_ESTAT_QSTAT(mbuf_rx_tpa_mapping_failed);
      UPDATE_ESTAT_QSTAT(mbuf_rx_sge_alloc_failed);
      UPDATE_ESTAT_QSTAT(mbuf_rx_sge_mapping_failed);

      /* track the number of allocated mbufs */
      UPDATE_ESTAT_QSTAT(mbuf_alloc_tx);
      UPDATE_ESTAT_QSTAT(mbuf_alloc_rx);
      UPDATE_ESTAT_QSTAT(mbuf_alloc_sge);
      UPDATE_ESTAT_QSTAT(mbuf_alloc_tpa);
   }
}

static vmk_Bool
qfle3_edebug_stats_stopped(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;

   if (SHMEM2_HAS(adapter, edebug_driver_if[1])) {
      val = SHMEM2_RD(adapter, edebug_driver_if[1]);

      if (val == EDEBUG_DRIVER_IF_OP_CODE_DISABLE_STAT) {
	 return (VMK_TRUE);
      }
   }

   return (VMK_FALSE);
}

static void
qfle3_stats_update(struct qfle3_adapter *adapter)
{
   vmk_uint32 *stats_comp = QFLE3_SP(adapter, stats_comp);
   int donot_recover_flag;
   
	donot_recover_flag = vmk_BitVectorTest(adapter->sp_rtnl_state, QFLE3_SP_RTNL_DONOT_RECOVER);
   if (donot_recover_flag) {
      QFLE3_INFO("Hold off on recovery!\n");
      adapter->stats_pending = 0;
   }
   if (qfle3_edebug_stats_stopped(adapter)) {
      QFLE3_DBG(QFLE3_DBG_STATS, "qfle3_edebug_stats_stopped!\n");
      return;
   }
#define IS_PF(x) (x)!=NULL
   if (IS_PF(adapter)) {
      if (*stats_comp != DMAE_COMP_VAL) {
         QFLE3_DBG(QFLE3_DBG_STATS, "*stats_comp != DMAE_COMP_VAL\n");
         goto TIMEOUT;
      }

      if (adapter->port.pmf) {
         qfle3_hw_stats_update(adapter);
      }

      if (qfle3_storm_stats_update(adapter)) {
 TIMEOUT:
         if (adapter->stats_pending++ == STATS_TIMEOUT) {
          /* if (if_getdrvflags(adapter->ifp) & IFF_DRV_RUNNING) { */
          /*	atomic_store_rel_long(&adapter->chip_tq_flags, CHIP_TQ_REINIT); */
          /*	taskqueue_enqueue(adapter->chip_tq, &adapter->chip_tq_task); */
          /* } */
          QFLE3_DBG(QFLE3_DBG_STATS, "adpater->stats_pending %d\n", adapter->stats_pending);
               QFLE3_ERR("stats are not updated for %d times\n", STATS_TIMEOUT);
               QFLE3_STATS_UNLOCK(adapter);
               QFLE3_TRIGGER_TX_TO(adapter, QFLE3_ERR_STATS_TO);
         }
         return;
      }
   } else {
      /*
       * VF doesn't collect HW statistics, and doesn't get completions,
       * performs only update.
       */
      qfle3_storm_stats_update(adapter);
   }

   qfle3_net_stats_update(adapter);
   qfle3_drv_stats_update(adapter);

   /* vf is done */
   /* if (IS_VF(adapter)) { */
   /*     return; */
   /* } */

   qfle3_hw_stats_post(adapter);
   qfle3_storm_stats_post(adapter);

   QFLE3_DBG(QFLE3_DBG_STATS, "qfle3_hw_stats_post and qfle3_storm_stats_post finished!\n");


}

void
qfle3_port_stats_stop(struct qfle3_adapter *adapter)
{
   struct dmae_command *dmae;
   vmk_uint32 opcode;
   int loader_idx = PMF_DMAE_C(adapter);
   vmk_uint32 *stats_comp = QFLE3_SP(adapter, stats_comp);

   adapter->executer_idx = 0;

   opcode = qfle3_dmae_opcode(adapter, DMAE_SRC_PCI, DMAE_DST_GRC, VMK_FALSE, 0);

   if (adapter->port.port_stx) {
      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);

      if (adapter->func_stx) {
	 dmae->opcode = qfle3_dmae_opcode_add_comp(opcode, DMAE_COMP_GRC);
      } else {
	 dmae->opcode = qfle3_dmae_opcode_add_comp(opcode, DMAE_COMP_PCI);
      }

      dmae->src_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, port_stats));
      dmae->src_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, port_stats));
      dmae->dst_addr_lo = adapter->port.port_stx >> 2;
      dmae->dst_addr_hi = 0;
      dmae->len = qfle3_get_port_stats_dma_len(adapter);
      if (adapter->func_stx) {
	 dmae->comp_addr_lo = (dmae_reg_go_c[loader_idx] >> 2);
	 dmae->comp_addr_hi = 0;
	 dmae->comp_val = 1;
      } else {
	 dmae->comp_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, stats_comp));
	 dmae->comp_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, stats_comp));
	 dmae->comp_val = DMAE_COMP_VAL;

	 *stats_comp = 0;
      }
   }

   if (adapter->func_stx) {
      dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
      dmae->opcode = qfle3_dmae_opcode_add_comp(opcode, DMAE_COMP_PCI);
      dmae->src_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, func_stats));
      dmae->src_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, func_stats));
      dmae->dst_addr_lo = (adapter->func_stx >> 2);
      dmae->dst_addr_hi = 0;
      dmae->len = (sizeof(struct host_func_stats) >> 2);
      dmae->comp_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, stats_comp));
      dmae->comp_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, stats_comp));
      dmae->comp_val = DMAE_COMP_VAL;

      *stats_comp = 0;
   }
}

static void
qfle3_stats_stop(struct qfle3_adapter *adapter)
{
   vmk_uint8 update = VMK_FALSE;

   qfle3_stats_comp(adapter);

   if (adapter->port.pmf) {
      update = qfle3_hw_stats_update(adapter) == 0;
   }

   update |= qfle3_storm_stats_update(adapter) == 0;

   if (update) {
      qfle3_net_stats_update(adapter);

      if (adapter->port.pmf) {
	 qfle3_port_stats_stop(adapter);
      }

      qfle3_hw_stats_post(adapter);
      qfle3_stats_comp(adapter);
   }
}

static void
qfle3_stats_do_nothing(struct qfle3_adapter *adapter)
{
   return;
}

static const struct {
   void (*action)(struct qfle3_adapter *adapter);
   enum qfle3_stats_state next_state;
} qfle3_stats_stm[STATS_STATE_MAX][STATS_EVENT_MAX] = {
   {
      /* DISABLED PMF */ { qfle3_stats_pmf_update, STATS_STATE_DISABLED },
      /*      LINK_UP */ { qfle3_stats_start,      STATS_STATE_ENABLED },
      /*      UPDATE  */ { qfle3_stats_do_nothing, STATS_STATE_DISABLED },
      /*      STOP    */ { qfle3_stats_do_nothing, STATS_STATE_DISABLED }
   },
   {
      /* ENABLED  PMF */ { qfle3_stats_pmf_start,  STATS_STATE_ENABLED },
      /*      LINK_UP */ { qfle3_stats_restart,    STATS_STATE_ENABLED },
      /*      UPDATE  */ { qfle3_stats_update,     STATS_STATE_ENABLED },
      /*      STOP    */ { qfle3_stats_stop,       STATS_STATE_DISABLED }
   }
};

void qfle3_stats_handle(struct qfle3_adapter     *adapter,
			enum qfle3_stats_event event)
{
   enum qfle3_stats_state state;

   if (VMK_UNLIKELY(adapter->panic)) {
      return;
   }
   
   if(adapter->error_status & QFLE3_ERR_STATS_TO){
        QFLE3_INFO("Adapter in Stats Timeout state return without Execution\n");
        return;
   }

   state = adapter->stats_state;

   QFLE3_STATS_LOCK(adapter);
   qfle3_stats_stm[state][event].action(adapter);
   adapter->stats_state = qfle3_stats_stm[state][event].next_state;
   if(adapter->stats_pending != (STATS_TIMEOUT+1)){
      QFLE3_STATS_UNLOCK(adapter);
   } else {
      QFLE3_INFO("Lock already dropped before execution of Stats Pending recovery\n");
   }
}

int qfle3_stats_safe_exec(struct qfle3_adapter *adapter,
                          void (func_to_exec)(void *cookie),
                          void *cookie)
{
        int cnt = 10, rc = 0;

        /* Wait for statistics to end [while blocking further requests],
         * then run supplied function 'safely'.
         */
        QFLE3_DBG(QFLE3_DBG_IOV,"SRIOV: Executing\n");

        rc = QFLE3_STATS_LOCK(adapter);
        if (rc) {
                QFLE3_DBG(QFLE3_DBG_IOV,"Failed to take statistics lock for safe execution\n");
                return (rc);
        }

        qfle3_stats_comp(adapter);
        while (adapter->stats_pending && cnt--)
                if (qfle3_storm_stats_update(adapter))
                        vmk_WorldSleep(2000);;
        if (adapter->stats_pending) {
                QFLE3_DBG(QFLE3_DBG_IOV,"SRIOV: Failed to wait for stats pending to clear [possibly FW is stuck]\n");
                rc = -1;
                goto out;
        }

        func_to_exec(cookie);

out:
        /* No need to restart statistics - if they're enabled, the timer
         * will restart the statistics.
         */
        QFLE3_STATS_UNLOCK(adapter);
        return rc;
}

static void
qfle3_port_stats_base_init(struct qfle3_adapter *adapter)
{
   struct dmae_command *dmae;
   vmk_uint32 *stats_comp = QFLE3_SP(adapter, stats_comp);

   /* sanity */
   if (!adapter->port.pmf || !adapter->port.port_stx) {
      // print error
      QFLE3_DBG(QFLE3_DBG_STATS, "BUG!\n");
      return;
   }

   adapter->executer_idx = 0;

   dmae = QFLE3_SP(adapter, dmae[adapter->executer_idx++]);
   dmae->opcode = qfle3_dmae_opcode(adapter, DMAE_SRC_PCI, DMAE_DST_GRC,
				    VMK_TRUE, DMAE_COMP_PCI);
   dmae->src_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, port_stats));
   dmae->src_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, port_stats));
   dmae->dst_addr_lo = (adapter->port.port_stx >> 2);
   dmae->dst_addr_hi = 0;
   dmae->len = qfle3_get_port_stats_dma_len(adapter);
   dmae->comp_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, stats_comp));
   dmae->comp_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, stats_comp));
   dmae->comp_val = DMAE_COMP_VAL;

   *stats_comp = 0;
   qfle3_hw_stats_post(adapter);
   qfle3_stats_comp(adapter);
   QFLE3_DBG(QFLE3_DBG_STATS, "port base init qfle3_stats_comp finished\n");
}

/*
 * This function will prepare the statistics ramrod data the way
 * we will only have to increment the statistics counter and
 * send the ramrod each time we have to.
 */
static void
qfle3_prep_fw_stats_req(struct qfle3_adapter *adapter)
{
   struct stats_query_header *stats_hdr = &adapter->fw_stats_req->hdr;
   vmk_IOA cur_data_offset;
   struct stats_query_entry *cur_query_entry;

   stats_hdr->cmd_num = 0;
   stats_hdr->drv_stats_counter = 0;

   /*
    * The storm_counters struct contains the counters of completed
    * statistics requests per storm which are incremented by FW
    * each time it completes hadning a statistics ramrod. We will
    * check these counters in the timer handler and discard a
    * (statistics) ramrod completion.
    */
   QFLE3_DBG(QFLE3_DBG_STATS, "adapter->fw_stats_data_mapping %lx\n", adapter->fw_stats_data_mapping);
   cur_data_offset = (adapter->fw_stats_data_mapping +
		      vmk_offsetof(struct qfle3_fw_stats_data, storm_counters));
   QFLE3_DBG(QFLE3_DBG_STATS, "adapter->fw_stats_data_mapping %lx\n", adapter->fw_stats_data_mapping);

   stats_hdr->stats_counters_addrs.hi = htole32(U64_HI(cur_data_offset));
   stats_hdr->stats_counters_addrs.lo = htole32(U64_LO(cur_data_offset));

   /*
    * Prepare the first stats ramrod (will be completed with
    * the counters equal to zero) - init counters to somethig different.
    */
   vmk_Memset(&adapter->fw_stats_data->storm_counters, 0xff,
	      sizeof(struct stats_counter));

   /**** Port FW statistics data ****/
   cur_data_offset = (adapter->fw_stats_data_mapping +
		      vmk_offsetof(struct qfle3_fw_stats_data, port));

   cur_query_entry = &adapter->fw_stats_req->query[QFLE3_PORT_QUERY_IDX];

   cur_query_entry->kind = STATS_TYPE_PORT;
   /* For port query index is a DONT CARE */
   cur_query_entry->index = QFLE3_PORT(adapter);
   /* For port query funcID is a DONT CARE */
   cur_query_entry->funcID = htole16(QFLE3_FUNC(adapter));
   cur_query_entry->address.hi = htole32(U64_HI(cur_data_offset));
   cur_query_entry->address.lo = htole32(U64_LO(cur_data_offset));

   /**** PF FW statistics data ****/
   cur_data_offset = (adapter->fw_stats_data_mapping +
		      vmk_offsetof(struct qfle3_fw_stats_data, pf));

   cur_query_entry = &adapter->fw_stats_req->query[QFLE3_PF_QUERY_IDX];

   cur_query_entry->kind = STATS_TYPE_PF;
   /* For PF query index is a DONT CARE */
   cur_query_entry->index = QFLE3_PORT(adapter);
   cur_query_entry->funcID = htole16(QFLE3_FUNC(adapter));
   cur_query_entry->address.hi = htole32(U64_HI(cur_data_offset));
   cur_query_entry->address.lo = htole32(U64_LO(cur_data_offset));
   QFLE3_DBG(QFLE3_DBG_STATS, "cur_query_entry finished\n");

   stats_hdr->cmd_num = 2;
	/**** FCoE FW statistics data ****/
   //TODO: reenable-FCOE statistics
#if 0
	if (!NO_FCOE(adapter)) {
		cur_data_offset = adapter->fw_stats_data_mapping +
			vmk_offsetof(struct qfle3_fw_stats_data, fcoe);

		cur_query_entry =
			&adapter->fw_stats_req->query[QFLE3_FCOE_QUERY_IDX];

		cur_query_entry->kind = STATS_TYPE_FCOE;
		/* For FCoE query index is a DONT CARE */
		cur_query_entry->index = QFLE3_PORT(adapter);
		cur_query_entry->funcID = htole16(QFLE3_FUNC(adapter));
		cur_query_entry->address.hi =
			htole32(U64_HI(cur_data_offset));
		cur_query_entry->address.lo =
			htole32(U64_LO(cur_data_offset));
	}
#endif
}

void
qfle3_stats_init(struct qfle3_adapter *adapter)
{
   int /*abs*/port = QFLE3_PORT(adapter);
   int mb_idx = SC_FW_MB_IDX(adapter);
   int i;

   adapter->stats_pending = 0;
   adapter->executer_idx = 0;
   adapter->stats_counter = 0;

   /* port and func stats for management */
   if (!QFLE3_NOMCP(adapter)) {
      adapter->port.port_stx = SHMEM_RD(adapter, port_mb[port].port_stx);
      QFLE3_DBG(QFLE3_DBG_STATS, "adapter->port.port_stx %d", adapter->port.port_stx);
      adapter->func_stx = SHMEM_RD(adapter, func_mb[mb_idx].fw_mb_param);
      QFLE3_DBG(QFLE3_DBG_STATS, "adapter->func_stx %d", adapter->func_stx);
   } else {
      adapter->port.port_stx = 0;
      adapter->func_stx = 0;
   }

   QFLE3_DBG(QFLE3_DBG_STATS, "port_stx 0x%x func_stx 0x%x\n",
	     adapter->port.port_stx, adapter->func_stx);

   /* pmf should retrieve port statistics from SP on a non-init*/
   if (!adapter->stats_init && adapter->port.pmf && adapter->port.port_stx) {
      QFLE3_DBG(QFLE3_DBG_STATS, "Executing qfle3_stats_handle for STATS_EVENT_PMF\n");
      qfle3_stats_handle(adapter, STATS_EVENT_PMF);
   }

   port = QFLE3_PORT(adapter);
   /* port stats */
   vmk_Memset(&(adapter->port.old_nig_stats), 0, sizeof(struct nig_stats));
   adapter->port.old_nig_stats.brb_discard =
      REG_RD(adapter, NIG_REG_STAT0_BRB_DISCARD + port*0x38);

   QFLE3_DBG(QFLE3_DBG_STATS, " adapter->port.old_nig_stats.brb_discard %d\n", adapter->port.old_nig_stats.brb_discard);

   adapter->port.old_nig_stats.brb_truncate =
      REG_RD(adapter, NIG_REG_STAT0_BRB_TRUNCATE + port*0x38);

   QFLE3_DBG(QFLE3_DBG_STATS, " adapter->port.old_nig_stats.brb_truncate %d\n", adapter->port.old_nig_stats.brb_truncate);

   if (!CHIP_IS_E3(adapter)) {
      REG_RD_DMAE(adapter, NIG_REG_STAT0_EGRESS_MAC_PKT0 + port*0x50,
		  &(adapter->port.old_nig_stats.egress_mac_pkt0_lo), 2);
      REG_RD_DMAE(adapter, NIG_REG_STAT0_EGRESS_MAC_PKT1 + port*0x50,
		  &(adapter->port.old_nig_stats.egress_mac_pkt1_lo), 2);
   }

   /* function stats */
// num_queues Todo
   for (i = 0; i < QFLE3_NUM_ETH_QUEUES(adapter); i++) {
      QFLE3_DBG(QFLE3_DBG_STATS, "vmk_Memset adapter->fp[%d].old_tclient \n", i);
      vmk_Memset(&adapter->fp[i].old_tclient, 0, sizeof(adapter->fp[i].old_tclient));
      vmk_Memset(&adapter->fp[i].old_uclient, 0, sizeof(adapter->fp[i].old_uclient));
      vmk_Memset(&adapter->fp[i].old_xclient, 0, sizeof(adapter->fp[i].old_xclient));
      if (adapter->stats_init) {
	 vmk_Memset(&adapter->fp[i].eth_q_stats, 0,
		    sizeof(adapter->fp[i].eth_q_stats));
	 vmk_Memset(&adapter->fp[i].eth_q_stats_old, 0,
		    sizeof(adapter->fp[i].eth_q_stats_old));
      }
   }
   QFLE3_DBG(QFLE3_DBG_STATS, "per queue vmk_Memset finished %d\n", QFLE3_NUM_ETH_QUEUES(adapter));

   /* prepare statistics ramrod data */
   qfle3_prep_fw_stats_req(adapter);
   QFLE3_DBG(QFLE3_DBG_STATS, "qfle3_prep_fw_stats_req finished\n");

   if (adapter->stats_init) {
      vmk_Memset(&adapter->net_stats_old, 0, sizeof(adapter->net_stats_old));
      vmk_Memset(&adapter->fw_stats_old, 0, sizeof(adapter->fw_stats_old));
      vmk_Memset(&adapter->eth_stats_old, 0, sizeof(adapter->eth_stats_old));
      vmk_Memset(&adapter->eth_stats, 0, sizeof(adapter->eth_stats));
      vmk_Memset(&adapter->func_stats, 0, sizeof(adapter->func_stats));
      QFLE3_DBG(QFLE3_DBG_STATS, "adapter eth_stats fw_stats net_stats func_stats finished\n");

      /* Clean SP from previous statistics */
      if (adapter->func_stx) {
	 vmk_Memset(QFLE3_SP(adapter, func_stats), 0, sizeof(struct host_func_stats));
	 qfle3_func_stats_init(adapter);
	 QFLE3_DBG(QFLE3_DBG_STATS, "qfle3_func_stats_init finished\n");
	 qfle3_hw_stats_post(adapter);
	 qfle3_stats_comp(adapter);
	 QFLE3_DBG(QFLE3_DBG_STATS, "qfle3_stats_comp \n");
      }
   }

   adapter->stats_state = STATS_STATE_DISABLED;

   if (adapter->port.pmf && adapter->port.port_stx) {
      qfle3_port_stats_base_init(adapter);
   }

   /* mark the end of statistics initializiation */
   adapter->stats_init = VMK_FALSE;
}

void
qfle3_save_statistics(struct qfle3_adapter *adapter)
{
   int i;

   /* save queue statistics */
//num_queues Todo
   for (i = 0; i < QFLE3_NUM_ETH_QUEUES(adapter); i++) {
      struct qfle3_fastpath *fp = &adapter->fp[i];
      struct qfle3_eth_q_stats *qstats = &fp->eth_q_stats;
      struct qfle3_eth_q_stats_old *qstats_old = &fp->eth_q_stats_old;

      UPDATE_QSTAT_OLD(total_unicast_bytes_received_hi);
      UPDATE_QSTAT_OLD(total_unicast_bytes_received_lo);
      UPDATE_QSTAT_OLD(total_broadcast_bytes_received_hi);
      UPDATE_QSTAT_OLD(total_broadcast_bytes_received_lo);
      UPDATE_QSTAT_OLD(total_multicast_bytes_received_hi);
      UPDATE_QSTAT_OLD(total_multicast_bytes_received_lo);
      UPDATE_QSTAT_OLD(total_unicast_bytes_transmitted_hi);
      UPDATE_QSTAT_OLD(total_unicast_bytes_transmitted_lo);
      UPDATE_QSTAT_OLD(total_broadcast_bytes_transmitted_hi);
      UPDATE_QSTAT_OLD(total_broadcast_bytes_transmitted_lo);
      UPDATE_QSTAT_OLD(total_multicast_bytes_transmitted_hi);
      UPDATE_QSTAT_OLD(total_multicast_bytes_transmitted_lo);
      UPDATE_QSTAT_OLD(total_tpa_bytes_hi);
      UPDATE_QSTAT_OLD(total_tpa_bytes_lo);
   }

   /* store port firmware statistics */
   if (adapter->port.pmf) {
      struct qfle3_eth_stats *estats = &adapter->eth_stats;
      struct qfle3_fw_port_stats_old *fwstats = &adapter->fw_stats_old;
      struct host_port_stats *pstats = QFLE3_SP(adapter, port_stats);

      fwstats->pfc_frames_rx_hi = pstats->pfc_frames_rx_hi;
      fwstats->pfc_frames_rx_lo = pstats->pfc_frames_rx_lo;
      fwstats->pfc_frames_tx_hi = pstats->pfc_frames_tx_hi;
      fwstats->pfc_frames_tx_lo = pstats->pfc_frames_tx_lo;

      if (IS_MF(adapter)) {
	 UPDATE_FW_STAT_OLD(mac_filter_discard);
	 UPDATE_FW_STAT_OLD(mf_tag_discard);
	 UPDATE_FW_STAT_OLD(brb_truncate_discard);
	 UPDATE_FW_STAT_OLD(mac_discard);
      }
   }
}

