/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

/*
 * qfle3f_els.c: QLogic 10 Gigabit ESX Ethernet FCoE offload driver.
 *
 * This file contains helper routines that handle ELS requests
 * and responses.
 *
 */

#include "qfle3f.h"
#include "ql_fcoe_mgmt/FC_LS.h"
#include "ql_fcoe_mgmt/ql_fcoe.h"

static int qfle3f_initiate_els(struct qfle3f_rport *target, vmk_uint32 op,
			void *data, vmk_uint32 data_len,
			void (*cb_func)(struct qfle3f_els_cb_arg *cb_arg),
			struct qfle3f_els_cb_arg *cb_arg, vmk_uint32 timer_msec);

void qfle3f_fill_fc_hdr(FC_FRAME_HEADER *fc_hdr, vmk_uint8 r_ctl,
			vmk_uint32 initiatorPortID, vmk_uint32 did, vmk_uint8 fh_type,
			vmk_uint32 f_ctl, vmk_uint32 parm_offset)
{
    fc_hdr->s_id[2] = (initiatorPortID & 0x000000FF);
    fc_hdr->s_id[1] = (initiatorPortID & 0x0000FF00) >> 8;
    fc_hdr->s_id[0] = (initiatorPortID & 0x00FF0000) >> 16;

    fc_hdr->d_id[2] = (did & 0x000000FF);
    fc_hdr->d_id[1] = (did & 0x0000FF00) >> 8;
    fc_hdr->d_id[0] = (did & 0x00FF0000) >> 16;

	fc_hdr->r_ctl = r_ctl;
	fc_hdr->type = fh_type;

    fc_hdr->f_ctl[2] = (f_ctl & 0x000000FF);
    fc_hdr->f_ctl[1] = (f_ctl & 0x0000FF00) >> 8;
    fc_hdr->f_ctl[0] = (f_ctl & 0x00FF0000) >> 16;


	fc_hdr->cs_ctl = 0;
	fc_hdr->df_ctl = 0;
    vmk_Memcpy(fc_hdr->parameter, &parm_offset, sizeof(parm_offset));
}

static void qfle3f_rrq_compl(struct qfle3f_els_cb_arg *cb_arg)
{
	struct qfle3fCommand *orig_ioRequest;
	struct qfle3fCommand *rrq_req;
	struct qfle3fHBA *hba;
	int rc = 0;

	//BUG_ON(!cb_arg);

	rrq_req = cb_arg->ioRequest;
	hba = rrq_req->vhba;

	qfle3f_log(hba, LOG_ELS, "Entered rrq_compl callback");
	orig_ioRequest = cb_arg->abortedIORequest;
	//BUG_ON(!orig_ioRequest);
	qfle3f_log(hba, LOG_ELS, "rrq_compl: orig xid = 0x%x, rrq_xid = 0x%x ",
		orig_ioRequest->xid, rrq_req->xid);

	ql_vmk_ref_put(&orig_ioRequest->refcount, qfle3f_commandRelease, orig_ioRequest);

	if (vmk_BitVectorAtomicTestAndClear(rrq_req->req_flags, QFLE3F_FLAG_ELS_TIMEOUT)) {
		/*
		 * els req is timed out. cleanup the IO with FW and
		 * drop the completion. Remove from active_cmd_queue.
		 */
		qfle3f_log(hba, LOG_ELS, "rrq xid - 0x%x timed out, clean it up",
			   rrq_req->xid);

		if (rrq_req->on_active_queue) {
			//list_del_init(&rrq_req->link);
            vmk_ListRemove(&rrq_req->link);
			rrq_req->on_active_queue = 0;
			rc = qfle3f_initiateCleanup(rrq_req);
			//BUG_ON(rc);
		}
	}

	qfle3f_free(cb_arg);

}

int qfle3f_send_rrq(struct qfle3fCommand *abortedIORequest)
{

    struct RRQ_REQUEST_DATA rrq = { FC_ELS_RRQ };
	struct qfle3f_rport *target = abortedIORequest->target;
	struct qfle3f_els_cb_arg *cb_arg = NULL;
	struct qfle3fHBA *hba;
	vmk_uint32 initiatorPortID = target->initiatorPortID;
	vmk_uint32 r_a_tov = target->Sess->Fabric->r_a_tov;
	int rc;

	hba = target->hba;

	qfle3f_log(hba, LOG_ELS, "Sending RRQ orig_xid = 0x%x",
		   abortedIORequest->xid);
	vmk_Memset(&rrq, 0, sizeof(rrq));

	cb_arg = qfle3f_alloc(sizeof(struct qfle3f_els_cb_arg));
	if (!cb_arg) {
		qfle3f_err(hba, "Unable to allocate cb_arg for RRQ");
		rc = -1;
		goto rrq_err;
	}

	cb_arg->abortedIORequest = abortedIORequest;

	rrq.command = FC_ELS_RRQ;

    rrq.OriginatorID[2] = (initiatorPortID & 0x000000FF);
    rrq.OriginatorID[1] = (initiatorPortID & 0x0000FF00) >> 8;
    rrq.OriginatorID[0] = (initiatorPortID & 0x00FF0000) >> 16;

    rrq.ox_id[0] = (abortedIORequest->xid)  >> 8;
    rrq.ox_id[1] = (abortedIORequest->xid & 0xFF);

    rrq.rx_id[0] = ((abortedIORequest->task->rxwr_txrd.var_ctx.rx_id) & 0xFF);
    rrq.rx_id[1] = (((abortedIORequest->task->rxwr_txrd.var_ctx.rx_id) & 0xFF00) >> 8);

	rc = qfle3f_initiate_els(target, FC_ELS_RRQ, &rrq, sizeof(rrq),
				 qfle3f_rrq_compl, cb_arg,
				 r_a_tov);
rrq_err:
	if (rc) {
		qfle3f_log(hba, LOG_ELS, "RRQ failed - release orig io req 0x%x",
			abortedIORequest->xid);
		qfle3f_free(cb_arg);
		vmk_SpinlockLock(target->targetLock);
		ql_vmk_ref_put(&abortedIORequest->refcount,
					qfle3f_commandRelease, abortedIORequest);
		vmk_SpinlockUnlock(target->targetLock);
	}
	return rc;
}

static void qfle3f_els_logo_compl(struct qfle3f_els_cb_arg *cb_arg)
{
	struct qfle3fCommand *logo_req = cb_arg->ioRequest;
	struct qfle3fHBA *hba = logo_req->vhba;
	struct qfle3f_rport *target = logo_req->target;
	int rc = 0;

	struct qfle3f_mp_req *mp_req;
	unsigned char *buf;
	void *resp_buf;
	vmk_uint32 resp_len;

	qfle3f_log(hba, LOG_ELS, "Entered logo_compl callback: xid = 0x%x",
			logo_req->xid);

	vmk_BitVectorClear(target->flags, QFLE3F_FLAG_LOGO_REQ_PENDING);

	if (vmk_BitVectorAtomicTestAndClear(logo_req->req_flags, QFLE3F_FLAG_ELS_TIMEOUT)) {
		/*
		 * els req is timed out. cleanup the IO with FW and
		 * drop the completion. Remove from active_cmd_queue.
		 */
		qfle3f_log(hba, LOG_ELS, "logo xid - 0x%x timed out, clean it up",
				logo_req->xid);

		if (logo_req->on_active_queue) {
			vmk_ListRemove(&logo_req->link);
			logo_req->on_active_queue = 0;
			rc = qfle3f_initiateCleanup(logo_req);
		}
		// As ql_fcoe_mgmt maintains it's own timer for LOGO Request,
		// no need to tell notify them. Just cleanup from the Firmware.
		return;
	}

	mp_req = &(logo_req->mp_req);
	resp_len = mp_req->resp_len;
	resp_buf = mp_req->resp_buf;

	buf = qfle3f_alloc(VMK_PAGE_SIZE);
	if (!buf) {
		qfle3f_err(hba, "failed to alloc mp buf.");
		goto free_arg;
	}
	vmk_Memcpy(buf, &(mp_req->resp_fc_hdr), sizeof(FC_FRAME_HEADER));
	vmk_Memcpy(buf + sizeof(FC_FRAME_HEADER), resp_buf, resp_len);

	qfle3f_printBuffer(hba, (vmk_uint8 *) buf, resp_len+sizeof(FC_FRAME_HEADER),
			"LOGO Response(FC Header + Data)");
	qfle3f_processL2FrameCompletion(target, buf,
			resp_len+sizeof(FC_FRAME_HEADER), cb_arg->l2_oxid);

	qfle3f_free(buf);

free_arg:
	qfle3f_free(cb_arg);
}

int qfle3f_send_els_logo(struct qfle3f_rport *target, vmk_PktHandle *pkt)
{
	struct qfle3fHBA *hba = target->hba;
	vmk_uint8 *mapped_data = (char *)vmk_PktFrameMappedPointerGet(pkt);
	vmk_EthHdr *eh = NULL;
	struct FC_FRAME_HEADER *fh = NULL;
	struct qfle3f_els_cb_arg *cb_arg = NULL;
	vmk_uint32 r_a_tov = target->Sess->Fabric->r_a_tov;
	int rc = 0;

	struct fcoe_header {
		vmk_uint8 version[13];
		vmk_uint8 sof;
	};

	eh = (vmk_EthHdr *) mapped_data;
	struct fcoe_header *now = NULL;
	now = (struct fcoe_header  *) (eh+1);
	fh = (struct FC_FRAME_HEADER *)(now + 1);
	void *startOfDataFrame = (void *) (fh + 1);

	struct LOGO_DATA data = { FC_ELS_LOGO };
	vmk_Memcpy(&data, startOfDataFrame,sizeof(struct LOGO_DATA));

	cb_arg = qfle3f_alloc(sizeof(struct qfle3f_els_cb_arg));
	if (!cb_arg) {
		qfle3f_err(hba, "Unable to allocate cb_arg for LOGO. Dropping Frame");
		rc = -1;
		goto exit;
	}

	cb_arg->abortedIORequest = NULL;
	cb_arg->l2_oxid = (vmk_uint16)(fh->ox_id[0] | (fh->ox_id[1]<<8));

	qfle3f_log(hba, LOG_ELS, "ELS LOGO via offloaded session: "
			"0x%x targetPortID=0x%x fcid=0x%x %x %x\n",
			FC_ELS_LOGO, target->targetPortID,
			data.fc_id[0], data.fc_id[1], data.fc_id[2]);

	qfle3f_printBuffer(hba, (vmk_uint8 *) &data, sizeof(data), "LOGO Request Data");
	qfle3f_log(hba, LOG_ELS, "r_a_tov: %d", r_a_tov);

	rc = qfle3f_initiate_els(target, FC_ELS_LOGO, &data, sizeof(data),
			qfle3f_els_logo_compl, cb_arg, r_a_tov);

	vmk_BitVectorSet(target->flags, QFLE3F_FLAG_LOGO_REQ_PENDING);

exit:
	return rc;
}

static int qfle3f_initiate_els(struct qfle3f_rport *target, vmk_uint32 op,
			void *data, vmk_uint32 data_len,
			void (*cb_func)(struct qfle3f_els_cb_arg *cb_arg),
			struct qfle3f_els_cb_arg *cb_arg, vmk_uint32 timer_msec)
{
	struct qfle3fHBA *hba = target->hba;
	struct qfle3fCommand *els_req;
	struct qfle3f_mp_req *mp_req;
	FC_FRAME_HEADER *fc_hdr;
	struct fcoe_task_ctx_entry *task;
	struct fcoe_task_ctx_entry *task_page;
	int rc = 0;
	int task_idx, index;
	vmk_uint32 did, initiatorPortID;
	vmk_uint16 xid;

	qfle3f_log(hba, LOG_ELS, "Sending ELS");

	if ((!(vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY)) ||
	     (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_EXPL_LOGO)))
			&& (op != FC_ELS_LOGO)) {
		qfle3f_err(hba, "els 0x%x: target not ready", op);
		rc = -1;
		goto els_err;
	}

retry_els:
	els_req = qfle3f_elstmAlloc(target, QFLE3F_ELS);
	if (!els_req) {
        // TODO : Implement this
        /*
		if (time_after(jiffies, start + (10 * vmk_TimerCyclesPerSecond()))) {
			qfle3f_log(hba, LOG_ELS, "els: Failed els 0x%x", op);
			rc = -1;
			goto els_err;
		}
        */
		//msleep(20);
		vmk_WorldSleep(20 * VMK_USEC_PER_MSEC);
		goto retry_els;
	}

	qfle3f_log(hba, LOG_ELS, "initiate_els els_req = 0x%p cb_arg = %p oxid: 0x%x",
		els_req, cb_arg, els_req->xid);
	els_req->sc_cmd = NULL;
	els_req->target = target;
	els_req->cb_func = cb_func;
	cb_arg->ioRequest = els_req;
	els_req->cb_arg = cb_arg;
	els_req->data_xfer_len = data_len;

	mp_req = (struct qfle3f_mp_req *)&(els_req->mp_req);
	rc = qfle3f_initMPRequest(els_req);
	if (rc == VMK_FAILURE) {
		qfle3f_err(hba, "ELS MP request init failed");
		vmk_SpinlockLock(target->targetLock);
		ql_vmk_ref_put(&els_req->refcount, qfle3f_commandRelease, els_req);
		vmk_SpinlockUnlock(target->targetLock);
		goto els_err;
	} else {
		/* rc SUCCESS */
		rc = 0;
	}

	/* Fill ELS Payload */
	if ((op >= FC_ELS_RJT) && (op <= FC_ELS_AUTH_ELS)) {
		vmk_Memcpy(mp_req->req_buf, data, data_len);
	} else {
		qfle3f_err(hba, "Invalid ELS op 0x%x", op);
		els_req->cb_func = NULL;
		els_req->cb_arg = NULL;
		vmk_SpinlockLock(target->targetLock);
		ql_vmk_ref_put(&els_req->refcount, qfle3f_commandRelease, els_req);
		vmk_SpinlockUnlock(target->targetLock);
		rc = -1;
	}

	if (rc)
		goto els_err;

	/* Fill FC header */
	fc_hdr = &(mp_req->req_fc_hdr);

	did = target->targetPortID;
	initiatorPortID = target->initiatorPortID;

	qfle3f_fill_fc_hdr(fc_hdr, FC_R_CTL_EXTENDED_LINK_DATA_COMMAND_REQUEST,
                initiatorPortID, did,
                FC_TYPE_ELS, FC_F_CTL0_FirstSequence | FC_F_CTL0_EndSequence |
                FC_F_CTL0_SequenceInitiativeTransfer, 0);

	qfle3f_printBuffer(hba, (vmk_uint8 *) fc_hdr, sizeof(FC_FRAME_HEADER),
					"FC Header of LOGO Request");
	/* Obtain exchange id */
	xid = els_req->xid;
	task_idx = xid/QFLE3F_TASKS_PER_PAGE;
	index = xid % QFLE3F_TASKS_PER_PAGE;

	/* Initialize task context for this IO request */
	task_page = (struct fcoe_task_ctx_entry *) hba->taskContext[task_idx];
	task = &(task_page[index]);
	qfle3f_initializeMPTask(els_req, task);

	vmk_SpinlockLock(target->targetLock);

	if (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY)
			&& op != FC_ELS_LOGO) {
		qfle3f_err(hba, "initiate_els.. session not ready");
		els_req->cb_func = NULL;
		els_req->cb_arg = NULL;
		ql_vmk_ref_put(&els_req->refcount, qfle3f_commandRelease, els_req);
		vmk_SpinlockUnlock(target->targetLock);
		return -1;
	}

#ifdef IOERROR_DEBUGGING
	els_req->printIORelease = VMK_TRUE;
#endif
	if (timer_msec)
		qfle3f_commandTimerSet(els_req, timer_msec);

	qfle3f_addToSQ(target, xid);

	els_req->on_active_queue = 1;
    vmk_ListInsert(&els_req->link, vmk_ListAtRear(&target->elsQueue));

	/* Ring doorbell */
	qfle3f_log(hba, LOG_ELS, "Ringing doorbell for ELS req");
	qfle3f_ringDoorbell(target);
	vmk_SpinlockUnlock(target->targetLock);

els_err:
	return rc;
}

void qfle3f_process_els_compl(struct qfle3fCommand *els_req,
			      struct fcoe_task_ctx_entry *task, vmk_uint8 num_rq)
{
	struct qfle3f_mp_req *mp_req;
	struct qfle3fHBA *hba = els_req->vhba;
	FC_FRAME_HEADER *fc_hdr;
	vmk_uint64 *hdr;
	vmk_uint64 *temp_hdr;

	qfle3f_log(hba, LOG_ELS, "Entered process_els_compl xid = 0x%x "
				"cmd_type = %d",
			els_req->xid, els_req->cmd_type);

	if (vmk_BitVectorAtomicTestAndSet(els_req->req_flags,
                    QFLE3F_FLAG_ELS_DONE)) {
		qfle3f_log(hba, LOG_ELS, "Timer context finished processing this "
			   "els - 0x%x", els_req->xid);
		/* This IO doesnt receive cleanup completion */
		ql_vmk_ref_put(&els_req->refcount, qfle3f_commandRelease, els_req);
		return;
	}

	/* Cancel the timeout_work, as we received the response */
	if(ql_vmk_cancel_delayed_work_sync(&els_req->timeout_work) == VMK_OK)
		ql_vmk_ref_put(&els_req->refcount,
			 qfle3f_commandRelease, els_req); /* drop timer hold */

	if (els_req->on_active_queue) {
	    vmk_ListRemove(&els_req->link);
		els_req->on_active_queue = 0;
	}

	mp_req = &(els_req->mp_req);
	fc_hdr = &(mp_req->resp_fc_hdr);

	hdr = (vmk_uint64 *)fc_hdr;
	temp_hdr = (vmk_uint64 *)
		&task->rxwr_only.union_ctx.comp_info.mp_rsp.fc_hdr;
	hdr[0] = vmk_CPUToBE64(temp_hdr[0]);
	hdr[1] = vmk_CPUToBE64(temp_hdr[1]);
	hdr[2] = vmk_CPUToBE64(temp_hdr[2]);

	mp_req->resp_len = task->rxwr_only.union_ctx.comp_info.mp_rsp.mp_payload_len;
	qfle3f_log(hba, LOG_ELS, "els_compl: resp_len = %d", mp_req->resp_len);

	/* Parse ELS response */
	if ((els_req->cb_func) && (els_req->cb_arg)) {
		els_req->cb_func(els_req->cb_arg);
		els_req->cb_arg = NULL;
	}

	ql_vmk_ref_put(&els_req->refcount, qfle3f_commandRelease, els_req);
}

#ifdef FDMI_SUPPORT

/**
 * Function Name : qfle3f_els_status_to_string
 *
 * DESCRIPTION:
 * Return the user friendly string for the current
 * status of the PLOGI/RHBA/RPA
 *
 * PARAMETERS:
 *  cur_state	The current status of PLOGI/RHBA/RPA
 *
 * RETURN:
 * Return the string for the respective ELS state.
 */
static const char* qfle3f_els_status_to_string(int cur_state) {

	switch(cur_state) {

	case ZERO_STATE: return "ZERO STATE";
	case BEGIN_PROCESS: return "BEGIN PROCESS";
	case PLOGI_SEND_MGMT_SERVER: return "PLOGI SEND MGMT SERVER";
	case RHBA_FDMI2: return "RHBA FDMI2";
	case RHBA_FDMI: return "RHBA FDMI";
	case RPA_SMARTSAN_FDMI2: return "RPA SMARTSAN FDMI2";
	case RPA_FDMI2: return "RPA FDMI2";
	case RPA_FDMI: return "RPA FDMI";
	case ALL_DONE: return "ALL DONE";

	default: return "INVALID STATE";
	}
}

/**
 * Function Name : qfle3f_els_cmd_timeout
 *
 * DESCRIPTION:
 * One of the following ELS command timed out.
 *
 * PLOGI
 * RHBA
 * RPA
 *
 * PARAMETERS:
 *  work	The pointer to els_timeout_work.work
 *
 * RETURN:
 *  		NULL
 */
static void qfle3f_els_cmd_timeout(struct work_struct *work)
{
	struct qfle3fHBA *hba = container_of(work, struct qfle3fHBA,
						 els_timeout_work.work);

	vmk_SemaLock(&hba->backgroundMutex);

	switch (hba->FDMICurrentState) {
	case PLOGI_SEND_MGMT_SERVER:
		hba->FDMICurrentState = ALL_DONE;
		goto done;
	break;
	case RHBA_FDMI2:
		/* Try RHBA FDMI */
		hba->FDMICurrentState = RHBA_FDMI;
	break;

	case RHBA_FDMI:
		/* Try RPA FDMI */
		hba->FDMICurrentState = RPA_FDMI;
	break;

	case RPA_SMARTSAN_FDMI2:
		/* Try RPA FDMI2 */
		hba->FDMICurrentState = RPA_FDMI2;
	break;

	case RPA_FDMI2:
		/* Try RPA FDMI */
		hba->FDMICurrentState = RPA_FDMI;
	break;

	case RPA_FDMI:
		/* Completed */
		hba->FDMICurrentState = ALL_DONE;
		goto done;
	break;

	case BEGIN_PROCESS:
	case ZERO_STATE:
	case ALL_DONE:
	default:

	break;

	}

	qfle3f_log(hba, LOG_ELS, "bk_status = %s."
		" Send ELS command.",
		qfle3f_els_status_to_string(hba->FDMICurrentState));

	qfle3f_bk_send_els_cmd(hba);

done:
	vmk_SemaUnlock(&hba->backgroundMutex);
}

/**
 * Function Name : qfle3f_bk_response
 *
 * DESCRIPTION:
 *  Process the response of the following ELS command,
 *  send to the Fabric Mgmt Server(0xfffffa) and initiated
 *  by the qfle3f driver.
 *
 * PLOGI
 * RHBA
 * RPA
 *
 * PARAMETERS:
 *  sp		FC sequence associated with the specific exchange
 *  in_fp	Pointer to the FC Frame.
 *  hba_arg	The interface on which the ELS response
 *  		was recieved.
 *
 * RETURN:
 *  		NULL
 */
static void qfle3f_bk_response(struct fc_seq *sp, FC_FRAME_HEADER *in_fp,
                                  void *hba_arg)
{

	struct qfle3fHBA *hba = hba_arg;
	struct ct_rsp_hdr *rsp_hdr = NULL;
        vmk_uint8 op;

	vmk_SemaLock(&hba->backgroundMutex);

	cancel_delayed_work(&hba->els_timeout_work);

	qfle3f_log(hba, LOG_ELS, "FDMICurrentState = %s.",
		qfle3f_els_status_to_string(hba->FDMICurrentState));

        if (IS_ERR(in_fp)) {
                int err = PTR_ERR(in_fp);

                if (err == -FC_EX_CLOSED || err == -FC_EX_TIMEOUT) {
			hba->FDMICurrentState = ALL_DONE;
			qfle3f_log(hba, LOG_ELS,
                            "frame error %d", err);
                        goto done;
		}
                qfle3f_log(hba, LOG_ELS, "Cannot process RRQ, "
                            "frame error %d", err);
                return;
        }

	switch(hba->FDMICurrentState) {

	case PLOGI_SEND_MGMT_SERVER:
		op = fc_frame_payload_op(in_fp);
		if(op == ELS_LS_ACC)
			hba->FDMICurrentState = RHBA_FDMI2;
		else
			hba->FDMICurrentState = ZERO_STATE;
	break;

	case RHBA_FDMI2:
		rsp_hdr = fc_frame_payload_get(in_fp,
				sizeof(struct ct_rsp_hdr) - 3);
		if(vmk_BE16ToCPU(rsp_hdr->response) ==
		   FC_FS_ACC)
			hba->FDMICurrentState = RPA_SMARTSAN_FDMI2;
		else
			hba->FDMICurrentState = RHBA_FDMI;
	break;

	case RHBA_FDMI:
		rsp_hdr = fc_frame_payload_get(in_fp,
				sizeof(struct ct_rsp_hdr) - 3);
		if(vmk_BE16ToCPU(rsp_hdr->response) ==
		   FC_FS_ACC)
			hba->FDMICurrentState = RPA_FDMI;
		else
			hba->FDMICurrentState = RPA_SMARTSAN_FDMI2;
	break;

	case RPA_SMARTSAN_FDMI2:
		rsp_hdr = fc_frame_payload_get(in_fp,
				sizeof(struct ct_rsp_hdr) - 3);
		if(vmk_BE16ToCPU(rsp_hdr->response) ==
		   FC_FS_ACC) {
			hba->FDMICurrentState = ALL_DONE;
		} else
			hba->FDMICurrentState = RPA_FDMI2;
	break;

	case RPA_FDMI2:
		rsp_hdr = fc_frame_payload_get(in_fp,
				sizeof(struct ct_rsp_hdr) - 3);
		if(vmk_BE16ToCPU(rsp_hdr->response) ==
		   FC_FS_ACC) {
			hba->FDMICurrentState = ALL_DONE;
		} else
			hba->FDMICurrentState = RPA_FDMI;
	break;

	case RPA_FDMI:
		rsp_hdr = fc_frame_payload_get(in_fp,
				sizeof(struct ct_rsp_hdr) - 3);
		if(vmk_BE16ToCPU(rsp_hdr->response) ==
		   FC_FS_ACC)
			hba->FDMICurrentState = ALL_DONE;
		else
			hba->FDMICurrentState = ALL_DONE;
	break;

	default:
                qfle3f_log(hba, LOG_ELS,
                            "Error : Discard the frame.");
	break;

	}

	if(rsp_hdr) {
		qfle3f_log(hba, LOG_ELS,
			"vmk_BE16ToCPU(rsp_hdr->response) = 0x%x.",
			vmk_BE16ToCPU(rsp_hdr->response));

		qfle3f_log(hba, LOG_ELS,
			    "rsp_hdr->header.revision: %x. rsp_hdr->header.gs_type: %x.\n"
			    "rsp_hdr->header.gs_subtype: %x. rsp_hdr->header.options: %x. ",
			    rsp_hdr->header.revision, rsp_hdr->header.gs_type,
			    rsp_hdr->header.gs_subtype, rsp_hdr->header.options);

		qfle3f_log(hba, LOG_ELS,
			    "rsp_hdr->header.reserved: %x. rsp_hdr->response: %x.\n"
			    "rsp_hdr->reinitiatorPortIDual: %x. rsp_hdr->fragment_id: %x.",
			    rsp_hdr->header.reserved, rsp_hdr->response,
			    rsp_hdr->reinitiatorPortIDual, rsp_hdr->fragment_id);
	}

	qfle3f_bk_send_els_cmd(hba);

done:
	qfle3f_log(hba, LOG_ELS, "FDMICurrentState = %s.",
		qfle3f_els_status_to_string(hba->FDMICurrentState));

	vmk_SemaUnlock(&hba->backgroundMutex);

	fc_frame_free(in_fp);

}

/**
 * Function Name : qfle3f_send_plogi
 *
 * DESCRIPTION:
 *  Send a PLOGI(Register Host Attributes) ELS command
 *
 * PARAMETERS:
 *  hba		The interface to be used to send the
 *  		ELS command.
 *  lport	Pointer to the local port structure.
 *  did		The Destination Port ID.
 *
 * RETURN:
 * 0		SUCESS
 * 1		FAILURE
 *
 * NOTE:
 * Called should be holding the hba->backgroundMutex lock.
 *
 */
static int qfle3f_send_plogi(struct qfle3fHBA *hba,
			struct fc_lport *lport, int did)
{
	FC_FRAME_HEADER *fp;

	qfle3f_log(hba, LOG_ELS, "FDMICurrentState = %s",
		qfle3f_els_status_to_string(hba->FDMICurrentState));

	/* Create fc_frame for FLOGI */
	fp = ql_fcoe_frame_alloc(hba->qlfcoe_hbalport, sizeof(struct fc_els_flogi));
	if (!fp) {
		qfle3f_log(hba, LOG_ELS, "els_plogi FC Frame alloc failed");
		hba->FDMICurrentState = ZERO_STATE;
		return 1;
	}

	qfle3f_log(hba, LOG_ELS,
		"Sending PLOGI to MGMT Server");
	if (!fc_elsct_send(lport, FC_FID_MGMT_SERV,
				fp, FC_ELS_PLOGI,
				qfle3f_bk_response, hba,
				2 * lport->r_a_tov)) {
		qfle3f_log(hba, LOG_ERROR,
			"Unable to send the PLOGI to MGMT Server");
		hba->FDMICurrentState = ZERO_STATE;
		return 1;
	}

	return 0;
}

#define	FC_FDMI_RHBA	0x200
#define FC_FDMI_RPA	0x211

/**
 * Function Name : qfle3f_prep_ct_req
 *
 * DESCRIPTION:
 *  Prepare common CT request fields for SNS query.
 *
 *
 * PARAMETERS:
 *  flags	RHBA_FDMI2:     Send FDMI2 attributes
 *  		RHBA_FDMI:      Send FDMI attributes
 *  ct_req	CT request buffer
 *  cmd		GS command
 *  rsp_size	response size in bytes
 *
 * RETURN:
 *  ct_sns_req	Pointer to the Initializes CT SNS Request
 *
 */
static inline struct ct_sns_req *
qfle3f_prep_ct_fdmi_req(struct ct_sns_req *ct_req, uint16_t cmd,
    uint16_t rsp_size)
{
	vmk_Memset(ct_req, 0, sizeof(struct ct_sns_req));

	ct_req->header.revision = 0x01;
	ct_req->header.gs_type = 0xFA;
	ct_req->header.gs_subtype = 0x10;
	ct_req->command = vmk_CPUToBE16(cmd);
	ct_req->max_rsp_size = vmk_CPUToBE16((rsp_size - 16) / 4);

	return ct_req;
}

/**
 * Function Name : qfle3f_rhba_fdmi
 *
 * DESCRIPTION:
 *  Send a RHBA(Register Host Attributes) ELS command
 *
 *
 * PARAMETERS:
 *  hba		The interface to be used to send the
 *  		ELS command.
 *  lport	Pointer to the local port structure.
 *  did		The Destination ID
 *  flags	RHBA_FDMI2:     Send FDMI2 attributes
 *  		RHBA_FDMI:      Send FDMI attributes
 *
 * RETURN:
 *  0		SUCCESS
 *  1		FAILURE
 *
 * NOTE:
 * Called should be holding the hba->backgroundMutex lock.
 *
 */
static int qfle3f_rhba_fdmi(struct qfle3fHBA *hba,
		struct fc_lport *lport,
		int did, int flags)
{

	FC_FRAME_HEADER *fp;
	struct ct_sns_req *ct_req;
	int ret = 0, alen, size = 0;
	uint32_t max_frame_size;
	uint8_t *entries;
	struct ct_fdmi_hba_attr *eiter;
	uint8_t portName[WWN_SIZE], nodeName[WWN_SIZE];
	uint8_t *fc_hdr;
	struct	ethtool_drvinfo drv_info;
	int hdr_len = sizeof(FC_FRAME_HEADER);
    int count = 0;

	vmk_uint64_to_wwn(qfle3f_get_wwpn(hba), portName);
	vmk_uint64_to_wwn(qfle3f_get_wwnn(hba), nodeName);

	ct_req = qfle3f_alloc(sizeof(struct ct_sns_req));
	if(!ct_req) {
		qfle3f_log(hba, LOG_ERROR,
			"Unable to alloc memory for the CT Request");
		hba->FDMICurrentState = ZERO_STATE;
		return 1;
	}

	qfle3f_log(hba, LOG_ELS, "FDMICurrentState = %s",
		qfle3f_els_status_to_string(hba->FDMICurrentState));

	/* Prepare CT request */
	ct_req = qfle3f_prep_ct_fdmi_req(ct_req, RHBA_CMD,
	    RHBA_RSP_SIZE);

	/* Prepare FDMI command arguments -- attribute block, attributes. */
	vmk_Memcpy(ct_req->req.rhba.hba_identifier, portName, WWN_SIZE);
	ct_req->req.rhba.entry_count = __constant_vmk_CPUToBE32(1);
	vmk_Memcpy(ct_req->req.rhba.portName, portName, WWN_SIZE);
	entries = (uint8_t *)ct_req;

	/* size = 2 * WWN_SIZE + 4 + 4; */
	size = 40;

	if (hba->cnic && hba->cnic->netdev) {
		hba->cnic->netdev->ethtool_ops->get_drvinfo(hba->cnic->netdev,
			&drv_info);
	}

	/* Nodename. */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_NODE_NAME);
	eiter->len = __constant_vmk_CPUToBE16(4 + WWN_SIZE);
	vmk_Memcpy(eiter->a.nodeName, nodeName, WWN_SIZE);
	size += 4 + WWN_SIZE;
    count++;

	qfle3f_log(hba, LOG_ELS, "NODENAME=%02x%02x%02x%02x%02x%02x%02x%02x.",
	    eiter->a.nodeName[0], eiter->a.nodeName[1], eiter->a.nodeName[2],
	    eiter->a.nodeName[3], eiter->a.nodeName[4], eiter->a.nodeName[5],
	    eiter->a.nodeName[6], eiter->a.nodeName[7]);

	/* Manufacturer. */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_MANUFACTURER);
	vmk_StringCopy((char *)eiter->a.manufacturer, "QLogic Corporation", 19);
	alen = strlen((char *)eiter->a.manufacturer);
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS, "MANUFACTURER=%s.",
	    eiter->a.manufacturer);

	/* Model name. */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_MODEL);
	vmk_StringFormat(eiter->a.model, sizeof(eiter->a.model), NULL,
			 "BCM%s", hba->sym_name);
	alen = strnlen(eiter->a.model,
			sizeof(eiter->a.model));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS, "MODEL_NAME=%s.",
	    eiter->a.model);

	/* Model description. */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_MODEL_DESCRIPTION);
	sprintf(eiter->a.model_desc, "Broadcom NetXtreme II %s FCoE Driver",
		hba->sym_name);
	alen = strnlen(eiter->a.model_desc,
			sizeof(eiter->a.model_desc));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS, "MODEL_DESC=%s.",
	    eiter->a.model_desc);

	/* Hardware version. */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_HARDWARE_VERSION);
	vmk_StringCopy(eiter->a.hw_version, "MN", 2);
	alen = strnlen(eiter->a.hw_version, sizeof(eiter->a.hw_version));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS, "HARDWAREVER=%s.",
	    eiter->a.hw_version);

	/* Driver version. */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_DRIVER_VERSION);
	vmk_StringCopy(eiter->a.driver_version, QFLE3F_VERSION,
			sizeof(eiter->a.driver_version));
	alen = strnlen(eiter->a.driver_version,
			sizeof(eiter->a.driver_version));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS, "DRIVERVER=%s.",
	    eiter->a.driver_version);

	/* Option ROM version. */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_OPTION_ROM_VERSION);
	sprintf(eiter->a.orom_version, "0.0.0");
	alen = strnlen(eiter->a.orom_version,
			sizeof(eiter->a.orom_version));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS,"OPTROMVER=%s.",
	    eiter->a.orom_version);

	/* Firmware version */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_FIRMWARE_VERSION);
	sprintf(eiter->a.fw_version, drv_info.fw_version);
	alen = strnlen(eiter->a.fw_version,
			sizeof(eiter->a.fw_version));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS,"FIRMWAREVER=%s.",
	    eiter->a.fw_version);

	/* OS name and version */
	vmk_SystemVersionInfo sys_info;
	vmk_SystemGetVersionInfo(&sys_info);

	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	sprintf(eiter->a.os_version,
		"%s-%s (%s)",
		sys_info.productName, sys_info.productVersion,
		sys_info.buildVersion);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_OS_NAME_AND_VERSION);
	alen = strnlen(eiter->a.os_version,
			sizeof(eiter->a.os_version));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS,"OSNAMEANDVERSION=%s.",
	    eiter->a.os_version);

	/* Attributes */
	if(flags == RHBA_FDMI) {
		ct_req->req.rhba.attrs.count =
		    __constant_vmk_CPUToBE32(count);
		goto send_pkt;
	}

	/* Max CT Length */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	max_frame_size = 0x800;
	eiter->a.max_ct_len = vmk_CPUToBE32(max_frame_size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_MAXIMUM_CT_PAYLOAD_LENGTH);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;
    count++;

	qfle3f_log(hba, LOG_ELS,"MAXCTLEN=%x.",
	    eiter->a.max_ct_len);

	/* Node Sym Name */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
    vmk_StringFormat(eiter->a.node_sym_name, 255, NULL,
        "%s DVR:v%s", drv_info.fw_version, QFLE3F_VERSION);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_NODE_SYM_NAME);
	alen = strnlen(eiter->a.node_sym_name,
			sizeof(eiter->a.node_sym_name));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS,"NODESYMNAME=%s.",
	    eiter->a.node_sym_name);

	/* Vendor Specific Info */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->a.vendor_info = __constant_vmk_CPUToBE32(0x1077);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_VENDOR_SPECIFIC_INFO);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;
    count++;

	qfle3f_log(hba, LOG_ELS,"VENDORSPECIFICINFO=%d.",
	    eiter->a.vendor_info);

	/* Number of Ports */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->a.no_of_ports = __constant_vmk_CPUToBE32(1);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_NO_OF_PORTS);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;
    count++;

	qfle3f_log(hba, LOG_ELS,"NOOFPORTS=%d.",
	    eiter->a.no_of_ports);

	/* Fabric Name */
	struct fcoe_ctlr *fip = &hba->ctlr;
	struct fcoe_fcf *fcf, *next;

	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	list_for_each_entry_safe(fcf, next, &fip->fcfs, list) {
        vmk_uint64_to_wwn(fcf->switch_name, eiter->a.fabric_name);
        break;
    }
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_FABRIC_NAME);
	eiter->len = __constant_vmk_CPUToBE16(4 + WWN_SIZE);
	size += 4 + WWN_SIZE;
    count++;

	qfle3f_log(hba, LOG_ELS,"FABRIC NODENAME=%02x%02x%02x%02x%02x%02x%02x%02x.",
	    eiter->a.fabric_name[0], eiter->a.fabric_name[1], eiter->a.fabric_name[2],
	    eiter->a.fabric_name[3], eiter->a.fabric_name[4], eiter->a.fabric_name[5],
	    eiter->a.fabric_name[6], eiter->a.fabric_name[7]);

	/* Boot BIOS Name */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	sprintf(eiter->a.boot_bios_name,
		"0.0.0");
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_BOOT_BIOS_NAME);
	alen = strnlen(eiter->a.boot_bios_name, 256);
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = __constant_vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS,"BOOTBIOSNAME=%s.",
	    eiter->a.boot_bios_name);

	/* Boot BIOS State */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	eiter->a.boot_bios_state = __constant_vmk_CPUToBE32(0xFFFFFFFF);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_BOOT_BIOS_STATE);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;
    count++;

	qfle3f_log(hba, LOG_ELS,"BOOTBIOSSTATE=0x%x.",
	    eiter->a.boot_bios_state);

	/* Vendor Identifier */
	eiter = (struct ct_fdmi_hba_attr *) (entries + size);
	vmk_StringCopy(eiter->a.vendor_identifier, "QLogic",
		sizeof(eiter->a.vendor_identifier));
	eiter->type = __constant_vmk_CPUToBE16(FDMI_HBA_VENDOR_IDENTIFIER);
	alen = strnlen(eiter->a.vendor_identifier,
			sizeof(eiter->a.vendor_identifier));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = __constant_vmk_CPUToBE16(4 + alen);
	size += 4 + alen;
    count++;

	qfle3f_log(hba, LOG_ELS,"VENDORIDENTIFIER=%s.",
	    eiter->a.vendor_identifier);

	if(flags == RHBA_FDMI2) {
		ct_req->req.rhba.attrs.count =
		    __constant_vmk_CPUToBE32(count);
    }

send_pkt:

	fp = ql_fcoe_frame_alloc(hba->qlfcoe_hbalport, size);
	if (!fp) {
		qfle3f_log(hba, LOG_ELS, "ELS RHBA FC Frame alloc failed");
		ret = 1;
		goto end;
	}

	fc_hdr = (uint8_t *)fc_frame_header_get(fp);

	/* Fill the FC Header */
	fc_fill_fc_hdr(fp, FC_RCTL_DD_UNSOL_CTL, did,
			fc_host_targetPortID(lport->host), FC_TYPE_CT,
		       FC_FC_FIRST_SEQ | FC_FC_END_SEQ | FC_FC_SEQ_INIT, 0);

	 /* Copy FC Frame header and payload into the frame */
	vmk_Memcpy(fc_hdr + hdr_len, ct_req, size);

	/* Send the CT Packet through a new exchange */
	if (!lport->tt.exch_seq_send(lport, fp, qfle3f_bk_response,
				NULL, hba, 2 * lport->r_a_tov)) {
		qfle3f_log(hba, LOG_ERROR,
			"Unable to send the RHBA to MGMT Server");
		hba->FDMICurrentState = ZERO_STATE;
	}

	qfle3f_log(hba, LOG_ELS, "FDMICurrentState = %s",
		qfle3f_els_status_to_string(hba->FDMICurrentState));

end:
	qfle3f_free(ct_req);
	return ret;
}

/**
 * Function Name : qfle3f_rpa_fdmi
 *
 * DESCRIPTION:
 *  Send a RPA(Register Port Attributes) ELS command
 *
 *
 * PARAMETERS:
 *  hba		The interface to be used to send the
 *  		ELS command.
 *  lport	Pointer to the local port structure.
 *  did		The Destination ID
 *  flags	RPA_SMARTSAN_FDMI2: Send SMARTSAN Data + FDMI2 attributes
 *  		RPA_FDMI2:          Send FDMI2 attributes
 *  		RPA_FDMI:           Send FDMI attributes
 *
 * RETURN:
 *  0		SUCCESS
 *  1		FAILURE
 *
 * NOTE:
 * Called should be holding the hba->backgroundMutex lock.
 *
 */
static int qfle3f_rpa_fdmi(struct qfle3fHBA *hba,
		struct fc_lport *lport,
		int did, int flags)
{

	FC_FRAME_HEADER *fp;
	struct ct_sns_req *ct_req;
	int ret = 0, alen, size = 0;
	uint32_t max_frame_size;
	char    nodename_str[16];
	uint8_t *entries;
	struct ct_fdmi_port_attr *eiter;
	uint8_t portName[WWN_SIZE], nodeName[WWN_SIZE];
	uint8_t *fc_hdr;
	struct	ethtool_drvinfo drv_info;
	int hdr_len = sizeof(FC_FRAME_HEADER);
	int count;

	ct_req = qfle3f_alloc(sizeof(struct ct_sns_req));
	if(!ct_req) {
		qfle3f_log(hba, LOG_ERROR,
			"Unable to alloc memory for the CT Request");
		hba->FDMICurrentState = ZERO_STATE;
		return 1;
	}

	if(flags != RPA_SMARTSAN_FDMI2 && flags != RPA_FDMI2 &&
		flags != RPA_FDMI)
		return 1;

	vmk_uint64_to_wwn(qfle3f_get_wwpn(hba), portName);
	vmk_uint64_to_wwn(qfle3f_get_wwnn(hba), nodeName);

	qfle3f_log(hba, LOG_ELS, "FDMICurrentState = %s",
		qfle3f_els_status_to_string(hba->FDMICurrentState));

	sprintf(nodename_str,
		"%02x%02x%02x%02x%02x%02x%02x%02x",
		nodeName[0], nodeName[1], nodeName[2],
		nodeName[3], nodeName[4], nodeName[5],
		nodeName[6], nodeName[7]);

	/* Prepare CT request */
	ct_req = qfle3f_prep_ct_fdmi_req(ct_req, RPA_CMD,
	    RPA_RSP_SIZE);

	/* Prepare FDMI command arguments -- attribute block, attributes. */
	vmk_Memcpy(ct_req->req.rpa.portName, portName, WWN_SIZE);
	entries = (uint8_t *)ct_req;

	if(flags == RPA_SMARTSAN_FDMI2)
		count = FDMI2_SMARTSAN_PORT_ATTR_COUNT;
	else if(flags == RPA_FDMI2)
		count = FDMI2_PORT_ATTR_COUNT;
	else if (flags == RPA_FDMI)
		count = FDMI_PORT_ATTR_COUNT;

	ct_req->req.rpa.attrs.count =
		__constant_vmk_CPUToBE32(count);
	size = 28;

	if (hba->cnic && hba->cnic->netdev) {
		hba->cnic->netdev->ethtool_ops->get_drvinfo(hba->cnic->netdev,
			&drv_info);
	}

	/* FC4 types. */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_FC4_TYPES);
	eiter->len = __constant_vmk_CPUToBE16(4 + 32);
	{
		uint32_t bit_mask[2];
		uint32_t type_value, index;
		uint32_t *bit_maskp = bit_mask;

		vmk_Memset(bit_mask, 0, 8);

		index = 0x8 >> 5;
		type_value = 1 << (0x8 % 32);
		bit_maskp[index] = vmk_CPUToBE32(type_value);
		vmk_Memcpy(eiter->a.fc4_types, bit_mask, 8);
	}
	size += 4 + 32;

	qfle3f_log(hba, LOG_ELS,"FC4TYPES=%d:%d:%d:%d:%d:%d:%d:%d.",
	    eiter->a.fc4_types[0], eiter->a.fc4_types[1],
	    eiter->a.fc4_types[2], eiter->a.fc4_types[3],
	    eiter->a.fc4_types[4], eiter->a.fc4_types[5],
	    eiter->a.fc4_types[6], eiter->a.fc4_types[7]);

	/* Supported speed. */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SUPPORT_SPEED);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	eiter->a.sup_speed = __constant_vmk_CPUToBE32(
	    FDMI_PORT_SPEED_10GB);

	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS," SUPPORTED_SPEED=%x.",
	    eiter->a.sup_speed);

	/* Current speed. */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_CURRENT_SPEED);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	switch (lport->link_speed) {
	case FC_PORTSPEED_1GBIT:
		eiter->a.cur_speed =
		    __constant_vmk_CPUToBE32(FDMI_PORT_SPEED_1GB);
		break;
	case FC_PORTSPEED_2GBIT:
		eiter->a.cur_speed =
		    __constant_vmk_CPUToBE32(FDMI_PORT_SPEED_2GB);
		break;
	case FC_PORTSPEED_10GBIT:
		eiter->a.cur_speed =
		    __constant_vmk_CPUToBE32(FDMI_PORT_SPEED_10GB);
		break;
	default:
		eiter->a.cur_speed =
		    __constant_vmk_CPUToBE32(FDMI_PORT_SPEED_UNKNOWN);
		break;
	}
	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS," CURRENT_SPEED=%x.",
	    eiter->a.cur_speed);

	/* Max frame size. */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_MAX_FRAME_SIZE);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	max_frame_size = 0x800;
	eiter->a.max_frame_size = vmk_CPUToBE32(max_frame_size);
	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS," MAX_FRAME_SIZE=%x.",
	    eiter->a.max_frame_size);

	/* OS device name. */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_OS_DEVICE_NAME);
	sprintf(eiter->a.os_dev_name, "%s", vmklnx_get_vmhbaName(lport->host));
	alen = strnlen(eiter->a.os_dev_name,
			sizeof(eiter->a.os_dev_name));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;

	qfle3f_log(hba, LOG_ELS," OS_DEVICE_NAME=%s.",
	    eiter->a.os_dev_name);

	/* Hostname. */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_HOST_NAME);
	vmk_Memcpy(eiter->a.host_name, nodename_str, 2 * WWN_SIZE);
	alen = strnlen(eiter->a.host_name,
			sizeof(eiter->a.host_name));
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;

	qfle3f_log(hba, LOG_ELS," HOSTNAME=%s.",
	    eiter->a.host_name);

	/* No need to register fdmi2 port attributes */
	if(flags == RPA_FDMI)
		goto send_pkt;

	/* Node Name */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_NODE_NAME);
	eiter->len = __constant_vmk_CPUToBE16(4 + WWN_SIZE);
	vmk_Memcpy(eiter->a.nodeName, nodeName, WWN_SIZE);
	size += 4 + WWN_SIZE;

	qfle3f_log(hba, LOG_ELS," NODENAME=%02x%02x%02x%02x%02x%02x%02x%02x.",
	    eiter->a.nodeName[0], eiter->a.nodeName[1], eiter->a.nodeName[2],
	    eiter->a.nodeName[3], eiter->a.nodeName[4], eiter->a.nodeName[5],
	    eiter->a.nodeName[6], eiter->a.nodeName[7]);

	/* Port Name */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_PORT_NAME);
	eiter->len = __constant_vmk_CPUToBE16(4 + WWN_SIZE);
	vmk_Memcpy(eiter->a.portName, portName, WWN_SIZE);
	size += 4 + WWN_SIZE;

	qfle3f_log(hba, LOG_ELS," PORTNAME=%02x%02x%02x%02x%02x%02x%02x%02x.",
	    eiter->a.portName[0], eiter->a.portName[1], eiter->a.portName[2],
	    eiter->a.portName[3], eiter->a.portName[4], eiter->a.portName[5],
	    eiter->a.portName[6], eiter->a.portName[7]);

	/* PORT SYM NAME */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
    vmk_StringFormat(eiter->a.port_sym_name, 255, NULL,
        "%s DVR:v%s port", drv_info.fw_version, QFLE3F_VERSION);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SYM_NAME);
	alen = strnlen(eiter->a.port_sym_name, 256);
	alen += (alen & 3) ? (4 - (alen & 3)) : 4;
	eiter->len = vmk_CPUToBE16(4 + alen);
	size += 4 + alen;

	qfle3f_log(hba, LOG_ELS," PORTSYMNAME=%s.",
	    eiter->a.port_sym_name);

	/* PORT TYPE */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->a.portType = __constant_vmk_CPUToBE32(NS_NX_PORT_TYPE);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_TYPE);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS," PORT_TYPE=%d.",
	    eiter->a.portType);

	/* CLASS OF SERVICE */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SUPPORTED_COS);
	eiter->a.supported_cos = __constant_vmk_CPUToBE32(BNX_FC_CLASS_3);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS,"SUPPORTEDCOS=%x.",
	    eiter->a.supported_cos);

	/* PORT FABRIC NAME */
	struct fcoe_ctlr *fip = &hba->ctlr;
	struct fcoe_fcf *fcf, *next;

	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	list_for_each_entry_safe(fcf, next, &fip->fcfs, list) {
        vmk_uint64_to_wwn(fcf->fabric_name, eiter->a.port_fabric_name);
        break;
    }
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_FABRIC_NAME);
	eiter->len = __constant_vmk_CPUToBE16(4 + WWN_SIZE);
	size += 4 + WWN_SIZE;

	qfle3f_log(hba, LOG_ELS,
	    "PORTFABRICNAME=%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x.",
	    eiter->a.port_fabric_name[0], eiter->a.port_fabric_name[1],
	    eiter->a.port_fabric_name[2], eiter->a.port_fabric_name[3],
	    eiter->a.port_fabric_name[4], eiter->a.port_fabric_name[5],
	    eiter->a.port_fabric_name[6], eiter->a.port_fabric_name[7]);

	/* PORT ACT FC4 TYPE */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_ACT_FC4_TYPE);
	{
		uint32_t bit_mask[2];
		uint32_t type_value, index;
		uint32_t *bit_maskp = bit_mask;

		vmk_Memset(bit_mask, 0, 8);

		index = 0x8 >> 5;
		type_value = 1 << (0x8 % 32);
		bit_maskp[index] = vmk_CPUToBE32(type_value);
		vmk_Memcpy(eiter->a.port_act_fc4_type, bit_mask, 8);
	}

	eiter->len = __constant_vmk_CPUToBE16(4 + 32);
	size += 4 + 32;

	qfle3f_log(hba, LOG_ELS,"PORTACTFC4TYPE=%d:%d:%d:%d:%d:%d:%d:%d.",
	    eiter->a.port_act_fc4_type[0], eiter->a.port_act_fc4_type[1],
	    eiter->a.port_act_fc4_type[2], eiter->a.port_act_fc4_type[3],
	    eiter->a.port_act_fc4_type[4], eiter->a.port_act_fc4_type[5],
	    eiter->a.port_act_fc4_type[6], eiter->a.port_act_fc4_type[7]);

	/* PORT STATE */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_STATE);
	eiter->a.portState = __constant_vmk_CPUToBE32(HBA_PORTSTATE_ONLINE);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS," PORTSTATE=%d.",
	    eiter->a.portState);

	/* NUM Discovered PORTS */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_NUM_OF_PORTS);
	eiter->a.num_discovered_ports = __constant_vmk_CPUToBE32(hba->num_ofld_sess);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS," NUMOFDISCPORTS=%d.",
	    eiter->a.num_discovered_ports);


	uint32_t targetPortID = fc_host_targetPortID(lport->host);

	/* PORT IDENTIFIER */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_PORT_IDENTIFIER);
	eiter->a.targetPortIDentifier[0] = (targetPortID >> 24) & 0xFF;
	eiter->a.targetPortIDentifier[1] =(targetPortID >> 16) & 0xFF;
	eiter->a.targetPortIDentifier[2] = (targetPortID >> 8) & 0xFF;
	eiter->a.targetPortIDentifier[3] = targetPortID & 0xFF;
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS," PORTIDENTIFIER=0x%x:0x%x:0x%x:0x%x.",
	    eiter->a.targetPortIDentifier[0], eiter->a.targetPortIDentifier[1],
	    eiter->a.targetPortIDentifier[2], eiter->a.targetPortIDentifier[3]);

	if(flags == RPA_FDMI2)
		goto send_pkt;

	/* SMARTSAN SERVICE CATEGORY */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SMARTSAN_SERVICE_CATEGORY);
	vmk_Memset(eiter->a.smart_san_service_category, 0, 256);
	vmk_Memcpy(eiter->a.smart_san_service_category, "Smart SAN Initiator", 256);
	alen = strnlen(eiter->a.smart_san_service_category,
			sizeof(eiter->a.smart_san_service_category));
    alen += (alen & 3) ? (4 - (alen & 3)) : 4;
    eiter->len = __constant_vmk_CPUToBE16(4 + alen);
    size += 4 + alen;

	qfle3f_log(hba, LOG_ELS,"SMARTSANSERVICECATEGORY=%s.",
	    eiter->a.smart_san_service_category);

	/* SMARTSAN GUID */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SMARTSAN_GUID);
	vmk_Memcpy(eiter->a.smart_san_guid, nodeName, WWN_SIZE);
	vmk_Memcpy(eiter->a.smart_san_guid + WWN_SIZE, portName,
		WWN_SIZE);
	eiter->len = __constant_vmk_CPUToBE16(4 + 16);
	size += 4 + 16;

	qfle3f_log(hba, LOG_ELS,"SMARTSANGUID=%02x%02x%02x%02x%02x%02x%02x%02x-"
	    "%02x%02x%02x%02x%02x%02x%02x%02x.",
	    eiter->a.smart_san_guid[0], eiter->a.smart_san_guid[1], eiter->a.smart_san_guid[2],
	    eiter->a.smart_san_guid[3], eiter->a.smart_san_guid[4], eiter->a.smart_san_guid[5],
	    eiter->a.smart_san_guid[6], eiter->a.smart_san_guid[7], eiter->a.smart_san_guid[8],
	    eiter->a.smart_san_guid[9], eiter->a.smart_san_guid[10], eiter->a.smart_san_guid[11],
	    eiter->a.smart_san_guid[12], eiter->a.smart_san_guid[13], eiter->a.smart_san_guid[14],
	    eiter->a.smart_san_guid[15]);

	/* SMARTSAN VERSION */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SMARTSAN_VERSION);
	vmk_StringFormat(eiter->a.smart_san_version, 256, NULL,
			 "Smart SAN Version 1.0");
	alen = strnlen(eiter->a.smart_san_version,
			sizeof(eiter->a.smart_san_version));
    alen += (alen & 3) ? (4 - (alen & 3)) : 4;
    eiter->len = __constant_vmk_CPUToBE16(4 + alen);
    size += 4 + alen;

	qfle3f_log(hba, LOG_ELS,"SMARTSANVERSION=%s.",
	    (char *) eiter->a.smart_san_version);

	/* SMARTSAN PRODUCT NAME */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SMARTSAN_PRODUCT_NAME);
	vmk_StringFormat(eiter->a.smart_san_product_name, 16, NULL,
			 "BCM%s", hba->sym_name);
	eiter->len = __constant_vmk_CPUToBE16(4 + 16);
	size += 4 + 16;

	qfle3f_log(hba, LOG_ELS, "SMARTSANPRODUCTNAME=%s.",
	    eiter->a.smart_san_product_name);

	/* SMARTSAN PORT INFO */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SMARTSAN_PORT_INFO);
	eiter->a.smart_san_port_info = __constant_vmk_CPUToBE32(1);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS,"SMARTSANPORTINFO=%d.",
	    eiter->a.smart_san_port_info);

	/* SMARTSAN QOS SUPPORT */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SMARTSAN_QOS_SUPPORT);
	eiter->a.smart_san_qos_support = __constant_vmk_CPUToBE32(1);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS,"SMARTSANQOSSUPPORT=%d.",
	    eiter->a.smart_san_qos_support);

	/* SMARTSAN SECURITY SUPPORT */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SMARTSAN_SECURITY_SUPPORT);
	eiter->a.smart_san_security_support = __constant_vmk_CPUToBE32(1);
	eiter->len = __constant_vmk_CPUToBE16(4 + 4);
	size += 4 + 4;

	qfle3f_log(hba, LOG_ELS,"SMARTSANSECURITYSUPPORT=%d.",
	    eiter->a.smart_san_security_support);

	/* TODO: Currently not sending smartsan connected targets ports info.
	 *       Should fix this once HP wants us to add this to Smart SAN
	 *       features.
	 *       Currently decrementing count by 1.
	 */
	count = count - 1;
    ct_req->req.rpa.attrs.count = vmk_CPUToBE32(count);

	/* TODO: Currently not sending smartsan connected targets ports info.
	 *       Should fix this once HP wants us to add this to Smart SAN
	 *       features.
	 */
	goto send_pkt;

	/* SMARTSAN CONNECTED TARGET PORTS */
	eiter = (struct ct_fdmi_port_attr *) (entries + size);
	eiter->type = __constant_vmk_CPUToBE16(FDMI_PORT_SMARTSAN_CONN_TARGET_PORTS);
	eiter->a.smart_san_connected_target_ports[0] = 0;
	eiter->a.smart_san_connected_target_ports[3] = 0;
	eiter->len = __constant_vmk_CPUToBE16(4 + 8);
	size += 4 + 8;

	qfle3f_log(hba, LOG_ELS,"SMARTSANCONNECTEDTARGETPORTS=0x%x0x%x0x%x0x%x.",
	    eiter->a.smart_san_connected_target_ports[0],
	    eiter->a.smart_san_connected_target_ports[1],
	    eiter->a.smart_san_connected_target_ports[2],
	    eiter->a.smart_san_connected_target_ports[3]);

send_pkt:
	fp = ql_fcoe_frame_alloc(hba->qlfcoe_hbalport, size);
	if (!fp) {
		qfle3f_log(hba, LOG_ELS, "ELS RPA FC Frame alloc failed");
		ret = 1;
		goto end;
	}

	fc_hdr = fc_frame_header_get(fp);

	/* Fill the FC Header */
	fc_fill_fc_hdr(fp, FC_RCTL_DD_UNSOL_CTL, did,
			fc_host_targetPortID(lport->host), FC_TYPE_CT,
		       FC_FC_FIRST_SEQ | FC_FC_END_SEQ | FC_FC_SEQ_INIT, 0);

	 /* Copy FC Frame header and payload into the frame */
	vmk_Memcpy(fc_hdr + hdr_len, ct_req, size);

	/* Send the CT Packet through a new exchange */
	if (!lport->tt.exch_seq_send(lport, fp, qfle3f_bk_response,
				NULL, hba, 2 * lport->r_a_tov)) {
		qfle3f_log(hba, LOG_ERROR,
			"Unable to send the RPA to MGMT Server");
		hba->FDMICurrentState = ZERO_STATE;
	}

	qfle3f_log(hba, LOG_ELS, "FDMICurrentState = %s",
		qfle3f_els_status_to_string(hba->FDMICurrentState));

end:
	qfle3f_free(ct_req);
	return ret;
}

/**
 * Function Name : qfle3f_bk_send_els_cmd
 *
 * DESCRIPTION:
 *  Send a ELS command to the Fabric Mgmt Server(0xfffffa).
 *  Currently it only supports three kind of ELS commands.
 *
 *  PLOGI
 *  RHBA
 *  RPA
 *
 * PARAMETERS:
 *  hba		The interface to be used to send the
 *  		ELS command.
 *
 * RETURN:
 *  0		SUCCESS
 *  1		FAILURE
 *
 * NOTE:
 * Called should be holding the hba->backgroundMutex lock.
 *
 */
int qfle3f_bk_send_els_cmd(struct qfle3fHBA *hba) {
	struct qfle3f_port      *port = hba->port;
	struct fc_lport         *lport = port->lport;
	int rval;

	qfle3f_log(hba, LOG_ELS, "FDMICurrentState = %s",
		qfle3f_els_status_to_string(hba->FDMICurrentState));

	if((hba->FDMICurrentState == ZERO_STATE) ||
	    (hba->FDMICurrentState == ALL_DONE)) {
		return -1;
	}

	switch (hba->FDMICurrentState) {

	case PLOGI_SEND_MGMT_SERVER:
		rval = qfle3f_send_plogi(hba, lport, FC_FID_MGMT_SERV);
		if(!rval) {
			INIT_DELAYED_WORK(&hba->els_timeout_work,
				qfle3f_els_cmd_timeout);
			queue_delayed_work(hba->els_timeout_wq,
					&hba->els_timeout_work,
					msecs_to_jiffies(2 * lport->r_a_tov));
		}
	break;

	case RHBA_FDMI2:
	case RHBA_FDMI:
		rval = qfle3f_rhba_fdmi(hba, lport, FC_FID_MGMT_SERV,
					hba->FDMICurrentState);
		if(!rval) {
			INIT_DELAYED_WORK(&hba->els_timeout_work,
				qfle3f_els_cmd_timeout);
			queue_delayed_work(hba->els_timeout_wq,
					&hba->els_timeout_work,
					msecs_to_jiffies(2 * lport->r_a_tov));
		}
	break;

	case RPA_SMARTSAN_FDMI2:
	case RPA_FDMI2:
	case RPA_FDMI:
		rval = qfle3f_rpa_fdmi(hba, lport, FC_FID_MGMT_SERV,
				hba->FDMICurrentState);
		if(!rval) {
			INIT_DELAYED_WORK(&hba->els_timeout_work,
				qfle3f_els_cmd_timeout);
			queue_delayed_work(hba->els_timeout_wq,
					&hba->els_timeout_work,
					msecs_to_jiffies(2 * lport->r_a_tov));
		}
	break;

	case ZERO_STATE:
	case BEGIN_PROCESS:
	case ALL_DONE:
	default:
		qfle3f_log(hba, LOG_ERROR,
			"Nothing to do.");
	break;

	}

	qfle3f_log(hba, LOG_ELS, "FDMICurrentState = %s",
			qfle3f_els_status_to_string(hba->FDMICurrentState));

	return 0;
}

#endif
