/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#include <vmkapi.h>
#include <qfle3f.h>
#include <qfle3f_vmk.h>
#include <qfle3f_version.h>
#include <qcnic_if.h>
#include "ql_fcoe_mgmt/ql_fcoe.h"

extern vmk_MgmtApiSignature ql_fcoe_sig;
extern vmk_MgmtHandle qed_fcapi_vmkmgmt_api_handle;
extern VMK_ReturnStatus qfle3fDumpCallback( void *, vmk_Bool);

#ifdef NPIV_SUPPORT
extern vmk_ScsiVportOps qfle3f_vportops;
#endif

/* for vmkmgt API */
struct qfle3fDriverInfo_t qfle3fDriverInfo;
vmk_LogComponent qfle3f_vmkLog;

unsigned long qfle3f_debug_level = 0;
VMK_MODPARAM(qfle3f_debug_level, long,
            "Enable more messaging from the driver.");

int qfle3f_user_overwrote_devloss_val = 1;
int qfle3f_devloss_tmo = 0;
VMK_MODPARAM(qfle3f_devloss_tmo, int,
		" Time after device disappears until SCSI layer is made aware of device being unavailable\n"
		"    Default value is 20 seconds\n"
		"    Valid values are from 1 to 120 seconds\n"
		"    Our recommendation would be to not change this value\n");

int qfle3f_max_luns = QFLE3F_MAX_LUN;
VMK_MODPARAM(qfle3f_max_luns, int,
        "parameter to adjust max LUNs supported by driver.");

#define MAX_Q_DEPTH 32
int qfle3f_queue_depth = MAX_Q_DEPTH;
VMK_MODPARAM(qfle3f_queue_depth, int,
        "parameter to adjust per LUN max Q depth, default use OS settings.");

int qfle3f_enable_cvl_fix = 1;
VMK_MODPARAM(qfle3f_enable_cvl_fix, int,
        "parameter to enable/disable insertion of VxPort ID CVL.");

int qfle3f_enable_r_a_tov = 0;
VMK_MODPARAM(qfle3f_enable_r_a_tov, int,
        "parameter to enable/disable user defined R_A_TOV.");

int qfle3f_r_a_tov = 10;
VMK_MODPARAM(qfle3f_r_a_tov, int,
            "parameter to set the user define R_A_TOV value."
                " This parameter is only applicable when qfle3f_enable_r_a_tov is set to 1.");

int qfle3f_default_vlan = 1;
VMK_MODPARAM(qfle3f_default_vlan, int,
            "parameter to fallback to default VLAN 1002"
                " incase the FIP VLAN Discovery fails."
                " This is enabled by default.");

int qfle3f_create_vmkMgmt_Entry = 1;
VMK_MODPARAM(qfle3f_create_vmkMgmt_Entry, int,
            "Create the vmkMgmt Interface."
            " 1 = Enable."
            " 0 = Disable.");

extern struct cnic_ulp_ops qfle3f_cnic_cb;

extern struct ql_fcoe_driver_info qlfcoe_driver_info;

extern VMK_ReturnStatus qfle3f_registerPreBootVport(struct qfle3fHBA *);

struct cnic_ll2_cb_ops qfle3_ll2_cb_ops = {
	.rx_cb = qfle3f_rx_cb,
	.tx_cb = NULL,
};

/**
 * qfle3f_find_hba_for_cnic - maps cnic instance to qfle3f hba instance
 *
 * @cnic:	Pointer to cnic device instance
 *
 **/
struct qfle3fHBA *qfle3f_find_hba_for_cnic(struct cnic_dev *cnic)
{
	struct qfle3fHBA *hba;
	vmk_ListLinks *current, *nextPtr;

	/* Called with qfle3fDriverInfo.drvLock held */
	VMK_LIST_FORALL_SAFE(&qfle3fDriverInfo.qfle3fHostList, current, nextPtr) {
		hba = VMK_LIST_ENTRY(current, struct qfle3fHBA, link);
		if (hba->type == HBA_PHYSICAL && hba->cnic == cnic)
			return hba;
	}
	return NULL;
}

void qfle3f_deviceDown(struct qfle3f_rport *target);

void qfle3f_deviceDown(struct qfle3f_rport *target) {
    struct qfle3fHBA *hba = target->hba;
    VMK_ReturnStatus status;
    qfle3f_log(hba, LOG_SESS, "Waking up.");

    vmk_BitVectorSet(target->flags, QFLE3F_FLAG_DEV_LOSS);

    status = vmk_ScsiNotifyPathStateChangeAsync(
                        hba->scsiAdapter,
						target->vmk_channel,
						target->vmk_targetID, -1);
	qfle3f_notice(hba, "Notified Path State Change(%s): "
			"Target port DEAD: C=0x%x TGT ID=%d",
			vmk_StatusToString(status),
			target->vmk_channel, target->vmk_targetID);
	ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);
}

VMK_ReturnStatus
qfle3f_scheduleDevlossTimer(struct qfle3f_rport *target)
{
    struct qfle3fHBA *hba = target->hba;
    VMK_ReturnStatus status = VMK_OK;
    vmk_ListLinks *current, *next_item;

	qfle3f_log(hba, LOG_INIT, "TGT=%x portType=0x%x",
                    target->vmk_targetID, target->portType);

    // This reference would be release during dev loss time expiration
    // or when we cancel this timer.
    ql_vmk_ref_get(&target->refcount);

	/* TODO: Will lead to a deadlock if qfle3f_freeTarget gets invoked */
	vmk_SpinlockLock(hba->scsiScanListLock);
	VMK_LIST_FORALL_SAFE(&hba->scsiScanList, current, next_item) {
		struct qfle3f_rport *currTarget = VMK_LIST_ENTRY(current, struct qfle3f_rport, scsiScanList);
		if (currTarget == target) {
			vmk_ListRemove(&target->scsiScanList);
			ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);
			break;
		}
	}
	vmk_SpinlockUnlock(hba->scsiScanListLock);

    if(vmk_TimerIsPending(target->devLossTimer) == VMK_FALSE) {
        qfle3f_log(hba, LOG_SESS, "Scheduling a timer(%d seconds) for dev loss timeout.",
                hba->devLossTimeout);
        status = vmk_TimerSchedule(qfle3fDriverInfo.timerQueue,
                    (void *)qfle3f_deviceDown,
                    (void *) target,
                    (hba->devLossTimeout) * VMK_USEC_PER_SEC,
                    VMK_TIMER_DEFAULT_TOLERANCE,
                    VMK_TIMER_ATTR_NONE,
                    VMK_LOCKDOMAIN_INVALID,
                    VMK_SPINLOCK_UNRANKED,
                    &target->devLossTimer);
    } else {
		ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);
	}
    qfle3f_log(hba, LOG_SESS, "Return: %s", vmk_StatusToString(status));
    return status;
}

VMK_ReturnStatus
qfle3f_cancelDevlossTimer(struct qfle3f_rport *target, vmk_Bool wait)
{
    struct qfle3fHBA *hba = target->hba;
    VMK_ReturnStatus status = VMK_FAILURE;

    if(vmk_TimerIsPending(target->devLossTimer) == VMK_TRUE) {
		status = vmk_TimerCancel(target->devLossTimer, wait);
        if(status == VMK_OK) {
	        ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);
        } else {
			qfle3f_err(hba, "Unable to cancel devloss timer: 0x%x:0x%x",
					target->vmk_channel, target->vmk_targetID);
		}
    }

    return status;
}

VMK_ReturnStatus qfle3f_queue_scsi_scan(struct qfle3f_rport *target, vmk_Bool incReference)
{
	struct qfle3fHBA *hba = target->hba;
    VMK_ReturnStatus status = VMK_FAILURE;

	qfle3f_log(hba, LOG_INIT, "TGT= %x=%x portType=0x%x", target->fcoe_conn_id,
                    target->vmk_targetID, target->portType);

	/* if (target->portType == FCT_TARGET) { */
	if (1) {
		qfle3f_log(hba, LOG_INIT,"SV: is-unlinked=0x%x",
			vmk_ListIsUnlinkedElement(&target->scsiScanList));

		if(vmk_SpinlockLock(hba->scsiScanListLock) == VMK_DEATH_PENDING)
			return VMK_DEATH_PENDING;

		if (vmk_ListIsUnlinkedElement(&target->scsiScanList) == VMK_TRUE &&
				(vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFFLOADED))) {
			qfle3f_log(hba, LOG_SESS, "Adding target[%d] to scsi scan list: %p",
					target->vmk_targetID, target);
			vmk_ListInsert(&target->scsiScanList,
					vmk_ListAtRear(&hba->scsiScanList));
			if(incReference)
				ql_vmk_ref_get(&target->refcount);
			status = VMK_OK;
		}
		vmk_SpinlockUnlock(hba->scsiScanListLock);
	}

	return status;
}

VMK_ReturnStatus qfle3f_scsi_scan(void *data)
{
	struct qfle3fHBA *hba;
	vmk_ListLinks local_scsiScanList;
	vmk_ListLinks *current, *next_ptr;
	//vmk_Name name;
	struct qfle3f_rport *target = NULL;
	//qfle3f_link_port_t *link_port = NULL;
	VMK_ReturnStatus status = VMK_OK;

	hba = (struct qfle3fHBA *)data;

    qfle3f_log(hba, LOG_INIT,"Entering scsi scan world =0x%x.",
			hba->scsiScanWorldID);

	while (1) {

		if(vmk_AtomicRead64(&hba->wakeupPending) == VMK_FALSE) {
		    status = vmk_WorldWait(VMK_EVENT_NONE, VMK_LOCK_INVALID,
				VMK_TIMEOUT_UNLIMITED_MS, "scsi scan world sleeping");

		    /* Only exit the world if VMK_DEATH_PENDING is returned */
		    if(status == VMK_DEATH_PENDING)
			    goto exit_world;

		    if(status != VMK_OK)
			    continue;
        }

		qfle3f_log(hba, LOG_INIT, "waking scsi-scan world=0x%x status = %s.",
			hba->scsiScanWorldID, vmk_StatusToString(status));

		vmk_AtomicWrite64(&hba->wakeupPending, VMK_FALSE);

		if(hba->ioAllowed == VMK_FALSE) {
			qfle3f_log(hba, LOG_INIT,"IO is not yet allowed on the scsi adapter...");
			continue;
		}

		vmk_ListInit(&local_scsiScanList);
		if(vmk_SpinlockLock(hba->scsiScanListLock) == VMK_DEATH_PENDING)
			goto exit_world;
		if (vmk_ListIsEmpty(&hba->scsiScanList) == VMK_FALSE) {
			qfle3f_log(hba, LOG_INIT,"atleast one target pending for scan...");
			vmk_ListSplitBefore(&hba->scsiScanList, &local_scsiScanList,
					vmk_ListLast(&hba->scsiScanList));
		}

		vmk_SpinlockUnlock(hba->scsiScanListLock);

		VMK_LIST_FORALL_SAFE(&local_scsiScanList, current, next_ptr) {
			target = VMK_LIST_ENTRY(current, struct qfle3f_rport, scsiScanList);
			qfle3f_log(hba, LOG_INIT,"work on target=%p",target);
			if (!target)
				continue;

			vmk_ListRemove(&target->scsiScanList);

			if(!vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_READY)) {
				qfle3f_log(hba, LOG_INIT,"io not allowed, so queuing again...");
				ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);

				if(vmk_WorldSleep(100*VMK_USEC_PER_MSEC) == VMK_DEATH_PENDING)
					goto exit_world;

				VMK_ReturnStatus localStatus = qfle3f_queue_scsi_scan(target, VMK_FALSE);
				if(localStatus == VMK_DEATH_PENDING) {
					goto exit_world;
				}
				continue;
			}

			if (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY)) {
				qfle3f_log(hba, LOG_SESS, "Target[%d] Session not ready. No need to notify",
						target->vmk_targetID);
				ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);
				continue;
			}

			qfle3f_log(hba, LOG_INIT,"Invoking scan for %s: TGT ID=0x%d",
				(char *)&hba->scsiAdapter->name,
				target->vmk_targetID);
			status = vmk_ScsiScanAndClaimPaths(
					&target->hba->scsiAdapter->name, 0,
					target->vmk_targetID, VMK_SCSI_PATH_ANY_LUN);

			if (status == VMK_OK) {
				status = vmk_ScsiNotifyPathStateChangeAsync(
						target->hba->scsiAdapter,
						target->vmk_channel,
						target->vmk_targetID, -1);
				qfle3f_notice(target->hba, "Notified Path State Change(%s): "
						"Target port ACTIVE: C=0x%x TGT ID=%d",
						vmk_StatusToString(status),
						target->vmk_channel, target->vmk_targetID);
			}

			if(vmk_WorldSleep(100*VMK_USEC_PER_MSEC) == VMK_DEATH_PENDING) {
				vmk_WarningMessage("VMK_DEATH_PENDING %d\n", __LINE__);
				ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);
				goto exit_world;
			}
			if (status != VMK_OK) {
				VMK_ReturnStatus localStatus = VMK_OK;
				localStatus = qfle3f_queue_scsi_scan(target, VMK_FALSE);

				if(localStatus == VMK_DEATH_PENDING) {
					vmk_WarningMessage("VMK_DEATH_PENDING %d\n", __LINE__);
					goto exit_world;
				}
			} else {
				ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);
			}
		}
	} /* end while */

exit_world:
	qfle3f_log(hba, LOG_INIT, "Exiting the scsi scan world.");
	return VMK_OK;
}


VMK_ReturnStatus qfle3f_vport_create_from_nvram(struct qfle3fHBA *);

static VMK_ReturnStatus qfle3fAttachDevice(vmk_Device vmkdev)
{
    VMK_ReturnStatus status = VMK_OK;
    struct qfle3fHBA *hba = NULL;
	int rc = 0;
	struct cnic_dev *cnic = NULL;

    qfle3f_log(hba, LOG_INIT, "Entered");

    status = vmk_DeviceGetRegisteringDriverData(vmkdev,
            (vmk_AddrCookie *)&cnic);
    if ((status != VMK_OK) || (!cnic)) {
        qfle3f_warning(hba, "Failed to get vmk_DeviceGetRegisteringDriverData %s",
                    vmk_StatusToString(status));
        return status;
    }

    qfle3f_log(hba, LOG_INIT, "cnic = %p", cnic);

	hba = qfle3f_interface_create(cnic, 0);
    if (hba == NULL) {
        status = VMK_FAILURE;
        qfle3f_warning(hba, "Failed to alloc hba %s", vmk_StatusToString(status));
        return status;
    }

	/* NPIV: populate physical adapter fields. */
	hba->type = HBA_PHYSICAL;
	hba->phba = NULL;

	hba->pdev = cnic->pdev;
    hba->deviceCreatedByCnicDriver = vmkdev;
    hba->qDepth = 1024;

    qfle3f_log(hba, LOG_INIT, "Query PCI Device");
    status = vmk_PCIQueryDeviceAddr(hba->pdev ,&hba->vmk_pci_addr);
    if (status != VMK_OK) {
        qfle3f_warning(hba, "FAILED PCI device query :%s",
            vmk_StatusToString(status));
    } else {
        qfle3f_warning(hba, "Enumerated PCI Function");
    }

	status = vmk_PCIQueryDeviceID(hba->pdev, &hba->pdev_id);
	if (status != VMK_OK) {
		qfle3f_warning(hba, "FAILED PCI device Id query :%s",
            vmk_StatusToString(status));
	}


    qfle3f_log(hba, LOG_INIT, "Calling scsi adapter alloc.");
    hba->scsiAdapter = vmk_ScsiAllocateAdapter();
    if (hba->scsiAdapter == NULL) {
        status = VMK_NO_MEMORY;
        qfle3f_warning(hba, "Could not obtain memory for scsi hba %s",
                    vmk_StatusToString(status));
        goto alloc_scsi_fail;
    }

    hba->nicDevice = cnic->ldev;
	hba->wwpn = cnic->fcoe_wwpn;
	hba->wwnn = cnic->fcoe_wwnn;

	hba->host_no = hba->scsiAdapter->hostNum;
	hba->vmhbaName = vmk_ScsiGetAdapterName(hba->scsiAdapter);

    status = qfle3f_scsiAdapterSetup(hba);
    if (status != VMK_OK) {
	goto setup_adapter_fail;
    }

    status = vmk_DeviceSetAttachedDriverData(vmkdev, hba);
    if (status != VMK_OK) {
        qfle3f_warning(hba, "Failed to save driver private data %s",
                    vmk_StatusToString(status));
        goto set_data_failed;
    }

    qfle3f_log(hba, LOG_INIT, "cnic->fip_mac: ");
    qfle3f_log(hba, LOG_INIT, VMK_ETH_ADDR_FMT_STR,
                    VMK_ETH_ADDR_FMT_ARGS(cnic->fip_mac));
    vmk_Memcpy(hba->dataSourceMacAddress, cnic->fip_mac, VMK_ETH_ADDR_LENGTH);
    qfle3f_log(hba, LOG_INIT, "hba->dataSourceMacAddress: ");
    qfle3f_log(hba, LOG_INIT, VMK_ETH_ADDR_FMT_STR,
                    VMK_ETH_ADDR_FMT_ARGS(hba->dataSourceMacAddress));

    qfle3f_log(hba, LOG_INIT, "Setting the ql_fcoe_mgmt Interface");
	status = qfle3f_interface_setup(hba);
	if(status != VMK_OK) {
        qfle3f_warning(hba, "Interface setup failed %s",
                    vmk_StatusToString(status));
		goto qlfcoe_hba_alloc_fail;
	}

    qfle3f_log(hba, LOG_INIT, "Creating the scsi scan world");
	vmk_Name name;
	vmk_ListInit(&hba->scsiScanList);
	vmk_NameFormat(&name, "scsiScanListLock-%u", hba->instance);
	qfle3f_vmk_spin_lock_init(&hba->scsiScanListLock, LOCK_RANK_HIGHEST,
		(const char *)&name);

	vmk_AtomicWrite64(&hba->wakeupPending, VMK_FALSE);
	/* Scsi scan world per scsi hba */
	vmk_WorldProps scsi_scan_props;
	char scsi_scan_world_name[20];

	vmk_Memset(&scsi_scan_props, 0, sizeof(vmk_WorldProps));
	scsi_scan_props.heapID = qfle3fDriverInfo.moduleHeapID;

	scsi_scan_props.moduleID = vmk_ModuleCurrentID;
	scsi_scan_props.startFunction = (void *)qfle3f_scsi_scan;
	scsi_scan_props.data = (void *)hba;
	scsi_scan_props.schedClass = VMK_WORLD_SCHED_CLASS_QUICK;
	vmk_StringFormat(scsi_scan_world_name, 20, NULL,
		"qfle3f-scsi-scan-%u", hba->instance);
	scsi_scan_props.name = scsi_scan_world_name;
	status = vmk_WorldCreate(&scsi_scan_props, &(hba->scsiScanWorldID));
	if (status != VMK_OK) {
		qfle3f_warning(hba, "FAILED scsi_scan_world create :%s",
		    vmk_StatusToString(status));
		goto scsi_scan_thread_fail;
	}

    qfle3f_log(hba, LOG_INIT, "scsi scan world created. world id = %d.",
                hba->scsiScanWorldID);


	vmk_BitVectorClear(hba->flags2, QFLE3F_CNIC_REGISTERED);

    qfle3f_log(hba, LOG_INIT, "Calling the cnic register device");
    qfle3f_log(hba, LOG_INIT, "cnic : %p qfle3f_cnic_cb = %p",
                        hba->cnic, &qfle3f_cnic_cb);
   	rc = hba->cnic->register_device(hba->cnic, CNIC_ULP_FCOE,
	    					(void *) hba, &qfle3f_cnic_cb);
    status = rc;
        qfle3f_log(hba, LOG_INIT, "Returned from cnic register device:%s",
                        vmk_StatusToString(status));
    if (status) {
    	qfle3f_warning(hba, "register_device failed, status = %s",
                            vmk_StatusToString(status));
	    goto register_device_failed;
   	} else {
    	vmk_BitVectorSet(hba->flags2, QFLE3F_CNIC_REGISTERED);
	}
	cnic->ll2_ops.register_cb_ops(cnic, &qfle3_ll2_cb_ops, 0);

    qfle3f_log(hba, LOG_INIT, "Setting the timer work queue");
	hba->timerWorkQueue =
			ql_vmk_create_singlethread_workqueue("qfle3f_timer_wq",
						VMK_WORLD_SCHED_CLASS_QUICK);
	if (VMK_UNLIKELY(!hba->timerWorkQueue)) {
		qfle3f_err(hba, "cannot create timer_wq");
		status = VMK_FAILURE;
		goto timerWorkQueue_creation_failed;
	}

#ifdef __FCOE_IF_RESTART_WQ__
	hba->fcoeInterfaceRestartWorkQueue =
			ql_vmk_create_singlethread_workqueue("qfle3f_fcoe_if_restart",
							VMK_WORLD_SCHED_CLASS_QUICK);
	if (VMK_UNLIKELY(!hba->fcoeInterfaceRestartWorkQueue)) {
		qfle3f_err(hba, "cannot create fcoeInterfaceRestartWorkQueue");
		rc = -3;
		status = VMK_FAILURE;
		goto fcoeInterfaceRestartWorkQueue_creation_failed;
	}
#endif

    hba->ioAllowed = VMK_FALSE;

    qfle3f_log(hba, LOG_INIT, "Allocation cmd mgr.");
	hba->commandManager = qfle3f_commandManagerAlloc(hba, QFLE3F_MIN_XID,
					    QFLE3F_MAX_XID);
	if (VMK_UNLIKELY(!hba->commandManager)) {
		qfle3f_err(hba, "qfle3f_commandManagerAlloc() failed");
		status = VMK_NO_MEMORY;
		goto commandManager_alloc_failed;
	}

#ifdef NPIV_SUPPORT
	/* Enable NPIV support by default. */
	vmk_BitVectorSet(hba->flags2, QFLE3F_NPIV_SUPPORTED);
#endif

    /* Check if we want both of these flags.
     * If not, delete one of these flags
     */
    vmk_BitVectorSet(hba->hbaState, ADAPTER_STATE_READY);

    qfle3f_log(hba, LOG_INIT, "hba->qDepth = %d", hba->qDepth);

    vmk_HashProperties hashTableProps;
    vmk_Memset(&hashTableProps, 0, sizeof(hashTableProps));

    hashTableProps.moduleID = vmk_ModuleCurrentID;
    hashTableProps.heapID = vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
    hashTableProps.keyType = VMK_HASH_KEY_TYPE_OPAQUE;
    hashTableProps.keyFlags = VMK_HASH_KEY_FLAGS_LOCAL_COPY;
    hashTableProps.keySize = sizeof(vmk_uint32);
    hashTableProps.nbEntries = QFLE3_FCOE_NUM_CONNECTIONS;
    hashTableProps.acquire = NULL;
    hashTableProps.release = NULL;

    int size = vmk_HashGetAllocSize(QFLE3_FCOE_NUM_CONNECTIONS);
    qfle3f_notice(hba, "best estimate amount for Hash Table: %d", size);

    status = vmk_HashAlloc(&hashTableProps, &hba->connIdHashTable);
    if(status != VMK_OK) {
        qfle3f_err(hba, "HashTable Alloc failed");
        goto hashTableAllocFailed;
    }

	hba->devLossTimeout = qfle3f_devloss_tmo;

	/* Pre-Boot adapters. */
	qfle3f_vport_create_from_nvram(hba);

    qfle3f_log(hba, LOG_INIT, "returned (VMK_OK)");
    return VMK_OK;

/* Failure scenario */
vmk_HashRelease(hba->connIdHashTable);
hashTableAllocFailed:
qfle3f_commandManagerFree(hba->commandManager);
commandManager_alloc_failed:
ql_vmk_destroy_singlethread_workqueue(hba->fcoeInterfaceRestartWorkQueue);
fcoeInterfaceRestartWorkQueue_creation_failed:
ql_vmk_destroy_singlethread_workqueue(hba->timerWorkQueue);
timerWorkQueue_creation_failed:
register_device_failed:
//cnic_registeration_failed:
	vmk_WorldDestroy(hba->scsiScanWorldID);
    vmk_WorldWaitForDeath(hba->scsiScanWorldID);
scsi_scan_thread_fail:
qlfcoe_hba_alloc_fail:
set_data_failed:
    qfle3f_scsiAdapterDestroy(hba);
setup_adapter_fail:
    vmk_ScsiFreeAdapter(hba->scsiAdapter);
alloc_scsi_fail:
	qfle3f_interface_destroy(hba);
    vmk_LogMessage("Returned (%s)", vmk_StatusToString(status));
    return status;
}

VMK_ReturnStatus qfle3f_RemoveDevice(vmk_Device device);
vmk_DeviceOps qfle3fDeviceOps = {
	.removeDevice = qfle3f_RemoveDevice
};

VMK_ReturnStatus qfle3f_RemoveDevice(vmk_Device device)
{
	VMK_ReturnStatus status;
   	vmk_AddrCookie cookie;
    struct qfle3fHBA *hba = NULL;

	qfle3f_log(hba, LOG_INIT, "Enter.");

	status = vmk_DeviceGetRegisteringDriverData(device, &cookie);
	if (status != VMK_OK) {
		return status;
	}

	hba = (struct qfle3fHBA *) cookie.ptr;
	hba->vmkScsiAdapterDevice = NULL;

	status = vmk_DeviceUnregister(device);

	return status;
}
static VMK_ReturnStatus qfle3fScanDevice(vmk_Device vmkdev)
{
    VMK_ReturnStatus status = VMK_OK;
	vmk_BusType bus_type;
	vmk_Name bus_name;
	struct qfle3fHBA *hba = NULL, *vhba = NULL;
	struct cnic_dev *cnic = NULL;
	vmk_ListLinks *current = NULL, *nextPtr = NULL;
    vmk_Device parent = vmkdev;
	vmk_DeviceID devId;
	vmk_DeviceProps deviceProps;

    qfle3f_log(hba, LOG_INIT, "Enter");

    status = vmk_DeviceGetRegisteringDriverData(vmkdev,
            (vmk_AddrCookie *)&cnic);
    if ((status != VMK_OK) || (!cnic)) {
		qfle3f_warning(hba,
				"Failed to get vmk_DeviceGetRegisteringDriverData %s",
				vmk_StatusToString(status));
        return status;
    }

    if(cnic == NULL) {
		qfle3f_warning(hba, "cnic is NULL.");
		return VMK_FAILURE;
    }

    qfle3f_log(hba, LOG_INIT, "cnic = %p", cnic);
	status = vmk_DeviceGetAttachedDriverData(vmkdev,
			(vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		qfle3f_warning(hba, "Failed to get cnic_dev: %s",
				vmk_StatusToString(status));
		return status;
	}

    if(hba == NULL) {
		qfle3f_warning(hba, "hba is NULL.");
		return status;
    }

    if(hba->cnic != cnic) {
		qfle3f_warning(hba, "hba->cnic(%p) not equal to cnic(%p).",
                    hba->cnic, cnic);
		return status;
    }

	status = vmk_NameInitialize(&bus_name, VMK_LOGICAL_BUS_NAME);
	if (status != VMK_OK) {
		qfle3f_warning(hba, "Failed to initialize bus name: %s",
				vmk_StatusToString(status));
		return status;
	}

	status = vmk_BusTypeFind(&bus_name, &bus_type);
	if (status != VMK_OK) {
		qfle3f_warning(hba,"Failed to BusTypeFind: %s",
				vmk_StatusToString(status));
		return status;
	}

	/* Create Logical device for iSCSI protocol driver. */
	status = vmk_LogicalCreateBusAddress(
			qfle3fDriverInfo.qfle3fDriver, parent, 0,
			&devId.busAddress, &devId.busAddressLen);

	if (status != VMK_OK) {
		qfle3f_warning(hba, "Failed to created logicalbus: %s",
			vmk_StatusToString(status));
		goto  fail_register_logical_bus;
	}

	devId.busType = bus_type;
	devId.busIdentifier = VMK_SCSI_PSA_DRIVER_BUS_ID;
	devId.busIdentifierLen = vmk_Strnlen(devId.busIdentifier,
			VMK_MISC_NAME_MAX);
	deviceProps.registeringDriver = qfle3fDriverInfo.qfle3fDriver;
	deviceProps.deviceID = &devId;
	deviceProps.deviceOps = &qfle3fDeviceOps;
	deviceProps.registeringDriverData.ptr = hba;
	deviceProps.registrationData.ptr = hba->scsiAdapter;

	status = vmk_DeviceRegister(&deviceProps, parent, &hba->vmkScsiAdapterDevice);
	if (status != VMK_OK) {
		qfle3f_warning(hba, "Failed to register device: %s", vmk_StatusToString(status));
	} else {
		qfle3f_log(hba, LOG_INIT, "Registered Logical device (local %s:global %s) with device layer.",
			vmk_ScsiGetAdapterName(hba->scsiAdapter),
			devId.busAddress);

#ifdef NPIV_SUPPORT
		/* Register vport handling support with Logical device */
		if (vmk_BitVectorTest(hba->flags2, QFLE3F_NPIV_SUPPORTED)) {
			qfle3f_log(hba, LOG_INIT, "Register Vport Ops.");
			status = vmk_ScsiRegisterVportOps(hba->vmkScsiAdapterDevice,
					&qfle3f_vportops);
			if (status != VMK_OK) {
				qfle3f_warning(hba, "Failed to register VportOps: %s",
						vmk_StatusToString(status));
			}
		}
#endif
	}

	vmk_LogicalFreeBusAddress(qfle3fDriverInfo.qfle3fDriver,
			devId.busAddress);

	if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_QUIESCE)) {
		qfle3f_log(hba, LOG_INIT, "Reset the quiesce state");
		vmk_BitVectorClear(hba->hbaState, ADAPTER_STATE_QUIESCE);
	}

    vmk_uint32 event = NETDEV_DOWN;
    qfle3f_log(hba, LOG_INIT, "Link state is cnic->uplinkLinkState.state = %d",
                    cnic->uplinkLinkState.state);

    if(cnic->uplinkLinkState.state == VMK_LINK_STATE_UP) {
        event = NETDEV_UP;
    } else if(cnic->uplinkLinkState.state == VMK_LINK_STATE_DOWN) {
        event = NETDEV_DOWN;
    }

    qfle3f_cnic_cb.indicate_netevent(hba, event);

	/* Register pre-boot vports. */
	/* As this is init path, vportListLck not needed. */
	VMK_LIST_FORALL_SAFE(&hba->vportList, current, nextPtr) {
		vhba = VMK_LIST_ENTRY(current, struct qfle3fHBA, vportList);

		qfle3f_log(vhba, LOG_NPIV, "Register pre-boot vport: %p.", vhba);
		status = qfle3f_registerPreBootVport(vhba);
		if (status) {
			qfle3f_warning(vhba, "pre-boot register failed. status=%s",
					vmk_StatusToString(status));
			//SV: TODO: qfle3f_deletePreBootVport();
		}
	}

    qfle3f_log(hba, LOG_INIT, "Exit");
	status = VMK_OK;

fail_register_logical_bus:
	vmk_BusTypeRelease(bus_type);

	return status;
}

static VMK_ReturnStatus qfle3fDetachDevice(vmk_Device vmkdev)
{
	struct qfle3fHBA *hba = NULL, *vhba = NULL;
	struct cnic_dev *cnic = NULL;
	VMK_ReturnStatus status = VMK_OK;
	struct qfle3f_rport *target = NULL;
	vmk_uint32 i = 0;
	vmk_uint64 target_ref = 0;

	 qfle3f_log(hba, LOG_INIT, "enter");

	status = vmk_DeviceGetRegisteringDriverData(vmkdev,
			(vmk_AddrCookie *)&cnic);

	if ((status != VMK_OK) || (!cnic)) {
		qfle3f_warning(hba, "failed to get vmk_DeviceGetRegisteringDriverData %s",
					vmk_StatusToString(status));
		return status;
	}

	qfle3f_log(hba, LOG_INIT, "cnic = %p", cnic);

	hba = qfle3f_find_hba_for_cnic(cnic);

	if (!hba) {
		qfle3f_warning(hba, "failed to find corresponding hba for cnic = %p.\n", cnic);
		return VMK_FAILURE;
	}

	qfle3f_notice(hba, "target check work.\n");

	for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
			target = hba->targetOffloadList[i];
			if(target) {
				target_ref = vmk_AtomicRead64(&target->refcount);
					vmk_WarningMessage("%s:target %d is not freed with refcount = 0x%lx. "
							"Free it now\n",
							vmk_ScsiGetAdapterName(hba->scsiAdapter),
							target->vmk_targetID, target_ref);
					status = qfle3f_cancelDevlossTimer(target, VMK_TRUE);
					if (status == VMK_OK) {
						qfle3f_notice(hba, "timer cancelled successfully");
						target_ref = target_ref - 1;
					}

					if (target_ref >= 1) {
						vmk_AtomicWrite64(&target->refcount, 1);
						ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);
					}

			}
	}
	qfle3f_notice(hba, "done vmk_ScsiFreeTarget.\n");

	qfle3f_notice(hba, "set HBA detach flag.\n");

	if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_DETACH)) {
		return VMK_OK;
	}
	vmk_BitVectorSet(hba->hbaState, ADAPTER_STATE_DETACH);

	/* GSS: TODO: Implement Preboot NPIV cleanup here */
	vmk_SpinlockLock(hba->vportListLck);
	for(i = 0; i < QFLE3F_MAX_NPIV; i++) {
		if (hba->vhba_address[i] != NULL) {
			vhba = hba->vhba_address[i];

			qfle3f_log(vhba, LOG_NPIV, "cleanup vport");
			/* Destroying the flogi poll world */
			status = vmk_WorldDestroy(vhba->flogiPollWorldID);
			if(status != VMK_OK) {
				qfle3f_log(vhba, LOG_NPIV, "unable to find flogi poll world");
			} else {
				qfle3f_log(vhba, LOG_NPIV, "successfully posted work to destroy flogi poll world");
			}
			status = vmk_WorldWaitForDeath(vhba->flogiPollWorldID);
			if(status != VMK_OK) {
				qfle3f_log(vhba, LOG_NPIV, "unable to destroy the flogi poll world");
				return VMK_BAD_PARAM;
			}
		}
	}
	vmk_SpinlockUnlock(hba->vportListLck);

	if (vmk_BitVectorAtomicTestAndClear(hba->flags2,
						QFLE3F_CNIC_REGISTERED)) {
		cnic->ll2_ops.unregister_cb_ops(hba->cnic);
		qfle3f_notice(hba, "Unregistering device from CNIC.\n");
		hba->cnic->unregister_device(hba->cnic, CNIC_ULP_FCOE);
	}

	qfle3f_notice(hba, "Hash Release  connIdHashTable.\n");
	vmk_HashRelease(hba->connIdHashTable);
	qfle3f_notice(hba, "qfle3f_commandManagerFree.\n");
	qfle3f_commandManagerFree(hba->commandManager);

#ifdef __FCOE_IF_RESTART_WQ__
	qfle3f_notice(hba, "fcoeInterfaceRestartWorkQueue destroy.\n");
	ql_vmk_destroy_singlethread_workqueue(hba->fcoeInterfaceRestartWorkQueue);
#endif

	qfle3f_notice(hba, "timerWorkQueue destroy.\n");
	ql_vmk_destroy_singlethread_workqueue(hba->timerWorkQueue);

	qfle3f_notice(hba, "scsiScanWorldID destroy.\n");
	vmk_WorldDestroy(hba->scsiScanWorldID);
	status = vmk_WorldWaitForDeath(hba->scsiScanWorldID);
	if(status != VMK_OK) {
		qfle3f_err(hba, "Unable to destroy the scsi scan world.\n");
	}
	hba->scsiScanWorldID = VMK_INVALID_WORLD_ID;

	qfle3f_notice(hba, "ioDMAEngine destroy.\n");
	vmk_DMAEngineDestroy(hba->ioDMAEngine);

	qfle3f_notice(hba, "Free mgmtAdapter.t.fcoe is done");
	qfle3f_free(hba->scsiAdapter->mgmtAdapter.t.fcoe);

	qfle3f_notice(hba, "vmk_ScsiFreeAdapter.\n");
	vmk_ScsiFreeAdapter(hba->scsiAdapter);


	if (vmk_BitVectorAtomicTestAndClear(hba->flags2,
		QFLE3F_CNIC_REGISTERED))
			hba->cnic->unregister_device(hba->cnic, CNIC_ULP_FCOE);


	qfle3f_interface_destroy(hba);

	vmk_LogMessage("%s: exit", __func__);

    return VMK_OK;
}

void qfle3f_wait_for_upload(struct qfle3fHBA *hba)
{
	vmk_uint32 count = 1;
	vmk_uint32 numSecs = 10;
	vmk_uint32 numMins = 5 * 60;
	while(1) {
		if(hba->num_ofld_sess) {
			/* The result will be true once every numSecs (10) */
			if(!(count % (2 * numSecs))) {
				qfle3f_warning(hba, "Waiting for uploads to complete, num_ofld_sess = %d", hba->num_ofld_sess);
			}
			/* Bring the system down after 5 mins */
			if (count >= (2 * numMins)) {
				vmk_Panic("Failed to upload all sessions");
			}
		} else {
			break;
		}
		/* 500 ms */
		vmk_WorldSleep(500 * VMK_USEC_PER_MSEC);
		count++;
	}
}

static VMK_ReturnStatus qfle3fQuiesceDevice(vmk_Device vmkdev)
{
	struct cnic_dev *cnic;
    vmk_uint32 event;
	struct qfle3fHBA *hba = NULL;
	vmk_uint8 wwpn_array[WWN_SIZE], wwnn_array[WWN_SIZE];
	VMK_ReturnStatus status;

	qfle3f_warning(hba, "Enter");

	status = vmk_DeviceGetRegisteringDriverData(vmkdev, (vmk_AddrCookie *)&cnic);
    if ((status != VMK_OK) || (!cnic)) {
        qfle3f_warning(hba, "Failed to get vmk_DeviceGetRegisteringDriverData %s",
                    vmk_StatusToString(status));
        return status;
    }

    if(cnic == NULL) {
		qfle3f_warning(hba, "cnic is NULL.");
		return VMK_FAILURE;
    }

    qfle3f_log(hba, LOG_INIT, "cnic = %p", cnic);
	status = vmk_DeviceGetAttachedDriverData(vmkdev,
			(vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		qfle3f_warning(hba, "Failed to get cnic_dev: %s", vmk_StatusToString(status));
		return status;
	}

    if(hba == NULL) {
		qfle3f_warning(hba, "hba is NULL.");
		return status;
    }

	if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_QUIESCE)) {
		return VMK_OK;
	}
	vmk_BitVectorSet(hba->hbaState, ADAPTER_STATE_QUIESCE);

	vmk_BitVectorClear(hba->hbaState, ADAPTER_STATE_UP);
	vmk_BitVectorClear(hba->hbaState, ADAPTER_STATE_GOING_DOWN);
	vmk_BitVectorClear(hba->hbaState, ADAPTER_STATE_READY);

	vmk_LogMessage("Number of Offloaded Sessions =%d", hba->num_ofld_sess);

	/* Initiate a link down to kick off session upload/cleanup */
	if(cnic->uplinkLinkState.state == VMK_LINK_STATE_UP) {
		event = NETDEV_GOING_DOWN;
		qfle3f_cnic_cb.indicate_netevent(hba, event);
	}

	qfle3f_wait_for_upload(hba);
	vmk_LogMessage("All sessions are uploaded, num_ofld_sess =%d", hba->num_ofld_sess);

	/* Initiate Session cleanup after waiting for fabric cleanup. */
	qfle3f_u64ToCharArray_BE(hba->wwpn, WWN_SIZE, wwpn_array);
	qfle3f_u64ToCharArray_BE(hba->wwnn, WWN_SIZE, wwnn_array);
	ql_wait_for_fabric_deletion(hba->qlfcoeAdapter, wwpn_array, wwnn_array, PHYSICAL_FABRIC);
	ql_fcoe_interface_destroy(hba->qlfcoeAdapter);

	qfle3f_stop(hba);
	qfle3f_fw_destroy(hba);

	qfle3f_warning(hba, "Exit");
	return VMK_OK;
}

static VMK_ReturnStatus qfle3fStartDevice(vmk_Device vmkdev)
{
    struct qfle3fHBA *hba = NULL;
	struct cnic_dev *cnic;
	VMK_ReturnStatus status;

	status = vmk_DeviceGetRegisteringDriverData(vmkdev, (vmk_AddrCookie *)&cnic);
	if ((status != VMK_OK) || (!cnic)) {
		qfle3f_warning(hba, "Failed to get vmk_DeviceGetRegisteringDriverData %s",
				vmk_StatusToString(status));
		return status;
	}

	if (cnic == NULL) {
		qfle3f_warning(hba, "cnic is NULL.");
		return VMK_FAILURE;
	}

	qfle3f_log(hba, LOG_INIT, "cnic = %p", cnic);

	status = vmk_DeviceGetAttachedDriverData(vmkdev,
			(vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		qfle3f_warning(hba, "Failed to get cnic_dev: %s", vmk_StatusToString(status));
		return status;
	}

	if (hba == NULL) {
		qfle3f_warning(hba, "hba is NULL.");
		return status;
	}

	qfle3f_log(hba, LOG_INIT, "Enter");

	qfle3f_log(hba, LOG_INIT, "Exit");

	return VMK_OK;
}

static void qfle3fForgetDevice(vmk_Device vmkdev)
{
	struct qfle3fHBA *hba = NULL;
	struct cnic_dev *cnic;
	VMK_ReturnStatus status;

	status = vmk_DeviceGetRegisteringDriverData(vmkdev, (vmk_AddrCookie *)&cnic);
	if ((status != VMK_OK) || (!cnic)) {
		qfle3f_warning(hba, "Failed to get vmk_DeviceGetRegisteringDriverData %s",
				vmk_StatusToString(status));
		return;
	}

	if (cnic == NULL) {
		qfle3f_warning(hba, "cnic is NULL.");
		return;
	}

	qfle3f_log(hba, LOG_INIT, "cnic = %p", cnic);

	status = vmk_DeviceGetAttachedDriverData(vmkdev,
			(vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		qfle3f_warning(hba, "Failed to get cnic_dev: %s", vmk_StatusToString(status));
		return;
	}

	if (hba == NULL) {
		qfle3f_warning(hba, "hba is NULL.");
		return;
	}

	qfle3f_log(hba, LOG_INIT, "Enter");

	qfle3f_log(hba, LOG_INIT, "Exit");

	return;
}

/*
 * Sequence of call made when the Qlogic PCI device is discovered.
 * - attachDevice
 * - startDevice
 * - scanDevice
 */
static vmk_DriverOps pciDriverOps = {
	.attachDevice	= qfle3fAttachDevice,
	.scanDevice	= qfle3fScanDevice,
	.detachDevice	= qfle3fDetachDevice,
	.quiesceDevice	= qfle3fQuiesceDevice,
	.startDevice = qfle3fStartDevice,
	.forgetDevice = qfle3fForgetDevice,
};


int init_module(void)
{
	vmk_HeapID heap_id;
	vmk_Name heap_name;
	vmk_HeapCreateProps heapProps;
	VMK_ReturnStatus vmk_stat = VMK_OK;
#if (VMWARE_ESX_DDK_VERSION <= 65000)
	vmk_ModuleID moduleID;
#endif
	vmk_DriverProps pci_drv_props;
	vmk_LogProperties log_props;
	vmk_TimerQueueProps timerqueue_props;
	vmk_ByteCount worldHeapSize = 0, worldHeapAlignment = 0;
	vmk_ByteCount heapMax = 0;
	//vmk_MgmtProps mgmtProps;
#if (VMWARE_ESX_DDK_VERSION >= 65000)
	vmk_MgmtProps mgmtProps;
#endif

    vmk_LogMessage("qfle3f(version: %s)", QFLE3F_VERSION);

   /*
    * Register module with vmkernel
    */
#if (VMWARE_ESX_DDK_VERSION <= 65000)
   vmk_stat = vmk_ModuleRegister(&moduleID, VMKAPI_REVISION);

   if (vmk_stat != VMK_OK) {
      vmk_WarningMessage("vmk_ModuleRegister failed: %s",
                         vmk_StatusToString(vmk_stat));
      vmk_WarningMessage("Could not load %s module",
                          QFLE3F_DRIVER_NAME);
      goto module_register_fail;
   }
   qfle3fDriverInfo.moduleID = moduleID;
#else
   qfle3fDriverInfo.moduleID = vmk_ModuleCurrentID;
#endif

   /*
    * Set driver name here
    */
   vmk_stat = vmk_NameInitialize(&qfle3fDriverInfo.driverName,
                                 QFLE3F_DRIVER_NAME);

   /* Creating driver heap */
   heapProps.type = VMK_HEAP_TYPE_SIMPLE;
   vmk_NameFormat(&heap_name, "%s-gen", QFLE3F_DRIVER_NAME);
   vmk_stat = vmk_NameInitialize(&heapProps.name, (const char*)&heap_name);
   heapProps.module = vmk_ModuleCurrentID;
   heapProps.initial = QFLE3F_HEAP_INITIAL_SIZE;
   heapProps.max = QFLE3F_HEAP_MAXIMUM_SIZE;
   heapProps.creationTimeoutMS = VMK_TIMEOUT_NONBLOCKING;

	/*
	 * The max World number is 4096. Driver will create two World for each phyport and
	 * create one World for NPIV port. There are 16 phyport on HBA and each phyport
	 * contains 254 NPIV port. So the total number is 16*2 + 254*16.
	 */
    worldHeapSize = vmk_WorldCreateAllocSize(&worldHeapAlignment);
    vmk_HeapAllocationDescriptor heapAllocDesc[] = {
            {QFLE3F_HEAP_MAXIMUM_SIZE, 0, 1},
            {worldHeapSize, worldHeapAlignment, 4096}
    };
    vmk_stat = vmk_HeapDetermineMaxSize(heapAllocDesc,
                sizeof(heapAllocDesc)/sizeof(heapAllocDesc[0]),
                &heapMax);

    if (vmk_stat != VMK_OK) {
        vmk_WarningMessage("vmk_HeapDetermineMaxSize failed: %s",
                            vmk_StatusToString(vmk_stat));
        goto heap_create_fail;
    }

	heapProps.max = heapMax;

	vmk_LogMessage("vmk_HeapCreate max size : 0x%x",
					heapProps.max);

   vmk_stat = vmk_HeapCreate(&heapProps, &heap_id);
   if (vmk_stat != VMK_OK) {
      vmk_WarningMessage("qfle3f : genereal dma creation failed: %s",
                         vmk_StatusToString(vmk_stat));
      goto heap_create_fail;
   }
   qfle3fDriverInfo.moduleHeapID = heap_id;

	/*
	 * Set module heap
	 */
	vmk_ModuleSetHeapID(vmk_ModuleCurrentID, heap_id);
	VMK_ASSERT(vmk_ModuleGetHeapID(vmk_ModuleCurrentID) == heap_id);


    /* Create DMA Heap */
   heapProps.type = VMK_HEAP_TYPE_CUSTOM;
   vmk_NameFormat(&heap_name, "%s-dma", QFLE3F_DRIVER_NAME);
   vmk_stat = vmk_NameInitialize(&heapProps.name, (const char*)&heap_name);
   heapProps.module = vmk_ModuleCurrentID;
   heapProps.typeSpecific.custom.physContiguity = VMK_MEM_PHYS_CONTIGUOUS;
   heapProps.typeSpecific.custom.physRange = VMK_PHYS_ADDR_ANY;
   heapProps.initial = QFLE3F_HEAP_INITIAL_SIZE;
   heapProps.max = QFLE3F_HEAP_MAXIMUM_SIZE;
   heapProps.creationTimeoutMS = VMK_TIMEOUT_NONBLOCKING;

   vmk_stat = vmk_HeapCreate(&heapProps, &heap_id);
   if (vmk_stat != VMK_OK) {
      vmk_WarningMessage("qfle3f : dma heap creation failed: %s",
                         vmk_StatusToString(vmk_stat));
      goto dma_heap_create_fail;
   }
   qfle3fDriverInfo.moduleDMAHeapID = heap_id;

   /*
    * Register driver log
    */
   vmk_stat = vmk_NameInitialize(&log_props.name, QFLE3F_DRIVER_NAME);
   VMK_ASSERT(vmk_stat == VMK_OK);
   if (vmk_stat != VMK_OK) {
      vmk_WarningMessage("qfle3f : vmk_NameInitialize failed for vmk_LogRegister (%s)",
                         vmk_StatusToString(vmk_stat));
      goto log_create_fail;
   }
   log_props.module = vmk_ModuleCurrentID;
   log_props.heap = heap_id;
   log_props.defaultLevel = 0;
   log_props.throttle = NULL;
   vmk_stat = vmk_LogRegister(&log_props, &qfle3f_vmkLog);
   if (vmk_stat != VMK_OK) {
      vmk_WarningMessage("qfle3f : vmk_LogRegister failed %s",
                        vmk_StatusToString(vmk_stat));
      goto log_create_fail;
   }
   vmk_LogSetCurrentLogLevel(qfle3f_vmkLog, 0);

	/* Create Timer Queue */
	vmk_stat = vmk_NameInitialize(&timerqueue_props.name, "qfle3f_timerq");
	VMK_ASSERT(vmk_stat == VMK_OK);
	timerqueue_props.moduleID = vmk_ModuleCurrentID;
	timerqueue_props.heapID = vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
	timerqueue_props.attribs = VMK_TIMER_QUEUE_ATTR_NONE;

	vmk_stat = vmk_TimerQueueCreate(&timerqueue_props,
		&qfle3fDriverInfo.timerQueue);
	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("vmk_TimerQueueCreate Failed: %s",
			vmk_StatusToString(vmk_stat));
		goto timerqueue_failed;
	}

	vmk_stat = vmk_TimerQueueCreate(&timerqueue_props,
		&qfle3fDriverInfo.delayedTQ);
	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("Delayed TQ creation Failed: %s",
			vmk_StatusToString(vmk_stat));
		goto delayed_timerqueue_failed;
	}

   /*
    * Create lock domain
    */
   vmk_stat = vmk_LockDomainCreate(vmk_ModuleCurrentID, heap_id,
                                   &qfle3fDriverInfo.driverName,
				            &qfle3fDriverInfo.lockDomain);
   if (vmk_stat != VMK_OK) {
      vmk_WarningMessage("qfle3f : vmk_LockDomainCreate failed %s",
                        vmk_StatusToString(vmk_stat));
      goto lock_dom_fail;
   }

   /*
    * Create a global driver lock to access global data
    */
	vmk_stat = vmk_BinarySemaCreate(&qfle3fDriverInfo.drvLock,
									vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
									"qfle3fDriverInfo.drvLock");
	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("qfle3f : Failed to create global lock %s",
                        vmk_StatusToString(vmk_stat));
		goto semaphore_creation_failed;
	}

	/* Initialize the host list */
	vmk_ListInit(&qfle3fDriverInfo.qfle3fHostList);

   vmk_Memset(&pci_drv_props, 0, sizeof(vmk_DriverProps));
   vmk_NameCopy(&pci_drv_props.name, &(qfle3fDriverInfo.driverName));
   pci_drv_props.moduleID = vmk_ModuleCurrentID;
   pci_drv_props.ops = &pciDriverOps;

   /* Register PCI Callback function */
   vmk_stat = vmk_DriverRegister(&pci_drv_props,
                &qfle3fDriverInfo.qfle3fDriver);

   if (vmk_stat != VMK_OK) {
      vmk_WarningMessage("qfle3f: Unable to register PCI callback function %s",
                        vmk_StatusToString(vmk_stat));
      goto pci_callback_register_fail;
   }

	/* Initialize driver dump file */
	vmk_stat = vmk_DumpAddFileCallback(
			vmk_ModuleCurrentID,
			qfle3fDriverInfo.moduleHeapID,
			"qfle3f_dump",
			qfle3fDumpCallback,
			NULL,
			"QFLE3F",
			&qfle3fDriverInfo.dumpHandler);

	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("Unable to add file callback for dump : %s",
				vmk_StatusToString(vmk_stat));
	}

    qfle3fDriverInfo.hostInstanceMap = vmk_BitVectorAlloc(
        qfle3fDriverInfo.moduleHeapID, QFLE3F_MAX_HBAS);
    if (qfle3fDriverInfo.hostInstanceMap == NULL) {
        vmk_AlertMessage("qfle3f: "
            "Failed to allocate bit-vector: hostInstanceMap");
        goto hostInstanceMapAllocFailed;
    }
    vmk_BitVectorZap(qfle3fDriverInfo.hostInstanceMap);

#if (VMWARE_ESX_DDK_VERSION >= 65000)
	mgmtProps.modId = vmk_ModuleCurrentID;
	mgmtProps.heapId = qfle3fDriverInfo.moduleHeapID;
	mgmtProps.sig = &ql_fcoe_sig;
	mgmtProps.cleanupFn = NULL;
	mgmtProps.sessionAnnounceFn = NULL;
	mgmtProps.sessionCleanupFn = NULL;
	mgmtProps.handleCookie = 0;
	vmk_stat = vmk_MgmtInit(&mgmtProps, &qed_fcapi_vmkmgmt_api_handle);
#else
	vmk_stat = vmk_MgmtInit(moduleID,
			qfle3fDriverInfo.moduleHeapID,
			&ql_fcoe_sig,
			NULL,
			0,
			&qed_fcapi_vmkmgmt_api_handle);
#endif
	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("qfle3f: "
				"Unable to init common vmkmgmt-api. status=%s",
				vmk_StatusToString(vmk_stat));
	}

    if (qfle3f_create_vmkMgmt_Entry == 1) {
		vmk_LogMessage("Calling qfle3fKeyValueInit");
		vmk_stat = qfle3fKeyValueInit();
		if (vmk_stat != VMK_OK) {
			vmk_LogMessage("Failed to initialize KeyValue with status = %s",
					vmk_StatusToString(vmk_stat));
			goto KeyValueInitFailed;
		}
	}

    vmk_LogMessage("Calling ql_fcoe_module_init()");
    ql_fcoe_module_init();


	/* Validate module parameters */
	if(qfle3f_r_a_tov == 0) {
		vmk_LogMessage("qfle3f_r_a_tov is set to 0, "
				"setting it to the default value of 10.");
		qfle3f_r_a_tov = 10;
	}

	if(qfle3f_enable_r_a_tov == 1) {
		vmk_LogMessage("User define R_A_TOV value = %d.",
				qfle3f_r_a_tov);
	}

	if(qfle3f_devloss_tmo==0) {
		qfle3f_user_overwrote_devloss_val = 0;
		qfle3f_devloss_tmo = QFC_DEV_LOSS_TMO;
	}

	if(qfle3f_devloss_tmo<QFC_MIN_DEV_LOSS_TMO || qfle3f_devloss_tmo>QFC_MAX_DEV_LOSS_TMO) {
		vmk_WarningMessage("qfle3f_devloss_tmo value of %d is an invalid value. "
						"Setting it back to default value of 20 seconds",
						qfle3f_devloss_tmo);
		qfle3f_user_overwrote_devloss_val = 0;
		qfle3f_devloss_tmo = QFC_DEV_LOSS_TMO;
	}

	return VMK_OK;
KeyValueInitFailed:
	vmk_MgmtDestroy(qed_fcapi_vmkmgmt_api_handle);
hostInstanceMapAllocFailed:
pci_callback_register_fail:
   vmk_SemaDestroy(&qfle3fDriverInfo.drvLock);
semaphore_creation_failed:
   vmk_LockDomainDestroy(qfle3fDriverInfo.lockDomain);
lock_dom_fail:
	vmk_TimerQueueDestroy(qfle3fDriverInfo.delayedTQ);
delayed_timerqueue_failed:
	vmk_TimerQueueDestroy(qfle3fDriverInfo.timerQueue);
timerqueue_failed:
   vmk_LogUnregister(qfle3f_vmkLog);
dma_heap_create_fail:
   vmk_HeapDestroy(qfle3fDriverInfo.moduleDMAHeapID);
log_create_fail:
   vmk_HeapDestroy(qfle3fDriverInfo.moduleHeapID);
heap_create_fail:
#if (VMWARE_ESX_DDK_VERSION <= 65000)
   vmk_ModuleUnregister(vmk_ModuleCurrentID);
module_register_fail:
#endif
   return VMK_FAILURE;
}

void cleanup_module(void)
{
    /* Unregister the driver */
    VMK_ReturnStatus status;

    vmk_WarningMessage("%s: ql_fcoe_module_exit", __func__);
	ql_fcoe_module_exit();
    if(qfle3f_create_vmkMgmt_Entry == 1) {
        vmk_LogMessage("freeing qfle3fKeyValue");
        vmk_MgmtDestroy(qfle3fDriverInfo.kvMgmtModuleParm);
    }
    vmk_LogMessage("%s: mgmtDestroy qed_fcapi_vmkmgmt_api_handle", __func__);
	vmk_MgmtDestroy(qed_fcapi_vmkmgmt_api_handle);
    vmk_LogMessage("qfle3f: freeing hostInstanceMap");
	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, qfle3fDriverInfo.hostInstanceMap);
	vmk_DumpDeleteFileCallback(qfle3fDriverInfo.dumpHandler);
    vmk_LogMessage("%s: driver unregister", __func__);
    vmk_DriverUnregister(qfle3fDriverInfo.qfle3fDriver);
    vmk_LogMessage("qfle3f: destroying semaphore");
    vmk_SemaDestroy(&qfle3fDriverInfo.drvLock);
    vmk_LogMessage("qfle3f: destroying lockDomain");
    vmk_LockDomainDestroy(qfle3fDriverInfo.lockDomain);
    vmk_LogMessage("qfle3f: destroying delayedTQ");
	vmk_TimerQueueDestroy(qfle3fDriverInfo.delayedTQ);
    vmk_LogMessage("qfle3f: destroying timerQueue");
	vmk_TimerQueueDestroy(qfle3fDriverInfo.timerQueue);
    vmk_LogMessage("qfle3f: log Unregister");
    status = vmk_LogUnregister(qfle3f_vmkLog);
    if(status != VMK_OK) {
        vmk_WarningMessage("qfle3f : unable to unregister log : %s",
                            vmk_StatusToString(status));
    }
    vmk_LogMessage("qfle3f: destroying DMA Heap");
    vmk_HeapDestroy(qfle3fDriverInfo.moduleDMAHeapID);
    vmk_LogMessage("qfle3f: destroying Heap");
    vmk_HeapDestroy(qfle3fDriverInfo.moduleHeapID);
    vmk_LogMessage("%s: module unregister", __func__);
#if (VMWARE_ESX_DDK_VERSION <= 65000)
    status = vmk_ModuleUnregister(qfle3fDriverInfo.moduleID);
    if(status != VMK_OK) {
        vmk_WarningMessage("qfle3f : unable to unregister module : %s",
                            vmk_StatusToString(status));
    }
#endif

    return;
}
