/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#include <vmkapi.h>
#include "qfle3i.h"
#include "ql_iscsi_vmkmgmt.h"
#include "ql_iscsi_basetypes.h"

#define QED_PT_CMD_TOV      (66)

#define CMD_IOCTL_COMP_STATUS(Cmnd) ((Cmnd)->status.plugin)
#define CMD_IOCTL_SCSI_STATUS(Cmnd) ((Cmnd)->lbc)
#define CMD_IOCTL_ACTUAL_SNSLEN(Cmnd)       ((Cmnd)->lba)

#define SS_MASK             0xfff   /* Reserved bits BIT_12-BIT_15*/
#define SS_RESIDUAL_UNDER       BIT_11
#define SS_RESIDUAL_OVER        BIT_10
#define SS_SENSE_LEN_VALID      BIT_9
#define SS_RESPONSE_INFO_LEN_VALID  BIT_8
#define SS_CHECK_CONDITION      BIT_1

/*
 * Status entry completion status
 */
#define CS_COMPLETE     0x0 /* No errors */
#define CS_INCOMPLETE       0x1 /* Incomplete transfer of cmd. */
#define CS_DMA          0x2 /* A DMA direction error. */
#define CS_TRANSPORT        0x3 /* Transport error. */
#define CS_RESET        0x4 /* SCSI bus reset occurred */
#define CS_ABORTED      0x5 /* System aborted command. */
#define CS_TIMEOUT      0x6 /* Timeout error. */
#define CS_DATA_OVERRUN     0x7 /* Data overrun. */
#if (VMWARE_ESX_DDK_VERSION >= 60000)
#define CS_DIF_ERROR        0xC /* DIF error detected  */
#endif

#define CS_DATA_UNDERRUN    0x15    /* Data Underrun. */
#define CS_QUEUE_FULL       0x1C    /* Queue Full. */
#define CS_PORT_UNAVAILABLE 0x28    /* Port unavailable */
                    /* (selection timeout) */
#define CS_PORT_LOGGED_OUT  0x29    /* Port Logged Out */
#define CS_PORT_CONFIG_CHG  0x2A    /* Port Configuration Changed */
#define CS_PORT_BUSY        0x2B    /* Port Busy */
#define CS_COMPLETE_CHKCOND 0x30    /* Error? */
#define CS_BAD_PAYLOAD      0x80    /* Driver defined */
#define CS_UNKNOWN      0x81    /* Driver defined */
#define CS_RETRY        0x82    /* Driver defined */
#define CS_LOOP_DOWN_ABORT  0x83    /* Driver defined */

#define STATUS_MASK     0xfe
#define IOCTL_INVALID_STATUS 0xffff

extern unsigned int qfle3i_ssan_feature;
extern vmk_uint32 adapter_count;

vmk_MgmtApiSignature ql_iscsi_sig;
vmk_MgmtHandle qed_iscsi_vmkmgmt_api_handle;

static void qfle3i_scsi_pt_done(struct vmk_ScsiCommand *pscsi_cmd)
{
	struct qfle3i_hba *hba;

	hba = (struct qfle3i_hba *)pscsi_cmd->doneData;

	vmk_SpinlockLock(hba->ioctl_cmpl_lock);
	hba->SCSIPT_InProgress = 0;
	hba->cmpl_done = 1;
	vmk_SpinlockUnlock(hba->ioctl_cmpl_lock);
	vmk_WorldWakeup((vmk_WorldEventID)&hba->ioctl_cmpl_sem);
}

static VMK_ReturnStatus qlfe3i_find_hba_by_inst(UINT64 inst,
		struct qfle3i_hba **find_hba)
{
	struct qfle3i_hba *hba;

	hba = hba_list[inst];
	if (!hba) {
		vmk_LogMessage("HBA is not found for the inst = 0x%lx\n", inst);
		return VMK_FAILURE;
	} else {
		if (hba->instance == inst) {
			*find_hba = hba;
			return VMK_OK;
		} else {
			vmk_LogMessage("HBA is not found for the inst = 0x%lx\n", inst);
			return VMK_FAILURE;
		}
	}
	return VMK_OK;
}

static VMK_ReturnStatus qlfe3i_find_hba_by_hostno(UINT64 host_no,
		struct qfle3i_hba **find_hba)
{
	struct qfle3i_hba *hba;
	UINT64 inst;

	for (inst = 0; inst < QFLE3I_MAX_HBA; inst++) {
		hba = hba_list[inst];
		if (!hba)
			continue;

		if (hba->scsiAdapter->hostNum != host_no)
			continue;

		*find_hba = hba;
		return VMK_OK;
	}
	vmk_WarningMessage("%s: %d: HBA is not found for the host = 0x%lx\n",
		__func__, __LINE__, host_no);
	return VMK_FAILURE;
}

int ql_iscsi_ssan_support(UINT64 api_cookie, UINT64 instance_id,
		struct ssan_support *smartsan_support)
{
	int fw_support = 1;

	if (!fw_support)
		smartsan_support->status = SSAN_NO_ADAPTER_SUPPORT;
	else if (!qfle3i_ssan_feature)
		smartsan_support->status = SSAN_DISABLED_IN_DRIVER;
	else {
		smartsan_support->status = 0;
		smartsan_support->hba_cnt = adapter_count;
	}
	return VMK_OK;
}

int ql_iscsi_read_rdp(UINT64 api_cookie, UINT64 instance_id,
				struct ssan_rdp_attributes *rdp)
{
	struct qfle3i_hba *hba = NULL;
	struct sfp_diagnostic sfp_diag;
	struct link_error_status lesb;
	struct ssan_rdp_attr *rdp_attr;
	int rc;
	int len;
	UINT8 *entry;

	if (qlfe3i_find_hba_by_hostno(rdp->hostno, &hba) != VMK_OK) {
		vmk_WarningMessage("%s:HBA is not found, hostno:%d \n",
						__func__, rdp->hostno);
		return VMK_FAILURE;
	}

	rc = hba->cnic->drv_get_sfp_diagnostic(hba->cnic, &sfp_diag);
	if (rc != VMK_OK) {
		PRINT_ERR(hba, "failed to read SFP parameters\n");
		return rc;
	}
	QFLE3I_DBG(DBG_APP, hba, "temp:0x%x vcc:0x%x txb:0x%x txp:0x%x rxp:0x%x"
					"cap:0x%x opr:0x%x\n",
					sfp_diag.temperature, sfp_diag.vcc, sfp_diag.tx_bias,
					sfp_diag.tx_power, sfp_diag.rx_power, sfp_diag.speed_cap,
					sfp_diag.opr_speed);
#if 0
	rc = hba->cnic->drv_get_link_error(hba->cnic, &lesb);
	if (rc != VMK_OK) {
		PRINT_ERR(hba, "failed to read LESB.\n");
		return rc;
	}
#endif

	entry = (UINT8 *)rdp;
	rdp->count = SSAN_RDP_ATTR_COUNT;
	rdp->total_len = sizeof(rdp->hostno) + sizeof(rdp->count);

	rdp_attr = (struct ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_MAC_ADDR);
	vmk_Memcpy(rdp_attr->a.mac_addr, hba->mac_addr, VMK_ETH_ADDR_LENGTH);
	len = VMK_ETH_ADDR_LENGTH;
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);

	QFLE3I_DBG(DBG_APP, hba,
	"mac-addr: %02hhx:%02hhx:%02hhx:%02hhx:%02hhx:%02hhx len:0x%x totallen:%d \n",
	rdp_attr->a.mac_addr[0], rdp_attr->a.mac_addr[1], rdp_attr->a.mac_addr[2],
	rdp_attr->a.mac_addr[3], rdp_attr->a.mac_addr[4], rdp_attr->a.mac_addr[5],
	rdp_attr->len, rdp->total_len);

	rdp_attr = (struct ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_SFP_TEMP);
	rdp_attr->a.sfp_temp = htonl(sfp_diag.temperature);
	len = sizeof(rdp_attr->a.sfp_temp);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);
	QFLE3I_DBG(DBG_APP, hba, "temp:0x%x len:0x%x totallen:%d\n",
				rdp_attr->a.sfp_temp, rdp_attr->len, rdp->total_len);

	rdp_attr = (struct ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_SFP_VCC);
	rdp_attr->a.sfp_vcc = htonl(sfp_diag.vcc);
	len = sizeof(rdp_attr->a.sfp_vcc);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);
	QFLE3I_DBG(DBG_APP, hba, "vcc:0x%x len:0x%x totallen:%d\n",
				rdp_attr->a.sfp_vcc, rdp_attr->len, rdp->total_len);

	rdp_attr = (struct ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_SFP_TX_BIAS);
	rdp_attr->a.sfp_tx_bias = htonl(sfp_diag.tx_bias);
	len = sizeof(rdp_attr->a.sfp_tx_bias);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);
	QFLE3I_DBG(DBG_APP, hba, "tx_bias:0x%x len:0x%x totallen:%d\n",
				rdp_attr->a.sfp_tx_bias, rdp_attr->len, rdp->total_len);

	rdp_attr = (struct ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_SFP_TX_POWER);
	rdp_attr->a.sfp_tx_power = htonl(sfp_diag.tx_power);
	len = sizeof(rdp_attr->a.sfp_tx_power);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);
	QFLE3I_DBG(DBG_APP, hba, "tx_power:0x%x len:0x%x totallen:%d\n",
				rdp_attr->a.sfp_tx_power, rdp_attr->len, rdp->total_len);

	rdp_attr = (struct ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_SFP_RX_POWER);
	rdp_attr->a.sfp_rx_power = htonl(sfp_diag.rx_power);
	len = sizeof(rdp_attr->a.sfp_rx_power);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);
	QFLE3I_DBG(DBG_APP, hba, "rx_power:0x%x len:0x%x totallen:%d\n",
				rdp_attr->a.sfp_rx_power, rdp_attr->len, rdp->total_len);

	rdp_attr = (struct ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_PORT_SPEED_CAP);
	//rdp_attr->a.port_cap = htonl(sfp_diag.speed_cap);
	rdp_attr->a.port_cap = htonl(0x10);
	len = sizeof(rdp_attr->a.port_cap);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);

	rdp_attr = (struct ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_PORT_OPR_SPEED);
	//rdp_attr->a.opr_speed = htonl(sfp_diag.opr_speed);
	rdp_attr->a.opr_speed = htonl(0x10);
	len = sizeof(rdp_attr->a.opr_speed);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);

#if 0
	rdp_attr = (ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_LINK_FAIL_CNT);
	rdp_attr->a.link_fail_cnt = htonl(lesb.link_fail_cnt);
	len = sizeof(rdp_attr->a.link_fail_cnt);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);

	rdp_attr = (ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_VLINK_FAIL_CNT);
	rdp_attr->a.vlink_fail_cnt = htonl(lesb.vlink_fail_cnt);
	len = sizeof(rdp_attr->a.vlink_fail_cnt);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);

	rdp_attr = (ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_SYM_ERR_DCC);
	rdp_attr->a.sym_err_dcc = htonl(lesb.sym_err_dcc);
	len = sizeof(rdp_attr->a.sym_err_dcc);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);

	rdp_attr = (ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_ERR_BLK_CNT);
	rdp_attr->a.err_blk_cnt = htonl(lesb.err_blk_cnt);
	len = sizeof(rdp_attr->a.err_blk_cnt);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);

	rdp_attr = (ssan_rdp_attr *)(entry + rdp->total_len);
	rdp_attr->type = htons(SSAN_RDP_FRAME_CHK_SEQ);
	rdp_attr->a.frame_chk_seq = htonl(frame_chk_seq);
	len = sizeof(rdp_attr->a.frame_chk_seq);
	len += (len % 4) ? (4 - (len % 4)): 0;
	rdp_attr->len = htons(4 + len); /* 4 for type and len variable */
	rdp->total_len += (4 + len);
#endif
	return VMK_OK;
}

int ql_iscsi_get_hba_cnt(UINT64 api_cookie, UINT64 instance_id,
	struct ql_iscsi_hba_cnt_cb *get_hba_cnt)
{
	get_hba_cnt->hba_cnt = adapter_count;
	get_hba_cnt->status = EXT_STATUS_OK;

	return VMK_OK;
}

int ql_iscsi_get_hostno(UINT64 api_cookie, UINT64 instance_id,
	struct ql_iscsi_hostno_cb *get_host_no)
{
	struct qfle3i_hba *hba = NULL;

	if (qlfe3i_find_hba_by_inst(get_host_no->inst, &hba) != VMK_OK) {
		get_host_no->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	get_host_no->host_no = hba->scsiAdapter->hostNum;
	get_host_no->status = EXT_STATUS_OK;

	return VMK_OK;
}

int ql_iscsi_get_pci_info(UINT64 api_cookie, UINT64 instance_id,
	struct ql_iscsi_pci_info_cb *get_pci_info)
{
	struct qfle3i_hba *hba = NULL;
	struct ql_iscsi_pci_info *qfle3i_pci_info = &get_pci_info->pci;
	enum ql_iscsi_pci_bus_speed pcie_speed;
	enum ql_iscsi_pcie_link_width pcie_width;
	vmk_uint32 link_speed, val = 0;
	vmk_PCIDeviceID *qfle3i_pci_dev = NULL;
	struct cnic_dev *cnic;

	if (qlfe3i_find_hba_by_hostno(get_pci_info->host_no, &hba) != VMK_OK) {
		get_pci_info->status = EXT_STATUS_DEV_NOT_FOUND;
		vmk_WarningMessage("%s: %d: HBA is not found\n", __func__, __LINE__);
		return VMK_OK;
	}
	QFLE3I_DBG(DBG_APP, hba, "Enter\n");

	cnic = hba->cnic;
	qfle3i_pci_dev = &hba->pcidev_id;

	qfle3i_pci_info->vendor_id = qfle3i_pci_dev->vendorID;
	qfle3i_pci_info->device_id = qfle3i_pci_dev->deviceID;
	qfle3i_pci_info->subvendor_id = qfle3i_pci_dev->subVendorID;
	qfle3i_pci_info->subsystem_id = qfle3i_pci_dev->subDeviceID;
//	qfle3i_pci_info->domain_num = ;
	qfle3i_pci_info->bus_num = hba->sbdf.bus;
	qfle3i_pci_info->device_num = hba->sbdf.dev;
	qfle3i_pci_info->function_num = hba->sbdf.fn;
	qfle3i_pci_info->revision = qfle3i_pci_dev->progIFRevID;

	if (cnic->pf_id & 0x1)
		qfle3i_pci_info->port_num = (cnic->pf_num & 0x3) + 1;
	else
		qfle3i_pci_info->port_num = (cnic->pf_num & 0x1) + 1;

	 vmk_PCIReadConfig(vmk_ModuleCurrentID, hba->pcidev,
		VMK_PCI_CONFIG_ACCESS_32, QL_ISCSI_PCICFG_LINK_CONTROL, &val);

	 pcie_width = (enum ql_iscsi_pcie_link_width) ((val & QL_ISCSI_PCICFG_LINK_WIDTH) >>
		QL_ISCSI_PCICFG_LINK_WIDTH_SHIFT);
	 pcie_speed = (val & QL_ISCSI_PCICFG_LINK_SPEED) >> QL_ISCSI_PCICFG_LINK_SPEED_SHIFT;

	 switch (pcie_speed) {
		 case 3:
			 qfle3i_pci_info->pcie_speed = QL_ISCSI_PCIE_SPEED_8_0GT;
			 break;
		case 2:
			 qfle3i_pci_info->pcie_speed = QL_ISCSI_PCIE_SPEED_5_0GT;
			 break;
		default:
			 qfle3i_pci_info->pcie_speed = QL_ISCSI_PCIE_SPEED_2_5GT;
			 break;
	 }

	 qfle3i_pci_info->pcie_width = pcie_width;

	 get_pci_info->status = EXT_STATUS_OK;

	return VMK_OK;
}

int ql_iscsi_get_port_info(UINT64 api_cookie, UINT64 instance_id,
		struct ql_iscsi_port_info_cb *get_port_info)
{
	struct qfle3i_hba *hba = NULL;
	struct ql_iscsi_port_info *qfle3i_port_info = &get_port_info->port;
	vmk_UplinkSharedData *sharedData;

	if (qlfe3i_find_hba_by_hostno(get_port_info->host_no, &hba) != VMK_OK) {
		vmk_WarningMessage("%s: %d: HBA is not found\n", __func__, __LINE__);
		get_port_info->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	QFLE3I_DBG(DBG_APP, hba, "Enter\n");

	vmk_Memcpy(qfle3i_port_info->manufacturer, "QLogic Corporation", QLEVIS_SIZE_64);
	vmk_StringFormat(qfle3i_port_info->model_name, QLEVIS_SIZE_32, NULL, "ISP%04x",
			 hba->pcidev_id.deviceID);
	vmk_Memcpy(qfle3i_port_info->model_desc, "QLogic 10G ISCSI Adapter",
			QLEVIS_SIZE_128);
	//vmk_Memcpy(qfle3i_port_info->serial_num, (void*)&qfc_func->cdev->chip_num, QLEVIS_SIZE_32);
	vmk_Memcpy(qfle3i_port_info->driver_version, DRV_MODULE_VERSION,
			QLEVIS_SIZE_32);
	vmk_Strncpy(qfle3i_port_info->driver_date, DRV_MODULE_RELDATE, QLEVIS_SIZE_32);
	vmk_Strncpy(qfle3i_port_info->driver_name, QFLE3I_DRIVER_NAME, QLEVIS_SIZE_16);
	qfle3i_port_info->driver_loaded = 1;

	sharedData = hba->cnic->uplinkSharedData;
	vmk_NameCopy((vmk_Name *)qfle3i_port_info->fw_version,
			&sharedData->driverInfo.firmwareVersion);

	if (hba->name_string)
		vmk_Strncpy(qfle3i_port_info->iqn, hba->name_string, QLEVIS_SIZE_256);

	vmk_Memcpy(qfle3i_port_info->mac, (void*)hba->mac_addr, QLEVIS_SIZE_6);
	vmk_Memcpy(qfle3i_port_info->ipv4addr, (void*)hba->cnic->ipv4_addr,
			QLEVIS_SIZE_4);
	vmk_Memcpy(qfle3i_port_info->subnet, (void*)hba->cnic->subnet_mask, QLEVIS_SIZE_4);
	//vmk_Memcpy(qfle3i_port_info->ipv6addr, (void*)hba->cnic->ipv6_addr.s6_addr,
	//		QLEVIS_SIZE_16);

	qfle3i_port_info->port_sup_speed = hba->cnic->link_supported_speeds;

	qfle3i_port_info->port_speed = sharedData->link.speed;

	qfle3i_port_info->port_state = sharedData->link.state;

	qfle3i_port_info->session_cnt = hba->num_active_sess;
	qfle3i_port_info->mtu = hba->mtu_supported;
	qfle3i_port_info->vlan =  hba->cnic->vlan_id;
	vmk_Strncpy(qfle3i_port_info->hba_name,
		vmk_ScsiGetAdapterName(hba->scsiAdapter), QLEVIS_SIZE_32);

	get_port_info->status = EXT_STATUS_OK;

	return VMK_OK;
}

int ql_iscsi_get_sess_info(UINT64 api_cookie, UINT64 instance_id,
		struct ql_iscsi_sess_info_cb *get_sess_info)
{
	struct qfle3i_hba *hba = NULL;
	struct ql_iscsi_sess_info *sess_info = &get_sess_info->sess;
	struct qfle3i_sess *sess = NULL;
	int ret, sess_cnt;
	vmk_uint32 l5_cid, cid;

	if (qlfe3i_find_hba_by_hostno(get_sess_info->host_no, &hba) != VMK_OK) {
		vmk_WarningMessage("%s: %d: HBA is not found\n", __func__, __LINE__);
		get_sess_info->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	QFLE3I_DBG(DBG_APP, hba, "Enter %s, sess_num:%d\n",
				 __func__, sess_info->sess_num);

	sess_cnt = 0;
	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(sess, &hba->active_sess, link,
		struct qfle3i_sess) {

		if (sess_cnt != sess_info->sess_num) {
			sess_cnt++;
			continue;
		}

		if (sess->lead_conn)
			QFLE3I_DBG(DBG_APP, hba,
					"Get session info for sess:0x%p conn:%p iscsi_conn_cid:0x%x \n",
					sess, sess->lead_conn, sess->lead_conn->iscsi_conn_cid);

		l5_cid = cid = 0xFF;
		if (sess->lead_conn && sess->lead_conn->ep) {
			l5_cid = sess->lead_conn->ep->ep_iscsi_cid;
			cid = sess->lead_conn->ep->ep_cid;

			if (vmk_BitVectorTest(sess->lead_conn->ep->cm_sk->flags, SK_F_IPV6)) {
				vmk_Memcpy(sess_info->tgt_ipv6addr, sess->lead_conn->ep->cm_sk->dst_ip,
						QLEVIS_SIZE_16);
			} else
				vmk_Memcpy(sess_info->tgt_ipv4addr, sess->lead_conn->ep->cm_sk->dst_ip,
						QLEVIS_SIZE_4);
		}

		sess_info->sess_unique = (vmk_uint64)sess;
		if (hba->name_string)
			vmk_Strncpy(sess_info->init_iqn, hba->name_string, QLEVIS_SIZE_256);

		vmk_Memcpy(sess_info->init_ipv4addr, (void *)hba->cnic->ipv4_addr,
				QLEVIS_SIZE_4);

//		vmk_Memcpy(sess_info->init_ipv6addr, (void *)hba->cnic->ipv6_addr.s6_addr,
//				QLEVIS_SIZE_16);

		vmk_Memcpy(sess_info->init_mac, hba->mac_addr, QLEVIS_SIZE_6);

		//		sess_info->ipv4_port = sess;
		//		sess_info->ipv6_port = sess;

		if (sess->target_name)
			vmk_Memcpy(sess_info->tgt_iqn, sess->target_name, QLEVIS_SIZE_256);

		//		memcpy(sess_info->mac, sess->mac, QLEVIS_SIZE_6);
		vmk_Memcpy(sess_info->isid, sess->isid, QLEVIS_SIZE_6);

		sess_info->state = sess->state;
		sess_info->recovery_state = sess->recovery_state;
		sess_info->old_recovery_state = sess->old_recovery_state;
		sess_info->tmf_active = vmk_AtomicRead64(&sess->tmf_active);
		sess_info->do_recovery_inprogess = vmk_AtomicRead64(&sess->do_recovery_inprogess);
		sess_info->device_offline = vmk_AtomicRead64(&sess->device_offline);
		sess_info->max_iscsi_tasks = sess->max_iscsi_tasks;
		sess_info->num_free_cmds = sess->num_free_cmds;
		sess_info->allocated_cmds = sess->allocated_cmds;
		sess_info->total_cmds_allocated = sess->total_cmds_allocated;
		sess_info->total_cmds_freed = sess->total_cmds_freed;
		sess_info->sq_size = sess->sq_size;
		sess_info->login_noop_pending = vmk_AtomicRead64(&sess->login_noop_pending);
		sess_info->tmf_pending = vmk_AtomicRead64(&sess->tmf_pending);
		sess_info->logout_pending = vmk_AtomicRead64(&sess->logout_pending);
		sess_info->nop_resp_pending = vmk_AtomicRead64(&sess->nop_resp_pending);
		sess_info->pend_cmd_count = sess->pend_cmd_count;
		sess_info->active_cmd_count = sess->active_cmd_count;
		sess_info->cmd_cleanup_req = sess->cmd_cleanup_req;
		sess_info->cmd_cleanup_cmpl = sess->cmd_cleanup_cmpl;
		sess_info->total_cmds_sent = sess->total_cmds_sent;
		sess_info->total_cmds_queued = sess->total_cmds_queued;
		sess_info->total_cmds_completed = sess->total_cmds_completed;
		sess_info->total_cmds_failed = sess->total_cmds_failed;
		sess_info->total_cmds_completed_by_chip = sess->total_cmds_completed_by_chip;
		sess_info->cmdsn = sess->pend_cmd_count;
		sess_info->exp_cmdsn = sess->exp_cmdsn;
		sess_info->max_cmdsn = sess->max_cmdsn;
		sess_info->initial_r2t = sess->initial_r2t;
		sess_info->max_r2t = sess->max_r2t;
		sess_info->imm_data = sess->imm_data;
		sess_info->first_burst_len = sess->first_burst_len;
		sess_info->time2wait = sess->time2wait;
		sess_info->time2retain = sess->time2retain;
		sess_info->pdu_inorder = sess->pdu_inorder;
		sess_info->dataseq_inorder = sess->dataseq_inorder;
		sess_info->erl = sess->erl;
		sess_info->tgt_prtl_grp = sess->tgt_prtl_grp;
		sess_info->num_active_conn = sess->num_active_conn;
		sess_info->max_conns = sess->max_conns;
		sess_info->last_nooput_requested = sess->last_nooput_requested;
		sess_info->last_nooput_posted = sess->last_nooput_posted;
		sess_info->last_noopin_indicated = sess->last_noopin_indicated;
		sess_info->last_noopin_processed = sess->last_noopin_processed;
		sess_info->last_nooput_sn = sess->last_nooput_sn;
		sess_info->noopout_resp_count = sess->noopout_resp_count;
		sess_info->unsol_noopout_count = sess->unsol_noopout_count;
		sess_info->noopout_requested_count = sess->noopout_requested_count;
		sess_info->noopout_posted_count = sess->noopout_posted_count;
		sess_info->noopin_indicated_count = sess->noopin_indicated_count;
		sess_info->noopin_processed_count = sess->noopin_processed_count;
		sess_info->tgt_noopin_count = sess->tgt_noopin_count;
		sess_info->alloc_scsi_task_failed = sess->alloc_scsi_task_failed;
		sess_info->cid = cid;
		sess_info->iscsi_cid = l5_cid;

		break;
	}
	vmk_SpinlockUnlock(hba->lock);

	get_sess_info->status = EXT_STATUS_OK;

	return VMK_OK;
}

int ql_iscsi_get_tgt_info(UINT64 api_cookie, UINT64 instance_id,
	struct ql_iscsi_tgt_info_cb *get_tgt_info)
{
	struct qfle3i_hba *hba = NULL;
	struct  ql_iscsi_tgt_info *tgt_info = &get_tgt_info->tgt;
	struct qfle3i_sess *sess = NULL;
	int ret, sess_cnt, tgt_cnt;

	if (qlfe3i_find_hba_by_hostno(get_tgt_info->host_no, &hba) != VMK_OK) {
		vmk_WarningMessage("%s: %d: HBA is not found\n", __func__, __LINE__);
		get_tgt_info->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	QFLE3I_DBG(DBG_APP, hba, "Enter\n");

	tgt_cnt = 0;
	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(sess, &hba->active_sess, link,
			struct qfle3i_sess) {
		if (tgt_cnt != tgt_info->scsi_tgt_num) {
			tgt_cnt++;
			continue;
		}

		if (sess->target_name)
			vmk_Memcpy(tgt_info->iqn, sess->target_name, QLEVIS_SIZE_256);
		//		tgt_info->mac
		vmk_Memcpy(tgt_info->isid, sess->isid, QLEVIS_SIZE_6);

		if (sess->lead_conn && sess->lead_conn->ep) {
			if (vmk_BitVectorTest(sess->lead_conn->ep->cm_sk->flags, SK_F_IPV6)) {
				vmk_Memcpy(tgt_info->ipv6addr, sess->lead_conn->ep->cm_sk->dst_ip,
						QLEVIS_SIZE_16);
			} else
				vmk_Memcpy(tgt_info->ipv4addr, sess->lead_conn->ep->cm_sk->dst_ip,
						QLEVIS_SIZE_4);
			break;
		}
	}
	vmk_SpinlockUnlock(hba->lock);

	get_tgt_info->status = EXT_STATUS_OK;

	return VMK_OK;
}

int
qfle3i_scsi_pass_dma_map(struct qfle3i_hba *hba, void *mem, struct qla_dmamem *buffer, vmk_uint32 size)
{
    VMK_ReturnStatus status = VMK_OK;
    vmk_DMAMapErrorInfo err;

    if (mem == NULL) {
        goto sgCreate;
    }

    vmk_Memset(buffer, 0, sizeof(struct qla_dmamem));

    status = vmk_SgCreateOpsHandle(qfle3i_driver_info.heap_id,
        &buffer->SgOps_handle, NULL, NULL);

    if (status != VMK_OK)
        goto sgCreate;

    /* Allocate and initialize a Input sg array with machine-addresses */
    status = vmk_SgAllocWithInit(buffer->SgOps_handle,
        &(buffer->MachSg), mem, size);

    if (status != VMK_OK)
        goto sgInit;

    buffer->dma_size = size;
    buffer->virtaddr = mem;

    /* Map the sg array with the DMA Engine of the device */
    status = vmk_DMAMapSg(hba->dmaEngine,
        VMK_DMA_DIRECTION_BIDIRECTIONAL, buffer->SgOps_handle,
        buffer->MachSg, &(buffer->MachSg), &err);

    if (status != VMK_OK) {
        vmk_LogMessage("failed to map %p size %d to IO address, %s",
            mem, size, vmk_DMAMapErrorReasonToString(err.reason));
        goto sgAlloc;
	}

	return VMK_OK;

sgAlloc:
    vmk_SgFree(buffer->SgOps_handle, buffer->MachSg);
sgInit:
    vmk_SgDestroyOpsHandle(buffer->SgOps_handle);
sgCreate:
    vmk_LogMessage("Unable to map DMA buffer!");

    return VMK_FAILURE;
}

void qfle3i_scsi_pass_dma_unmap(struct qfle3i_hba *hba, struct qla_dmamem *buffer)
{
    if (buffer->virtaddr == NULL)
        return;
    vmk_DMAUnmapSg(hba->dmaEngine, VMK_DMA_DIRECTION_BIDIRECTIONAL,
        buffer->SgOps_handle, buffer->MachSg);
    vmk_SgFree(buffer->SgOps_handle, buffer->MachSg);
    vmk_SgDestroyOpsHandle(buffer->SgOps_handle);
}

void *qfle3i_dma_alloc(struct qfle3i_hba *hba, struct qla_dmamem *buffer, vmk_uint32 size)
{
    void *map = NULL;

    if (size > VMK_PAGE_SIZE)
        map = vmk_HeapAlign(qfle3i_driver_info.heap_id,
                            size, VMK_PAGE_SIZE);
    else
        map = vmk_HeapAlloc(qfle3i_driver_info.heap_id, size);

    if (map == NULL)
        return map;

    vmk_Memset(map, 0, size);

    if (qfle3i_scsi_pass_dma_map(hba, map, buffer, size) != 0) {
        vmk_HeapFree(qfle3i_driver_info.heap_id, map);
        map = NULL;
    }

    return map;
}

void qfle3i_dma_free(struct qfle3i_hba *hba, struct qla_dmamem *buffer)
{
    if (buffer->virtaddr == NULL)
        return;
    qfle3i_scsi_pass_dma_unmap(hba, buffer);
    vmk_HeapFree(qfle3i_driver_info.heap_id, buffer->virtaddr);
    buffer->virtaddr = NULL;
}

int ql_iscsi_send_scsi_cmd(UINT64 api_cookie, UINT64 instance_id,
		struct ql_iscsi_scsi_cmd_cb *scsi_cmd)
{
	struct qfle3i_hba *hba = NULL;
	struct ql_iscsi_scsi_cmd *ioctl_scsi_cmd = &scsi_cmd->cmnd;
	struct qfle3i_scsi_task *scsi_task;
	struct qfle3i_conn *conn;
	struct vmk_ScsiCommand *pscsi_cmd = NULL;
	vmk_uint32 cmd_start, cmd_cur, cmd_wait_time;
	qfle3i_iscsilun *iscsilun;
	qfle3i_sess *sess;
	int tgt_cnt = 0, lun_allocated = 0, transfer_len;
	VMK_ReturnStatus status;

	if (qlfe3i_find_hba_by_hostno(scsi_cmd->host_no, &hba) != VMK_OK) {
		vmk_WarningMessage("%s: %d: HBA is not found\n", __func__, __LINE__);
		scsi_cmd->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	QFLE3I_DBG(DBG_APP, hba, "Enter\n");

	iscsilun = NULL;

	pscsi_cmd = vmk_HeapAlloc(qfle3i_driver_info.heap_id, sizeof(struct vmk_ScsiCommand));
	if (!pscsi_cmd) {
		scsi_cmd->status = EXT_STATUS_NO_MEMORY;
		vmk_LogMessage("Error in allocation memory\n");
		return VMK_NO_MEMORY;
	} else
		vmk_Memset(pscsi_cmd, 0, sizeof(struct vmk_ScsiCommand));


	QFLE3I_DBG(DBG_APP, hba, "bus = 0x%x tgt = 0x%x, lun = 0x%x, direction = 0x%x," \
		"cdb_len = 0x%x, cdb = %02x %02x %02x %02x sense_len = 0x%x\n",
		ioctl_scsi_cmd->bus,
		ioctl_scsi_cmd->tgt, ioctl_scsi_cmd->lun,
		ioctl_scsi_cmd->direction, ioctl_scsi_cmd->cdb_len,
		ioctl_scsi_cmd->cdb[0], ioctl_scsi_cmd->cdb[1],
		ioctl_scsi_cmd->cdb[2], ioctl_scsi_cmd->cdb[3],
		ioctl_scsi_cmd->sense_len);

	tgt_cnt = 0;
	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(sess, &hba->active_sess, link,
			struct qfle3i_sess) {
		if (tgt_cnt != ioctl_scsi_cmd->tgt) {
			tgt_cnt++;
			continue;
		} else
			break;
		tgt_cnt = 0;
	}
	vmk_SpinlockUnlock(hba->lock);

//	if (tgt_cnt) {
//		vmk_LogMessage("Target not found\n");
//		scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
//		goto free_request;
//	}

	if ((!sess) || (!hba) || sess->hba != hba) {
		vmk_WarningMessage("%s: %d: Invalid Session\n",
			__func__, __LINE__);
		scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
		goto free_request;
	}

	switch (ioctl_scsi_cmd->cdb_len) {
		case 0x6:
			pscsi_cmd->cdbLen = 6;
			break;
		case 0x0A:
			pscsi_cmd->cdbLen = 0x0A;
			break;
		case 0x0C:
			pscsi_cmd->cdbLen = 0x0C;
			break;
		case 0x10:
			pscsi_cmd->cdbLen = 0x10;
			break;
		default:
			vmk_WarningMessage("Unsupported Cdb Length = 0x%x\n",
					ioctl_scsi_cmd->cdb_len);
			scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
			goto free_request;
	}
	vmk_Memcpy(pscsi_cmd->cdb, ioctl_scsi_cmd->cdb,
			pscsi_cmd->cdbLen);

	switch (ioctl_scsi_cmd->direction) {
		case QL_ISCSI_DATA_DIRECTION_OUT:
			pscsi_cmd->dataDirection = VMK_SCSI_COMMAND_DIRECTION_WRITE;
			break;
		case QL_ISCSI_DATA_DIRECTION_IN:
			pscsi_cmd->dataDirection = VMK_SCSI_COMMAND_DIRECTION_READ;
			break;
		default :
			pscsi_cmd->dataDirection = VMK_SCSI_COMMAND_DIRECTION_NONE;
			break;
	}


	if (hba->ioctl_mem) {
		if (ioctl_scsi_cmd->data_len &&
				ioctl_scsi_cmd->data_len != hba->ioctl_mem_phys.dma_size) {
			qfle3i_dma_free(hba, &hba->ioctl_mem_phys);
			hba->ioctl_mem = qfle3i_dma_alloc(hba,
					&hba->ioctl_mem_phys, ioctl_scsi_cmd->data_len);
		}
	} else {
		hba->ioctl_mem = qfle3i_dma_alloc(hba,
			&hba->ioctl_mem_phys,
			ioctl_scsi_cmd->data_len ? ioctl_scsi_cmd->data_len : VMK_PAGE_SIZE);
	}

	if (!hba->ioctl_mem) {
		scsi_cmd->status = EXT_STATUS_NO_MEMORY;
		vmk_WarningMessage("Can not alloc requested(0x%x) DMA buffer\n",
				ioctl_scsi_cmd->data_len);
		goto free_request;
	}

	if (pscsi_cmd->dataDirection == VMK_SCSI_COMMAND_DIRECTION_WRITE) {
		vmk_Memcpy(hba->ioctl_mem,
				ioctl_scsi_cmd->data_buffer,
				ioctl_scsi_cmd->data_len);
	}

	iscsilun = qfle3i_get_iscsilun(sess, ioctl_scsi_cmd->lun);
	if (!iscsilun && (ioctl_scsi_cmd->cdb[0] == 0xA0)) {
		iscsilun = qfle3i_allocate_lun(sess, sess->channel_id, sess->target_id, 0);
		if(!iscsilun) {
			vmk_WarningMessage("%s: %d: Unable to Allocate memory  for LUN\n",
					__func__, __LINE__);
			scsi_cmd->status = EXT_STATUS_NO_MEMORY;
			goto free_request;
		}
		lun_allocated = 1;
	} else if (!iscsilun && (ioctl_scsi_cmd->cdb[0] != 0xA0)) {
		vmk_WarningMessage("%s: %d: could not find  LUN 0x%x in sess %p ch 0x%x tgt 0x%x\n",
			__func__, __LINE__, ioctl_scsi_cmd->lun, sess, sess->channel_id, sess->target_id);
		scsi_cmd->status = EXT_STATUS_NO_MEMORY;
		goto free_request;
	}

	pscsi_cmd->done = qfle3i_scsi_pt_done;
	pscsi_cmd->doneData = hba;
	pscsi_cmd->sgArray = hba->ioctl_mem_phys.MachSg;

	if (sess->state == QFLE3I_SESS_DESTROYED) {
		vmk_WarningMessage("Invalid Session \n");
		scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
		goto scsi_cmd_error;
	}

	if (sess->state == QFLE3I_SESS_IN_SHUTDOWN ||
			sess->state == QFLE3I_SESS_IN_LOGOUT || !sess->lead_conn) {
		vmk_WarningMessage("Invalid Session \n");
		scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
		goto scsi_cmd_error;
	}

	vmk_SpinlockLock(sess->lock);
	if (sess->recovery_state) {
		if (sess->recovery_state & ISCSI_SESS_RECOVERY_START)
			vmk_WarningMessage("Session recover starts.\n");

		vmk_SpinlockUnlock(sess->lock);
		scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
		goto scsi_cmd_error;
	}

	conn = sess->lead_conn;
	if (!conn || !conn->ep ||
			conn->ep->state != EP_STATE_ULP_UPDATE_COMPL) {
		vmk_SpinlockUnlock(sess->lock);
		vmk_WarningMessage("Invalid Session \n");
		scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
		goto scsi_cmd_error;
	}

	scsi_task = qfle3i_alloc_scsi_task(sess);
	if (!scsi_task) {
		sess->alloc_scsi_task_failed++;
		vmk_SpinlockUnlock(sess->lock);
		vmk_WarningMessage("unavailable to allocate memory for scsi_task \n");
		scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
		goto scsi_cmd_error;
	}

	scsi_task->scsi_cmd = pscsi_cmd;
	scsi_task->ilun = iscsilun;
	scsi_task->app_cmd = 1;

	vmk_ListInsert(&scsi_task->link, vmk_ListAtRear(&sess->pend_cmd_list));
	sess->pend_cmd_count++;

	if (sess->hba->max_scsi_task_queued < sess->pend_cmd_count) {
		sess->hba->max_scsi_task_queued = sess->pend_cmd_count;
	}
	sess->total_cmds_queued++;

	vmk_SpinlockUnlock(sess->lock);

	hba->cmpl_done = 0;
	hba->SCSIPT_InProgress = 1;

	if (vmk_AtomicRead64(&conn->worker_enabled)) {
		vmk_AtomicWrite64(&conn->lastSched, 12);
		QFLE3I_DBG(DBG_APP, hba,
				"scheduling conn tasklet, cid:%d, cmd:0x%x, sc:%p\n",
				conn->iscsi_conn_cid, pscsi_cmd->cdb[0], pscsi_cmd);
		ql_vmk_tasklet_schedule(&conn->conn_tasklet);
	}

	ql_vmk_spin_lock_init(&hba->ioctl_cmpl_lock, LOCK_RANK_MEDIUM, "ioctl_cmpl_lock");
	vmk_SpinlockLock(hba->ioctl_cmpl_lock);
	QFLE3I_DBG(DBG_APP, hba, "Waiting for completion\n");
	cmd_start = vmk_GetTimerCycles() / vmk_TimerCyclesPerSecond();
	cmd_wait_time = QED_PT_CMD_TOV;

wait_again:
	status = vmk_WorldWait((vmk_WorldEventID)&hba->ioctl_cmpl_sem,
			hba->ioctl_cmpl_lock, cmd_wait_time * VMK_MSEC_PER_SEC,
			"waiting for IOCTL cmd completion");

	/* If VMK_BAD_PARAM is returned, still holding the lock. */
	if(status != VMK_BAD_PARAM) {
		vmk_SpinlockLock(hba->ioctl_cmpl_lock);
	}

	cmd_cur = vmk_GetTimerCycles() / vmk_TimerCyclesPerSecond();

	if (!hba->cmpl_done) {
		if ((cmd_cur - cmd_start) < cmd_wait_time) {
			cmd_wait_time -= (cmd_cur - cmd_start);
			cmd_start = cmd_cur;
			goto wait_again;
		} else {
			vmk_WarningMessage("IOCTL cmd timeout");
			vmk_SpinlockUnlock(hba->ioctl_cmpl_lock);
			vmk_SpinlockDestroy(hba->ioctl_cmpl_lock);
			scsi_cmd->status = EXT_STATUS_BUSY;
			goto scsi_cmd_error;
		}
	}
	vmk_SpinlockUnlock(hba->ioctl_cmpl_lock);
	vmk_SpinlockDestroy(hba->ioctl_cmpl_lock);

	if (hba->SCSIPT_InProgress == 1) {
		vmk_WarningMessage("%s: %d: ERROR passthru command timeout\n",
			__func__, __LINE__);
		scsi_cmd->status = EXT_STATUS_BUSY;
		goto scsi_cmd_error;
	}

	if (ioctl_scsi_cmd->direction == QL_ISCSI_DATA_DIRECTION_IN) {
		 QFLE3I_DBG(DBG_APP, hba, "Start copying data\n");
		transfer_len = ioctl_scsi_cmd->data_len;
		 QFLE3I_DBG(DBG_APP, hba, "Final Transfer Len = 0x%x\n", transfer_len);
		vmk_Memcpy(ioctl_scsi_cmd->data_buffer,
				hba->ioctl_mem, transfer_len);
	}

scsi_cmd_error:
	if (hba->ioctl_mem) {
		qfle3i_dma_free(hba, &hba->ioctl_mem_phys);
		hba->ioctl_mem = NULL;
	}

	if(iscsilun && lun_allocated)
		qfle3i_free_iscsilun(iscsilun);

free_request:
	if (pscsi_cmd) {
		vmk_HeapFree(qfle3i_driver_info.heap_id, pscsi_cmd);
		pscsi_cmd = NULL;
	}

	return VMK_OK;
}
