//
//  Downloader.m
//  Cosmic Debris
//
//  Created by John Schilling on 11/22/04.
//  Copyright 2004 John Schilling. All rights reserved.
//

#import "Downloader.h"


@implementation Downloader


#pragma mark
#pragma mark INSTANCE METHODS
#pragma mark

- (id)initWithURL:(NSString *)URL delegate:(id)delegate
{
    if( ![super init] ) return nil;
    
    _URL        = [[NSMutableString alloc] initWithString:URL];
    _delegate   = delegate;
    _data       = [[NSMutableData alloc] initWithLength:0];
    
    return self;
}

- (void)cleanup
{
    if (_download) [_download release];
    _download = nil;
    
    if (_data) [_data setLength:0];
    if (_data) [_data release];
    _data = nil;
    
    [_URL release];
    _URL = nil;
}

- (void)dealloc
{
    [super dealloc];
}







#pragma mark
#pragma mark ACCESSOR METHODS
#pragma mark

- (void)setURL:(NSString *)URL
{
    [_URL setString:URL];
}






#pragma mark
#pragma mark DOWNLOADING METHODS
#pragma mark


- (void)downloadDataWithTimeoutInterval:(float)timeoutInterval
{
    if (_isDownloading) return;
    
    NSURLRequest *theRequest = [NSURLRequest requestWithURL: [NSURL URLWithString: _URL]
                                                cachePolicy: NSURLRequestReloadIgnoringCacheData
                                            timeoutInterval: timeoutInterval];
                                             
    _download   = [[NSURLConnection alloc] initWithRequest: theRequest
                                                  delegate: self];


    
    if (_download) {
    
        _isDownloading = YES;
        
        if ([_delegate respondsToSelector:@selector( downloaderDidBegin: )]) {
            [_delegate downloaderDidBegin:self];
        }
        
    } else {
    
        if ([_delegate respondsToSelector:@selector( downloaderDidFailWithError:error: )]) {
            [_delegate downloaderDidFailWithError:self error:@"Error: cannot create NSURLConnection"];
        }
        [self resetDownload];
        
    }
}

- (void)cancelDownloading
{
    if (_download) [_download cancel];
    [self resetDownload];
}

- (void)resetDownload
{
    [_download release];
    _download = nil;

    _bytesReceived  = 0;
    _expectedLength = 0;
    _isDownloading  = NO;
    
    [_data setLength:0];
    
    if ([_delegate respondsToSelector:@selector( downloaderDidReset: )]) {
        [_delegate downloaderDidReset:self];
    }
}

- (void)resetSyncDownload
{
    _download = nil;
    
    _bytesReceived  = 0;
    _expectedLength = 0;
    _isDownloading  = NO;
    
    [_data setLength: 0];
    
    if( [_delegate respondsToSelector: @selector( downloaderDidReset: )] )
    {
        [_delegate downloaderDidReset: self];
    }
}

- (BOOL)isDownloading
{
    return _isDownloading;
}








#pragma mark
#pragma mark NSURLDownload DELEGATE METHODS
#pragma mark

- (void)connection:(NSURLConnection *)connection didFailWithError:(NSError *)error
{
    NSString *errorMsg = [NSString stringWithFormat:@"Download failed: %@ %@",
                            [error localizedDescription],
                            [[error userInfo] objectForKey:NSErrorFailingURLStringKey]];
    
    if ([_delegate respondsToSelector:@selector( downloaderDidFailWithError:error: )]) {
        [_delegate downloaderDidFailWithError:self error:errorMsg];
    }
    
    [self resetDownload];
}

- (void)connection:(NSURLConnection *)connection didReceiveResponse:(NSURLResponse *)response
{
    _expectedLength = [response expectedContentLength];
    //[_data setLength:0];
    if ([_delegate respondsToSelector:@selector( downloaderDidConnect:expectedLength: )]) {
        [_delegate downloaderDidConnect:self expectedLength:_expectedLength];
    }
}

- (NSURLRequest *)connection:(NSURLConnection *)connection 
            willSendRequest:(NSURLRequest *)request
            redirectResponse:(NSURLResponse *)redirectResponse
{
    if (floor(NSAppKitVersionNumber) >= 949) {
        return request;
    } else {
        return nil;
    }
}

-(NSCachedURLResponse *)connection:(NSURLConnection *)connection 
            willCacheResponse:(NSCachedURLResponse *)cachedResponse
{
    return nil;
}

- (void)connection:(NSURLConnection *)connection didReceiveData:(NSData *)data
{    
    if (_expectedLength > 0) {
        _bytesReceived += [data length];
        if ([_delegate respondsToSelector:@selector( downloaderDidReceiveData:bytesReceived:expectedLength: )]) {
            [_delegate downloaderDidReceiveData:self bytesReceived:_bytesReceived expectedLength:_expectedLength];
        }
    }
    [_data appendData:data];
}

- (void)connectionDidFinishLoading:(NSURLConnection *)connection
{

    if ([_data length] > 0) {
        if (_expectedLength > 0 && ([_data length] != _expectedLength)) {
            NSString *errorMsg = [NSString stringWithFormat:@"Error: unexpected content length (%d : %d)", [_data length], _expectedLength];
            if ([_delegate respondsToSelector:@selector( downloaderDidFailWithError:error: )]) {
                [_delegate downloaderDidFailWithError:self error:errorMsg];
            }
            [self resetDownload];
        } else {
            if ([_delegate respondsToSelector:@selector( downloaderDidFinishWithData:data: )]) {
                [_delegate downloaderDidFinishWithData:self data:_data];
            }
            [self resetDownload];
        }
    } else {
        if ([_delegate respondsToSelector:@selector( downloaderDidFailWithError:error: )]) {
            [_delegate downloaderDidFailWithError:self error:@"Error: data length is zero"];
        }
        [self resetDownload];
    }
}



@end
