/*
    Platypus - program for creating Mac OS X application wrappers around scripts
    Copyright (C) 2003-2010 Sveinbjorn Thordarson <sveinbjornt@simnet.is>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

// PlatypusAppSpec is a data wrapper class around an NSDictionary containing
// all the information / specifications for creating a Platypus application.


#import "PlatypusAppSpec.h"

@implementation PlatypusAppSpec

/*****************************************
 - init / dealloc functions
*****************************************/

- (id)init
{
	if (self = [super init]) 
	{
		properties = [[NSMutableDictionary alloc] init];
    }
    return self;
}

-(id)initWithDefaults
{
	if (self = [super init]) 
	{
		properties = [[NSMutableDictionary alloc] init];
    }
	[self setDefaults];
	return self;
}

-(id)initWithDictionary: (NSDictionary *)dict
{
	if (self = [super init]) 
	{
		properties = [[NSMutableDictionary alloc] init];
		[properties setObject: PROGRAM_STAMP forKey: @"Creator"];
		[properties addEntriesFromDictionary: dict];
    }
	return self;
}

-(id)initWithProfile: (NSString *)filePath
{
	return [self initWithDictionary: [NSMutableDictionary dictionaryWithContentsOfFile: filePath]];
}

-(void)dealloc
{
	if (properties) { [properties release]; }
	[super dealloc];
}

#pragma mark -

/**********************************
	init a spec with default values for everything
**********************************/

-(void)setDefaults
{
	// stamp the spec with the creator
	[properties setObject: PROGRAM_STAMP												forKey: @"Creator"];

	//prior properties
	[properties setObject: CMDLINE_EXEC_PATH											forKey: @"ExecutablePath"];
	[properties setObject: CMDLINE_NIB_PATH												forKey: @"NibPath"];
	[properties setObject: [DEFAULT_DESTINATION_PATH stringByExpandingTildeInPath]		forKey: @"Destination"];

	[properties setValue: [NSNumber numberWithBool: NO]									forKey: @"DestinationOverride"];
	[properties setValue: [NSNumber numberWithBool: NO]									forKey: @"DevelopmentVersion"];
	[properties setValue: [NSNumber numberWithBool: NO]									forKey: @"OptimizeApplication"];
	
	// primary attributes	
	[properties setObject: @"MyApp"														forKey: @"Name"];
	[properties setObject: @""															forKey: @"ScriptPath"];
	[properties setObject: DEFAULT_OUTPUT_TYPE											forKey: @"Output"];
	[properties setObject: [NSNumber numberWithBool: YES]								forKey: @"HasCustomIcon"];
	[properties setObject: CMDLINE_ICON_PATH											forKey: @"CustomIconPath"];
	
	// secondary attributes
	[properties setObject: @"/bin/sh"													forKey: @"Interpreter"];
	[properties setObject: [NSMutableArray array]										forKey: @"Parameters"];
	[properties setObject: @"1.0"														forKey: @"Version"];
	[properties setObject: @"????"														forKey: @"Signature"];
	[properties setObject: [NSString stringWithFormat: @"org.%@.MyApp", NSUserName()]	forKey: @"Identifier"];
	[properties setObject: NSFullUserName()												forKey: @"Author"];
	
	[properties setValue: [NSNumber numberWithBool: NO]									forKey: @"AppPathAsFirstArg"];
	[properties setValue: [NSNumber numberWithBool: NO]									forKey: @"Droppable"];
	[properties setValue: [NSNumber numberWithBool: NO]									forKey: @"Secure"];
	[properties setValue: [NSNumber numberWithBool: NO]									forKey: @"Authentication"];
	[properties setValue: [NSNumber numberWithBool: YES]								forKey: @"RemainRunning"];
	[properties setValue: [NSNumber numberWithBool: NO]									forKey: @"ShowInDock"];
	
	// bundled files
	[properties setObject: [NSMutableArray array]										forKey: @"BundledFiles"];
	
	// environment
	[properties setObject: [NSMutableDictionary 
							dictionaryWithObject: PROGRAM_STAMP forKey: @"Creator"]		forKey: @"Environment"];

	// suffixes / file types
	[properties setObject: [NSMutableArray arrayWithObject: @"*"]						forKey: @"Suffixes"];
	[properties setObject: [NSMutableArray arrayWithObjects: @"****", @"fold", NULL]	forKey: @"FileTypes"];
	[properties setObject: @"Viewer"													forKey: @"Role"];

	// text output settings
	[properties setObject: [NSNumber numberWithInt: DEFAULT_OUTPUT_TXT_ENCODING]		forKey: @"TextEncoding"];
	[properties setObject: DEFAULT_OUTPUT_FONT											forKey: @"TextFont"];
	[properties setObject: [NSNumber numberWithFloat: DEFAULT_OUTPUT_FONTSIZE]			forKey: @"TextSize"];
	[properties setObject: DEFAULT_OUTPUT_FG_COLOR										forKey: @"TextForeground"];
	[properties setObject: DEFAULT_OUTPUT_BG_COLOR										forKey: @"TextBackground"];

	// status item settings
	[properties setObject: @"Text"														forKey: @"StatusItemDisplayType"];
	[properties setObject: @"MyApp"														forKey: @"StatusItemTitle"];
	[properties setObject: [NSData data]												forKey: @"StatusItemIcon"];
}


/****************************************

	This function creates the Platypus app
	based on the data contained in the spec.
	It's long...

****************************************/

-(BOOL)create
{
	int	      i;
	NSString *contentsPath, *macosPath, *resourcesPath, *lprojPath, *tmpPath = NSTemporaryDirectory();
	NSString *execDestinationPath, *infoPlistPath, *iconPath, *bundledFileDestPath, *nibDestPath;
	NSString *execPath, *bundledFilePath;
	NSString *appSettingsPlistPath;
	NSString *infoPlistStrings;
	NSString *b_enc_script = @"";
	NSMutableDictionary	*appSettingsPlist;
	NSFileManager *fileManager = [NSFileManager defaultManager];
	
	/////// MAKE SURE CONDITIONS ARE ACCEPTABLE //////
	
	// make sure we can write to temp path
	if (![fileManager isWritableFileAtPath: tmpPath])
	{
		error = [NSString stringWithFormat: @"Could not write to the temp directory '%@'.", tmpPath]; 
		return 0;
	}

	//check if app already exists
	if ([fileManager fileExistsAtPath: [properties objectForKey: @"Destination"]] && ![[properties objectForKey: @"DestinationOverride"] boolValue])
	{
		error = @"App already exists at path.";
		return 0;
	}
		
	////////////////////////// CREATE THE FOLDER HIERARCHY /////////////////////////////////////
	
	// we begin by creating the application bundle at temp path
	
	//Application.app bundle
	tmpPath = [tmpPath stringByAppendingString: [[properties objectForKey: @"Destination"] lastPathComponent]];
	[fileManager createDirectoryAtPath: tmpPath attributes:nil];
	
	//.app/Contents
	contentsPath = [tmpPath stringByAppendingString:@"/Contents"];
	[fileManager createDirectoryAtPath: contentsPath attributes:nil];
	
	//.app/Contents/MacOS
	macosPath = [contentsPath stringByAppendingString:@"/MacOS"];
	[fileManager createDirectoryAtPath: macosPath attributes:nil];
	
	//.app/Contents/Resources
	resourcesPath = [contentsPath stringByAppendingString:@"/Resources"];
	[fileManager createDirectoryAtPath: resourcesPath attributes:nil];
	
	//.app/Contents/Resources/English.lproj 
	lprojPath = [resourcesPath stringByAppendingString:@"/English.lproj"];
	[fileManager createDirectoryAtPath: lprojPath attributes:nil];
			
	////////////////////////// COPY FILES TO THE APP BUNDLE //////////////////////////////////
	
	//copy exec file
	//.app/Contents/Resources/MacOS/ScriptExec
	execPath = [properties objectForKey: @"ExecutablePath"];
	execDestinationPath = [macosPath stringByAppendingString:@"/"];
	execDestinationPath = [execDestinationPath stringByAppendingString: [properties objectForKey: @"Name"]]; 
	[fileManager copyPath:execPath toPath:execDestinationPath handler:nil];
	
	//copy nib file to app bundle
	//.app/Contents/Resources/English.lproj/MainMenu.nib
	nibDestPath = [lprojPath stringByAppendingString:@"/MainMenu.nib"];
	[fileManager copyPath: [properties objectForKey: @"NibPath"] toPath: nibDestPath handler: NULL];
		
	// if optimize application is set, we see if we can compile the nib file
	if ([[properties objectForKey: @"OptimizeApplication"] boolValue] == YES && [fileManager fileExistsAtPath: IBTOOL_PATH])
	{
		NSTask *ibToolTask = [[NSTask alloc] init];
		[ibToolTask setLaunchPath: IBTOOL_PATH];
		[ibToolTask setArguments: [NSArray arrayWithObjects: @"--strip", nibDestPath, nibDestPath, NULL]];
		[ibToolTask launch];
		[ibToolTask waitUntilExit];
		[ibToolTask release];
	}
	
	//create InfoPlist.strings file
	//.app/Contents/Resources/English.lproj/InfoPlist.strings
	infoPlistStrings = [NSString stringWithFormat:
						@"CFBundleName = \"%@\";\nCFBundleShortVersionString = \"%@\";\nCFBundleGetInfoString = \"%@ version %@ Copyright %d %@\";\nNSHumanReadableCopyright = \"Copyright %d %@.\";",  
										  [properties objectForKey: @"Name"], 
										  [properties objectForKey: @"Version"], 
										  [properties objectForKey: @"Name"],
										  [properties objectForKey: @"Version"], 
										  [[NSCalendarDate calendarDate] yearOfCommonEra], 
										  [properties objectForKey: @"Author"], 
										  [[NSCalendarDate calendarDate] yearOfCommonEra], 
										  [properties objectForKey: @"Author"]
						];
	[infoPlistStrings writeToFile:  [lprojPath stringByAppendingString:@"/InfoPlist.strings"] atomically: YES];
	
	// create script file in app bundle
	//.app/Contents/Resources/script
	if ([[properties objectForKey: @"Secure"] boolValue])
		b_enc_script = [NSData dataWithContentsOfFile: [properties objectForKey: @"ScriptPath"]];
	else
	{
		NSString *scriptFilePath = [resourcesPath stringByAppendingString:@"/script"];
		// make a symbolic link instead of copying script if this is a dev version
		if ([[properties objectForKey: @"DevelopmentVersion"] boolValue] == YES)
			[fileManager createSymbolicLinkAtPath: scriptFilePath pathContent: [properties objectForKey: @"ScriptPath"]];
		else
			[fileManager copyPath: [properties objectForKey: @"ScriptPath"] toPath: scriptFilePath handler:nil];
	}
		
	//create AppSettings.plist file
	//.app/Contents/Resources/AppSettings.plist
	appSettingsPlist = [NSMutableDictionary dictionaryWithCapacity: PROGRAM_MAX_LIST_ITEMS];
	[appSettingsPlist setObject: [properties objectForKey: @"AppPathAsFirstArg"] forKey: @"AppPathAsFirstArg"];
	[appSettingsPlist setObject: [properties objectForKey: @"Authentication"] forKey: @"RequiresAdminPrivileges"];
	[appSettingsPlist setObject: [properties objectForKey: @"Droppable"] forKey: @"Droppable"];
	[appSettingsPlist setObject: [properties objectForKey: @"RemainRunning"] forKey: @"RemainRunningAfterCompletion"];
	[appSettingsPlist setObject: [properties objectForKey: @"Secure"] forKey: @"Secure"];
	[appSettingsPlist setObject: [properties objectForKey: @"Output"] forKey: @"OutputType"];
	[appSettingsPlist setObject: [properties objectForKey: @"Interpreter"] forKey: @"ScriptInterpreter"];
	[appSettingsPlist setObject: PROGRAM_STAMP forKey: @"Creator"];
	[appSettingsPlist setObject: [properties objectForKey: @"Parameters"] forKey: @"InterpreterParams"];
	
	// we need only set text settings for the output types that use this information
	if ([[properties objectForKey: @"Output"] isEqualToString: @"Progress Bar"] ||
		[[properties objectForKey: @"Output"] isEqualToString: @"Text Window"] ||
		[[properties objectForKey: @"Output"] isEqualToString: @"Status Menu"])
	{
		[appSettingsPlist setObject: [properties objectForKey: @"TextFont"] forKey: @"TextFont"];
		[appSettingsPlist setObject: [properties objectForKey: @"TextSize"] forKey: @"TextSize"];
		[appSettingsPlist setObject: [properties objectForKey: @"TextForeground"] forKey: @"TextForeground"];
		[appSettingsPlist setObject: [properties objectForKey: @"TextBackground"] forKey: @"TextBackground"];
		[appSettingsPlist setObject: [properties objectForKey: @"TextEncoding"] forKey: @"TextEncoding"];
	}
	
	// likewise, status menu settings are only written if that is the output type
	if ([[properties objectForKey: @"Output"] isEqualToString: @"Status Menu"] == YES)
	{
		[appSettingsPlist setObject: [properties objectForKey: @"StatusItemDisplayType"] forKey: @"StatusItemDisplayType"];
		[appSettingsPlist setObject: [properties objectForKey: @"StatusItemTitle"] forKey: @"StatusItemTitle"];
		[appSettingsPlist setObject: [properties objectForKey: @"StatusItemIcon"] forKey: @"StatusItemIcon"];
	}
	
	// we  set the suffixes/file types in the AppSettings.plist if app is droppable
	if ([[properties objectForKey: @"Droppable"] boolValue] == YES)
	{		
		[appSettingsPlist setObject: [properties objectForKey:@"Suffixes"] forKey: @"DropSuffixes"];
		[appSettingsPlist setObject: [properties objectForKey:@"FileTypes"] forKey: @"DropTypes"];
	}
	
	if ([[properties objectForKey: @"Secure"] boolValue])
		[appSettingsPlist setObject: [NSKeyedArchiver archivedDataWithRootObject: b_enc_script] forKey: @"TextSettings"];
	
	appSettingsPlistPath = [resourcesPath stringByAppendingString:@"/AppSettings.plist"];
	[appSettingsPlist writeToFile: appSettingsPlistPath atomically: YES];//write it
	
	//create icon
	//.app/Contents/Resources/appIcon.icns
	iconPath = [resourcesPath stringByAppendingString:@"/appIcon.icns"];
	if ([[properties objectForKey: @"HasCustomIcon"] boolValue] == YES)
		[fileManager copyPath: [properties objectForKey: @"CustomIconPath"] toPath: iconPath handler: NULL];
	else
	{
		NSImage *img = [[NSImage alloc] initWithData: [properties objectForKey: @"Icon"]];
		if (img == NULL)
			NSLog(@"Could not init icon image");
		else
		{
			[self writeIcon: img toPath: iconPath];
			[img release];
		}
	}

	//create Info.plist file
	//.app/Contents/Info.plist
	infoPlistPath = [contentsPath stringByAppendingString:@"/Info.plist"];
	// create the Info.plist dictionary
	NSMutableDictionary *infoPlist = [NSMutableDictionary dictionaryWithObjectsAndKeys: 
							@"English", @"CFBundleDevelopmentRegion",
							[properties objectForKey: @"Name"], @"CFBundleExecutable", 
							[properties objectForKey: @"Name"], @"CFBundleName",
							[properties objectForKey: @"Name"], @"CFBundleDisplayName",
							[NSString stringWithFormat: @"%@ %@ Copyright %d %@", [properties objectForKey: @"Name"], [properties objectForKey: @"Version"], [[NSCalendarDate calendarDate] yearOfCommonEra], [properties objectForKey: @"Author"] ], @"CFBundleGetInfoString", 
							[NSString stringWithFormat: @"%@ %@ Copyright %d %@", [properties objectForKey: @"Name"], [properties objectForKey: @"Version"], [[NSCalendarDate calendarDate] yearOfCommonEra], [properties objectForKey: @"Author"] ], @"NSHumanReadableCopyright", 
							@"appIcon.icns", @"CFBundleIconFile",  
							//[properties objectForKey: @"Version"], @"CFBundleVersion",
							[properties objectForKey: @"Version"], @"CFBundleShortVersionString", 
							[properties objectForKey: @"Identifier"], @"CFBundleIdentifier",  
							[properties objectForKey: @"ShowInDock"], @"LSUIElement",
							@"6.0", @"CFBundleInfoDictionaryVersion",
							@"APPL", @"CFBundlePackageType",
							[properties objectForKey: @"Signature"], @"CFBundleSignature",
							[NSNumber numberWithBool: NO], @"LSHasLocalizedDisplayName",
							[properties objectForKey: @"Environment"], @"LSEnvironment",
							[NSNumber numberWithBool: NO], @"NSAppleScriptEnabled",
							@"MainMenu", @"NSMainNibFile",
							PROGRAM_MIN_SYS_VERSION, @"LSMinimumSystemVersion",
							@"NSApplication", @"NSPrincipalClass",  nil];
	
	// if droppable, we declare the accepted file types
	if ([[properties objectForKey: @"Droppable"] boolValue] == YES)
	{
		NSMutableDictionary	*typesAndSuffixesDict = [NSMutableDictionary dictionaryWithObjectsAndKeys:
						[properties objectForKey: @"Suffixes"], @"CFBundleTypeExtensions",//extensions
						[properties objectForKey: @"FileTypes"], @"CFBundleTypeOSTypes",//os types
						[properties objectForKey: @"Role"], @"CFBundleTypeRole", nil];//viewer or editor?
		[infoPlist setObject: [NSArray arrayWithObject: typesAndSuffixesDict] forKey: @"CFBundleDocumentTypes"];
		
		// we also set the suffixes/file types in the AppSettings.plist
		[appSettingsPlist setObject: [properties objectForKey:@"Suffixes"] forKey: @"DropSuffixes"];
		[appSettingsPlist setObject: [properties objectForKey:@"FileTypes"] forKey: @"DropTypes"];
	}
		
	// finally, write the Info.plist file
	[infoPlist writeToFile: infoPlistPath atomically: YES];
			
	//copy files in file list to the Resources folder
	//.app/Contents/Resources/*
	for (i = 0; i < [[properties objectForKey: @"BundledFiles"] count]; i++)
	{
		bundledFilePath = [[properties objectForKey: @"BundledFiles"] objectAtIndex: i];
		bundledFileDestPath = [resourcesPath stringByAppendingString:@"/"];
		bundledFileDestPath = [bundledFileDestPath stringByAppendingString: [bundledFilePath lastPathComponent]];
		
		// if it's a development version, we just symlink it
		if ([[properties objectForKey: @"DevelopmentVersion"] boolValue] == YES)
			[fileManager createSymbolicLinkAtPath: bundledFileDestPath pathContent: bundledFilePath];
		else // else we copy it 
			[fileManager copyPath: bundledFilePath toPath: bundledFileDestPath handler: NULL];
	}

	////////////////////////////////// COPY APP OVER TO FINAL DESTINATION /////////////////////////////////
	
	// we've now created the application bundle in the temporary directory
	// now it's time to move it to the destination specified by the user
	
	// first, let's see if there's anything there.  If we have override set on, we just delete that stuff.
	if ([fileManager fileExistsAtPath: [properties objectForKey: @"Destination"]] && [[properties objectForKey: @"DestinationOverride"] boolValue])
		[fileManager removeFileAtPath: [properties objectForKey: @"Destination"] handler: NULL];

	//if delete wasn't a success and there's still something there
	if ([fileManager fileExistsAtPath: [properties objectForKey: @"Destination"]]) 
	{
		[fileManager removeFileAtPath: tmpPath handler: NULL];
		error = @"Could not remove pre-existing item at destination path";
		return 0;
	}
	
	// now, move the newly created app to the destination
	[fileManager movePath: tmpPath toPath: [properties objectForKey: @"Destination"] handler: NULL];//move
	if (![fileManager fileExistsAtPath: [properties objectForKey: @"Destination"]]) //if move wasn't a success
	{
		[fileManager removeFileAtPath: tmpPath handler: NULL];
		error = @"Failed to create application at the specified destination";
		return 0;
	}

	// notify workspace that the file changed
	[[NSWorkspace sharedWorkspace] noteFileSystemChanged:  [properties objectForKey: @"Destination"]];
	
	return 1;
}

/************************

	Make sure the data
	in the spec isn't
	insane

************************/

-(BOOL)verify
{
	BOOL isDir;
	
	if (![[properties objectForKey: @"Destination"] hasSuffix: @"app"])
	{
		error = @"Destination must end with .app";
		return 0;
	}

	if ([[properties objectForKey: @"Name"] isEqualToString: @""])
	{
		error = @"Empty app name";
		return 0;
	}
	
	if (![[NSFileManager defaultManager] fileExistsAtPath: [properties objectForKey: @"ScriptPath"] isDirectory:&isDir] || isDir)
	{
		error = [NSString stringWithFormat: @"Script not found at path '%@'", [properties objectForKey: @"ScriptPath"] ];
		return 0;
	}
	
	if (![[NSFileManager defaultManager] fileExistsAtPath: [properties objectForKey: @"NibPath"] isDirectory:&isDir] || !isDir)
	{
		error = [NSString stringWithFormat: @"Nib not found at path '%@'", [properties objectForKey: @"NibPath"]];
		return 0;
	}
	
	if (![[NSFileManager defaultManager] fileExistsAtPath: [properties objectForKey: @"ExecutablePath"] isDirectory:&isDir] || isDir)
	{
		error = [NSString stringWithFormat: @"Executable not found at path '%@'", [properties objectForKey: @"ExecutablePath"]];
		return 0;
	}
	
	//make sure destination directory exists
	if (![[NSFileManager defaultManager] fileExistsAtPath: [[properties objectForKey: @"Destination"] stringByDeletingLastPathComponent] isDirectory: &isDir] || !isDir)
	{
		error = [NSString stringWithFormat: @"Destination directory '%@' does not exist.", [[properties objectForKey: @"Destination"] stringByDeletingLastPathComponent]];
		return 0;
	}
	
	//make sure we have write privileges for the destination directory
	if (![[NSFileManager defaultManager] isWritableFileAtPath: [[properties objectForKey: @"Destination"] stringByDeletingLastPathComponent]])
	{
		error = [NSString stringWithFormat: @"Don't have permission to write to the destination directory '%@'", [properties objectForKey: @"Destination"]] ;
		return 0;
	}
	
	return 1;
}

/************************
 Dump properties array to a file
************************/

-(void)dump: (NSString *)filePath
{
	[properties writeToFile: filePath atomically: YES];
}

/****************************
 Accessor functions
*****************************/

-(void)setProperty: (id)property forKey: (NSString *)theKey
{
	[properties setObject: property forKey: theKey];
}

-(id)propertyForKey: (NSString *)theKey
{
	return [properties objectForKey: theKey];
}

-(void)addProperties: (NSDictionary *)dict
{
	[properties addEntriesFromDictionary: dict];
}

-(NSDictionary *)properties
{
	return [properties retain];
}

-(NSString *)getError
{
	return error;
}

/*****************************************
 - Write an NSImage as icon to a path
*****************************************/

- (void)writeIcon: (NSImage *)img toPath: (NSString *)path
{
	IconFamily *iconFam = [[IconFamily alloc] initWithThumbnailsOfImage: img];
	if (iconFam == NULL) 
	{ 
		NSLog(@"Error generating icon");
		return; 
	}
	[iconFam writeToFile: path];
	[iconFam release];
}

@end
