/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Andy Jefferson - coding standards
2008 Andy Jefferson - add check on incompatible types for comparison
2008 Andy Jefferson - add handling of inconsistent datastore types
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.math.BigInteger;
import java.util.List;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of a Numeric expression in a Query.
 *
 * @version $Revision: 1.20 $
 **/
public class NumericExpression extends ScalarExpression
{
    /**
     * Constructor
     * @param qs the QueryExpression
     */
    protected NumericExpression(QueryExpression qs)
    {
        super(qs);
    }

    /**
     * 
     * @param qs the QueryExpression
     * @param mapping the mapping associated to this expression
     * @param te the TableExpression where this expression refers to
     */
    public NumericExpression(QueryExpression qs, JavaTypeMapping mapping, LogicSetExpression te)
    {
        super(qs, mapping, te);
    }

    /**
     * Generates statement as e.g. FUNCTION_NAME(arg[,argN]). The function returns a numeric value
     * @param functionName
     * @param args ScalarExpression list
     */
    public NumericExpression(String functionName, List args)
    {
        super(functionName, args);
    }
    
    /**
     * Generates statement as e.g. FUNCTION_NAME(arg AS type[,argN as typeN])
     * @param functionName Name of function
     * @param args ScalarExpression list
     * @param types String or ScalarExpression list
     */
    public NumericExpression(String functionName, List args, List types)
    {
        super(functionName, args, types);
    }

    /**
     * Perform a function <code>op</code> on <code>operand</code> 
     * @param op operator
     * @param operand operand
     */    
    public NumericExpression(MonadicOperator op, ScalarExpression operand)
    {
        super(op, operand);
    }
    
    /**
     * Performs a function on two arguments.
     * op(operand1,operand2)
     * operand1 op operand2 
     * @param operand1 the first expression
     * @param op the operator between operands
     * @param operand2 the second expression
     */
    public NumericExpression(ScalarExpression operand1, DyadicOperator op, ScalarExpression operand2)
    {
        super(operand1, op, operand2);

        // Assign a mapping to this expression otherwise it will have nothing to use for get/set operations (where needed)
        mapping = (operand1.getMapping() != null ? operand1.getMapping() : operand2.getMapping());
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        // Allow for parameter comparison
        assertValidTypeForParameterComparison(expr, NumericExpression.class);
        expr = getConsistentTypeForParameterComparison(expr);

        if (expr instanceof NullLiteral)
        {
            return expr.eq(this);
        }
        else if (expr instanceof NumericExpression || expr instanceof SubqueryExpression)
        {
            return new BooleanExpression(this, OP_EQ, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_EQ, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        // Allow for parameter comparison
        assertValidTypeForParameterComparison(expr, NumericExpression.class);
        expr = getConsistentTypeForParameterComparison(expr);

        if (expr instanceof NullLiteral)
        {
            return expr.noteq(this);
        }
        else if (expr instanceof NumericExpression || expr instanceof SubqueryExpression)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_NOTEQ, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }
        else
        {
            return super.noteq(expr);
        }
    }

    public BooleanExpression lt(ScalarExpression expr)
    {
        // Allow for parameter comparison
        assertValidTypeForParameterComparison(expr, NumericExpression.class);
        expr = getConsistentTypeForParameterComparison(expr);

        if (expr instanceof NumericExpression || expr instanceof SubqueryExpression)
        {
            return new BooleanExpression(this, OP_LT, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_LT, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }            
        else
        {
            return super.lt(expr);
        }
    }

    public BooleanExpression lteq(ScalarExpression expr)
    {
        // Allow for parameter comparison
        assertValidTypeForParameterComparison(expr, NumericExpression.class);
        expr = getConsistentTypeForParameterComparison(expr);

        if (expr instanceof NumericExpression || expr instanceof SubqueryExpression)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_LTEQ, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }              
        else
        {
            return super.lteq(expr);
        }
    }

    public BooleanExpression gt(ScalarExpression expr)
    {
        // Allow for parameter comparison
        assertValidTypeForParameterComparison(expr, NumericExpression.class);
        expr = getConsistentTypeForParameterComparison(expr);

        if (expr instanceof NumericExpression || expr instanceof SubqueryExpression)
        {
            return new BooleanExpression(this, OP_GT, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_GT, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }
        else
        {
            return super.gt(expr);
        }
    }

    public BooleanExpression gteq(ScalarExpression expr)
    {
        // Allow for parameter comparison
        assertValidTypeForParameterComparison(expr, NumericExpression.class);
        expr = getConsistentTypeForParameterComparison(expr);

        if (expr instanceof NumericExpression || expr instanceof SubqueryExpression)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_GTEQ, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }              
        else
        {
            return super.gteq(expr);
        }
    }

    public BooleanExpression in(ScalarExpression expr) 
    { 
        return new BooleanExpression(this, OP_IN, expr); 
    } 

    public ScalarExpression add(ScalarExpression expr)
    {
        if (expr instanceof NumericExpression)
        {
            return new NumericExpression(this, OP_ADD, expr);
        }
        else if (expr instanceof StringExpression)
        {
            return new StringExpression(qs.getStoreManager().getDatastoreAdapter().toStringExpression(this), OP_CONCAT, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new NumericExpression(this, OP_ADD, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }
        else if (expr instanceof NullLiteral)
        {
            return expr;
        }
        else
        {
            return super.add(expr);
        }
    }

    public ScalarExpression sub(ScalarExpression expr)
    {
        if (expr instanceof NumericExpression)
        {
            return new NumericExpression(this, OP_SUB, expr);
        }
        else if (expr instanceof CharacterExpression )
        {
            return new NumericExpression(this, OP_SUB, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }          
        else
        {
            return super.sub(expr);
        }
    }

    public ScalarExpression mul(ScalarExpression expr)
    {
        if (expr instanceof NumericExpression)
        {
            return new NumericExpression(this, OP_MUL, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new NumericExpression(this, OP_MUL, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }                
        else
        {
            return super.mul(expr);
        }
    }

    public ScalarExpression div(ScalarExpression expr)
    {
        if (expr instanceof NumericExpression)
        {
            return new NumericExpression(this, OP_DIV, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new NumericExpression(this, OP_DIV, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }                       
        else
        {
            return super.div(expr);
        }
    }

    /**
     * Method to return a modulus expression.
     * @param expr The expression to modulus against
     * @return The modulus expression
     */
    public ScalarExpression mod(ScalarExpression expr)
    {
        if (expr instanceof NumericExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().modOperator(this, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().modOperator(this, qs.getStoreManager().getDatastoreAdapter().toNumericExpression((CharacterExpression) expr));
        }        
        else
        {
            return super.mod(expr);
        }
    }

    public ScalarExpression neg()
    {
        return new NumericExpression(OP_NEG, this);
    }

    public ScalarExpression com()
    {
        return this.neg().sub(new IntegerLiteral(qs,mapping,BigInteger.ONE,false));
    }
}