/**********************************************************************
Copyright (c) 2002 Kelly Grizzle (TJDO) and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 


Contributors:
2002 Mike Martin (TJDO)
2003 Andy Jefferson - coding standards
2005 Andy Jefferson - added support for result set type/concurrency
2007 Andy Jefferson - removed RDBMS dependency
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.datanucleus.ObjectManager;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.mapped.mapping.MappingHelper;

/**
 * Representation of a snippet of an SQL statement.
 * May contain parameters.
 */
public class StatementText
{
    private StringBuffer statementText;
    private List<Parameter> parameters = null;
    private boolean encloseInParentheses = false;
    private String postpend;
    private List appended;

    /**
     * Constructor
     **/
    public StatementText()
    {
        appended = new ArrayList();
    }

    /**
     * Constructor
     * @param initialStatementText 
     */
    public StatementText(String initialStatementText)
    {
        this();
        append(initialStatementText);
    }
    
    /**
     * Convenience method to reset the SQL for the statement.
     * This is used when updating an expression internally, and need to regenerate
     * the statement.
     */
    public void clearStatement()
    {
        statementText = null;
        appended.clear();
    }

    /**
     * Whether to enclose this statement within parentheses
     */
    public void encloseInParentheses()
    {
        statementText = null;
        this.encloseInParentheses = true;
    }

    /**
     * Set a String to the end of the statement.  
     * @param s the string
     * @return the StatementText
     */
    public StatementText postpend(String s)
    {
        statementText = null;
        postpend = s;
        return this;
    }    

    /**
     * Append a char  
     * @param c the char
     * @return the StatementText
     */    
    public StatementText append(char c)
    {
        statementText = null;
        appended.add(new Character(c));
        return this;
    }

    /**
     * Append a char  
     * @param s the String
     * @return the StatementText
     */    
    public StatementText append(String s)
    {
        statementText = null;
        appended.add(s);
        return this;
    }

    /**
     * Append a QueryExpression
     * @param qsc the QueryExpression
     * @return the StatementText
     */    
    public StatementText append(QueryExpression qsc)
    {
        statementText = null;
        appended.add(qsc);
        return this;
    }

    /**
     * Append a StatementText
     * @param st the StatementText
     * @return the StatementText
     */    
    public StatementText append(StatementText st, int mode)
    {
        statementText = null;
        appended.add(st.toStatementString(mode));
        if (st.parameters != null)
        {
            if (parameters == null)
            {
                parameters = new ArrayList();
            }
            parameters.addAll(st.parameters);
        }
        return this;
    }

    /**
     * Append a ScalarExpression
     * @param expr the ScalarExpression
     * @return the StatementText
     */    
    public StatementText append(ScalarExpression expr)
    {
        statementText = null;
        appended.add(expr);
        return this;
    }
    
    /**
     * Append a parameter.
     * @param mapping the mapping
     * @param value the parameter value
     * @return the StatementText
     */
    public StatementText appendParameter(JavaTypeMapping mapping, Object value)
    {
        statementText = null;
        appended.add(new Parameter(mapping,value));
        return this;
    }

    /**
     * Method to set the parameters in the datastore statement.
     * @param om ObjectManager
     * @param datastoreStatement The datastore "statement"
     */
    public void applyParametersToStatement(ObjectManager om, Object datastoreStatement)
    {
        if (parameters != null)
        {
            int num = 1;

            Iterator<Parameter> i = parameters.iterator();
            while (i.hasNext())
            {
                Parameter param = i.next();
                JavaTypeMapping mapping = param.mapping;
                Object value = param.value;

                mapping.setObject(om, datastoreStatement, MappingHelper.getMappingIndices(num, mapping), value);
                if (mapping.getNumberOfDatastoreFields() > 0)
                {
                    num += mapping.getNumberOfDatastoreFields();
                }
                else
                {
                    num += 1;
                }
            }
        }
    }

    /**
     * Accessor for the SQL of the statement.
     * @param mode (0=PROJECTION;1=FILTER) - which means WHAT exactly ?
     * @return The SQL text
     */
    public String toStatementString(int mode)
    {
        if (statementText == null)
        {
            statementText = new StringBuffer();
            if (encloseInParentheses)
            {
                statementText.append("(");
            }
            for (int i = 0; i < appended.size(); i++)
            {
                Object item = appended.get(i);
                if (item instanceof ScalarExpression)
                {
                    ScalarExpression expr = (ScalarExpression) item;
                    StatementText st = expr.toStatementText(mode);
                    statementText.append(st.toStatementString(mode));

                    if (st.parameters != null)
                    {
                        if (parameters == null)
                        {
                            parameters = new ArrayList();
                        }
                        parameters.addAll(st.parameters);
                    }
                }
                else if (item instanceof Parameter)
                {
                    Parameter param = (Parameter) item;
                    statementText.append('?');

                    if (parameters == null)
                    {
                        parameters = new ArrayList();
                    }
                    parameters.add(param);
                }
                else if (item instanceof QueryExpression)
                {
                    QueryExpression qe = (QueryExpression) item;
                    StatementText st = qe.toStatementText(false); // TODO use the same lock as the caller
                    statementText.append(st.toStatementString(mode));
                    if (st.parameters != null)
                    {
                        if (parameters == null)
                        {
                            parameters = new ArrayList();
                        }
                        parameters.addAll(st.parameters);
                    }
                }
                else if (item instanceof StatementText)
                {
                    StatementText st = (StatementText) item;
                    statementText.append(st.toStatementString(mode));
                    if (st.parameters != null)
                    {
                        if (parameters == null)
                        {
                            parameters = new ArrayList();
                        }
                        parameters.addAll(st.parameters);
                    }
                }
                else
                {
                    statementText.append(item);
                }
            }
            if (encloseInParentheses)
            {
                statementText.append(")");
            }
            statementText.append((postpend == null ? "" : postpend));
        }
        return statementText.toString();
    }

    /**
     * Accessor for the string form of the statement.
     * @return String form of the statement
     */
    public String toString()
    {
        //the mode should be indifferent, so we use projection, whatever that means
        return toStatementString(ScalarExpression.PROJECTION);
    }

    /**
     * Internal class to keep parameters
     */
    private class Parameter
    {
        final JavaTypeMapping mapping;
        final Object value;

        public Parameter(JavaTypeMapping mapping, Object value)
        {
            this.mapping = mapping;
            this.value = value;
        }
    }
}