/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.enhancer.asm;

import javax.jdo.spi.PersistenceCapable;

import org.datanucleus.enhancer.ClassEnhancer;
import org.datanucleus.enhancer.ClassMethod;
import org.datanucleus.enhancer.DataNucleusEnhancer;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.util.Localiser;
import org.objectweb.asm.AnnotationVisitor;
import org.objectweb.asm.Attribute;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.Label;
import org.objectweb.asm.MethodAdapter;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;

/**
 * Adapter for property setter methods in JDO-enabled classes.
 * This adapter processes the setXXX method and
 * <ul>
 * <li>Creates jdoSetXXX with the same code as was present in setXXX</li>
 * <li>Changes setXXX to have the code below</li>
 * </ul>
 * When detachable this will be (CHECK_WRITE variant)
 * <pre>
 * void setZZZ(YYY zzz)
 * {
 *     if (jdoFlags != 0 && jdoStateManager != null)
 *         jdoStateManager.setStringField(this, 2, jdoGetZZZ(), zzz);
 *     else
 *     {
 *         jdoSetXXX(zzz);
 *         if (jdoIsDetached() == true)
 *             ((BitSet) jdoDetachedState[3]).set(2);
 *     }
 * }
 * </pre>
 * and when not detachable
 * <pre>
 * void setZZZ(YYY zzz)
 * {
 *     if (jdoFlags > 0 && jdoStateManager != null)
 *         jdoStateManager.setObjectField(this, 2, jdoGetZZZ(), zzz);
 *     jdoSetXXX(zzz);
 * }
 * </pre>
 * There are other variants for MEDIATE_WRITE and NORMAL_WRITE
 */
public class JdoPropertySetterAdapter extends MethodAdapter
{
    /** Localisation of messages. */
    protected static Localiser LOCALISER = Localiser.getInstance("org.datanucleus.enhancer.Localisation", ClassEnhancer.class.getClassLoader());

    /** The enhancer for this class. */
    protected ASMClassEnhancer enhancer;

    /** Name for the method being adapted. */
    protected String methodName;

    /** Descriptor for the method being adapted. */
    protected String methodDescriptor;

    /** MetaData for the property. */
    protected AbstractMemberMetaData mmd;

    /** Visitor for the jdoSetXXX method. */
    protected MethodVisitor jdoVisitor = null;

    /**
     * Constructor for the method adapter.
     * @param mv MethodVisitor
     * @param enhancer ClassEnhancer for the class with the method
     * @param methodName Name of the method
     * @param methodDesc Method descriptor
     * @param mmd MetaData for the property
     * @param cv ClassVisitor
     */
    public JdoPropertySetterAdapter(MethodVisitor mv, ASMClassEnhancer enhancer, String methodName, String methodDesc,
            AbstractMemberMetaData mmd, ClassVisitor cv)
    {
        super(mv);
        this.enhancer = enhancer;
        this.methodName = methodName;
        this.methodDescriptor = methodDesc;
        this.mmd = mmd;

        // Generate jdoSetXXX method to include code that this setXXX currently has
        int access = (mmd.isPublic() ? Opcodes.ACC_PUBLIC : 0) | 
            (mmd.isProtected() ? Opcodes.ACC_PROTECTED : 0) | 
            (mmd.isPrivate() ? Opcodes.ACC_PRIVATE : 0) |
            (mmd.isAbstract() ? Opcodes.ACC_ABSTRACT : 0);
        this.jdoVisitor = cv.visitMethod(access, "jdoSet" + mmd.getName(), methodDesc, null, null);
    }

    /**
     * Method called at the end of visiting the setXXX method.
     * This is used to add the jdoSetXXX method with the same code as is present originally in the setXXX method.
     */
    public void visitEnd()
    {
        jdoVisitor.visitEnd();
        if (DataNucleusEnhancer.LOGGER.isDebugEnabled())
        {
            String msg = ClassMethod.getMethodAdditionMessage("jdoSet" + mmd.getName(), null, new Object[]{mmd.getType()}, new String[] {"val"});
            DataNucleusEnhancer.LOGGER.debug(LOCALISER.msg("Enhancer.AddMethod", msg));
        }

        if (!mmd.isAbstract())
        {
            // Property is not abstract so generate the setXXX method to use the jdoSetXXX we just added
            generateSetXXXMethod(mv, mmd, enhancer.getASMClassName(), enhancer.getClassDescriptor());
        }
    }

    /**
     * Convenience method to use the MethodVisitor to generate the code for the method setXXX() for the
     * property with the specified MetaData.
     * @param mv MethodVisitor
     * @param mmd MetaData for the property
     * @param asmClassName ASM class name for the owning class
     * @param asmClassDesc ASM descriptor for the owning class
     */
    public static void generateSetXXXMethod(MethodVisitor mv, AbstractMemberMetaData mmd,
            String asmClassName, String asmClassDesc)
    {
        String[] argNames = new String[] {"objPC", "val"};
        String fieldTypeDesc = Type.getDescriptor(mmd.getType());

        mv.visitCode();

        AbstractClassMetaData cmd = mmd.getAbstractClassMetaData();
        if ((mmd.getJdoFieldFlag() & PersistenceCapable.MEDIATE_WRITE) == PersistenceCapable.MEDIATE_WRITE)
        {
            // MEDIATE_WRITE
            Label startLabel = new Label();
            mv.visitLabel(startLabel);

            // "if (objPC.jdoStateManager == null) objPC.ZZZ = zzz;"
            mv.visitVarInsn(Opcodes.ALOAD, 0);
            mv.visitFieldInsn(Opcodes.GETFIELD, asmClassName,
                ClassEnhancer.FN_StateManager, "Ljavax/jdo/spi/StateManager;");
            Label l1 = new Label();
            mv.visitJumpInsn(Opcodes.IFNONNULL, l1);
            mv.visitVarInsn(Opcodes.ALOAD, 0);
            ASMUtils.addLoadForType(mv, mmd.getType(), 1);
            mv.visitMethodInsn(Opcodes.INVOKEVIRTUAL, asmClassName,
                "jdoSet" + mmd.getName(), "(" + fieldTypeDesc + ")V");
            Label l3 = new Label();
            mv.visitJumpInsn(Opcodes.GOTO, l3);

            // "else objPC.jdoStateManager.setYYYField(objPC, 0, objPC.ZZZ, zzz);"
            mv.visitLabel(l1);
            mv.visitVarInsn(Opcodes.ALOAD, 0);
            mv.visitFieldInsn(Opcodes.GETFIELD, asmClassName,
                ClassEnhancer.FN_StateManager, "Ljavax/jdo/spi/StateManager;");
            mv.visitVarInsn(Opcodes.ALOAD, 0);
            ASMUtils.addBIPUSHToMethod(mv, mmd.getFieldId());
            if (cmd.getPersistenceCapableSuperclass() != null)
            {
                mv.visitFieldInsn(Opcodes.GETSTATIC, asmClassName, "jdoInheritedFieldCount", "I");
                mv.visitInsn(Opcodes.IADD);
            }

            mv.visitVarInsn(Opcodes.ALOAD, 0);
            mv.visitMethodInsn(Opcodes.INVOKEVIRTUAL, asmClassName, 
                "jdoGet" + mmd.getName(), "()" + fieldTypeDesc);
            ASMUtils.addLoadForType(mv, mmd.getType(), 1);
            String jdoMethodName = "set" + ASMUtils.getTypeNameForJDOMethod(mmd.getType()) + "Field";
            String argTypeDesc = fieldTypeDesc;
            if (jdoMethodName.equals("setObjectField"))
            {
                argTypeDesc = ASMUtils.CD_Object;
            }
            mv.visitMethodInsn(Opcodes.INVOKEINTERFACE, "javax/jdo/spi/StateManager",
                jdoMethodName, "(Ljavax/jdo/spi/PersistenceCapable;I" + argTypeDesc + argTypeDesc + ")V");
            mv.visitLabel(l3);

            if (cmd.isDetachable())
            {
                // "if (objPC.jdoIsDetached() == true)"
                mv.visitVarInsn(Opcodes.ALOAD, 0);
                mv.visitMethodInsn(Opcodes.INVOKEVIRTUAL, asmClassName,
                    ClassEnhancer.MN_JdoIsDetached, "()Z");
                Label l6 = new Label();
                mv.visitJumpInsn(Opcodes.IFEQ, l6);

                // "((BitSet) objPC.jdoDetachedState[3]).set(0);"
                mv.visitVarInsn(Opcodes.ALOAD, 0);
                mv.visitFieldInsn(Opcodes.GETFIELD, asmClassName,
                    ClassEnhancer.FN_JdoDetachedState, "[Ljava/lang/Object;");
                mv.visitInsn(Opcodes.ICONST_3);
                mv.visitInsn(Opcodes.AALOAD);
                mv.visitTypeInsn(Opcodes.CHECKCAST, "java/util/BitSet");
                ASMUtils.addBIPUSHToMethod(mv, mmd.getFieldId());
                if (cmd.getPersistenceCapableSuperclass() != null)
                {
                    mv.visitFieldInsn(Opcodes.GETSTATIC, asmClassName,
                        ClassEnhancer.FN_JdoInheritedFieldCount, "I");
                    mv.visitInsn(Opcodes.IADD);
                }
                mv.visitMethodInsn(Opcodes.INVOKEVIRTUAL, "java/util/BitSet", "set", "(I)V");
                mv.visitLabel(l6);
            }
            mv.visitInsn(Opcodes.RETURN);

            Label endLabel = new Label();
            mv.visitLabel(endLabel);
            mv.visitLocalVariable(argNames[0], asmClassDesc, null, startLabel, endLabel, 0);
            mv.visitLocalVariable(argNames[1], fieldTypeDesc, null, startLabel, endLabel, 1);
            mv.visitMaxs(5, 2);
        }
        else if ((mmd.getJdoFieldFlag() & PersistenceCapable.CHECK_WRITE) == PersistenceCapable.CHECK_WRITE)
        {
            // CHECK_WRITE
            Label startLabel = new Label();
            mv.visitLabel(startLabel);

            // "if (objPC.jdoFlags != 0 && objPC.jdoStateManager != null)"
            mv.visitVarInsn(Opcodes.ALOAD, 0);
            mv.visitFieldInsn(Opcodes.GETFIELD, asmClassName, ClassEnhancer.FN_Flag, "B");
            Label l1 = new Label();
            mv.visitJumpInsn(Opcodes.IFEQ, l1);
            mv.visitVarInsn(Opcodes.ALOAD, 0);
            mv.visitFieldInsn(Opcodes.GETFIELD, asmClassName,
                ClassEnhancer.FN_StateManager, "Ljavax/jdo/spi/StateManager;");
            mv.visitJumpInsn(Opcodes.IFNULL, l1);

            // "objPC.jdoStateManager.setYYYField(objPC, 8, objPC.ZZZ, val);"
            mv.visitVarInsn(Opcodes.ALOAD, 0);
            mv.visitFieldInsn(Opcodes.GETFIELD, asmClassName,
                ClassEnhancer.FN_StateManager, "Ljavax/jdo/spi/StateManager;");
            mv.visitVarInsn(Opcodes.ALOAD, 0);
            ASMUtils.addBIPUSHToMethod(mv, mmd.getFieldId());
            if (cmd.getPersistenceCapableSuperclass() != null)
            {
                mv.visitFieldInsn(Opcodes.GETSTATIC, asmClassName,
                    ClassEnhancer.FN_JdoInheritedFieldCount, "I");
                mv.visitInsn(Opcodes.IADD);
            }

            mv.visitVarInsn(Opcodes.ALOAD, 0);
            mv.visitMethodInsn(Opcodes.INVOKEVIRTUAL, asmClassName, 
                "jdoGet" + mmd.getName(), "()" + fieldTypeDesc);
            ASMUtils.addLoadForType(mv, mmd.getType(), 1);
            String jdoMethodName = "set" + ASMUtils.getTypeNameForJDOMethod(mmd.getType()) + "Field";
            String argTypeDesc = fieldTypeDesc;
            if (jdoMethodName.equals("setObjectField"))
            {
                argTypeDesc = ASMUtils.CD_Object;
            }
            mv.visitMethodInsn(Opcodes.INVOKEINTERFACE, "javax/jdo/spi/StateManager",
                jdoMethodName, "(Ljavax/jdo/spi/PersistenceCapable;I" + argTypeDesc + argTypeDesc + ")V");
            Label l3 = new Label();
            mv.visitJumpInsn(Opcodes.GOTO, l3);

            // "objPC.text = val;"
            mv.visitLabel(l1);
            mv.visitVarInsn(Opcodes.ALOAD, 0);
            ASMUtils.addLoadForType(mv, mmd.getType(), 1);
            mv.visitMethodInsn(Opcodes.INVOKEVIRTUAL, asmClassName,
                "jdoSet" + mmd.getName(), "(" + fieldTypeDesc + ")V");

            if (cmd.isDetachable())
            {
                // "if (objPC.jdoIsDetached() == true)  ((BitSet) objPC.jdoDetachedState[3]).set(8);"
                mv.visitVarInsn(Opcodes.ALOAD, 0);
                mv.visitMethodInsn(Opcodes.INVOKEVIRTUAL, asmClassName,
                    ClassEnhancer.MN_JdoIsDetached, "()Z");
                mv.visitJumpInsn(Opcodes.IFEQ, l3);
                mv.visitVarInsn(Opcodes.ALOAD, 0);
                mv.visitFieldInsn(Opcodes.GETFIELD, asmClassName,
                    ClassEnhancer.FN_JdoDetachedState, "[Ljava/lang/Object;");
                mv.visitInsn(Opcodes.ICONST_3);
                mv.visitInsn(Opcodes.AALOAD);
                mv.visitTypeInsn(Opcodes.CHECKCAST, "java/util/BitSet");
                ASMUtils.addBIPUSHToMethod(mv, mmd.getFieldId());
                if (cmd.getPersistenceCapableSuperclass() != null)
                {
                    mv.visitFieldInsn(Opcodes.GETSTATIC, asmClassName,
                        ClassEnhancer.FN_JdoInheritedFieldCount, "I");
                    mv.visitInsn(Opcodes.IADD);
                }
                mv.visitMethodInsn(Opcodes.INVOKEVIRTUAL, "java/util/BitSet", "set", "(I)V");
            }

            mv.visitLabel(l3);
            mv.visitInsn(Opcodes.RETURN);

            Label endLabel = new Label();
            mv.visitLabel(endLabel);
            mv.visitLocalVariable(argNames[0], asmClassDesc, null, startLabel, endLabel, 0);
            mv.visitLocalVariable(argNames[1], fieldTypeDesc, null, startLabel, endLabel, 1);
            mv.visitMaxs(5, 2);
        }
        else
        {
            // NORMAL
            Label startLabel = new Label();
            mv.visitLabel(startLabel);

            mv.visitVarInsn(Opcodes.ALOAD, 0);
            ASMUtils.addLoadForType(mv, mmd.getType(), 1);
            mv.visitMethodInsn(Opcodes.INVOKEVIRTUAL, asmClassName,
                "jdoSet" + mmd.getName(), "(" + fieldTypeDesc + ")V");
            mv.visitInsn(Opcodes.RETURN);

            Label endLabel = new Label();
            mv.visitLabel(endLabel);
            mv.visitLocalVariable(argNames[0], asmClassDesc, null, startLabel, endLabel, 0);
            mv.visitLocalVariable(argNames[1], fieldTypeDesc, null, startLabel, endLabel, 1);
            mv.visitMaxs(2, 2);
        }

        mv.visitEnd();
    }

    public AnnotationVisitor visitAnnotation(String arg0, boolean arg1)
    {
        // Keep any annotation on the setXXX method
        return mv.visitAnnotation(arg0, arg1);
    }

    public AnnotationVisitor visitAnnotationDefault()
    {
        return jdoVisitor.visitAnnotationDefault();
    }

    public void visitAttribute(Attribute arg0)
    {
        jdoVisitor.visitAttribute(arg0);
    }

    public void visitCode()
    {
        jdoVisitor.visitCode();
    }

    public void visitFieldInsn(int arg0, String arg1, String arg2, String arg3)
    {
        jdoVisitor.visitFieldInsn(arg0, arg1, arg2, arg3);
    }

    public void visitFrame(int arg0, int arg1, Object[] arg2, int arg3, Object[] arg4)
    {
        jdoVisitor.visitFrame(arg0, arg1, arg2, arg3, arg4);
    }

    public void visitIincInsn(int arg0, int arg1)
    {
        jdoVisitor.visitIincInsn(arg0, arg1);
    }

    public void visitInsn(int arg0)
    {
        jdoVisitor.visitInsn(arg0);
    }

    public void visitIntInsn(int arg0, int arg1)
    {
        jdoVisitor.visitIntInsn(arg0, arg1);
    }

    public void visitJumpInsn(int arg0, Label arg1)
    {
        jdoVisitor.visitJumpInsn(arg0, arg1);
    }

    public void visitLabel(Label arg0)
    {
        jdoVisitor.visitLabel(arg0);
    }

    public void visitLdcInsn(Object arg0)
    {
        jdoVisitor.visitLdcInsn(arg0);
    }

    public void visitLineNumber(int arg0, Label arg1)
    {
        jdoVisitor.visitLineNumber(arg0, arg1);
    }

    public void visitLocalVariable(String arg0, String arg1, String arg2, Label arg3, Label arg4, int arg5)
    {
        jdoVisitor.visitLocalVariable(arg0, arg1, arg2, arg3, arg4, arg5);
    }

    public void visitLookupSwitchInsn(Label arg0, int[] arg1, Label[] arg2)
    {
        jdoVisitor.visitLookupSwitchInsn(arg0, arg1, arg2);
    }

    public void visitMaxs(int arg0, int arg1)
    {
        jdoVisitor.visitMaxs(arg0, arg1);
    }

    public void visitMethodInsn(int arg0, String arg1, String arg2, String arg3)
    {
        jdoVisitor.visitMethodInsn(arg0, arg1, arg2, arg3);
    }

    public void visitMultiANewArrayInsn(String arg0, int arg1)
    {
        jdoVisitor.visitMultiANewArrayInsn(arg0, arg1);
    }

    public AnnotationVisitor visitParameterAnnotation(int arg0, String arg1, boolean arg2)
    {
        return jdoVisitor.visitParameterAnnotation(arg0, arg1, arg2);
    }

    public void visitTableSwitchInsn(int arg0, int arg1, Label arg2, Label[] arg3)
    {
        jdoVisitor.visitTableSwitchInsn(arg0, arg1, arg2, arg3);
    }

    public void visitTryCatchBlock(Label arg0, Label arg1, Label arg2, String arg3)
    {
        jdoVisitor.visitTryCatchBlock(arg0, arg1, arg2, arg3);
    }

    public void visitTypeInsn(int arg0, String arg1)
    {
        jdoVisitor.visitTypeInsn(arg0, arg1);
    }

    public void visitVarInsn(int arg0, int arg1)
    {
        jdoVisitor.visitVarInsn(arg0, arg1);
    }
}