package com.google.appengine.api.internal;

import java.util.Collections;
import java.util.List;

/** Utilities for dealing with repackaged classes. Not part of the public API. */
public final class Repackaged {
  private Repackaged() {}

  static final String REPACKAGED_PREFIX = "com.google.appengine.repackaged.";

  /**
   * Returns an immutable copy of the given list if it is of a repackaged class, otherwise returns
   * the original list.
   */
  public static <E> List<E> copyIfRepackagedElseOriginal(List<E> list) {
    return isRepackaged(list) ? ImmutableCopy.list(list) : list;
  }

  /**
   * Returns an immutable copy of the given list if it is of a repackaged class, otherwise returns
   * an unmodifiable wrapper around the original list.
   */
  public static <E> List<E> copyIfRepackagedElseUnmodifiable(List<E> list) {
    return isRepackaged(list) ? ImmutableCopy.list(list) : Collections.unmodifiableList(list);
  }

  /** Returns true if the given object has a class that is the result of App Engine repackaging. */
  public static boolean isRepackaged(Object object) {
    return object != null && object.getClass().getName().startsWith(REPACKAGED_PREFIX);
  }
}
