/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;


/**
 * Perform a exists function on a Query Expression returning a true boolean value if one or more elements are retrieved.
 *
 * @version $Revision: 1.6 $
 **/
public class ExistsExpression extends BooleanExpression
{
    private final QueryExpression argumentExpression;
    private final boolean truthTest;

    /**
     * Constructs a expression for existence of the argument expression.
     * @param qs the QueryExpression
     * @param argumentExpression the QueryExpression returning records to the exists function
     */
    public ExistsExpression(QueryExpression qs, QueryExpression argumentExpression)
    {
        this(qs, argumentExpression, true);
    }

    /**
     * Constructs a expression for existence (or not) of the argument expression.
     * Create an "EXISTS", or "NOT EXISTS" depending on the truth flag.
     * @param qs the QueryExpression
     * @param argumentExpression the QueryExpression returning records to the exists function
     * @param truthTest true to perform a "exists", and false to perform a "not exists"
     */
    public ExistsExpression(QueryExpression qs, QueryExpression argumentExpression, boolean truthTest)
    {
        super(qs);

        this.argumentExpression = argumentExpression;
        this.truthTest = truthTest;
        if (!truthTest)
        {
            st.append("NOT ");
        }

        argumentExpression.setExistsSubQuery(true);
        st.append("EXISTS (").append(argumentExpression).append(')');
    }

    public BooleanExpression not()
    {
        return new ExistsExpression(qs, argumentExpression, !truthTest);
    }

    public BooleanExpression and(ScalarExpression expr)
    {
        if (expr instanceof ExistsExpression)
        {
            ExistsExpression expr2 = (ExistsExpression)expr;
            if (expr2.qs == this.argumentExpression)
            {
                // TODO What condition is this trying to cater for ???? Example???
                this.argumentExpression.andCondition(expr2);
                return this;
            }
            else if (this.qs == expr2.argumentExpression)
            {
                // TODO What condition is this trying to cater for ???? Example???
                expr2.argumentExpression.andCondition(this);
                return expr2;
            }
            return super.and(expr);
        }
        else if (expr.qs != this.qs && expr.qs != this.argumentExpression)
        {
            // TODO What condition is this trying to cater for ???? Example???
            return super.and(expr);
        }
        else
        {
            // TODO What condition is this trying to cater for ???? Example???
            argumentExpression.andCondition((BooleanExpression)expr,true);
            return this;
        }
    }
}