/**********************************************************************
Copyright (c) 2009 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus.api.jpa.criteria;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Selection;
import javax.persistence.criteria.Predicate.BooleanOperator;

import org.datanucleus.query.expression.DyadicExpression;
import org.datanucleus.query.expression.InvokeExpression;
import org.datanucleus.query.expression.Literal;

/**
 * Implementation of JPA2 Criteria "Expression".
 */
public class ExpressionImpl<T> implements Selection<T>, Expression<T>, Serializable
{
    private final Class<T> cls;
    private String alias;

    /** The underlying (generic) query expression. */
    org.datanucleus.query.expression.Expression queryExpr;

    public ExpressionImpl(Class<T> cls)
    {
        this.cls = cls;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.Expression#as(java.lang.Class)
     */
    public <X> Expression<X> as(Class<X> cls)
    {
        ExpressionImpl<X> expr = new ExpressionImpl<X>(cls);
        expr.queryExpr = queryExpr;
        return expr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.Expression#in(java.util.Collection)
     */
    public Predicate in(Collection<?> coll)
    {
        PredicateImpl pred = new PredicateImpl(BooleanOperator.OR);
        pred.queryExpr = queryExpr;
        Iterator<?> iter = coll.iterator();
        while (iter.hasNext())
        {
            PredicateImpl predSub = new PredicateImpl();
            Object obj = iter.next();
            predSub.queryExpr = new DyadicExpression(((ExpressionImpl)this).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_EQ, new Literal(obj));
            pred.append(predSub);
        }
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.Expression#in(javax.persistence.criteria.Expression<?>[])
     */
    public Predicate in(Expression<?>... exprs)
    {
        PredicateImpl pred = new PredicateImpl(BooleanOperator.OR);
        pred.queryExpr = queryExpr;
        for (int i=0;i<exprs.length;i++)
        {
            PredicateImpl predSub = new PredicateImpl();
            predSub.queryExpr = new DyadicExpression(((ExpressionImpl)this).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_EQ,
                ((ExpressionImpl)exprs[i]).getQueryExpression());
            pred.append(predSub);
        }
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.Expression#in(javax.persistence.criteria.Expression)
     */
    public Predicate in(Expression<Collection<?>> collExpr)
    {
        PredicateImpl pred = new PredicateImpl();
        List args = new ArrayList();
        args.add(this.getQueryExpression());
        org.datanucleus.query.expression.Expression queryExpr =
            new InvokeExpression(((ExpressionImpl)collExpr).getQueryExpression(), "contains", args);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.Expression#in(java.lang.Object[])
     */
    public Predicate in(Object... expr)
    {
        PredicateImpl pred = new PredicateImpl(BooleanOperator.OR);
        pred.queryExpr = queryExpr;
        for (int i=0;i<expr.length;i++)
        {
            PredicateImpl predSub = new PredicateImpl();
            predSub.queryExpr = new DyadicExpression(((ExpressionImpl)this).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_EQ, new Literal(expr[i]));
            pred.append(predSub);
        }
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.Expression#isNotNull()
     */
    public Predicate isNotNull()
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(null);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_NOTEQ,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.Expression#isNull()
     */
    public Predicate isNull()
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(null);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(this.getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_EQ,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.Selection#alias(java.lang.String)
     */
    public Selection<T> alias(String alias)
    {
        this.alias = alias;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.Selection#getCompoundSelectionItems()
     */
    public List<Selection<?>> getCompoundSelectionItems()
    {
        // TODO Implement this
        throw new UnsupportedOperationException(
            "Not yet implemented. Provide a testcase that uses this and raise a JIRA attaching your testcase");
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.Selection#isCompoundSelection()
     */
    public boolean isCompoundSelection()
    {
        // TODO Implement this
        throw new UnsupportedOperationException(
            "Not yet implemented. Provide a testcase that uses this and raise a JIRA attaching your testcase");
    }

    /* (non-Javadoc)
     * @see javax.persistence.TupleElement#getAlias()
     */
    public String getAlias()
    {
        return alias;
    }

    /* (non-Javadoc)
     * @see javax.persistence.TupleElement#getJavaType()
     */
    public Class<? extends T> getJavaType()
    {
        return cls;
    }

    /**
     * Accessor for the underlying (generic) query expression.
     * @return The query expression
     */
    public org.datanucleus.query.expression.Expression getQueryExpression()
    {
        return queryExpr;
    }

    /**
     * Method to print out the expression as it would appear in JPQL single-string form.
     * @return The JPQL single string form of this expression
     */
    public String toString()
    {
        return JPQLHelper.getJPQLForExpression(queryExpr);
    }
}