/**********************************************************************
Copyright (c) 2002 Kelly Grizzle and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2002 Mike Martin - unknown changes
2003 Erik Bengtson - removed exist() operation
2004 Andy Jefferson - added getHighestFieldNumber()
2005 Andy Jefferson - javadocs
2007 Xuan Baldauf - Contrib of notifyMadePersistentClean()
2007 Xuan Baldauf - Contrib of internalXXXFieldXXX() methods
2008 Andy Jefferson - removed all deps on org.datanucleus.store.mapped
    ...
**********************************************************************/
package org.datanucleus;

import java.io.PrintWriter;
import java.util.Set;

import javax.jdo.spi.PersistenceCapable;

import org.datanucleus.cache.CachedPC;
import org.datanucleus.exceptions.NucleusObjectNotFoundException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.state.ActivityState;
import org.datanucleus.state.FetchPlanState;
import org.datanucleus.state.RelationshipManager;
import org.datanucleus.store.FieldValues;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.fieldmanager.FieldManager;

/**
 * StateManager for persistable objects. 
 * Makes the assumption that a StateManager corresponds to ONE persistable object.
 */
public interface StateManager
{
    // Types of PC objects that can be managed by this StateManager
    /** PC **/
    public static int PC = 0;
    /** Embedded (or serialised) PC **/
    public static int EMBEDDED_PC = 1;
    /** Embedded (or serialised) Collection Element PC **/
    public static int EMBEDDED_COLLECTION_ELEMENT_PC = 2;
    /** Embedded (or serialised) Map Key PC **/
    public static int EMBEDDED_MAP_KEY_PC = 3;
    /** Embedded (or serialised) Map Value PC **/
    public static int EMBEDDED_MAP_VALUE_PC = 4;

    /**
     * Initialises a state manager to manage a hollow instance having the given object ID and the given
     * (optional) field values. This constructor is used for creating new instances of existing persistent
     * objects, and consequently shouldnt be used when the StoreManager controls the creation of such objects
     * (such as in an ODBMS).
     * @param id the identity of the object.
     * @param fv the initial field values of the object (optional)
     * @param pcClass Class of the object that this will manage the state for
     */
    void initialiseForHollow(Object id, FieldValues fv, Class pcClass);

    /**
     * Initialises a state manager to manage a HOLLOW / P_CLEAN instance having the given FieldValues.
     * This constructor is used for creating new instances of existing persistent objects using application 
     * identity, and consequently shouldnt be used when the StoreManager controls the creation of such objects
     * (such as in an ODBMS).
     * @param fv the initial field values of the object.
     * @param pcClass Class of the object that this will manage the state for
     */
    void initialiseForHollowAppId(FieldValues fv, Class pcClass);

    /**
     * Initialises a state manager to manage the given hollow instance having the given object ID.
     * Unlike the {@link #initialiseForHollow} method, this method does not create a new instance and instead 
     * takes a pre-constructed instance.
     * @param id the identity of the object.
     * @param pc the object to be managed.
     */
    void initialiseForHollowPreConstructed(Object id, Object pc);

    /**
     * Initialises a state manager to manage the passed persistent instance having the given object ID.
     * Used where we have retrieved a PC object from a datastore directly (not field-by-field), for example on
     * an object datastore. This initialiser will not add StateManagers to all related PCs. This must be done by
     * any calling process. This simply adds the StateManager to the specified object and records the id, setting
     * all fields of the object as loaded.
     * @param id the identity of the object.
     * @param pc The object to be managed
     */
    void initialiseForPersistentClean(Object id, Object pc);

    /**
     * Initialises a state manager to manage a PersistenceCapable instance that will be EMBEDDED/SERIALISED 
     * into another PersistenceCapable object. The instance will not be assigned an identity in the process 
     * since it is a SCO.
     * @param pc The PersistenceCapable to manage (see copyPc also)
     * @param copyPc Whether the SM should manage a copy of the passed PC or that one
     */
    void initialiseForEmbedded(Object pc, boolean copyPc);

    /**
     * Initialises a state manager to manage a transient instance that is becoming newly persistent.
     * A new object ID for the instance is obtained from the store manager and the object is inserted
     * in the data store.
     * <p>This constructor is used for assigning state managers to existing
     * instances that are transitioning to a persistent state.
     * @param pc the instance being make persistent.
     * @param preInsertChanges Any changes to make before inserting
     */
    void initialiseForPersistentNew(Object pc, FieldValues preInsertChanges);

    /**
     * Initialises a state manager to manage a Transactional Transient instance.
     * A new object ID for the instance is obtained from the store manager and the object is inserted in the data store.
     * <p>
     * This constructor is used for assigning state managers to Transient
     * instances that are transitioning to a transient clean state.
     * @param pc the instance being make persistent.
     */
    void initialiseForTransactionalTransient(Object pc);

    /**
     * Initialises the StateManager to manage a PersistenceCapable object in detached state.
     * @param pc the detach object.
     * @param id the identity of the object.
     * @param version the detached version
     */
    void initialiseForDetached(Object pc, Object id, Object version);

    /**
     * Initialise to create a StateManager for a PersistenceCapable object, assigning the specified id to the object. 
     * This is used when getting objects out of the L2 Cache, where they have no StateManager assigned, and returning 
     * them as associated with a particular PM.
     * @param cachedPC Cache object for persistable object from L2 cache
     * @param id Id to assign to the PersistenceCapable object
     * @param pcClass Class of the object that this will manage the state for
     */
    void initialiseForCachedPC(CachedPC cachedPC, Object id, Class pcClass);

    /**
     * Initialises the StateManager to manage a PersistenceCapable object that is not persistent but that
     * is about to be deleted. The initial state will be P_NEW, but when the delete call comes in will be
     * P_NEW_DELETED. The object will not be enlisted in the transaction.
     * @param pc the object to delete
     */
    void initialiseForPNewToBeDeleted(Object pc);

    /**
     * Accessor for the object managed by this StateManager.
     * @return The object
     */
    Object getObject();

    /**
     * Accessor for the id of the object managed by this StateManager.
     * @return The identity of the object
     */
    Object getInternalObjectId();

    /**
     * return a copy from the object Id
     * @param obj the persistable object
     * @return the object id
     */
    Object getExternalObjectId(Object obj);

    /**
      * Returns the ObjectManager that owns the StateManager instance
      * @return The ObjectManager
     */
    ObjectManager getObjectManager();

    /**
     * Accessor for the manager for the store.
     * @return Store Manager
     */
    StoreManager getStoreManager();

    /**
     * Accessor for the manager for MetaData.
     * @return MetaData manager
     */
    MetaDataManager getMetaDataManager();

    /**
     * Method to mark the specified (absolute) field number as dirty.
     * @param fieldNumber The (absolute) field number of the field
     */
    void makeDirty(int fieldNumber);

    /**
     * Accessor for the names of all dirty fields.
     * @return Names of all dirty fields
     */
    String[] getDirtyFieldNames();

    /**
     * Accessor for the names of all loaded fields.
     * @return Names of all loaded fields
     */
    String[] getLoadedFieldNames();

    /**
     * Marks the given field dirty for issuing an update after the insert.
     * @param pc The Persistable object
     * @param fieldNumber The no of field to mark as dirty. 
     */
    void updateFieldAfterInsert(Object pc, int fieldNumber);

    /**
     * Update the acitvity state.
     * @param activityState the activity state
     */
    void changeActivityState(ActivityState activityState);

    /**
     * Method to run reachability from this StateManager.
     * @param reachables List of reachable StateManagers so far
     */
    void runReachability(Set reachables);

    /**
     * Method to make the managed object transactional.
     */
    void makeTransactional();

    /**
     * Method to make the managed object nontransactional.
     */
    void makeNontransactional();

    /**
     * Method to make the managed object transient.
     * @param state Object containing the state of any fetch plan processing
     */
    void makeTransient(FetchPlanState state);

    /**
     * Method to make the managed object persistent.
     */
    void makePersistent();
    
    /**
     * Method to make Transactional Transient instances persistent
     */
    void makePersistentTransactionalTransient();

    /**
     * Method to delete the object from persistence.
     */
    void deletePersistent();

    /**
     * Method to attach to this the detached persistable instance
     * @param detachedPC the detached persistable instance to be attached
     * @param embedded Whether it is embedded
     * @return The attached copy
     */
    Object attachCopy(Object detachedPC, boolean embedded);

    /**
     * Method to attach the object managed by this StateManager.
     * @param embedded Whether it is embedded
     */
    void attach(boolean embedded);

    /**
     * Method to make detached copy of this instance
     * @param state State for the detachment process
     * @return the detached PersistenceCapable instance
     */
    Object detachCopy(FetchPlanState state);

    /**
     * Method to detach the PersistenceCapable object.
     * @param state State for the detachment process
     */
    void detach(FetchPlanState state);

    /**
     * Method to return an L2 cacheable object representing the managed object.
     * @return The object suitable for L2 caching
     */
    CachedPC cache();

    /**
     * Validates whether the persistence capable instance exists in the
     * datastore. If the instance does not exist in the datastore, this method
     * will fail raising a NucleusObjectNotFoundException.
     */
    void validate();

    /**
     * Method to change the object state to evicted.
     */
    void evict();

    /**
     * Method to refresh the values of the currently loaded fields in the managed object.
     */
    void refresh();

    /**
     * Method to retrieve the fields for this object.
     * @param fgOnly Whether to retrieve just the current fetch plan fields
     */
    void retrieve(boolean fgOnly);

    /**
     * Method to retrieve the object.
     * @param fetchPlan the fetch plan to load fields
     **/
    void retrieve(FetchPlan fetchPlan);

    /**
     * Convenience interceptor to allow operations to be performed before the begin is performed
     * @param tx The transaction
     */
    void preBegin(Transaction tx);
    
    /**
     * Convenience interceptor to allow operations to be performed after the commit is performed
     * but before returning control to the application.
     * @param tx The transaction
     */
    void postCommit(Transaction tx);

    /**
     * Convenience interceptor to allow operations to be performed before any rollback is
     * performed.
     * @param tx The transaction
     */
    void preRollback(Transaction tx);

    /**
     * Method to flush all changes to the datastore.
     */
    void flush();

    /**
     * Accessor for the relationship manager (if present).
     * @return The Relationship manager
     */
    RelationshipManager getRelationshipManager();

    /**
     * Method to process all managed relations for this object.
     */
    void checkManagedRelations();

    /**
     * Method to process all managed relations for this object.
     */
    void processManagedRelations();

    /**
     * Method to clear any state related to "managed relationships" stored for the object.
     * This removes all original values stored for bidirectional fields that were changed in the
     * previous flush-cycle. This is called when ObjectManager.flush() completes.
     */
    void clearManagedRelations();

    /**
     * Method to return the current value of the specified field.
     * @param fieldNumber (absolute) field number of the field
     * @return The current value
     */
    Object provideField(int fieldNumber);

    /**
     * Method to obtain updated field values from the passed FieldManager.
     * @param fieldNumbers The numbers of the fields
     * @param fm The fieldManager
     */
    void provideFields(int fieldNumbers[], FieldManager fm);

    /**
     * Convenience method to change the value of a field that is assumed loaded.
     * Will mark the object/field as dirty if it isnt previously.
     * Only for use in management of relations.
     * @param fieldNumber Number of field
     * @param newValue The new value
     */
    void replaceFieldValue(int fieldNumber, Object newValue);

    /**
     * Method to change the value of the specified field.
     * @param fieldNumber (absolute) field number of the field
     * @param value The new value.
     * @param makeDirty Whether to make the field dirty when replacing it
     */
    void replaceField(int fieldNumber, Object value, boolean makeDirty);

    /**
     * Method to update the data in the object with the values from the passed FieldManager
     * @param fieldNumbers (absolute) field numbers of the fields to update
     * @param fm The FieldManager
     * @param replaceWhenDirty Whether to replace these fields if the field is dirty
     */
    void replaceFields(int fieldNumbers[], FieldManager fm, boolean replaceWhenDirty);

    /**
     * Method to update the data in the object with the values from the passed FieldManager
     * @param fieldNumbers (absolute) field numbers of the fields to update
     * @param fm The FieldManager
     */
    void replaceFields(int fieldNumbers[], FieldManager fm);

    /**
     * Method to update the data in the object with the values from the passed
     * FieldManager. Only non loaded fields are updated
     * @param fieldNumbers (absolute) field numbers of the fields to update
     * @param fm The FieldManager
     */
    void replaceNonLoadedFields(int fieldNumbers[], FieldManager fm);

    /**
     * Method to replace all loaded SCO fields with wrappers.
     * If the loaded field already uses a SCO wrapper nothing happens to that field.
     */
    void replaceAllLoadedSCOFieldsWithWrappers();

    /**
     * Method to replace all loaded (wrapped) SCO fields with unwrapped values.
     * If the loaded field doesnt use a SCO wrapper nothing happens to that field.
     */
    void replaceAllLoadedSCOFieldsWithValues();

    /**
     * Method to wrap a SCO field (if not wrapped currently) and return the wrapped value.
     * If the field is not a SCO field will just return the value.
     * If "replaceFieldIfChanged" is set, we replace the value in the object with the wrapped value.
     * @param fieldNumber Number of the field
     * @param value The value to give it
     * @param forInsert Whether the creation of any wrapper should insert this value into the datastore
     * @param forUpdate Whether the creation of any wrapper should update the datastore with this value
     * @param replaceFieldIfChanged Whether to replace the field in the object if wrapping the value
     * @return The wrapper (or original value if not wrappable)
     */
    Object wrapSCOField(int fieldNumber, Object value, boolean forInsert, boolean forUpdate, boolean replaceFieldIfChanged);

    /**
     * Method to unwrap a SCO field (if it is wrapped currently) and return the unwrapped value.
     * If the field is not a SCO field will just return the value.
     * If "replaceFieldIfChanged" is set, we replace the value in the object with the unwrapped value.
     * @param fieldNumber The field number
     * @param value The value to unwrap for this field
     * @param replaceFieldIfChanged Whether to replace the field value in the object if unwrapping the value
     * @return The unwrapped field value
     */
    Object unwrapSCOField(int fieldNumber, Object value, boolean replaceFieldIfChanged);

    /**
     * Accessor for the referenced PC object when we are attaching or detaching.
     * When attaching and this is the detached object this returns the newly attached object.
     * When attaching and this is the newly attached object this returns the detached object.
     * When detaching and this is the newly detached object this returns the attached object.
     * When detaching and this is the attached object this returns the newly detached object.
     * @return The referenced object (or null).
     */
    Object getReferencedPC();

    /**
     * Tests whether this object is new yet waiting to be flushed to the datastore.
     * @return true if this instance is waiting to be flushed
     */
    boolean isWaitingToBeFlushedToDatastore();

    /**
     * Method to allow the setting of the id of the PC object. This is used when it is obtained after persisting
     * the object to the datastore. In the case of RDBMS, this may be via auto-increment, or in the case of ODBMS
     * this may be an accessor for the id after storing.
     * @param id the id received from the datastore. May be an OID, or the key value for an OID, or an application id.
     */
    void setPostStoreNewObjectId(Object id);

    /**
     * Sets the value for the version column in the datastore. Update the transactional version too
     * @param version The version
     */
    void setVersion(Object version);

    /**
     * Return the object representing the transactional version of the calling instance.
     * @param pc the calling persistable instance
     * @return the object representing the version of the calling instance
     */    
    Object getTransactionalVersion(Object pc);  
    
    /**
     * Sets the value for the version column in a transaction not yet committed
     * @param optimisticTransactionalVersion
     */
    void setTransactionalVersion(Object optimisticTransactionalVersion);

    /**
     * Accessor for the highest field number
     * @return Highest field number
     */
    int getHighestFieldNumber();

    /**
     * Accessor for the ClassMetaData for this object.
     * @return The ClassMetaData.
     **/
    AbstractClassMetaData getClassMetaData();
    
    /**
     * Nullify fields with reference to PersistenceCapable or SCO instances 
     */
    void nullifyFields();

    /**
     * Convenience method to load the specified field if not loaded.
     * @param fieldNumber Absolute field number
     */
    void loadField(int fieldNumber);

    /**
     * Fetchs from the database all fields that are not currently loaded and that are in the current
     * fetch group. Called by lifecycle transitions.
     */
    void loadUnloadedFieldsInFetchPlan();

    /**
     * Method to load all unloaded fields in the FetchPlan.
     * Recurses through the FetchPlan objects and loads fields of sub-objects where needed.
     * @param state The FetchPlan state
     */
    void loadFieldsInFetchPlan(FetchPlanState state);

    /**
     * Loads all unloaded fields of the managed class that are in the current FetchPlan.
     * Called by life-cycle transitions.
     * @param fetchPlan The FetchPlan
     * @since 1.1
     */
    void loadUnloadedFieldsOfClassInFetchPlan(FetchPlan fetchPlan);

    /**
     * Fetch from the database all fields that are not currently loaded regardless of whether
     * they are in the current fetch group or not. Called by lifecycle transitions.
     */
    void loadUnloadedFields();

    /**
     * Method that will unload all fields that are not in the FetchPlan.
     */
    void unloadNonFetchPlanFields();

    /**
     * Convenience method to reset the detached state in the current object.
     */
    void resetDetachState();

    /**
     * Disconnect the StateManager from the PersistenceManager and PC object.
     */
    void disconnect();
    
    //called by lifecycle ops
    void evictFromTransaction();

    void enlistInTransaction();
    
    /**
     * Refreshes from the database all fields currently loaded.
     * Called by life-cycle transitions.
     */
    void refreshLoadedFields();

    /**
     * Method to clear all saved fields on the object.
     **/
    void clearSavedFields();
    
    /**
     * Refreshes from the database all fields in fetch plan.
     * Called by life-cycle transitions.
     */
    void refreshFieldsInFetchPlan();
    
    /**
     * Method to clear all fields that are not part of the primary key of the object.
     **/
    void clearNonPrimaryKeyFields();
    
    /**
     * Method to restore all fields of the object.
     **/
    void restoreFields();

    /**
     * Method to save all fields of the object.
     **/
    void saveFields();
    
    /**
     * Method to clear all fields of the object.
     **/
    void clearFields();
    
    /**
     * Registers the pc class in the cache
     */
    void registerTransactional();

    /**
     * Accessor for the Restore Values flag 
     * @return Whether to restore values
     */
    boolean isRestoreValues();
    
    /**
     * Method to clear all loaded flags on the object.
     **/
    void clearLoadedFlags();

    //used by SCO classes
    /**
     * Method to register an owner StateManager with this embedded/serialised object.
     * @param ownerSM The owning State Manager.
     * @param ownerFieldNumber The field number in the owner that the embedded/serialised object is stored as
     */
    void addEmbeddedOwner(StateManager ownerSM, int ownerFieldNumber);
    
    //used by PM classes
    /**
     * Convenience method to load the passed field values.
     * Loads the fields using any required fetch plan and calls jdoPostLoad() as appropriate.
     * @param fv Field Values to load (including any fetch plan to use when loading)
     */
    void loadFieldValues(FieldValues fv);

    /**
     * Look to the database to determine which
     * class this object is. This parameter is a hint. Set false, if it's
     * already determined the correct pcClass for this pc "object" in a certain
     * level in the hierarchy. Set to true and it will look to the database.
     * @param fv the initial field values of the object.
     */
    void checkInheritance(FieldValues fv);

    /**
     * Convenience method to retrieve the detach state from the passed State Manager's object
     * @param sm The State Manager
     */
    void retrieveDetachState(org.datanucleus.StateManager sm);

    //used by embedded mappings
    /**
     * Method to set this StateManager as managing an embedded/serialised object.
     * @param embeddedType The type of object being managed
     */
    void setPcObjectType(int embeddedType);

    /**
     * Accessor for the PC object type.
     * @return PC Object type (PC, embedded PC, etc)
     */
    int getPcObjectType();

    /**
     * Accessor for the overall owner StateManagers of the managed object when embedded.
     * @return Owning StateManagers when embedded (if any)
     */
    StateManager[] getEmbeddedOwners();

    //used by container mappings
    /**
     * Method to set the storing PC flag.
     */
    void setStoringPC();

    /**
     * Method to unset the storing PC flag.
     */
    void unsetStoringPC();
 
    //used by scostore
    /**
     * Convenience accessor for whether this StateManager manages an embedded/serialised object.
     * @return Whether the managed object is embedded/serialised.
     */
    boolean isEmbedded();

    /**
     * Method to set the value for an external field stored against this object.
     * This is for a situation such as in ORM where this object can have an "external" foreign-key
     * provided by an owning object (e.g 1-N uni relation and this is the element with no knowledge
     * of the owner, so the associated value is the FK value).
     * @param key The key for this associated information
     * @param value The value to store
     */
    void setAssociatedValue(Object key, Object value);

    /**
     * Accessor for the value of an external field.
     * This is for a situation such as in ORM where this object can have an "external" foreign-key
     * provided by an owning object (e.g 1-N uni relation and this is the element with no knowledge
     * of the owner, so the associated value is the FK value).
     * @param key The key for this associated information
     * @return The value stored (if any) against this key
     */
    Object getAssociatedValue(Object key);

    /**
     * Convenience method to load a field from the datastore.
     * Used in attaching fields and checking their old values (so we don't
     * want any postLoad method being called).
     * TODO Merge this with one of the loadXXXFields methods.
     * @param fieldNumber The field number.
     */
    void loadFieldFromDatastore(int fieldNumber);

    /**
     * Tests whether this object is being inserted.
     * @return true if this instance is inserting.
     */
    boolean isInserting();

    /**
     * Tests whether this object is in the process of being deleted.
     * @return true if this instance is being deleted.
     */
    boolean isDeleting();

    /**
     * Tests whether this object is in the process of being detached.
     * @return true if this instance is being detached.
     */
    boolean isDetaching();

    /**
     * Convenience method to return if we are in the phase of performing postInsert updates
     * due to related objects having been inserted and so allowing the field to be inserted.
     * @return Whether we are updating for postInsert
     */
    boolean isUpdatingFieldForPostInsert();

    /**
     * Locate this object in the datastore.
     * @throws NucleusObjectNotFoundException if the object doesnt exist.
     */
    void locate();

    /**
     * Returns whether all fields are loaded.
     * @return Returns true if all fields are loaded.
     */
    public boolean getAllFieldsLoaded();

    /**
     * Creates a copy of the internal dirtyFields array.
     * TODO Change this to return an array of the field numbers that are dirty
     * @return a copy of the internal dirtyFields array.
     */
    public boolean[] getDirtyFields();

    /**
     * Accessor for the field numbers of all dirty fields in this managed instance.
     * @return Field numbers of all (currently) dirty fields
     */
    public int[] getDirtyFieldNumbers();

    /**
     * Accessor for the field numbers of all loaded fields in this managed instance.
     * @return Field numbers of all (currently) loaded fields
     */
    public int[] getLoadedFieldNumbers();

    /**
     * Accessor for the fields
     * @return boolean array of loaded state in order of absolute field numbers
     */
    public boolean[] getLoadedFields();
    
    /**
     * Mark the specified field as not loaded so that it will be reloaded on next access.
     * @param fieldName Name of the field
     */
    void unloadField(String fieldName);

    /**
     * Accessor for whether a field is currently loaded.
     * Just returns the status, unlike "isLoaded" which also loads it if not.
     * @param fieldNumber The (absolute) field number
     * @return Whether it is loaded
     */
    boolean isFieldLoaded(int fieldNumber);

    /**
     * Convenience method to update our object with the field values from the passed object.
     * Objects need to be of the same type, and the other object should not have a StateManager.
     * @param pc The object that we should copy fields from
     */
    void copyFieldsFromObject(PersistenceCapable pc, int[] fieldNumbers);

    /**
     * Diagnostic method to dump the current state to the provided PrintWriter.
     * @param out The PrintWriter
     */
    void dump(PrintWriter out);

    // ---------------------------------- JDO-specific ------------------------------------
    // TODO Move these to JDOStateManagerImpl or similar
    /**
     * Accessor for whether the managed object is deleted.
     * @param pc The PC object
     * @return Whether it is deleted
     */
    boolean isDeleted(PersistenceCapable pc);

    /**
     * Accessor for whether the managed object is new.
     * @param pc The PC object
     * @return Whether it is new
     */
    boolean isNew(PersistenceCapable pc);

    /**
     * Accessor for the object id.
     * @param pc The PC object
     * @return The (external) id
     */
    Object getObjectId(PersistenceCapable pc);

    /**
     * Method to swap the managed object for the supplied object.
     * This is of particular use for object datastores where the datastore is responsible for creating
     * the in-memory object and where we have a temporary object that we want to swap for the datastore
     * generated object. Makes no change to what fields are loaded/dirty etc, just swaps the managed object.
     * @param pc The PersistenceCapable to use
     */
    void replaceManagedPC(PersistenceCapable pc);
    
    Object getVersion(PersistenceCapable pc);
    
    boolean isLoaded(PersistenceCapable pc, int fieldNumber);
    
    void setObjectField(PersistenceCapable pc, int fieldNumber, Object oldValue, Object newValue);
}