/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2003
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.swing;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.MouseListener;
import java.util.Enumeration;
import javax.swing.BorderFactory;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.border.Border;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

/**
 * A nicer looking swing table. It uses vertical grid lines,
 * but unlike the default swing table implementation, rows
 * which are still empty, get their grid line painted just the
 * same. Additionally, a nicer border is applied and the table
 * headers have a slightly thicker border as well.
 * 
 * @author Chr. Clemens Lee, Kenneth Westelinck
 * @version $Id: CCLTable.java,v 1.1 2003/05/01 16:44:13 clemens Exp clemens $
 */
public class CCLTable extends TableScrollPane
{
    private JTable _table = null;
    private int _resizeMode = -1;
    
    /*
     * The ScrollableJTable shows the same resizebehaviour of a normal table.
     * AUTO_SIZE_LAST_COLUMN_TO_FIT is an additional parameter, specific for
     * this component.
     */
    public static final int AUTO_SIZE_LAST_COLUMN_TO_FIT = 1;
    public static final int AUTO_RESIZE_ALL_COLUMNS =
        JTable.AUTO_RESIZE_ALL_COLUMNS + AUTO_SIZE_LAST_COLUMN_TO_FIT;
    public static final int AUTO_RESIZE_LAST_COLUMN =
        JTable.AUTO_RESIZE_LAST_COLUMN + AUTO_SIZE_LAST_COLUMN_TO_FIT;
    public static final int AUTO_RESIZE_NEXT_COLUMN =
        JTable.AUTO_RESIZE_NEXT_COLUMN + AUTO_SIZE_LAST_COLUMN_TO_FIT;
    public static final int AUTO_RESIZE_OFF =
        JTable.AUTO_RESIZE_OFF + AUTO_SIZE_LAST_COLUMN_TO_FIT;
    public static final int AUTO_RESIZE_SUBSEQUENT_COLUMNS =
        JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS + AUTO_SIZE_LAST_COLUMN_TO_FIT;

    /**
     * Create a ScrollableJtable based on the model.
     *
     * @param   font   the font used for the table header.
     */
    public CCLTable(TableModel aModel, Font font ) 
    {
    	super( null );
        
        _table = new JTable(aModel);
        _table.setShowGrid(false);

        SwingUtil.setCCLButtonBorderOnTableHeader
               ( _table, font );

        setViewportView( _table );
		setVerticalScrollBarPolicy  ( JScrollPane.VERTICAL_SCROLLBAR_ALWAYS  );
	    setHorizontalScrollBarPolicy( JScrollPane.HORIZONTAL_SCROLLBAR_NEVER );

        // both are necessary to let our own grid shine through
		getViewport().setOpaque(false);
        _table.setOpaque(false);

        int preferredWidth = _table.getColumnModel().getTotalColumnWidth();
        _table.setPreferredScrollableViewportSize(
                                                 new Dimension(preferredWidth, 400));

        //*
        setBorder( BorderFactory.createCompoundBorder( SwingUtil.createCCLBorder()
                                                                , new CCLLineBorder( getBackground().darker(), new Insets( 0, 0, 0, 1 ) ) ) );
        // */
    }

    /**
     * Return the JTable component that is behind this ScrollableJTable.
     * @return the table
     */
    public JTable getTable() {
        return _table;
    }

    /**
     * Set the resizemode. Comparable with the modes provided with JTable.
     * @param aResizeMode the resize mode
     */
    public void setAutoResizeMode(int aResizeMode) {
        _resizeMode = aResizeMode;
        if (aResizeMode != AUTO_SIZE_LAST_COLUMN_TO_FIT) {
            _table.setAutoResizeMode(aResizeMode - AUTO_SIZE_LAST_COLUMN_TO_FIT);
        }
    }

    /**
     * @see javax.swing.JTable#getCellRect
     */
    public Rectangle getCellRect( int row, int column, boolean includeSpacing )
    {
        return _table.getCellRect( row, column, includeSpacing );
    }

    /**
     * Gets forwarded to contained table.
     */
    public void addMouseListener( MouseListener listener )
    {
        _table.addMouseListener( listener );
    }

    /**
     * Gets forwarded to contained table.
     */
    public ListSelectionModel getSelectionModel()
    {
        return _table.getSelectionModel();
    }

    /**
     * Gets forwarded to contained table.
     */
    public void setSelectionMode( int selectionMode )
    {
        _table.setSelectionMode( selectionMode );
    }

    /**
     * Gets forwarded to contained table.
     */
    public void scrollRectToVisible( Rectangle rectangle )
    {
        _table.scrollRectToVisible( rectangle );
    }
    
	/**
	 * Returns resize mode used for this table.
	 */
	protected int getResizeMode() {
		return _resizeMode;
	}

}

/**
 * This class implements the vertical grid lines painting.
 */    
class TableScrollPane extends JScrollPane 
{
    private JTable _table = null;

    /**
     * Construct the scrollpane using a table.
     */
    public TableScrollPane(JTable aTable) {
        super(aTable);
        _table = aTable;
    }

    /**
     * This overrides the component's paint method. It draws vertical lines
     * to seperate the table's columns.
     * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
     */
    public void paintComponent(Graphics g) 
    {
        paintComponent(g, _table, this);
    }

    /**
     * This overrides the component's paint method. It draws vertical lines
     * to seperate the table's columns.    
     */
    static public void paintComponent(Graphics g, JTable table, JScrollPane scrollpane) 
    {
        Rectangle d = scrollpane.getViewport().getVisibleRect();
        //Rectangle d = scrollpane.getViewport().getViewRect();

        d.width += scrollpane.getBorder().getBorderInsets( scrollpane ).left;
        d.height += 100;
        g.setColor(Color.white);
        g.fillRect(d.x, d.y, d.width, d.height);
        g.setColor( scrollpane.getBackground() );
        g.fillRect(d.x + d.width, d.y, 100, 100 );
        g.setColor(table.getGridColor());
		
        TableColumnModel model = table.getColumnModel();
        int offset = scrollpane.getBorder().getBorderInsets( scrollpane ).left - 1;
        Enumeration eColumns = model.getColumns();
        while (eColumns.hasMoreElements()) 
        {
            TableColumn column = (TableColumn) eColumns.nextElement();
            if (!eColumns.hasMoreElements()) {
                // skip last one
                break;
            }
            offset += column.getWidth();
            g.drawLine(offset, 0, offset, d.height);
        }
        
        // do not do super, paint border ourselves
        Border border = scrollpane.getViewportBorder();
        if (border != null) 
        {
            Rectangle r = scrollpane.getViewportBorderBounds();
            border.paintBorder(scrollpane, g, r.x, r.y, r.width, r.height);
        }
    }
    
    /**
     * @see javax.swing.JScrollPane#setViewportView(java.awt.Component)
     */
    public void setViewportView(Component view) {
        if ( view instanceof JTable )
        {
            _table = (JTable) view;
        }
        super.setViewportView(view);
    }
}
