/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.settings;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.CharBuffer;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.AccessDeniedException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Base64;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import javax.security.auth.DestroyFailedException;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.store.BufferedChecksumIndexInput;
import org.apache.lucene.store.ChecksumIndexInput;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.SimpleFSDirectory;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.cli.UserException;
import org.elasticsearch.common.Randomness;
import org.elasticsearch.common.settings.SecureSetting;
import org.elasticsearch.common.settings.SecureSettings;
import org.elasticsearch.common.settings.SecureString;
import org.elasticsearch.common.settings.Setting;

public class KeyStoreWrapper
implements SecureSettings {
    public static final Setting<SecureString> SEED_SETTING = SecureSetting.secureString("keystore.seed", null, new Setting.Property[0]);
    private static final char[] SEED_CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789~!@#$%^&*-_=+?".toCharArray();
    private static final String KEYSTORE_FILENAME = "elasticsearch.keystore";
    private static final int FORMAT_VERSION = 2;
    private static final int MIN_FORMAT_VERSION = 1;
    private static final String NEW_KEYSTORE_TYPE = "PKCS12";
    private static final String NEW_KEYSTORE_STRING_KEY_ALGO = "PBE";
    private static final String NEW_KEYSTORE_FILE_KEY_ALGO = "PBE";
    private static final CharsetEncoder ASCII_ENCODER = StandardCharsets.US_ASCII.newEncoder();
    private final int formatVersion;
    private final boolean hasPassword;
    private final String type;
    private final SecretKeyFactory stringFactory;
    private final SecretKeyFactory fileFactory;
    private final Map<String, KeyType> settingTypes;
    private final byte[] keystoreBytes;
    private final SetOnce<KeyStore> keystore = new SetOnce();
    private final SetOnce<KeyStore.PasswordProtection> keystorePassword = new SetOnce();

    private KeyStoreWrapper(int formatVersion, boolean hasPassword, String type, String stringKeyAlgo, String fileKeyAlgo, Map<String, KeyType> settingTypes, byte[] keystoreBytes) {
        this.formatVersion = formatVersion;
        this.hasPassword = hasPassword;
        this.type = type;
        try {
            this.stringFactory = SecretKeyFactory.getInstance(stringKeyAlgo);
            this.fileFactory = SecretKeyFactory.getInstance(fileKeyAlgo);
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
        this.settingTypes = settingTypes;
        this.keystoreBytes = keystoreBytes;
    }

    public static Path keystorePath(Path configDir) {
        return configDir.resolve(KEYSTORE_FILENAME);
    }

    public static KeyStoreWrapper create(char[] password) throws Exception {
        KeyStoreWrapper wrapper = new KeyStoreWrapper(2, password.length != 0, NEW_KEYSTORE_TYPE, "PBE", "PBE", new HashMap<String, KeyType>(), null);
        KeyStore keyStore = KeyStore.getInstance(NEW_KEYSTORE_TYPE);
        keyStore.load(null, null);
        wrapper.keystore.set((Object)keyStore);
        wrapper.keystorePassword.set((Object)new KeyStore.PasswordProtection(password));
        KeyStoreWrapper.addBootstrapSeed(wrapper);
        return wrapper;
    }

    public static void addBootstrapSeed(KeyStoreWrapper wrapper) throws GeneralSecurityException {
        assert (!wrapper.getSettingNames().contains(SEED_SETTING.getKey()));
        SecureRandom random = Randomness.createSecure();
        int passwordLength = 20;
        char[] characters = new char[passwordLength];
        for (int i = 0; i < passwordLength; ++i) {
            characters[i] = SEED_CHARS[random.nextInt(SEED_CHARS.length)];
        }
        wrapper.setString(SEED_SETTING.getKey(), characters);
        Arrays.fill(characters, '\u0000');
    }

    public static KeyStoreWrapper load(Path configDir) throws IOException {
        Path keystoreFile = KeyStoreWrapper.keystorePath(configDir);
        if (!Files.exists(keystoreFile, new LinkOption[0])) {
            return null;
        }
        SimpleFSDirectory directory = new SimpleFSDirectory(configDir);
        try (IndexInput indexInput = directory.openInput(KEYSTORE_FILENAME, IOContext.READONCE);){
            boolean hasPassword;
            BufferedChecksumIndexInput input = new BufferedChecksumIndexInput(indexInput);
            int formatVersion = CodecUtil.checkHeader((DataInput)input, (String)KEYSTORE_FILENAME, (int)1, (int)2);
            byte hasPasswordByte = input.readByte();
            boolean bl = hasPassword = hasPasswordByte == 1;
            if (!hasPassword && hasPasswordByte != 0) {
                throw new IllegalStateException("hasPassword boolean is corrupt: " + String.format(Locale.ROOT, "%02x", hasPasswordByte));
            }
            String type = input.readString();
            String stringKeyAlgo = input.readString();
            String fileKeyAlgo = formatVersion >= 2 ? input.readString() : "PBE";
            Map<Object, Object> settingTypes = formatVersion >= 2 ? input.readMapOfStrings().entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> KeyType.valueOf((String)e.getValue()))) : new HashMap();
            byte[] keystoreBytes = new byte[input.readInt()];
            input.readBytes(keystoreBytes, 0, keystoreBytes.length);
            CodecUtil.checkFooter((ChecksumIndexInput)input);
            KeyStoreWrapper keyStoreWrapper = new KeyStoreWrapper(formatVersion, hasPassword, type, stringKeyAlgo, fileKeyAlgo, settingTypes, keystoreBytes);
            return keyStoreWrapper;
        }
    }

    public static void upgrade(KeyStoreWrapper wrapper, Path configDir) throws Exception {
        if (wrapper.getSettingNames().contains(SEED_SETTING.getKey())) {
            return;
        }
        KeyStoreWrapper.addBootstrapSeed(wrapper);
        wrapper.save(configDir);
    }

    @Override
    public boolean isLoaded() {
        return this.keystore.get() != null;
    }

    public boolean hasPassword() {
        return this.hasPassword;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void decrypt(char[] password) throws GeneralSecurityException, IOException {
        if (this.keystore.get() != null) {
            throw new IllegalStateException("Keystore has already been decrypted");
        }
        this.keystore.set((Object)KeyStore.getInstance(this.type));
        try (ByteArrayInputStream in = new ByteArrayInputStream(this.keystoreBytes);){
            ((KeyStore)this.keystore.get()).load(in, password);
        }
        finally {
            Arrays.fill(this.keystoreBytes, (byte)0);
        }
        this.keystorePassword.set((Object)new KeyStore.PasswordProtection(password));
        Arrays.fill(password, '\u0000');
        Enumeration<String> aliases = ((KeyStore)this.keystore.get()).aliases();
        if (this.formatVersion == 1) {
            while (aliases.hasMoreElements()) {
                this.settingTypes.put(aliases.nextElement(), KeyType.STRING);
            }
        } else {
            HashSet<String> expectedSettings = new HashSet<String>(this.settingTypes.keySet());
            while (aliases.hasMoreElements()) {
                String settingName = aliases.nextElement();
                if (expectedSettings.remove(settingName)) continue;
                throw new SecurityException("Keystore has been corrupted or tampered with");
            }
            if (!expectedSettings.isEmpty()) {
                throw new SecurityException("Keystore has been corrupted or tampered with");
            }
        }
    }

    public void save(Path configDir) throws Exception {
        assert (this.isLoaded());
        char[] password = ((KeyStore.PasswordProtection)this.keystorePassword.get()).getPassword();
        SimpleFSDirectory directory = new SimpleFSDirectory(configDir);
        String tmpFile = "elasticsearch.keystore.tmp";
        try (IndexOutput output = directory.createOutput(tmpFile, IOContext.DEFAULT);){
            CodecUtil.writeHeader((DataOutput)output, (String)KEYSTORE_FILENAME, (int)2);
            output.writeByte(password.length == 0 ? (byte)0 : 1);
            output.writeString(NEW_KEYSTORE_TYPE);
            output.writeString("PBE");
            output.writeString("PBE");
            output.writeMapOfStrings(this.settingTypes.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> ((KeyType)((Object)((Object)e.getValue()))).name())));
            assert (this.type.equals(NEW_KEYSTORE_TYPE)) : "keystore type changed";
            assert (this.stringFactory.getAlgorithm().equals("PBE")) : "string pbe algo changed";
            assert (this.fileFactory.getAlgorithm().equals("PBE")) : "file pbe algo changed";
            ByteArrayOutputStream keystoreBytesStream = new ByteArrayOutputStream();
            ((KeyStore)this.keystore.get()).store(keystoreBytesStream, password);
            byte[] keystoreBytes = keystoreBytesStream.toByteArray();
            output.writeInt(keystoreBytes.length);
            output.writeBytes(keystoreBytes, keystoreBytes.length);
            CodecUtil.writeFooter((IndexOutput)output);
        }
        catch (AccessDeniedException e2) {
            String message = String.format(Locale.ROOT, "unable to create temporary keystore at [%s], please check filesystem permissions", configDir.resolve(tmpFile));
            throw new UserException(78, message, (Throwable)e2);
        }
        Path keystoreFile = KeyStoreWrapper.keystorePath(configDir);
        Files.move(configDir.resolve(tmpFile), keystoreFile, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        PosixFileAttributeView attrs = Files.getFileAttributeView(keystoreFile, PosixFileAttributeView.class, new LinkOption[0]);
        if (attrs != null) {
            attrs.setPermissions(PosixFilePermissions.fromString("rw-rw----"));
        }
    }

    @Override
    public Set<String> getSettingNames() {
        return this.settingTypes.keySet();
    }

    @Override
    public SecureString getString(String setting) throws GeneralSecurityException {
        assert (this.isLoaded());
        KeyStore.Entry entry = ((KeyStore)this.keystore.get()).getEntry(setting, (KeyStore.ProtectionParameter)this.keystorePassword.get());
        if (this.settingTypes.get(setting) != KeyType.STRING || !(entry instanceof KeyStore.SecretKeyEntry)) {
            throw new IllegalStateException("Secret setting " + setting + " is not a string");
        }
        KeyStore.SecretKeyEntry secretKeyEntry = (KeyStore.SecretKeyEntry)entry;
        PBEKeySpec keySpec = (PBEKeySpec)this.stringFactory.getKeySpec(secretKeyEntry.getSecretKey(), PBEKeySpec.class);
        SecureString value = new SecureString(keySpec.getPassword());
        keySpec.clearPassword();
        return value;
    }

    @Override
    public InputStream getFile(String setting) throws GeneralSecurityException {
        assert (this.isLoaded());
        KeyStore.Entry entry = ((KeyStore)this.keystore.get()).getEntry(setting, (KeyStore.ProtectionParameter)this.keystorePassword.get());
        if (this.settingTypes.get(setting) != KeyType.FILE || !(entry instanceof KeyStore.SecretKeyEntry)) {
            throw new IllegalStateException("Secret setting " + setting + " is not a file");
        }
        KeyStore.SecretKeyEntry secretKeyEntry = (KeyStore.SecretKeyEntry)entry;
        PBEKeySpec keySpec = (PBEKeySpec)this.fileFactory.getKeySpec(secretKeyEntry.getSecretKey(), PBEKeySpec.class);
        char[] chars = keySpec.getPassword();
        final byte[] bytes = new byte[chars.length];
        for (int i = 0; i < bytes.length; ++i) {
            bytes[i] = (byte)chars[i];
        }
        keySpec.clearPassword();
        ByteArrayInputStream bytesStream = new ByteArrayInputStream(bytes){

            @Override
            public void close() throws IOException {
                super.close();
                Arrays.fill(bytes, (byte)0);
            }
        };
        return Base64.getDecoder().wrap(bytesStream);
    }

    void setString(String setting, char[] value) throws GeneralSecurityException {
        assert (this.isLoaded());
        if (!ASCII_ENCODER.canEncode((CharSequence)CharBuffer.wrap(value))) {
            throw new IllegalArgumentException("Value must be ascii");
        }
        SecretKey secretKey = this.stringFactory.generateSecret(new PBEKeySpec(value));
        ((KeyStore)this.keystore.get()).setEntry(setting, new KeyStore.SecretKeyEntry(secretKey), (KeyStore.ProtectionParameter)this.keystorePassword.get());
        this.settingTypes.put(setting, KeyType.STRING);
    }

    void setFile(String setting, byte[] bytes) throws GeneralSecurityException {
        assert (this.isLoaded());
        bytes = Base64.getEncoder().encode(bytes);
        char[] chars = new char[bytes.length];
        for (int i = 0; i < chars.length; ++i) {
            chars[i] = (char)bytes[i];
        }
        SecretKey secretKey = this.stringFactory.generateSecret(new PBEKeySpec(chars));
        ((KeyStore)this.keystore.get()).setEntry(setting, new KeyStore.SecretKeyEntry(secretKey), (KeyStore.ProtectionParameter)this.keystorePassword.get());
        this.settingTypes.put(setting, KeyType.FILE);
    }

    void remove(String setting) throws KeyStoreException {
        assert (this.isLoaded());
        ((KeyStore)this.keystore.get()).deleteEntry(setting);
        this.settingTypes.remove(setting);
    }

    @Override
    public void close() throws IOException {
        try {
            if (this.keystorePassword.get() != null) {
                ((KeyStore.PasswordProtection)this.keystorePassword.get()).destroy();
            }
        }
        catch (DestroyFailedException e) {
            throw new IOException(e);
        }
    }

    private static enum KeyType {
        STRING,
        FILE;

    }
}

