#light

#if CLI_AT_MOST_1_1
#else
namespace Microsoft.FSharp.Control

    #nowarn "57"

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Control
    open Microsoft.FSharp.Collections

    [<Experimental("This type is part of an experimental feature called 'asynchronous workflows'")>]
    /// A computation, which, when run, will eventually produce a value of the given type, or else
    /// raise an exception. The value and/or exception is not returned to the caller immediately, but
    /// is rather passed to a continuation or exception continuation.
    ///
    /// Async computations are normally specified using the F# 'workflow' syntax for building computations.
    ///
    /// Operationally, async computations typically run partly in the .NET Thread Pool via ThreadPool.QueueUserWorkItem, 
    /// and, when waiting for asynchronous I/O, they are suspended as thunks using ThreadPool.RegisterWaitForSingleObject,
    /// waiting for the I/O completion. 
    ///
    /// Some primitive asynchronous computations necessarily end up executing blocking operations:
    /// these should be run on a pool of threads specifically dedicated to resolving blocking conditions,
    /// via UnblockedPrimitive. For example, FileOpen on Windows is, by design, a blocking operation.
    /// However frequently it is important to code as if this is asynchronous. This can be done by running the
    /// blocking operation via UnblockedPrimitive.
    ///
    /// When run, async computations belong to an AsyncGroup. This can usually be specified when the async computation
    /// is started. The only action on an AsyncGroup is to raise a cancellation condition for the AsyncGroup.
    /// Async values check the cancellation condition for their AsyncGroup regularly, though synchronous computations
    /// within an asynchronous computation will not automatically check this condition. This gives a user-level
    /// cooperative cancellation protocol.
    type Async<'res> with


        //---------- Sequencing, exceptions and results. See Async.AsyncBuilder.


        //---------- Running async values
        
        /// Run the asynchronous computation and await its result.
        ///
        /// If an exception occurs in the asynchronous computation then an exception is re-raised by this
        /// function.
        static member Run : computation:Async<'a> * ?asyncGroup : AsyncGroup * ?timeout : int * ?exitContext : bool -> 'a

        /// Start the asynchronous computation in the thread pool. When the result is available 
        /// execute the given postBack in the synchronization context of the thread that originally called Run.
        /// This will frequently be the GUI thread, and in that case the postBack will be executed by sending a 'BeginInvoke' to the
        /// GUI message loop.  If the System.Threading.SynchronizationContext.Current of the calling thread is 'null'
        /// then a FailureException is raised.
        ///
        /// If an exception occurs in the asynchronous computation then it is posted as a 
        /// function that raises an exception in the synchronization context of the thread that originally called RunThenPostBack. 
        static member SpawnThenPostBack : computation:Async<'a> * postBack:('a -> unit) * ?asyncGroup : AsyncGroup -> unit

        /// Start the asynchronous computation in the thread pool. Do not await its result.
        static member Spawn : computation:Async<unit> * ?asyncGroup : AsyncGroup -> unit

        /// Start the asynchronous computation in the .NET thread pool, initially as a CPU-intensive worker item. 
        /// Return a handle to the computation as an IAsyncFuture.
        static member AsyncFuture : computation:Async<'a> * ?asyncGroup : AsyncGroup -> IAsyncFuture<'a>
        

        //---------- Exceptions

        // TODO: deprecate this
        //[<System.Obsolete("Consider using 'async { try Choice1_2(p) with exn -> Choice2_2(exn) }' instead")>]
        static member Catch : Async<'a> -> Async<Choice<'a,exn>>


        //---------- Cancellation checking and performing actions on cancellation.

        /// "CancelCheck" generates an asynchronous computation that checks if the cancellation condition for the AsyncGroup to which this
        /// Async computation belongs has been set. If so the operation raises a System.OperationCanceledException.
        static member CancelCheck : unit -> Async<unit>

        /// "async { use! holder = Async.OnCancel f ... }" generates an asynchronous computation where, 
        /// if a cancellation happens any time during the execution of the asynchronous computation in the scope of 'holder',
        /// then action 'f' is executed on the thread that is performing the cancellation. You can use
        /// this to arrange for your own computation to be asynchronously notified that a cancellation has occurred, e.g.
        /// by setting a flag, or deregistering a pending I/O action.
        static member OnCancel : f: (string -> unit) -> Async<System.IDisposable>

        /// Raise the cancellation condition for the most recent set of Async computations started without any specific AsyncGroup.
        /// Replace the global group with a new global group for any async computations created after this point without 
        /// any specific AsyncGroup.
        static member CancelDefaultGroup :  ?message:string -> unit 

        /// "WhenCancelled p f" runs "p". If "p" is effectively cancelled before its termination then
        /// the process "f exn" is executed. 
        static member WhenCancelled : Async<'a> * (System.OperationCanceledException -> unit) -> Async<'a>

        //---------- Parallelism

        /// Start the asynchronous computation in the thread pool, and add it to the AsyncGroup of 
        /// of the asynchronous computation. Do not await its result. Exceptions in the asynchronous 
        /// computation are currently ignored.
        static member SpawnChild : Async<unit> -> Async<unit>

        /// Generate asynchronous computations for indexes 0..N-1. Do so in the indicated number of parallel groups, which 
        /// defaults to the physical processor count on the machine. If any raise an exception attempt to cancel the others.
        static member Generate : N:int * generator: (int -> Async<'b>) * ?numGroups: int -> Async<'b array>
        
        /// When the joint task is run, it will execute all the asynchronous computations, queueing each in the thread pool immediately.
        /// If any raise an exception then the overall computation will raise an exception, and attempt to cancel the others.
        /// All the sub-computations belong to an AsyncGroup that is a subsidiary of the AsyncGroup of the outer computations.
        static member Parallel : #seq<Async<'b>> -> Async<'b array>

        /// When the joint task is run, it will execute it will execute the two asynchronous computations, starting each in the thread pool.
        /// If any raise an exception then the overall computation will raise an exception, and attempt to cancel the others.
        /// All the sub-computations belong to an AsyncGroup that is a subsidiary of the AsyncGroup of the outer computations.
        static member Parallel2 : Async<'b> * Async<'c> -> Async<'b * 'c>

        /// When the joint task is run, it will execute it will execute the two asynchronous computations, starting each in the thread pool.
        /// If any raise an exception then the overall computation will raise an exception, and attempt to cancel the others.
        /// All the sub-computations belong to an AsyncGroup that is a subsidiary of the AsyncGroup of the outer computations.
        static member Parallel3 : Async<'b> * Async<'c> * Async<'d> -> Async<'b * 'c * 'd>

        //---------- Thread Control
        
        /// 'SwitchToNewThread' creates an asynchronous computation that, when run, creates a brand new thread and runs
        /// its continutation in that thread
        static member SwitchToNewThread : unit -> Async<unit> 
        
        /// 'SwitchToThreadPool p' creates an asynchronous computation that, when run, queues a CPU-intensive work item that runs
        /// its continutation.
        static member SwitchToThreadPool :  unit -> Async<unit> 

                    
        //---------- Building new primitive Async values

        /// Build a new Async task by specifying a callback that gets called when
        /// the task is run. The callback must eventually call either the continuation
        /// or the exception continuation.
        [<OverloadID("Primitive")>]
        static member Primitive : (('a -> unit) * (exn -> unit) -> unit) -> Async<'a>

        /// Build a new Async task by specifying a callback that gets called when
        /// the task is run. The callback must eventually call either the continuation,
        /// the exception continuation or the cancel exception.
        [<OverloadID("Primitive_with_ccont")>]
        static member Primitive : (('a -> unit) * (exn -> unit) * (System.OperationCanceledException -> unit) -> unit) -> Async<'a>

        /// Build a new Async task in terms of a Begin/End pair of actions in 
        /// the style frequently used in .NET APIs where the
        /// overall operation is not qualified by any arguments. For example, 
        ///     Async.BuildPrimitive(ws.BeginGetWeather,ws.EndGetWeather)
        [<OverloadID("BuildPrimitve_zero_arg")>]
        static member BuildPrimitive : (System.AsyncCallback * obj -> System.IAsyncResult) * (System.IAsyncResult -> 'a) -> Async<'a>

        /// Build a new Async task in terms of a Begin/End pair of actions in 
        /// the style frequently used in .NET APIs where the
        /// overall operation is qualified by one argument. For example, 
        ///     Async.BuildPrimitive(place,ws.BeginGetWeather,ws.EndGetWeather)
        [<OverloadID("BuildPrimitve_one_arg")>]
        static member BuildPrimitive : 'arg1 * ('arg1 * System.AsyncCallback * obj -> System.IAsyncResult) * (System.IAsyncResult -> 'a) -> Async<'a>

        /// Build a new Async task in terms of a Begin/End pair of actions in 
        /// the style frequently used in .NET APIs where the
        /// overall operation is qualified by two arguments. For example, 
        ///     Async.BuildPrimitive(arg1,arg2,ws.BeginGetWeather,ws.EndGetWeather)
        [<OverloadID("BuildPrimitve_two_arg")>]
        static member BuildPrimitive : 'arg1 * 'arg2 * ('arg1 * 'arg2 * System.AsyncCallback * obj -> System.IAsyncResult) * (System.IAsyncResult -> 'a) -> Async<'a>

        /// Build a new Async task in terms of a Begin/End pair of actions in 
        /// the style frequently used in .NET APIs where the
        /// overall operation is qualified by three arguments. For example, 
        ///     Async.BuildPrimitive(arg1,arg2,arg3,ws.BeginGetWeather,ws.EndGetWeather)
        [<OverloadID("BuildPrimitve_three_arg")>]
        static member BuildPrimitive : 'arg1 * 'arg2 * 'arg3 * ('arg1 * 'arg2 * 'arg3 * System.AsyncCallback * obj -> System.IAsyncResult) * (System.IAsyncResult -> 'a) -> Async<'a>
        

        //---------- Obsolete functions


        /// Run a primitive blocking operation on a brand new thread. Execute the continuation as a worker
        /// item in the thread pool.
        [<System.Obsolete("Consider using 'async { do! Async.SwitchToNewThread() ... }' instead")>]
        static member UnblockedPrimitive : (unit -> 'a) -> Async<'a>

        /// "Return x" generates an asynchronous computation that, when run, yields the value "x" immediately
        [<System.Obsolete("Consider using 'async { return ... }' instead, or call 'async.Return' directly")>]
        static member Return : 'a -> Async<'a>
        
        /// "Delay f" generates an asynchronous computation that, when run, 
        /// runs the asynchronous computation "f x"
        [<System.Obsolete("Consider using a computation expression 'async { ... }' or call 'async.Delay' directly")>]
        static member Delay : (unit -> Async<'a>) -> Async<'a>

        /// "Done" generates an asynchronous computation that, when run, yields the value 'unit' immediately
        [<System.Obsolete("Consider using 'async { return () }' instead")>]
        static member Done : Async<unit>

        [<System.Obsolete("Consider using a computation expression 'async { ... }' or call 'async.Bind' directly")>]
        static member Bind : p: Async<'a> * f: ('a -> Async<'b>) -> Async<'b>

        /// Generate an asynchronous computation that, when run, runs 'p', ignoring the result and
        /// returning the result '()'.
        [<System.Obsolete("Consider using a computation expression 'async { let! _ = p in return () }'")>]
        static member Ignore : p: Async<'a> -> Async<unit>


        //---------- Exceptions

        /// "Raise exn" generates an asynchronous computation that, when run terminates by raising 
        /// the given exception, i.e. the given exception is passed to the exception continuation.
        [<System.Obsolete("Consider using 'async { do raise ... }' instead")>]
        static member Raise : #exn -> Async<'b>

        [<System.Obsolete("Consider using 'async { try ... finally ... }' instead, or call 'async.TryFinally' directly")>]
        static member TryFinally :  Async<'a> * (unit -> unit) -> Async<'a> 
        
        [<System.Obsolete("Consider using 'async { use x = ... }' instead, or call 'async.Using' directly")>]
        static member Using : 'b * ('b -> Async<'a>) -> Async<'a> when 'b :> System.IDisposable

        [<System.Obsolete("This member has been renamed AsyncFuture")>]
        static member Future : computation:Async<'a>  * ?asyncGroup : AsyncGroup -> IAsyncFuture<'a>


    and 
        [<Experimental("This type is part of an experimental feature called 'asynchronous workflows'")>]
        /// A handle to an asynchronous computation. The Async.AsyncFuture member allows you to 
        /// spawn a computation and at a later point synchronize on its result.
        ///
        /// Each IAsyncFuture object may consume an OS resource in the form of a WaitHandle.
        IAsyncFuture<'a> =
          interface System.IDisposable 
          /// Wait for the completion of the operation and get its result
          abstract Value : 'a
          abstract AsIAsyncResult : System.IAsyncResult

     and 
         [<System.Obsolete("This type has been renamed IAsyncFuture")>]
         AsyncFuture<'a> = IAsyncFuture<'a>

    and 
        [<System.Obsolete("This type has been renamed IAsyncFuture")>]
        Future<'a> = IAsyncFuture<'a>
        
    and 
        [<Experimental("This type is part of an experimental feature called 'asynchronous workflows'")>]
        AsyncGroup =
          new : unit -> AsyncGroup
          /// Raise the cancellation condition for this group of computations
          member TriggerCancel : ?message:string -> unit


    [<Sealed>]
    type AsyncReplyChannel<'reply> =
        member Reply : 'reply -> unit
        [<System.Obsolete("This method has been renamed 'Reply'")>]
        member Post : 'reply -> unit
        
    [<System.Obsolete("The type IChannel has been replaced by the type AsyncReplyChannel")>]
    type IChannel<'msg> =
        abstract Post : 'msg -> unit
        
    [<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix);
      Experimental("This module is part of an experimental feature called 'asynchronous workflows'")>]
    module Async =
        
        type AsyncBuilder with
            /// Generate an asynchronous computation that, when run, enumerates the sequence 'seq'
            /// on demand and runs 'f' for each element.
            member For: seq:#seq<'a> * f:('a -> Async<unit>) -> Async<unit>

            /// Generate an asynchronous computation that, when run, just returns '()'
            member Zero : unit -> Async<unit> 

            /// Generate an asynchronous computation that, when run, first runs 'p1' and then runs 'p2', returning the result of 'p2'.
            member Combine : p1:Async<unit> * p2:Async<'a> -> Async<'a>

            /// Generate an asynchronous computation that, when run, runs 'p' repeatedly 
            /// until 'gd()' becomes false.
            member While : gd:(unit -> bool) * p:Async<unit> -> Async<unit>

            /// Generate an asynchronous computation that, when run, returns the result 'v'
            member Return : v:'a -> Async<'a>

            /// Generate an asynchronous computation that, when run, runs 'f()'
            member Delay : f:(unit -> Async<'a>) -> Async<'a>

            /// Generate asynchronous computation that, when run, runs 'f(resource)'. 
            /// The action 'resource.Dispose()' is executed as this computation yields its result
            /// or if the asynchronous computation exits by an exception or by cancellation.
            member Using: resource:'a * f:('a -> Async<'b>) -> Async<'b> when 'a :> System.IDisposable

            /// Generate an asynchronous computation that, when run, runs 'f v'
            member Let: v:'a * f:('a -> Async<'b>) -> Async<'b>

            /// Generate an asynchronous computation that, when run, runs 'p', and when 
            /// 'p' generates a result 'res', runs 'f res'.
            member Bind: p: Async<'a> * f: ('a -> Async<'b>) -> Async<'b>

            /// Generate asynchronous computation that, when run, firs runs 'presource' to produce
            /// a result 'resource', and then runs 'f(resource)'.
            /// The action 'resource.Dispose()' is executed as the second computation yields its result
            /// or if the asynchronous computation exits by an exception or by cancellation.
            member BindUsing: presource:Async<'a> * ('a -> Async<'b>) -> Async<'b> when 'a :> System.IDisposable
            
            /// Generate an asynchronous computation that, when run, runs 'p'. The action 'f' is executed 
            /// after 'p' completes, whether 'p' exits normally or by an exception. If 'f' raises an exception itself
            /// the original exception is discarded and the new exception becomes the overall result of the computation.
            member TryFinally : p:Async<'a> * (unit -> unit) -> Async<'a>

            /// Generate an asynchronous computation that, when run, runs 'p' and returns its result.
            /// If an exception happens then 'f(exn)' is called and the resulting computation executed instead.
            member TryWith : p:Async<'a> * f:(exn -> Async<'a>) -> Async<'a>

        [<Experimental("This value is part of an experimental feature called 'asynchronous workflows'")>]
        val async : AsyncBuilder

    [<Experimental("This value is part of an experimental feature called 'asynchronous workflows'")>]
    type AsyncWorker<'a> =
        new : Async<'a> * ?asyncGroup: AsyncGroup -> AsyncWorker<'a>
        member ProgressChanged : IEvent<int> 
        member Error : IEvent<exn> 
        member Completed : IEvent<'a> 
        member Canceled : IEvent<exn> 
        member RunAsync : unit -> bool
        member ReportProgress : int -> unit
        member CancelAsync : ?message:string -> unit

namespace Microsoft.FSharp.Control

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Control
    open Microsoft.FSharp.Collections

    module CommonExtensions =
        open System.IO
        
        type System.IO.Stream with 
            
            [<OverloadID("ReadAsync1")>]
            /// Return an asynchronous computation that will read from the stream into the given buffer. 
            member ReadAsync : buffer:byte[] * ?offset:int * ?count:int -> Async<int>
            
            [<OverloadID("ReadAsync2")>]
            /// Return an asynchronous computation that will read the given number of bytes from the stream 
            member ReadAsync : int -> Async<byte[]>
            
            /// Return an asynchronous computation that will write the given bytes to the stream 
            member WriteAsync : buffer:byte[] * ?offset:int * ?count:int -> Async<unit>

        type System.IO.StreamReader with 
            member ReadToEndAsync : unit -> Async<string>

        type System.IO.File with 
            /// Return an asynchronous computation that will open the file via a fresh blocking I/O thread.
            static member OpenTextAsync: path:string -> Async<StreamReader>

            /// Return an asynchronous computation that will open the file via a fresh blocking I/O thread.
            static member OpenReadAsync: path:string -> Async<FileStream>

            /// Return an asynchronous computation that will open the file via a fresh blocking I/O thread.
            static member OpenWriteAsync: path:string -> Async<FileStream>

            /// Return an asynchronous computation that will open the file via a fresh blocking I/O thread.
            static member AppendTextAsync: path:string -> Async<StreamWriter>

            /// Return an asynchronous computation that will open the file via a fresh blocking I/O thread.
            static member OpenAsync: path:string * mode:FileMode * ?access: FileAccess * ?share: FileShare -> Async<FileStream>

#if ASSUME_NO_SILVERLIGHT
        type System.Net.WebRequest with 
            /// Return an asynchronous computation that, when run, will wait for a response to the given WebRequest.
            member GetResponseAsync : unit -> Async<System.Net.WebResponse>
#endif

        type System.Threading.WaitHandle with 
            /// Return an asynchronous computation that will wait on the given WaitHandle. This is scheduled
            /// as a wait-item in the .NET thread pool, meaning that the operation will not block any threads
            /// for the duration of the wait.
            member WaitOneAsync: ?millisecondsTimeout:int -> Async<bool>

        type System.Threading.Timer with 
            /// Return an asynchronous computation that will sleep for the given time. This is scheduled
            /// using a System.Threading.Timer object, meaning that the operation will not block any threads
            /// for the duration of the wait.
            static member SleepAsync : millisecondsDueTime:int -> Async<unit>

        type Microsoft.FSharp.Control.IChannel<'msg> with

            [<System.Obsolete("This member has been renamed 'TryPostAndReply'")>]
            member TryPostSync : (Microsoft.FSharp.Control.IChannel<'reply> -> 'msg) * ?timeout : int * ?exitContext : bool-> 'reply option
            [<System.Obsolete("This member has been renamed 'PostAndReply'")>]
            member PostSync : (Microsoft.FSharp.Control.IChannel<'reply> -> 'msg) * ?timeout : int * ?exitContext : bool-> 'reply

            /// Post the message and wait for a single reply on the given channel. Return None if
            /// no reply within the timeout period.
            member TryPostAndReply : (Microsoft.FSharp.Control.IChannel<'reply> -> 'msg) * ?timeout : int * ?exitContext : bool-> 'reply option
            /// Post the message and wait for a single reply on the given channel. Raise an exception
            /// if no reply within the timeout period.
            member PostAndReply : (Microsoft.FSharp.Control.IChannel<'reply> -> 'msg) * ?timeout : int * ?exitContext : bool-> 'reply


namespace Microsoft.FSharp.Control.Mailboxes

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Control
    open Microsoft.FSharp.Collections


    [<Experimental("This type is part of an experimental feature called 'asynchronous workflows'")>]
    /// A MailboxProcessor is an asynchronous computation that includes the ability to read from a single dedicated
    /// channel (i.e. a single dedicated message queue). Anyone can send messages to a MailboxProcessor by using the Post method.
    /// A MailboxProcessor enters a state where it waits for the next message by calling its own Receive or TryReceive method.
    ///
    /// A MailboxProcessor can scan through all available messages using its own Scan or TryScan method, 
    /// by using a function that selects an asynchronous computation to 
    /// run based on a scan of the message queue. 
    ///
    /// A MailboxProcessor generally needs to use one or more of its 
    /// own Receive, TryReceive, Scan or TryScan methods. It also often 
    /// typically has to allow other asynchronous computations to send 
    /// messages back to the MailboxProcessor. As a result the creation functions
    /// are given a reference to the MailboxProcessor itself. 
    ///
    type MailboxProcessor<'msg> =

        /// Create an instance of a MailboxProcessor. The asynchronous computation executed by the
        /// processor is the one returned by the 'initial' function. This function is not executed until
        /// 'Start' is called.
        new :  initial:(MailboxProcessor<'msg> -> Async<unit>) * ?asyncGroup: AsyncGroup -> MailboxProcessor<'msg>

        /// Create an instance of a MailboxProcessor. The asynchronous computation executed by the
        /// processor is the one returned by the 'initial' function. This function is not executed until
        /// 'Start' is called.
        [<System.Obsolete("This member has been replaced by an object constructor. Use 'new MailboxProcessor<_>(...)' instead of 'MailboxProcessor<_>.Create(...)'")>]
        static member Create :  initial:(MailboxProcessor<'msg> -> Async<unit>) * ?asyncGroup: AsyncGroup -> MailboxProcessor<'msg>

        /// Create and start an instance of a MailboxProcessor. The asynchronous computation executed by the
        /// processor is the one returned by the 'initial' function. 
        static member Start  :  initial:(MailboxProcessor<'msg> -> Async<unit>) * ?asyncGroup: AsyncGroup -> MailboxProcessor<'msg>

        /// Post a message to the message queue of the MailboxProcessor, asynchronously
        member Post : 'msg -> unit

        /// Post a message to the message queue of the MailboxProcessor and await a reply on the channel
        member PostAndReply : (Microsoft.FSharp.Control.AsyncReplyChannel<'reply> -> 'msg) * ?timeout : int * ?exitContext : bool -> 'reply

        /// Post a message to the message queue of the MailboxProcessor and await a reply on the channel
        member TryPostAndReply : (Microsoft.FSharp.Control.AsyncReplyChannel<'reply> -> 'msg) * ?timeout : int * ?exitContext : bool -> 'reply option

        /// Post a message to the message queue of the MailboxProcessor and await an eventual reply on the channel. Return None if timeout exceeded.
        member PostAndReplyAsync : (Microsoft.FSharp.Control.AsyncReplyChannel<'reply> -> 'msg) * ?timeout : int * ?exitContext : bool -> Async<'reply>

        /// Post a message to the message queue of the MailboxProcessor and eventually await a reply on the channel. Return None if timeout exceeded.
        member TryPostAndReplyAsync : (Microsoft.FSharp.Control.AsyncReplyChannel<'reply> -> 'msg) * ?timeout : int * ?exitContext : bool -> Async<'reply  option>

        [<System.Obsolete("This member has been renamed 'PostAndReply'")>]
        member PostSync : (Microsoft.FSharp.Control.AsyncReplyChannel<'reply> -> 'msg) * ?timeout : int * ?exitContext : bool-> 'reply

        /// Return an asynchronous computation which will 
        /// consume the first message in arrival order. No thread
        /// is blocked while waiting for further messages. Raise a TimeoutException
        /// if the timeout is exceeded.
        member Receive : ?timeout:int -> Async<'msg>

        /// Return an asynchronous computation which will 
        /// consume the first message in arrival order. No thread
        /// is blocked while waiting for further messages. Return None
        /// if the timeout is exceeded.
        member TryReceive : ?timeout:int -> Async<'msg option>
        
        /// Return an asynchronous computation which will 
        /// look through messages in arrival order until 'scanner' returns a Some value. No thread
        /// is blocked while waiting for further messages. Raise a TimeoutException
        /// if the timeout is exceeded.
        member Scan : scanner:('msg -> (Async<'res>) option) * ?timeout:int -> Async<'res>

        /// Return an asynchronous computation which will 
        /// look through messages in arrival order until 'scanner' returns a Some value. No thread
        /// is blocked while waiting for further messages. Return None
        /// if the timeout is exceeded.
        member TryScan : scanner:('msg -> (Async<'res>) option) * ?timeout:int -> Async<'res option>

        member UnsafeMessageQueueContents : seq<'msg> 

        /// Start the MailboxProcessor
        member Start : unit -> unit
        // Raise a timeout exception if a message not received in this amount of time. Default -1 (infinite).
        member DefaultTimeout : int with get, set
        interface IChannel<'msg>


#endif

namespace Microsoft.FSharp.Control.SharedMemory

    open Microsoft.FSharp.Core

    module Helpers =
        val inline readLock: System.Threading.ReaderWriterLock -> (unit -> 'a) -> 'a 
        val inline writeLock: System.Threading.ReaderWriterLock -> (unit -> 'a) -> 'a 


