// (c) Microsoft Corporation 2005-2007. 
#light
namespace Microsoft.FSharp.Math

open Microsoft.FSharp.Collections
open Microsoft.FSharp.Core
open Microsoft.FSharp.Math
open System

// A type-class for numeric types
type INumeric<'a> =
    abstract Zero: 'a
    abstract One: 'a
    abstract Add: 'a * 'a -> 'a
    abstract Subtract: 'a * 'a -> 'a
    abstract Multiply : 'a * 'a -> 'a
    abstract Negate : 'a -> 'a
    abstract Sign : 'a -> int
    abstract Abs : 'a -> 'a    
    abstract ToString : 'a * string * System.IFormatProvider -> string
    abstract Parse : string * System.Globalization.NumberStyles * System.IFormatProvider -> 'a

type IIntegral<'a> =
    inherit INumeric<'a>
    abstract Modulus: 'a * 'a -> 'a
    abstract Divide : 'a * 'a -> 'a
    abstract DivMod : 'a * 'a -> 'a * 'a
    //abstract Remainder: 'a * 'a -> 'a
    //abstract Quotient : 'a * 'a -> 'a
    //abstract QuotRem: 'a * 'a -> 'a * 'a
    abstract ToBigInt : 'a -> bigint
    abstract OfBigInt : bigint -> 'a
  
type IFractional<'a> =
    inherit INumeric<'a>
    abstract Reciprocal : 'a -> 'a
    abstract Divide : 'a * 'a -> 'a
    // TODO :add these
    // NOTE: Haskell puts ToBigNum in a class called "Real" and OfBigNum (actually OfInt) in INumeric
    // 
    //abstract ToBigNum : 'a -> bignum
    //abstract OfBigNum : bignum -> 'a

// This is Haskell's name for this class.  I'd like a better name.
// Suggestion: IReal (since transcendentals are added here).
type IFloating<'a> =
    inherit IFractional<'a>
    abstract Pi : 'a
    abstract Exp : 'a -> 'a
    abstract Log : 'a -> 'a
    abstract Sqrt : 'a -> 'a
    abstract LogN : 'a * 'a -> 'a
    abstract Sin : 'a -> 'a
    abstract Cos : 'a -> 'a
    abstract Tan : 'a -> 'a
    abstract Asin : 'a -> 'a
    abstract Acos : 'a -> 'a
    abstract Atan : 'a -> 'a
    abstract Atan2 : 'a * 'a -> 'a
    abstract Sinh : 'a -> 'a
    abstract Cosh : 'a -> 'a
    abstract Tanh : 'a -> 'a

type INormFloat<'a> =
    abstract Norm : 'a -> float

// Corresponds to Haskell's RealFrac
(*
type IFractionalToInt<'a> =
  interface
    inherit IFractional<'a>
    abstract ProperFraction : 'a -> (bigint * 'a)
    
    // These can all be dereived from ProperFaction but are included for efficiency
    abstract Truncate       : 'a -> int
    abstract Ceiling        : int -> 'a
    abstract Floor          : int -> 'a
  end
*)
  
// Corresponds to Haskell's RealFloat.  Direct access to IEEE encoding not easy on .NET,
// hence many functions dropped
type IIEEE<'a> =
  interface
    inherit IFloating<'a>
    //inherit IFractionalToInt<'a>
    abstract PositiveInfinity : 'a
    abstract NegativeInfinity : 'a
    abstract NaN              : 'a
    abstract Epsilon          : 'a

    abstract IsNaN: 'a -> bool 
    abstract IsInfinite : 'a -> bool 
    //abstract IsDenormalized   : 'a -> bool 
    //abstract IsNegativeZero   : 'a -> bool 
    //abstract IsIEEE           : 'a -> bool 
  end


module Instances : 
  val Float32Numerics  : IFractional<float32>  // should be IIEEE<bignum>, but not enough in System.Math to do that portably
                                            // Thus should be IFractionalToInt<bignum>
  val FloatNumerics    : IIEEE<float>
  val Int32Numerics    : IIntegral<int32>
  val Int64Numerics    : IIntegral<int64>
  val BigIntNumerics   : IIntegral<bigint>
  val BigNumNumerics   : IFractional<bignum>  // should be IFractionalToInt<bignum>



namespace Microsoft.FSharp.Math.Types

    type INumeric<'a> = Microsoft.FSharp.Math.INumeric<'a>
    type IIntegral<'a> = Microsoft.FSharp.Math.IIntegral<'a>
    type IFractional<'a> = Microsoft.FSharp.Math.IFractional<'a>
    type IFloating<'a> = Microsoft.FSharp.Math.IFloating<'a>
    type IIEEE<'a> = Microsoft.FSharp.Math.IIEEE<'a>
