(* (c) Microsoft Corporation. All rights reserved *)

(*F# 
/// Various constants and utilities used when parsing the ILASM format for IL
module Microsoft.Research.AbstractIL.Internal.AsciiConstants
open Microsoft.Research.AbstractIL
open Microsoft.Research.AbstractIL.Internal
module Ilx = Microsoft.Research.AbstractIL.Extensions.ILX.Types 
module Il = Microsoft.Research.AbstractIL.IL 
F#*)  

open Ilx
open Il
(* -------------------------------------------------------------------- 
 * IL Parser state - must be initialized before parsing a module
 * -------------------------------------------------------------------- *)

val parse_ilGlobals: mscorlib_refs ref

(* -------------------------------------------------------------------- 
 * IL Lexer and pretty-printer tables
 * -------------------------------------------------------------------- *)
type instr_name = string list
type prefix = 
  | Prefix_Tail
  | Prefix_Volatile
  | Prefix_Unaligned of int
  | Prefix_Readonly
  | Prefix_Constrained of typ
type prefixes = prefix list

type ('a,'b) delay = ('a -> 'b)
type modifiable_instr = prefixes -> instr

type none_instr = unit -> modifiable_instr
type i32_instr = Nums.i32 -> modifiable_instr
type i32_i32_instr = Nums.i32 * Nums.i32 -> modifiable_instr
type arg_instr = Nums.u16 -> modifiable_instr
type loc_instr = Nums.u16 -> modifiable_instr
type env_instr = int -> modifiable_instr
type arg_typ_instr = Nums.u16 * typ -> modifiable_instr
type i64_instr = Nums.i64 -> modifiable_instr
type real_instr = ldc_info -> modifiable_instr
type field_instr = field_spec -> modifiable_instr
type method_instr = method_spec * varargs -> modifiable_instr
type unconditional_instr = code_label -> modifiable_instr
type conditional_instr = code_label * code_label -> modifiable_instr
type type_instr = typ -> modifiable_instr
type int_type_instr = int * typ -> modifiable_instr
type valuetype_instr = typ -> modifiable_instr
type string_instr = Bytes.bytes -> modifiable_instr
type sig_instr = callsig * varargs -> modifiable_instr
type tok_instr = ldtoken_info -> modifiable_instr
type switch_instr = code_label list * code_label -> modifiable_instr

type 'a instr_table = (string list * 'a) list
type 'a lazy_instr_table = 'a instr_table Lazy.t

val loc_instrs: loc_instr lazy_instr_table
val arg_instrs: arg_instr lazy_instr_table
val none_instrs: none_instr lazy_instr_table
val i64_instrs: i64_instr lazy_instr_table
val i32_instrs: i32_instr lazy_instr_table
val i32_i32_instrs: i32_i32_instr lazy_instr_table
val real_instrs: real_instr lazy_instr_table
val field_instrs: field_instr lazy_instr_table
val method_instrs: method_instr lazy_instr_table
val string_instrs: string_instr lazy_instr_table
val switch_instrs: switch_instr lazy_instr_table
val tok_instrs: tok_instr lazy_instr_table
val sig_instrs: sig_instr lazy_instr_table
val type_instrs: type_instr lazy_instr_table
val int_type_instrs: int_type_instr lazy_instr_table
val valuetype_instrs: valuetype_instr lazy_instr_table

val words_of_noarg_instr : (instr -> string list)
val is_noarg_instr : (instr -> bool)

val set_hasthis_in_mspec : hasthis -> method_spec -> method_spec

(* -------------------------------------------------------------------- 
 * Lexer state
 * -------------------------------------------------------------------- *)

val lexing_bytearray: bool ref



