(* --------------------------------------------------------------------
 * Define an extension of the IL algebra
 *
 * (c) Microsoft Corporation. All rights reserved 
 * -------------------------------------------------------------------- *)

(*F# 
module Microsoft.Research.AbstractIL.Extensions.ILX.Types
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
module Il = Microsoft.Research.AbstractIL.IL 
module Illib = Microsoft.Research.AbstractIL.Internal.Library 
F#*)  

open Illib
open Il
open Nums

type ilx_typ =
 | EType_erasable_array of array_shape * typ      (* -- Pseudo-array types *)

let rescope_ilx_typ scoref typ = 
  match typ with 
  | EType_erasable_array (s,ty) -> EType_erasable_array (s,rescope_typ scoref ty)

and inst_ilx_typ_aux n inst typ = 
  match typ with 
  | EType_erasable_array (a,t) -> EType_erasable_array (a,inst_typ_aux n inst t)


let (mk_ilx_ext_typ,is_ilx_ext_typ,dest_ilx_ext_typ) = 
  define_typ_extension  
    { typeExtRescope=rescope_ilx_typ;
      typeExtInstAux=inst_ilx_typ_aux }

let mk_ilx_typ ity = Type_other (mk_ilx_ext_typ ity)

(* --------------------------------------------------------------------
 * Define an extension of the IL instruction algebra
 * -------------------------------------------------------------------- *)

type alternative = 
    { altName: string;
      altFields: field_def array;
      altCustomAttrs: custom_attrs }

and classunion_ref = ClassunionRef of type_ref * alternative array * bool
and classunion_spec = ClassunionSpec of classunion_ref * genactuals

and lambdas = 
  | Lambdas_forall of genparam * lambdas
  | Lambdas_lambda of param * lambdas
  | Lambdas_return of typ
and closure_ref = ClosureRef of type_ref * lambdas * freevar list 
and closure_spec = ClosureSpec of closure_ref * genactuals

and freevar = 
    { fvName: string ; fvType: typ }
let mk_freevar (name,ty) = 
    { fvName=name;
      fvType=ty; }
let typ_of_freevar p = p.fvType
let name_of_freevar p = p.fvName

type apps = 
  | Apps_tyapp of typ * apps 
  | Apps_app of typ * apps 
  | Apps_done of typ

type ilx_instr = 
  | EI_lddata of classunion_spec * int * int
  | EI_isdata of classunion_spec * int
  | EI_brisdata of classunion_spec * int * code_label * code_label
  | EI_castdata of bool * classunion_spec * int
  | EI_stdata of classunion_spec * int * int
  | EI_datacase of (bool * classunion_spec * (int * code_label) list * code_label) (* last label is fallthrough, bool is whether to leave value on the stack for each case *)
  | EI_lddatatag of classunion_spec
  | EI_newdata of classunion_spec * int
  | EI_newclo of closure_spec
  | EI_castclo of closure_spec
  | EI_ilzero of typ
  | EI_isclo of closure_spec
  | EI_callclo of tailness * closure_spec * apps
  | EI_stclofld  of (closure_spec * int)  (* nb. leave these brackets *)
  | EI_ldenv  of int
  | EI_callfunc of tailness * apps
  | EI_ldftn_then_call of method_spec * (tailness * method_spec * varargs)  (* special: for internal use only *)
  | EI_ld_instance_ftn_then_newobj of method_spec * callsig * (method_spec * varargs)  (* special: for internal use only *)
  | EI_ldelem_any_erasable  of array_shape * typ (* indicates that the array being loaded from is a "erasable" array - see above notes on array bounds *)
  | EI_stelem_any_erasable  of array_shape * typ (* see above  *)
  | EI_newarr_erasable      of array_shape * typ (* see above  *)
  | EI_ldlen_multi      of i32 * i32

let destinations_of_ilx_instr i =
  match i with 
  |  (EI_brisdata (_,_,l1,l2)) ->  [l1; l2]
  |  (EI_callfunc (Tailcall,_)) |  (EI_callclo (Tailcall,_,_)) ->   []
  |  (EI_datacase (_,_,ls,l)) -> l:: (List.fold_right (fun (_,l) acc -> insert l acc) ls [])
  | _ -> []

let fallthrough_of_ilx_instr i = 
  match i with 
  |  (EI_brisdata (_,_,_,l)) 
  |  (EI_datacase (_,_,_,l)) -> Some l
  | _ -> None

let ilx_instr_is_tailcall i = 
  match i with 
  |  (EI_callfunc (Tailcall,_)) |  (EI_callclo (Tailcall,_,_)) -> true
  | _ -> false

let remap_ilx_labels lab2cl i = 
  match i with 
    | EI_brisdata (a,b,l1,l2) -> EI_brisdata (a,b,lab2cl l1,lab2cl l2)
    | EI_datacase (b,x,ls,l) -> EI_datacase (b,x,List.map (fun (y,l) -> (y,lab2cl l)) ls, lab2cl l)
    | _ -> i

let (mk_ilx_ext_instr,is_ilx_ext_instr,dest_ilx_ext_instr) = 
  define_instr_extension  
    { instrExtDests=destinations_of_ilx_instr;
      instrExtFallthrough=fallthrough_of_ilx_instr;
      instrExtIsTailcall=ilx_instr_is_tailcall;
      instrExtRelabel=remap_ilx_labels; }

let mk_IlxInstr i = I_other (mk_ilx_ext_instr i)

(* --------------------------------------------------------------------
 * Define an extension of the IL algebra of type definitions
 * -------------------------------------------------------------------- *)


type closure_info = 
    { cloStructure: lambdas;
      cloFreeVars: freevar list;  
      cloCode: (il_method_body Lazy.t);
      cloSource: source option}

and classunion_info = 
    { cudReprAccess: member_access; (* is the representation public? *)
      cudHelpersAccess: member_access; (* are the representation public? *)
      cudHelpers: bool; (* generate the helpers? *)
      cudAlternatives: alternative array;
      cudNullPermitted: bool;
      (* debug info for generated code for classunions *) 
      cudWhere: source option; }

type ilx_type_def_kind = 
 | ETypeDef_closure of closure_info
 | ETypeDef_classunion of classunion_info


let (mk_ilx_ext_type_def_kind,is_ilx_ext_type_def_kind,dest_ilx_ext_type_def_kind) = 
  (define_type_def_kind_extension Type_def_kind_extension : (ilx_type_def_kind -> ext_type_def_kind) * (ext_type_def_kind -> bool) * (ext_type_def_kind -> ilx_type_def_kind) )

let mk_ilx_type_def_kind i = TypeDef_other (mk_ilx_ext_type_def_kind i)

(* --------------------------------------------------------------------
 * Define these as extensions of the IL types
 * -------------------------------------------------------------------- *)

let dest_func_app = function Apps_app (d,r) -> d,r | _ -> failwith "dest_func_app"
let dest_tyfunc_app = function Apps_tyapp (b,c) -> b,c | _ -> failwith "dest_tyfunc_app"

let gen_mk_array_ty (shape,ty,flag) = 
  if flag then mk_ilx_typ(EType_erasable_array(shape,ty)) else Type_array(shape,ty)

let gen_is_array_ty ty = 
  match ty with 
    Type_array _ -> true
  | Type_other e when is_ilx_ext_typ e ->
      begin match dest_ilx_ext_typ e with 
      | EType_erasable_array _ -> true
      end
  | _ -> false

let gen_dest_array_ty ty =
  match ty with 
  | Type_array(shape,ty) -> (shape,ty,false)
  | Type_other e when is_ilx_ext_typ e ->
      begin match dest_ilx_ext_typ e with 
      | EType_erasable_array (shape,ty) -> (shape,ty,true)
      end
  | _ -> failwith "gen_dest_array_ty"


let mk_array_ty_old (shape,ty) = gen_mk_array_ty (shape,ty,true)


(* --------------------------------------------------------------------
 * MS-ILX: Closures
 * -------------------------------------------------------------------- *)

let rec inst_apps_aux n inst = function
    Apps_tyapp (ty,rty) -> Apps_tyapp(inst_genactual_aux n inst ty, inst_apps_aux n inst rty)
  | Apps_app (dty,rty) ->  Apps_app(inst_typ_aux n inst dty, inst_apps_aux n inst rty)
  | Apps_done rty ->  Apps_done(inst_typ_aux n inst rty)

let rec inst_lambdas_aux n inst = function
  | Lambdas_forall (b,rty) -> 
      Lambdas_forall(b, inst_lambdas_aux n inst rty)
  | Lambdas_lambda (p,rty) ->  
      Lambdas_lambda({ p with paramType=inst_typ_aux n inst p.paramType},inst_lambdas_aux n inst rty)
  | Lambdas_return rty ->  Lambdas_return(inst_typ_aux n inst rty)

let inst_lambdas i t = inst_lambdas_aux 0 i t

let cloref_of_clospec (ClosureSpec(cloref,_)) = cloref

let tref_of_clospec (ClosureSpec(ClosureRef(tref,_,_),_)) = tref
let formal_freevar_type_of_cloref (ClosureRef(_,_,fvs)) n = 
  typ_of_freevar (List.nth fvs n)
let formal_freevar_type_of_clospec x n = formal_freevar_type_of_cloref (cloref_of_clospec x) n
let actual_freevar_type_of_clospec (ClosureSpec(cloref,inst)) n = 
  inst_typ inst (formal_freevar_type_of_cloref cloref n)
let actual_freevars_of_clospec (ClosureSpec(ClosureRef(_,_,fvs),inst)) = 
  List.map (fun fv -> {fv with fvType = inst_typ inst fv.fvType}) fvs
let formal_freevars_of_clospec (ClosureSpec(ClosureRef(_,_,fvs),inst)) = fvs
let actual_lambdas_of_clospec (ClosureSpec(ClosureRef(_,lambdas,_),inst)) = inst_lambdas inst lambdas
let formal_lambdas_of_clospec (ClosureSpec(ClosureRef(_,lambdas,_),_)) = lambdas
let inst_of_clospec (ClosureSpec(_,inst)) = inst
let generalize_cloref gparams csig = ClosureSpec(csig, generalize_gparams gparams)



(* --------------------------------------------------------------------
 * MS-ILX: Unions
 * -------------------------------------------------------------------- *)

let objtype_of_cuspec (ClassunionSpec(ClassunionRef(tref,_,_),inst)) = mk_boxed_typ tref inst
let tref_of_cuspec (ClassunionSpec(ClassunionRef(tref,_,_),inst)) = tref
let inst_of_cuspec (ClassunionSpec(_,inst)) = inst
let altsarray_of_cuspec (ClassunionSpec(ClassunionRef(tref,alts,_),inst)) = alts
let nullPermitted_of_cuspec (ClassunionSpec(ClassunionRef(_,_,np),inst)) = np
let alts_of_cuspec cuspec = Array.to_list (altsarray_of_cuspec cuspec)
let alt_of_cuspec cuspec n = (altsarray_of_cuspec cuspec).(n)

let alt_is_nullary alt = (Array.length alt.altFields = 0)
let fdefs_of_alt alt = alt.altFields
let name_of_alt alt = alt.altName
let fdef_of_alt alt fidx = (fdefs_of_alt alt).(fidx)

let fdef_of_cuspec cuspec idx fidx = fdef_of_alt (alt_of_cuspec cuspec idx) fidx

let actual_typ_of_cuspec_field cuspec idx fidx =
  inst_typ (inst_of_cuspec cuspec) (fdef_of_cuspec cuspec idx fidx).fdType

