// (c) Microsoft Corporation. All rights reserved *)

#light

module Microsoft.FSharp.Compiler.Interactive.Shell
module Ilsupp = Microsoft.Research.AbstractIL.Internal.Support
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics
module Ilmorph = Microsoft.Research.AbstractIL.Morphs 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter
module Ilreflect = Microsoft.Research.AbstractIL.RuntimeWriter
module Il = Microsoft.Research.AbstractIL.IL
module Unilex = Microsoft.FSharp.Compiler.UnicodeLexing
   
open Microsoft.Research.AbstractIL
open Microsoft.Research.AbstractIL.Internal
open Microsoft.Research.AbstractIL.Extensions.ILX
open Microsoft.Research.AbstractIL.RuntimeWriter

open Microsoft.FSharp.Compiler
open Microsoft.FSharp.Core.Pervasives
open Microsoft.FSharp.Compatibility 

open System
open System.Runtime.InteropServices
open System.IO
open System.Text
open System.Threading
open System.Windows.Forms

open Microsoft.FSharp.Compiler.Interactive.Settings

open Lib
open Fscopts
open Ildiag
open Il
open Ilxgen
open Range
open Ast
open Tc
open Tast
open Tastops
open List
open Opt
open Env
open Printf
open Build
open Nameres
open Lexhelp
open PostTypecheckSemanticChecks

//----------------------------------------------------------------------------
// Hardbinding dependencies should we NGEN fsi.exe
//----------------------------------------------------------------------------

#if CLI_AT_MOST_1_1
#else
open System.Runtime.CompilerServices
[<Dependency("FSharp.Compiler",LoadHint.Always)>] do ()
[<Dependency("Microsoft.Research.AbstractIL",LoadHint.Always)>] do ()
[<Dependency("FSharp.Core",LoadHint.Always)>] do ()
[<Dependency("FSharp.Compatibility",LoadHint.Always)>] do ()
#endif

//----------------------------------------------------------------------------
// Timing support
//----------------------------------------------------------------------------

do Fscopts.product := "MSR F# Interactive"

#if CLI_AT_MOST_1_1
type TimeReporter() =
    let ptime = System.Diagnostics.Process.GetCurrentProcess()
    member tr.TimeOp(f) =
        let startTotal = ptime.TotalProcessorTime
        let res = f ()
        let total = ptime.TotalProcessorTime - startTotal
        printfn "CPU: %02d:%02d:%02d.%02d" 
            total.Hours total.Minutes total.Seconds total.Milliseconds
        res
    member tr.TimeOpIf flag f = if flag then tr.TimeOp(f) else f ()

#else

type TimeReporter() =
    let stopwatch = new System.Diagnostics.Stopwatch()
    let ptime = System.Diagnostics.Process.GetCurrentProcess()
    let numGC = System.GC.MaxGeneration
    member tr.TimeOp(f) =
        let startTotal = ptime.TotalProcessorTime
        let startGC = [| for i in 0 .. numGC -> System.GC.CollectionCount(i) |]
        stopwatch.Reset()
        stopwatch.Start()
        let res = f ()
        stopwatch.Stop()
        let total = ptime.TotalProcessorTime - startTotal
        let spanGC = [ for i in 0 .. numGC-> System.GC.CollectionCount(i) - startGC.[i] ]
        let elapsed = stopwatch.Elapsed in
        printfn "Real: %02d:%02d:%02d.%02d, CPU: %02d:%02d:%02d.%02d, GC %s" 
            elapsed.Hours elapsed.Minutes elapsed.Seconds elapsed.Milliseconds 
            total.Hours total.Minutes total.Seconds total.Milliseconds
            (String.concat ", " (List.mapi (sprintf "gen%d: %d") spanGC))
        res

    member tr.TimeOpIf flag f = if flag then tr.TimeOp(f) else f ()
#endif

let timeReporter = TimeReporter()

//----------------------------------------------------------------------------
// Console coloring
//----------------------------------------------------------------------------

let runningOnMono = 
        match (try Some(System.Reflection.Assembly.Load
                           (
#if CLI_AT_MOST_1_1
                            "Mono.Posix, Version=1.0.5000.0, Culture=neutral, PublicKeyToken=0738eb9f132ed756"
#else
                            "Mono.Posix, Version=2.0.0.0, Culture=neutral, PublicKeyToken=0738eb9f132ed756"
#endif
                    )) with _ -> None) with 
        | Some(monoPosix) -> true
        | _ -> false

open Layout
#if CLI_AT_MOST_1_1
let withErrorColor _ f = f()
#else
let ignoreFailureOnMono1_1_16 f = try f() with _ -> ()

let withErrorColor isWarn f =
  let c = try Console.ForegroundColor with _ -> ConsoleColor.White
  try 
    ignoreFailureOnMono1_1_16 (fun () -> Console.ForegroundColor <- (if isWarn then ConsoleColor.Cyan else ConsoleColor.Red)); 
    f();
  finally 
    ignoreFailureOnMono1_1_16 (fun () -> Console.ForegroundColor <- c)
#endif

//----------------------------------------------------------------------------
// value printing
//----------------------------------------------------------------------------

open System.Reflection
let callStaticMethod (ty:Type) name args =
    ty.InvokeMember(name, (BindingFlags.InvokeMethod ||| BindingFlags.Static ||| BindingFlags.Public ||| BindingFlags.NonPublic), null, null, CompatArray.of_list args)

let callGenericStaticMethod (ty:Type) name tyargs args =
#if CLI_AT_MOST_1_1
    callStaticMethod (ty:Type) name args
#else
    let m = ty.GetMethod(name,(BindingFlags.InvokeMethod ||| BindingFlags.Static ||| BindingFlags.Public ||| BindingFlags.NonPublic)) 
    let m = m.MakeGenericMethod(CompatArray.of_list tyargs) 
    m.Invoke(null,CompatArray.of_list args)
#endif

let saverPath  = ["Microsoft";"FSharp";"Compiler";"Interactive";"Internals";"saveIt"]

let handle f = 
    try f ()
    with 
    | :? ThreadAbortException -> Layout.wordL ""
    | e -> 
      Printf.printf "\n\nException raised during pretty printing.\nPlease report this so it can be fixed.\nTrace: %s\n" (e.ToString()); 
      Layout.wordL ""
    
let printVal (opts:FormatOptions) (x:obj) (ty:System.Type) = handle (fun () -> 
    // We do a dynamic invoke of any_to_layout with the right System.Type parameter for the static type of the saved value.
    // In principle this helps any_to_layout do the right thing as it descends through terms. In practice it means
    // it at least does the right thing for top level 'null' list and option values (but not for nested ones).
    //
    // The static type was saved into the location used by Internals.getSavedItType when Internals.saveIt was called.
    // Internals.saveIt has type ('a -> unit), and fetches the System.Type for 'a by using a (type 'a) call.
    // The funny thing here is that you might think that the driver (this file) knows more about the static types
    // than the compiled code does. But it doesn't! In particular, it's not that easy to get a System.Type value based on the
    // static type information we do have: we have no direct way to bind a F# TAST type or even an AbstractIL type to 
    // a System.Type value (I guess that functionality should be in ilreflect.fs).
    //
    // This will be more significant when we print values other then 'it'
    //
    let ass = System.Reflection.Assembly.Load(fslib())
    let pervasivesModule = ass.GetType(lib_FSLib_Pervasives_name)  
    let res : StructuredFormat.Layout = callGenericStaticMethod pervasivesModule "any_to_layout" [ty] [box opts; box x] |> unbox 
    res)
    
let invokePrinter printer denv vref = 
    let opts        = Microsoft.FSharp.Compiler.Interactive.Internals.getFsiPrintOptions()
    let savedIt     = Microsoft.FSharp.Compiler.Interactive.Internals.getSavedIt()
    let savedItType = Microsoft.FSharp.Compiler.Interactive.Internals.getSavedItType()
    let valL = printer opts savedIt savedItType 
    let fullL = (Tastops.NicePrint.valSpecL denv vref) ++ (wordL "=" $$ valL)
    output_layout opts stdout fullL;  
    stdout.WriteLine()
    
//----------------------------------------------------------------------------
// state -  for reporting, warnings, errors
//----------------------------------------------------------------------------

let errors = ref 0
let warningOptions = newWarningOptions()
let abortOnError () = 
    if !errors > 0 
    then (eprintf "stopped due to error\n"; stderr.Flush(); raise ReportedError) 
    else ()

//----------------------------------------------------------------------------
// cmd line - option state
//----------------------------------------------------------------------------

let fsharpBinariesDir = Filename.dirname Sys.executable_name

//----------------------------------------------------------------------------
// tcConfig - build the initial config
//----------------------------------------------------------------------------

let tcConfig = Build.newTcConfig(fsharpBinariesDir, warningOptions, 
                                 true, // long running: optimizeForMemory 
                                 Sys.getcwd() )
do tcConfig.useFsiAuxLib <- true
do tcConfig.includes <- (// BUG 890: #light does not start new block on RHS of <- assignment
                         let progFiles = System.Environment.GetFolderPath(System.Environment.SpecialFolder.ProgramFiles)
                         let windows   = Path.Combine(System.Environment.GetFolderPath(System.Environment.SpecialFolder.System),"..")
                         let rec path = function [] -> "" | [x] -> x | x::y::ys -> path (Path.Combine(x,y)::ys)
                         tcConfig.includes @
                         (* Hardwire paths to locate OWC11 and XCEED DLLs for plotting. *)
                         (* Proper fix to follow, e.g. to locate them via Assembly.Load calls? *)
                          [path [windows;"assembly";"GAC";"Microsoft.Office.Interop.Owc11";"11.0.0.0__71e9bce111e9429c"];
                           path [progFiles;"Xceed Components";"Bin";".NET";"Signed DLLs"]
                          ])

//----------------------------------------------------------------------------
// cmd line - state for options
//----------------------------------------------------------------------------

let readline = ref (not runningOnMono)
let gui        = ref true // override via "--gui", on by default 
let showILCode = ref false // show modul il code 
let showTypes  = ref true  // show types after each interaction?
let saveEmittedCode = ref false
let showBanner    = ref true
let fsiServerName = ref ""
let interact = ref true
let isServer() = !fsiServerName<>""  
let explicitArgs = ref []
let recordExplicitArg arg = explicitArgs := !explicitArgs @ [arg]

#if CLI_AT_MOST_1_1
#else
let fsiServerInputCodePage = ref None
let fsiServerOutputCodePage = ref None
#endif

let fsiUsage inputFiles =
   [ "--gui"       , Arg.Set gui, "Open a default top window, use main thread for all\n\tparsing and execution (on by default)";
     "--no-gui"    , Arg.Clear gui, "";
     "--exec"      , Arg.Clear interact, "Exit after loading the files or running the .fsx scripts given on the\n\tcommand line";
     "--"          , Arg.Rest recordExplicitArg, "Treat all remaining args as command line arguments, accessed\n\tusing fsi.CommandLineArgs";
     "--use"       , Arg.String (fun s -> inputFiles := !inputFiles @ [(s,true)]), "#use the given file on startup";
     "--load"      , Arg.String (fun s -> inputFiles := !inputFiles @ [(s,false)]), "#load the given file on startup";
     "--no-logo"   , Arg.Clear showBanner, "Don't show the splash text on startup";
     "--no-banner" , Arg.Clear showBanner, "";
     "--no-readline" , Arg.Clear readline, "Don't attempt to process individual keystrokes from the\n\tconsole";
     "--readline" , Arg.Set readline, "Attempt to process individual keystrokes from the\n\tconsole";
     "--quiet"     , Arg.Unit (fun () -> tcConfig.noFeedback <- true), "Suppress fsi writing to stdout";
     "--fsi-server", Arg.String (fun s -> fsiServerName := s), "FSI server mode on given named channel";
#if CLI_AT_LEAST_2_0
     "--no-debug-info", Arg.Int (fun n -> Fscopts.debuginfo := false), "\n\tTurn off generation of debug information for dynamic code"; 
     "--fsi-server-input-codepage", Arg.Int (fun n -> fsiServerInputCodePage := Some(n)), "\n\tModify the input codepage associated with tthe console"; 
     "--fsi-server-output-codepage", Arg.Int (fun n -> fsiServerOutputCodePage := Some(n)), "\n\tModify the output codepage associated with the console"; 
     "--fsi-server-no-unicode", Arg.Unit (fun () -> fsiServerOutputCodePage := None;  fsiServerInputCodePage := None), "\n\tDo not set the codepages associated with the console"
#endif

]

#if CLI_AT_LEAST_2_0
do Fscopts.debuginfo := true
#endif

//----------------------------------------------------------------------------
// printfs - user, error
//----------------------------------------------------------------------------

let nullOut = new StreamWriter(Stream.Null) :> TextWriter
/// uprintf to write usual responses to stdout (suppressed by --quiet)
let uprintf fmt = fprintf (if tcConfig.noFeedback then nullOut else System.Console.Out) fmt

/// eprintf to write errors to stderr (not suppressable (yet))
let eprintf fmt = eprintf fmt

//----------------------------------------------------------------------------
// Reporting - syphon input text
//----------------------------------------------------------------------------

let stdinMockFilename = "stdin"

let syphonText = new StringBuilder()
let syphonDump() =
    let text = syphonText.ToString()
    let lines = text.Split(CompatArray.of_list [ '\n' ])  
    CompatArray.iteri (fun i (s:string) -> dprintf2 "history %2d : %s\n" i s) lines

let syphonReset () = ignore (syphonText.Remove(0,syphonText.Length))

let syphon (str:string) = // syphonDump();
    ignore (syphonText.Append(str))  (* ; printf "syphon: %s\n" str *)

let syphonLine filename i =
    if filename<> stdinMockFilename then "" else
    let text = syphonText.ToString()
    // In Visual Studio, when sending a block of text, it  prefixes  with '# <line> "filename"\n'
    // and postfixes with '# 1 "stdin"\n'. To first, get errors filename context,
    // and second to get them back into stdin context (no position stack...).
    // To find an error line, trim upto the last stdinReset string the syphoned text.
    //printf "PrePrune:-->%s<--\n\n" text;
    let rec prune (text:string) =
      let stdinReset = "# 1 \"stdin\"\n"
      let idx = text.IndexOf(stdinReset)
      if idx <> -1 then
        prune (text.Substring(idx + stdinReset.Length))
      else
        text
   
    let text = prune text
    //printf "PostPrune:-->%s<--\n\n" text;
    let lines = text.Split(CompatArray.of_list [ '\n' ])
    if 0 < i && i <= lines.Length then CompatArray.get lines (i-1) else ""
 

//----------------------------------------------------------------------------
// Run "f x" on GUI thread, resorting to "dflt" if failure
//----------------------------------------------------------------------------

let workerThreadRunCodeOnWinFormsMainThread (mainForm : #Control) f = 
    if !progress then dprintf0 "workerThreadRunCodeOnWinFormsMainThread: entry...\n";                  

    // Hack: Mono's Control.Invoke returns a null result.  Hence avoid the problem by 
    // transferring the resulting state using a mutable location.
    // Note ownership of this mutable location gets temporarily transferred between threads,
    // but it is not concurrently read/written because Invoke is blocking.
    let mainFormInvokeResultHolder = ref None

    // Actually, Mono's Control.Invoke isn't even blocking (or wasn't on 1.1.15)!  So use a signal to indicate completion.
    // Indeed, we should probably do this anyway with a timeout so we can report progress from 
    // the GUI thread.
    use doneSignal = new AutoResetEvent(false)

    if !progress then dprintf0 "workerThreadRunCodeOnWinFormsMainThread: invoking...\n";                  
    // BLOCKING: This blocks the stdin-reader thread until the
    // form invocation has completed.  NOTE: does not block on Mono
    mainForm.Invoke(new MethodInvoker(fun () -> 
                               try 
                                  mainFormInvokeResultHolder := Some(f ());
                               finally 
                                  doneSignal.Set() |> ignore)) |> ignore;
    let handles = CompatArray.of_list [ (doneSignal :> WaitHandle) ] 
    if !progress then dprintf0 "workerThreadRunCodeOnWinFormsMainThread: Waiting for completion signal....\n";
    while not (doneSignal.WaitOne(new TimeSpan(0,0,1),true)) do 
      if !progress then dprintf0 "."; stdout.Flush()
    done;
    if !progress then dprintf1 "workerThreadRunCodeOnWinFormsMainThread: Got completion signal, res = %b\n" (Option.is_some !mainFormInvokeResultHolder);
    !mainFormInvokeResultHolder |> Option.get


//----------------------------------------------------------------------------
// Reporting - warnings, errors
//----------------------------------------------------------------------------
 
let ignoreAllErrors f = try f() with _ -> ()

let printError isWarn  err = 
    ignoreAllErrors(fun () -> 
        withErrorColor isWarn  (fun () ->
            stderr.WriteLine();
            buff stderr (output_err_context "  " syphonLine) err; 
            buff stderr (output_err false)  err;
            stderr.WriteLine()))

let reportError  err =
    printError false err
    incr errors;
    raise ReportedError // STOP ON FIRST ERROR (AVOIDS PARSER ERROR RECOVERY)
 
let reportWarning err =
    withErrorColor true (fun () -> 
        if reportWarningAsError warningOptions err then 
            reportError err 
        else if reportWarning warningOptions err then 
            stderr.WriteLine();
            buff stderr (output_err_context "  " syphonLine) err; 
            buff stderr (output_err true) err;
            stderr.WriteLine())

let _ = 
    errorHandler := reportError;
    warningHandler := reportWarning

let errorRecoveryPointMulti(e:exn) =
  try errorRecoveryPoint(e) with ReportedError -> ()


//----------------------------------------------------------------------------
// cmd line - parse options and process inputs
//----------------------------------------------------------------------------

/// Process command line, flags and collect filenames 
/// The parseArgs function calls imperative function to process "real" args 
/// Rather than start processing, just collect names, then process them. 
let sourceFiles = 
    let inputFilesAcc   = ref ([] : (string * bool) list) 
    let collect name = 
        let lower = String.lowercase name in 
        let fsx = Filename.check_suffix lower ".fsx" in
        inputFilesAcc := !inputFilesAcc @ [(name,fsx)] (* O(n^2), but n small... *)
    (try parseArgs collect (specs tcConfig (fsiUsage inputFilesAcc))  (List.tl (Array.to_list Sys.argv)) with e ->errorRecoveryPointMulti e; exit 1);
    !inputFilesAcc

do 
    let firstArg = 
        match sourceFiles with 
        | [] -> Sys.argv.[0] 
        | _ -> fst (List.hd (List.rev sourceFiles) )
    let args = CompatArray.of_list (firstArg :: !explicitArgs) 
    fsi.CommandLineArgs <- args


//----------------------------------------------------------------------------
// Banner
//----------------------------------------------------------------------------

let prompt = if isServer() then "SERVER-PROMPT>\n" else "> "
let banner() = 
    uprintf "\nMSR F# Interactive, (c) Microsoft Corporation, All Rights Reserved\n";
    uprintf "F# Version %s, compiling for .NET Framework Version %s\n\n" Ilxconfig.version (Ilsupp.clrVersion());
    uprintf "NOTE: \n";
    uprintf "NOTE: See 'fsi --help' for flags\n";
    uprintf "NOTE: \n";
    uprintf "NOTE: Commands: #r <string>;;    reference (dynamically load) the given DLL. \n";
    uprintf "NOTE:           #I <string>;;    add the given search path for referenced DLLs. \n";
    uprintf "NOTE:           #use <string>;;  accept input from the given file. \n";
    uprintf "NOTE:           #load <string> ...<string>;;\n";
    uprintf "NOTE:                            load the given file(s) as a compilation unit.\n";
    uprintf "NOTE:           #time;;          toggle timing on/off. \n";
    uprintf "NOTE:           #types;;         toggle display of types on/off. \n";
    uprintf "NOTE:           #quit;;          exit. \n";
    //do Printf.printf "NOTE:           #help          display help. \n"
    uprintf "NOTE: \n";
    uprintf "NOTE: Visit the F# website at http://research.microsoft.com/fsharp.\n";
    uprintf "NOTE: Bug reports to fsbugs@microsoft.com. Enjoy!\n";
    if isServer() then 
        uprintf "NOTE:\n";
        uprintf "NOTE: Visual Studio key bindings:\n";
        uprintf "NOTE:\n";  
        uprintf "NOTE:           Up/Down   = cycle history\n"; 
        uprintf "NOTE:           CTRL-C    = interrupt session\n";
        uprintf "NOTE:           ALT-ENTER = send selected source text to FSI session (adds ;;)\n";   
        ()


//----------------------------------------------------------------------------
// TabletPC warning, in case of imminent system hang!
//----------------------------------------------------------------------------

// Determining Whether a PC is a Tablet PC 
// http://msdn2.microsoft.com/en-us/library/ms700675(VS.85).aspx
// http://blogs.msdn.com/swick/archive/2007/11/04/tabletpc-development-gotchas-part3-semantics-of-getsystemmetrics-sm-tabletpc-a.aspx
[<DllImport("user32.dll", EntryPoint=("GetSystemMetrics"))>]
extern bool GetSystemMetrics(int);
let warningForTabletPC() =
  let SM_TABLETPC = 86
  if GetSystemMetrics(SM_TABLETPC) then
    uprintf "NOTE:\n"
    uprintf "NOTE: For Tablet PC (Input Service) users:\n"
    uprintf "NOTE:   If the 'Tablet PC Input Service' is running, the system might hang.\n"
    uprintf "NOTE:   See the XP hotfix http://support.microsoft.com/kb/925271\n" (* intended no full stop, keep URL clear *)

      
//----------------------------------------------------------------------------
// Set the input/output encoding. The use of a thread is due to a known bug on 
// on Vista where calls to System.Console.InputEncoding can block the process.
//----------------------------------------------------------------------------
 
let setServerCodePages() =     
#if CLI_AT_MOST_1_1
    ()
#else
    match !fsiServerInputCodePage, !fsiServerOutputCodePage with 
    | None,None -> ()
    | inputCodePageOpt,outputCodePageOpt -> 
        let successful = ref false 
        Async.Spawn (async { do match inputCodePageOpt with 
                                | None -> () 
                                | Some(n:int) -> 
                                      let encoding = System.Text.Encoding.GetEncoding(n) in 
                                      // Note this modifies the real honest-to-goodness settings for the current shell.
                                      // and the modifiations hang around even after the process has exited.
                                      System.Console.InputEncoding <- encoding 
                             do match outputCodePageOpt with 
                                | None -> () 
                                | Some(n:int) -> 

                                      let encoding = System.Text.Encoding.GetEncoding(n) in 
                                      // Note this modifies the real honest-to-goodness settings for the current shell.
                                      // and the modifiations hang around even after the process has exited.
                                      System.Console.OutputEncoding <- encoding
                             do successful := true  });
        for pause in [10;50;100;1000;2000;10000] do 
            if not !successful then 
                System.Threading.Thread.Sleep(pause);
        if not !successful then 
            System.Windows.Forms.MessageBox.Show("A problem occurred starting the F# Interactive process. This may be due to a known problem with background process console support for Unicode-enabled applications on some Windows systems. Try selecting Tools->Options->F# Interactive for Visual Studio and enter '--fsi-server-no-unicode'") |> ignore
#endif


//----------------------------------------------------------------------------
// Startup...
//----------------------------------------------------------------------------
                
do if !showBanner then banner()
do if not runningOnMono then warningForTabletPC()
do (try setServerCodePages() with e -> warning(e))
do if isNil sourceFiles then uprintf "\n%s" prompt else uprintf "\nloading..."          


//----------------------------------------------------------------------------
// Startup processing
//----------------------------------------------------------------------------

let consoleOpt = 
#if CLI_AT_MOST_1_1
#else
    try 
        // Probe to see if the console looks functional on this version of .NET
        let _ = System.Console.KeyAvailable 
        let _ = System.Console.ForegroundColor
        let _ = System.Console.CursorLeft <- System.Console.CursorLeft
        Some(new Microsoft.FSharp.Compiler.Interactive.ReadLineConsole(fun (s1,s2) -> Seq.empty))
    with e -> 
        (* warning(Failure("Note: there was a problem setting up custom readline console support. Consider starting fsi.exe with the --no-readline option")); *)
#endif
        None

/// This threading event gets set after the first-line-reader has finished its work
let consoleReaderStartupDone = new ManualResetEvent(false)

/// When using a key-reading console this holds the first line after it is read
let consoleFirstLine : string option ref = ref None

/// Peek on the standard input so that the user can type into it from a console window.
do if !interact then 
      (new Thread(fun () -> 
#if CLI_AT_MOST_1_1
#else
          match consoleOpt with 
          | Some console when !readline && not (isServer()) ->
              if isNil(sourceFiles) then (
                  if !progress then dprintf0 "first-line-reader-thread reading first line...\n";
                  consoleFirstLine := Some(console.ReadLine()); 
                  if !progress then dprintf1 "first-line-reader-thread got first line = %s...\n" (any_to_string !consoleFirstLine);
              );
              consoleReaderStartupDone.Set() |> ignore 
              if !progress then dprintf0 "first-line-reader-thread has set signal and exited.\n" ;
          | _ -> 
#endif
              ignore(Console.In.Peek());
              consoleReaderStartupDone.Set() |> ignore 
        )).Start()

/// FSI does a "startup" interaction to automatically page all the libary information.
/// This is mainly information for the typechecker environment.
/// Printing a prompt first means it should happen while the first line is being entered,
/// so effectively the background.
///
/// This ref controls the pre-printing of the prompt first time around.
let startedProper = ref false

/// Configure for .net version
let desiredCLIMetadataVersion,fullGenericsSupported = Fscopts.configureCLIVersion(false,tcConfig)

//----------------------------------------------------------------------------
// parsing - parseInteraction
//----------------------------------------------------------------------------

let parseInteraction tcConfig tokenizer =
  try 
    if !progress then dprintf0 "In parseInteraction...\n";

    let input = 
      Lexhelp.reusingLexbuf 
        (Lexfilter.get_lexbuf tokenizer)
        None 
        (fun () -> 
          let lexer = (Lexfilter.get_lexer tokenizer)
          let lexbuf = (Lexfilter.get_lexbuf tokenizer)
          Pars.interaction lexer lexbuf)
    Some input
  with e -> errorRecoveryPointMulti e; None      

//----------------------------------------------------------------------------
// tcImports, typechecker globals etc.
//----------------------------------------------------------------------------

let tcGlobals,tcImports =  
  try buildTcImports(tcConfig) 
  with e -> errorRecoveryPointMulti e; exit 1
let _ =  reportPhase "Build Initial TC Imports options"

let mscorlibRefs  = tcGlobals.ilg

//----------------------------------------------------------------------------
// global objects
//----------------------------------------------------------------------------

let niceNameGen = newNiceNameGenerator() 
let rangeStdin = rangeN "stdin" 0

/// Using an AbstractIL resource manager would cause space leaks, so 
/// we don't. 
let managerOpt = None

//----------------------------------------------------------------------------
// Final linking
//----------------------------------------------------------------------------

/// Add attributes 
let createModuleFragment assemblyName codegenResults =
    if !progress then dprintf0 "Creating main module...\n";
    let manifest_module = mk_simple_mainmod assemblyName (fsharpModuleName assemblyName) (!target = Dll) (mk_tdefs codegenResults.ilTypeDefs)
    { manifest_module 
      with modulManifest = 
            (let man = manifest_of_mainmod manifest_module 
            Some { man with  manifestCustomAttrs = mk_custom_attrs codegenResults.ilAssemAttrs }); }


//----------------------------------------------------------------------------
// Compute assembly names
//----------------------------------------------------------------------------

let outfile,pdbfile,assemblyName = "TMPFSCI.exe",None,"FSI-ASSEMBLY"

let asmB,modB = Ilreflect.mkDynamicAssemblyAndModule assemblyName !Fscopts.debuginfo

//----------------------------------------------------------------------------
// InteractionState
//----------------------------------------------------------------------------

type InteractionState =
    { optEnv    : Opt.env;
      emEnv     : Ilreflect.emEnv;
      tcState   : Build.tcState; 
      ilxGenEnv : Ilxgen.ilxGenEnv;
      timing    : bool;
      viewer    : (FormatOptions -> obj -> System.Type -> Layout.layout); }


//----------------------------------------------------------------------------
// processInput
//----------------------------------------------------------------------------

let processInputs i istate inputs showTypes isIncrementalFragment prefixPath =
    let optEnv    = istate.optEnv
    let emEnv     = istate.emEnv
    let tcState   = istate.tcState
    let ilxGenEnv = istate.ilxGenEnv

    // typecheck 
    let tcState,topCustomAttrs,declaredImpls,tcEnvAtEndOfLastInput = 
        try 
            let checkForNoErrors () = (!errors = 0) 
            typecheckClosedInputSet checkForNoErrors tcConfig tcImports tcGlobals (Some prefixPath) tcState inputs
        with e -> errorRecoveryPointMulti e; failwith "Type error"

    // Logging/debugging
    if !printAst then
        let show input =
            let l = any_to_layout FormatOptions.Default input 
            let l = squash_layout FormatOptions.Default l 
            output_layout FormatOptions.Default stdout l
        let (TAssembly(declaredImpls)) = declaredImpls
        for input in declaredImpls do 
            dprintf0 "AST:\n"; show input; dprintf0 "\n"

    reportPhase "Typechecked";
    abortOnError();    
     
    let importMap = { Import.g = tcGlobals; Import.assemMap= findCcu(tcConfig,tcImports,tcGlobals) } 

    // Echo the decls (reach inside wrapping) 
    if showTypes && not tcConfig.noFeedback then  
        let denv = denv_of_tenv tcState.tcsTcImplEnv
        // open the path for the fragment we just compiled 
        let denv = denv_add_open_path (path_of_lid prefixPath) denv 

        let (TAssembly(declaredImpls)) = declaredImpls
        for (TImplFile(qname,mexpr)) in declaredImpls do
            let responseL = NicePrint.inferred_sig_of_structL false denv mexpr 
            if not (Layout.isEmptyL responseL) then      
                uprintf "\n";
                Layout.renderL (Layout.channelR stdout) (Layout.squashTo 80 responseL);
                uprintf "\n"
            // note: no new line not after startup interaction 
            if !startedProper then uprintf "\n";

    // optimize: note we collect the incremental optimization environment 
    let optimizedImpls, optData, optEnv = 
        applyAllOptimizations (tcGlobals,outfile,importMap,isIncrementalFragment)  optEnv (tcState.tcsCcu,declaredImpls)
    abortOnError();
        
    // codegen: note we collect the incremental optimization environment 
    let fragName = text_of_lid prefixPath 
    let codegenResults = generateILX (isIncrementalFragment,true) tcGlobals importMap topCustomAttrs optimizedImpls tcState.tcsCcu fragName ilxGenEnv
    abortOnError();    
    //if assemAttrs <> [] or modulAttrs <> [] then warning(Failure("Assembly attributes are ignored by by F# Interactive"));

    // Each fragment is like a small separately compiled extension to a single source file. 
    // The incremental extension to the environment is dictated by the "signature" of the values as they come out 
    // of the type checker. Hence we add the declaredImpls (unoptimized) to the environment, rather than the 
    // optimizedImpls. 
    let ilxGenEnv = Ilxgen.add_incremental_local_mimpls isIncrementalFragment tcGlobals tcState.tcsCcu fragName ilxGenEnv declaredImpls in

    reportPhase "TAST -> ILX";
    abortOnError();    
        
    (* step *)    
    let ilxMainModule = createModuleFragment assemblyName codegenResults


    reportPhase "Linking";
    abortOnError();    
        
    (* step *)
    let (ilxMainModule : Il.modul) = Ilxerase.conv_module mscorlibRefs managerOpt ilxMainModule
    reportPhase "ILX -> IL"; 
    abortOnError();   
          
    let mainmod3 = Ilmorph.module_scoref2scoref_memoized (normalizeAssemblyRefs tcConfig tcImports) ilxMainModule
    reportPhase "Assembly refs Normalised"; 
    abortOnError();

    if !showILCode then 
        uprintf "\n--------------------\n";
        Ilprint.output_module stdout mainmod3;
        uprintf "\n--------------------\n"

    let emEnv,execs = Ilreflect.emitModuleFragment mscorlibRefs emEnv asmB modB mainmod3
    if !saveEmittedCode then asmB.Save(assemblyName ^ ".dll");
    abortOnError();
    reportPhase "Reflection.Emit";

  #if CLI_AT_MOST_1_1
  #else
    // Explicitly register the resources with the Sreflect module 
    // We would save them as resources into the dynamic assembly but there is missing 
    // functionality System.Reflection for dynamic modules that means they can't be read back out 
    //printf "#resources = %d\n" (length resources);
    for bytes in codegenResults.quotationResourceBytes do 
        Microsoft.FSharp.Quotations.Raw.ExplicitlyRegisterTopDefs (asmB :> Assembly) fragName bytes;
        
  #endif

    timeReporter.TimeOpIf istate.timing (fun () -> 
      execs.Iterate(fun exec -> 
        match exec() with 
        | Some(e) -> eprintf "%s\n" (e.ToString()); errors := 1; abortOnError() 
        | None -> ())) ;

    abortOnError();
    reportPhase "Run Bindings";

    let istate = {istate with  optEnv    = optEnv;
                               emEnv     = emEnv;
                               ilxGenEnv = ilxGenEnv;
                               tcState   = tcState  }
    istate,tcEnvAtEndOfLastInput


//----------------------------------------------------------------------------
// evalDefns, evalExpr
//----------------------------------------------------------------------------

let mkId      str  = {idText  = str;idRange = rangeStdin}
let modIdN    i = mkId ("FSI_" ^ sprintf "%04d" i) // shows exn traces, make clear and fixed width 
let newModPathN  i = [modIdN i]
let genIntI = let i = ref 0 in fun () -> incr i; !i 

let evalInputsFromLoadedFiles istate inputs =
    let i = genIntI()
    let prefix = newModPathN i 
    // Hack the path to include the qualifying name 
    let inputs = inputs |> List.map (prepend_path_to_input prefix) 
    let istate,_ = processInputs i istate inputs true false prefix 
    istate

let evalDefns istate showTypes defs =
    let ml       = false
    let filename = "stdin"
    let i = genIntI()
    let prefix = newModPathN i
    let prefixPath = path_of_lid prefix
    let impl = ModuleImpl (prefix,
                           true, (* true indicates module, false namespace *)
                           defs,
                           emptyXMLDoc,[],
                           None,
                           rangeStdin)
    let input = ImplFileInput(ImplFile(filename,(qname_of_unique_path rangeStdin prefixPath),[impl]))
    let istate,tcEnvAtEndOfLastInput = processInputs i istate [input] showTypes true prefix
    let tcState = istate.tcState 
    { istate with tcState = { tcState with tcsTcSigEnv = tcEnvAtEndOfLastInput;
                                           tcsTcImplEnv = tcEnvAtEndOfLastInput } } 
 
let evalExpr istate expr =
    let m = rangeStdin
    let itName = "it" 
    let itID  = mksyn_id m itName
    let itExp = Expr_lid_get (false,[itID],m)
    let exprA = expr
    let exprB = Expr_app(Expr_lid_get(false,map mkId saverPath,m), itExp,m)
    let mkBind pat expr = Binding(None,DoBinding,false,false,[],XMLDoc [],None,pat,BindingExpr([],None,expr),m)
    let bindingA = mkBind (mksyn_pat_var None itID) exprA (* let it = <expr> *)
    let bindingB = mkBind (Pat_wild m)         exprB (* let _  = <istate.viewer> it *)
    let defA = Def_let (false, [bindingA], m)
    let defB = Def_let (false, [bindingB], m)
    let istate = evalDefns istate false [defA;defB] 
    // Snarf the type for 'it' via the binding
    begin match istate.tcState.tcsTcImplEnv |> items_of_tenv |> Namemap.find itName with 
    | Item_val vref -> if not tcConfig.noFeedback then invokePrinter istate.viewer (denv_of_tenv istate.tcState.tcsTcImplEnv) (deref_val vref)
    | _ -> ()
    end;
    istate

let evalRequireDll istate m path = 
    // Check the file can be resolved before calling requireDLLReference 
    let _ = resolveLibFile tcConfig m path 
    requireDLLReference tcConfig m path;
    let tcState = istate.tcState 
    let tcEnv,(dllinfos,ccuinfos) = requireDLL tcConfig tcImports tcGlobals tcState.tcsTcImplEnv m path 
    let optEnv = List.fold_left add_external_ccu_to_optEnv istate.optEnv ccuinfos
    let ilxGenEnv = add_external_ccus tcGlobals istate.ilxGenEnv (ccuinfos |> List.map (fun ccuinfo -> ccuinfo.ccuinfo_ccu)) 
    dllinfos,
    { istate with tcState = {tcState with tcsTcSigEnv = tcEnv; tcsTcImplEnv = tcEnv };
                  optEnv = optEnv;
                  ilxGenEnv = ilxGenEnv }

let withImplicitHome dir tcConfig f = 
    let old = tcConfig.implicitIncludeDir 
    tcConfig.implicitIncludeDir <- dir;
    try f() 
    finally tcConfig.implicitIncludeDir <- old
  
let processMetaCommandsFromInputAsInteractiveCommands istate sourceFile inp =
    withImplicitHome (Filename.dirname sourceFile) tcConfig (fun () ->
       processMetaCommandsFromInput ((fun st m nm -> snd (evalRequireDll st m nm)),(fun st -> requireDLLReference tcConfig))  tcConfig inp istate)
  
let evalLoadFiles istate m sourceFiles =
    match sourceFiles with 
    | [] -> istate
    | _ -> 
      // use source file as if it were a module 
      uprintf "."; stdout.Flush();
      let sourceFiles = sourceFiles |> List.map (resolveSourceFile tcConfig m)
      let inputs = sourceFiles |> List.map (Fscopts.processInput tcConfig ["INTERACTIVE"] 0)  
      abortOnError();
      if List.exists (function None -> true | _ -> false) inputs then failwith "parse error";
      let inputs = List.map Option.get inputs 
      uprintf "."; stdout.Flush();
      let istate = List.fold_left2 processMetaCommandsFromInputAsInteractiveCommands istate sourceFiles inputs
      evalInputsFromLoadedFiles istate inputs  

//----------------------------------------------------------------------------
// identifier completion - namedItemInEnvL
//----------------------------------------------------------------------------

let completionsForPartialLID istate (prefix:string) =
    let lid,stem =
        if prefix.IndexOf(".") >= 0 then
            let parts = prefix.Split(CompatArray.of_list ['.'])
            let n = parts.Length
            CompatArray.sub parts 0 (n-1) |> CompatArray.to_list,CompatArray.get parts (n-1)
        else
            [],prefix   
    let tcState = istate.tcState (* folded through now? *)
    let amap = { Import.g = tcGlobals; Import.assemMap = findCcu(tcConfig,tcImports,tcGlobals) } 
    let m = rangeStdin
    let plid = lid
    let nItems = ptc_lid tcGlobals amap m (nenv_of_tenv tcState.tcsTcImplEnv) plid
    let names  = map name_of_item nItems
    let names  = filter (fun (name:string) -> name.StartsWith(stem)) names
    names

//----------------------------------------------------------------------------
// ctrl-c handling
//----------------------------------------------------------------------------

type ControlEventHandler = delegate of int -> bool

[<DllImport("kernel32.dll")>]
extern void SetConsoleCtrlHandler(ControlEventHandler callback,bool add)

// One strange case: when a TAE happens a strange thing 
// occurs the next read from stdin always returns
// 0 bytes, i.e. the channel will look as if it has been closed.  So we check
// for this condition explicitly.  We also recreate the lexbuf whenever CtrlC kicks.
type StdinState = StdinEOFPermittedBecauseCtrlCRecentlyPressed | StdinNormal
let stdinState = ref StdinNormal
let CTRL_C = 0 
let mainThread = Thread.CurrentThread 

type SignalProcessorState =  
    | CtrlCCanRaiseException 
    | CtrlCIgnored 

type KillerThreadRequest =  
    | ThreadAbortRequest 
    | NoRequest 
    | ExitRequest 
    | PrintInterruptRequest

let ctrlcState = ref CtrlCIgnored
let killThreadRequest = ref NoRequest
let ctrlEventHandlers = ref [] : ControlEventHandler list ref
let ctrlEventActions  = ref [] : (unit -> unit) list ref

// Currently used only on Mono/Posix
type PosixSignalProcessor() = 
    let mutable reinstate = (fun () -> ())
    member x.Reinstate with set(f) = reinstate <- f
    member x.Invoke(n:int) = 
         // we run this code once with n = -1 to make sure it is JITted before execution begins
         // since we are not allowed to JIT a signal handler.  THis also ensures the "Invoke"
         // method is not eliminated by dead-code elimination
         if n >= 0 then 
             reinstate();
             stdinState := StdinEOFPermittedBecauseCtrlCRecentlyPressed;
             killThreadRequest := if (!ctrlcState = CtrlCCanRaiseException) then ThreadAbortRequest else PrintInterruptRequest
   
// REVIEW: streamline all this code to use the same code on Windows and Posix.   
let installCtrlCHandler((threadToKill:Thread),(pauseMilliseconds:int)) = 
    if !progress then dprintf0 "installing CtrlC handler\n";
    // WINDOWS TECHNIQUE: .NET has more safe points, and you can do more when a safe point. 
    // Hence we actually start up the killer thread within the handler. 
    try 
        let raiseCtrlC() = 
            Printf.eprintf "\n- Interrupt\n";  
            stdinState := StdinEOFPermittedBecauseCtrlCRecentlyPressed;
            if (!ctrlcState = CtrlCCanRaiseException) then 
                killThreadRequest := ThreadAbortRequest;
                let killerThread = 
                    new Thread(new ThreadStart(fun () ->
                        // sleep long enough to allow ControlEventHandler handler on main thread to return 
                        // Also sleep to give computations a bit of time to terminate 
                        Thread.Sleep(pauseMilliseconds);
                        if (!killThreadRequest = ThreadAbortRequest) then 
                            if !progress then uprintf "\n- Aborting main thread...\n";  
                            killThreadRequest := NoRequest;
                            threadToKill.Abort();
                        ())) 
                killerThread.IsBackground <- true;
                killerThread.Start() 
    
        let ctrlEventHandler = new ControlEventHandler(fun i ->  if i = CTRL_C then (raiseCtrlC(); true) else false ) 
        ctrlEventHandlers := ctrlEventHandler :: !ctrlEventHandlers;
        ctrlEventActions  := raiseCtrlC       :: !ctrlEventActions;
        SetConsoleCtrlHandler(ctrlEventHandler,true);
        false // don't exit via kill thread
    with e -> 
        if !progress then eprintf "Failed to install ctrl-c handler using Windows technique - trying to install one using Unix signal handling...\n";
        // UNIX TECHNIQUE: We start up a killer thread, and it watches the mutable reference location.    
        // We can't have a dependency on Mono DLLs (indeed we don't even have them!)
        // So SOFT BIND the following code:
        // Mono.Unix.Native.Stdlib.signal(Mono.Unix.Native.Signum.SIGINT,new Mono.Unix.Native.SignalHandler(fun n -> PosixSignalProcessor.Invoke(n))) |> ignore;
        match (try Choice2_1(System.Reflection.Assembly.Load(
#if CLI_AT_MOST_1_1
                                                              "Mono.Posix, Version=1.0.5000.0, Culture=neutral, PublicKeyToken=0738eb9f132ed756"
#else
                                                              "Mono.Posix, Version=2.0.0.0, Culture=neutral, PublicKeyToken=0738eb9f132ed756"
#endif
                    )) with e -> Choice2_2 e) with 
        | Choice2_1(monoPosix) -> 
          try
            if !progress then eprintf "loading type Mono.Unix.Native.Stdlib...\n";
            let monoUnixStdlib = monoPosix.GetType("Mono.Unix.Native.Stdlib") 
            if !progress then eprintf "loading type Mono.Unix.Native.SignalHandler...\n";
            let monoUnixSignalHandler = monoPosix.GetType("Mono.Unix.Native.SignalHandler") 
            if !progress then eprintf "creating delegate...\n";
            let target = new PosixSignalProcessor() 
            target.Invoke(-1);
            let monoHandler = System.Delegate.CreateDelegate(monoUnixSignalHandler,target,"Invoke") 
            if !progress then eprintf "registering signal handler...\n";
            let monoSignalNumber = System.Enum.Parse(monoPosix.GetType("Mono.Unix.Native.Signum"),"SIGINT")
            let register () = callStaticMethod monoUnixStdlib "signal" [ monoSignalNumber; box monoHandler ]  |> ignore 
            target.Reinstate <- register;
            register();
            let killerThread = 
                new Thread(new ThreadStart(fun () ->
                    while true do 
                        //Printf.eprintf "\n- kill thread loop...\n"; stderr.Flush();  
                        Thread.Sleep(pauseMilliseconds*2);
                        match !killThreadRequest with 
                        | PrintInterruptRequest -> 
                            Printf.eprintf "\n- Interrupt\n"; stderr.Flush();  
                            killThreadRequest := NoRequest;
                        | ThreadAbortRequest -> 
                            Printf.eprintf "\n- Interrupt\n"; stderr.Flush();  
                            if !progress then uprintf "\n- Aborting main thread...\n";  
                            killThreadRequest := NoRequest;
                            threadToKill.Abort()
                        | ExitRequest -> 
                            // Mono has some wierd behaviour where it blocks on exit
                            // once CtrlC has ever been pressed.  Who knows why?  Perhaps something
                            // to do with having a signal handler installed, but it only happens _after_
                            // at least one CtrLC has been pressed.  Maybe raising a ThreadAbort causes
                            // exiting to have problems.
                            //
                            // Anyway, we make "#q" work this case by setting ExitRequest and brutally calling
                            // the process-wide 'exit'
                            Printf.eprintf "\n- Exit...\n"; stderr.Flush();  
                            callStaticMethod monoUnixStdlib "exit" [ box 0 ] |> ignore
                        | _ ->  ()
                    done)) 
            killerThread.IsBackground <- true;
            killerThread.Start();
            true // exit via kill thread to workaround block-on-exit bugs with Mono once a CtrlC has been pressed
          with e -> 
            eprintf "Failed to install ctrl-c handler. Ctrl-C handling will not be available (sorry). Error was:\n\t%s" e.Message 
            false
        | Choice2_2 e ->
          eprintf "Failed to install ctrl-c handler - Ctrl-C handling will not be available (sorry). Error was:\n\t%s" e.Message 
          false  

//----------------------------------------------------------------------------
// assembly finder
//----------------------------------------------------------------------------

do  AppDomain.CurrentDomain.add_AssemblyResolve(new ResolveEventHandler(fun _ args -> 
   try 
       // Grab the name of the assembly
       let assemName = CompatArray.get (args.Name.Split(CompatArray.of_list [ ',' ])) 0
       if !progress then uprintf "ATTEMPT MAGIC LOAD ON ASSEMBLY, assemName = %s\n" assemName;
       
       // Special case: Mono Windows Forms attempts to load an assembly called something like "Windows.Forms.resources"
       // We can't resolve this, so don't try.
       if assemName.EndsWith(".resources") then null else

       // Special case: Is this the global unique dynamic assembly for FSI code? In this case just
       // return the dynamic assembly itself.       
       if assemblyName = assemName then (asmB :> Reflection.Assembly) else

       // Otherwise continue
       let fileName = (assemName + ".dll") 
       let workingPath = 
           try Some(tryResolveLibFile tcConfig rangeStdin fileName)
           with FileNameNotResolved err -> 

               if !progress then uprintf "ATTEMPT LOAD (referencedDLLs), fileName = %s\n" fileName;
               /// Take a look through the files quoted, perhaps with explicit paths
               match (tcConfig.referencedDLLs 
                      |> List.first (fun (_,referencedDLL) -> 
                         if !progress then uprintf "ATTEMPT MAGIC LOAD ON FILE, referencedDLL = %s\n" referencedDLL;
                         if System.String.Compare(Filename.basename referencedDLL, fileName,true) = 0 then
                           try Some(tryResolveLibFile tcConfig rangeStdin referencedDLL)
                           with FileNameNotResolved err -> None
                         else None)) with
               | Some res -> Some res
               | None -> 
                   // Raise the exception wrapped by FileNameNotResolved. However give other assembly resolvers a chance to 
                   // resolve the assembly first by simply adding a failing handler to the end of the AssemblyResolve chain. 
                   // This is a specific request from customers who customize the 
                   // AssemblyResolve mechanism to do magic things like going to a distributed company file store
                   // to pick up DLLs.
                   //
                   // This is also a fix for bug 1171.
                   let rec failingResolveHandler = 
                        new ResolveEventHandler(fun _ -> 
                            // OK, we're running now so remove ourself from the list
                            (try AppDomain.CurrentDomain.remove_AssemblyResolve(failingResolveHandler) with _ -> ());
                            // now raise the exception
                            raise err)
                   AppDomain.CurrentDomain.add_AssemblyResolve(failingResolveHandler);
                   None
                   
       match workingPath with 
       | None -> null
       | Some path -> 
           if assemName <> "Mono.Posix" then uprintf "Binding session to '%s'...\n" path;
           System.Reflection.Assembly.LoadFrom(path)
           
   with e -> 
       errorRecoveryPointMulti(e); 
       null));;


//----------------------------------------------------------------------------
// interactive loop
//----------------------------------------------------------------------------

type stepStatus = CtrlC | EndOfFile | Completed | CompletedWithReportedError

let interactiveCatch f istate = 
    try
        (* reset error count *)
        errors := 0;  
        f istate
    with  e -> 
        errorRecoveryPointMulti(e); 
        istate,CompletedWithReportedError

//----------------------------------------------------------------------------
// Process one parsed interaction.  This runs on the GUI thread.
// It might be simpler if it ran on the parser thread.
//----------------------------------------------------------------------------

let rec mainThreadProcessParsedInteraction exitViaKillThread processInteractiveFile action istate = 
    try 
        if !progress then dprintf0 "In mainThreadProcessParsedInteraction...\n";                  
        ctrlcState := CtrlCCanRaiseException;
        let res = 
            // REVIEW: unify this processing of meta-commands with the two techniques used to process
            // the meta-commands from files.
            match action with 
            | Some (IDefns ([  ],_)) ->
                istate,Completed
            (* Detect the encoding for 'naked expressions' - see mk_Do in pars.mly *)    
            | Some (IDefns ([  Def_let(false,[Binding (None,StandaloneExpression,false,false,[],_,None,(Pat_wild _),BindingExpr([],None,expr),m)], _) ],_)) ->
                evalExpr  istate expr,Completed
            | Some (IDefns (defs,m))             -> 
                evalDefns istate true defs,Completed
            | Some (IHash ("use",[sourceFile],m))      -> 
                processInteractiveFile istate (sourceFile,m)
            | Some (IHash ("load",sourceFiles,m))      -> 
                evalLoadFiles istate m sourceFiles,Completed
            | Some (IHash (("reference" | "r"),[path],m))        -> 
                let dllinfos,istate = evalRequireDll istate m path 
                dllinfos |> List.iter (fun dllinfo -> uprintf "\n--> Referenced '%s'\n\n" dllinfo.dllinfo_filename);
                istate,Completed
            | Some (IHash ("I",[path],m))        -> 
                requireIncludePath tcConfig m path; 
                uprintf "\n--> Added '%s' to library include path\n\n" (makeAbsolute tcConfig path);
                istate,Completed
            | Some (IHash ("cd",[path],m))        -> 
                let path = (makeAbsolute tcConfig path) 
                if Directory.Exists(path) then 
                  tcConfig.implicitIncludeDir <- path
                else error(Error(sprintf "Directory '%s' doesn't exist" path,m));
                istate,Completed
            | Some (IHash ("time",[],m))       -> 
                uprintf "\n--> Timing now %s\n\n" (if istate.timing then "off" else "on");
                {istate with timing = not (istate.timing)},Completed
            | Some (IHash ("nowarn",[d],m))      -> 
                turnWarnOff tcConfig m d; istate,Completed
            | Some (IHash ("terms",[],m))      -> 
                showTerms := not (!showTerms); istate,Completed
            | Some (IHash ("types",[],m))      -> 
                showTypes := not (!showTypes); istate,Completed
            | Some (IHash ("ilcode",[],m))     -> 
                showILCode := not (!showILCode); istate,Completed
            | Some (IHash ("savedll",[],m))    -> 
                saveEmittedCode := true; istate,Completed
            | Some (IHash ("nosavedll",[],m))  -> 
                saveEmittedCode := true; istate,Completed
            | Some (IHash (("q" | "quit"),[],m))          -> 
                if exitViaKillThread then 
                    killThreadRequest := ExitRequest;
                    Thread.Sleep(1000)
                exit 0;
                
            | Some (IHash (c,arg,m))           -> 
                uprintf "Invalid command #%s %s\n" c (String.concat " " arg); 
                istate,Completed
            | None                           -> 
                istate,Completed 
        killThreadRequest := NoRequest;
        ctrlcState := CtrlCIgnored;
        res
    with
    | :? ThreadAbortException ->
       killThreadRequest := NoRequest;
       ctrlcState := CtrlCIgnored;
       (try Thread.ResetAbort() with _ -> ());
       (istate,CtrlC)
     |  e ->
       killThreadRequest := NoRequest;
       ctrlcState := CtrlCIgnored;
       errorRecoveryPointMulti(e);
       istate,CompletedWithReportedError

//----------------------------------------------------------------------------
// Parse then process one parsed interaction.  This initially runs on the parser
// thread, then calls runCodeOnMainThread to run on the GUI thread. 
// 'processAndRunOneInteractionFromLexbuf' calls the runCodeOnMainThread when it has completed 
// parsing and needs to typecheck and execute a definition.  Type-checking and execution 
// happens on the GUI thread.
//----------------------------------------------------------------------------

let rec processAndRunOneInteractionFromLexbuf exitViaKillThread runCodeOnMainThread istate tokenizer =
    let lexbuf = (Lexfilter.get_lexbuf tokenizer)
    if lexbuf.IsPastEndOfStream then 
        istate,(if !stdinState = StdinEOFPermittedBecauseCtrlCRecentlyPressed then (stdinState := StdinNormal; CtrlC) 
                else EndOfFile)
    else 
        if !startedProper then uprintf "%s" prompt; (* not after startup interaction *)
        startedProper := true; (* but next time around, back to normal *)
        istate |> interactiveCatch (fun istate -> 
            // BLOCKING POINT
            // When FSI.EXE is waiting for input from the console the 
            // parser thread is blocked somewhere deep this call. *)
            if !progress then dprintf0 "entering parseInteraction...\n";
            let action  = parseInteraction tcConfig tokenizer
            if !progress then dprintf0 "returned from parseInteraction...\n";
            // After we've unblocked and got something to run we switch 
            // over to the run-thread (e.g. the GUI thread) 
            if !progress then dprintf0 "calling runCodeOnMainThread...\n";
            let processInteractiveFile = mainThreadProcessInteractiveFile exitViaKillThread 
            let res = runCodeOnMainThread (mainThreadProcessParsedInteraction exitViaKillThread processInteractiveFile action) istate 
            if !progress then dprintf1 "Just called runCodeOnMainThread, res = %O...\n" res;
            res)
    
and mainThreadProcessInteractiveFile exitViaKillThread istate (sourceFile,m) =
    // Resolve the filename to an absolute filename
    let sourceFile = sourceFile |> resolveSourceFile tcConfig m 
    // During the processing of the file, further filenames are 
    // resolved relative to the home directory of the loaded file.
    withImplicitHome (Filename.dirname sourceFile) tcConfig (fun () ->
        // use source file containing maybe several ;;-interaction blocks 
        Unilex.usingUnicodeFileAsUTF8Lexbuf sourceFile tcConfig.inputCodePage (fun lexbuf ->
            Lexhelp.resetLexbufPos sourceFile lexbuf;
            let lightSyntaxStatus = LightSyntaxStatus (ref tcConfig.light)
            let skip = true 
            let defines = "INTERACTIVE"::tcConfig.ifdefs
            let lexargs = mkLexargs ((fun () -> tcConfig.implicitIncludeDir),sourceFile,defines, lightSyntaxStatus) in 
            let tokenizer = Lexfilter.create lightSyntaxStatus (Lex.token lexargs skip) lexbuf
            let rec run istate =
                let istate,cont = processAndRunOneInteractionFromLexbuf exitViaKillThread (fun f istate -> f istate) istate tokenizer
                if cont = Completed then run istate else istate,cont 
            let istate,cont = run istate 
            match cont with
            | Completed -> failwith "mainThreadProcessInteractiveFile: Completed expected to have relooped"
            | CompletedWithReportedError -> istate,CompletedWithReportedError
            | EndOfFile -> istate,Completed (* here file-EOF is normal, continue required *)
            | CtrlC     -> istate,CtrlC
      ))

//----------------------------------------------------------------------------
// GUI runCodeOnMainThread
//----------------------------------------------------------------------------

//type InteractionStateConverter = delegate of InteractionState -> InteractionState * stepStatus

///Use a dummy to access protected member
type DummyForm() = 
    inherit Form() as base
    member x.DoCreateHandle() = base.CreateHandle() 

//----------------------------------------------------------------------------
// initial state and welcome
//----------------------------------------------------------------------------

let initialInteractiveState =
    let _ =  reportPhase "Up until build inital optimization environment" 
    let optEnv0 = optEnvInit tcImports
    let _ =  reportPhase "Build initial optimization environment" 
    let emEnv = Ilreflect.emEnv0
    let tcEnv = initTcEnv rangeStdin tcConfig tcImports tcGlobals 
    let _ =  reportPhase "initTcEnv" 
    let ccuName = assemblyName 
    let tcState = typecheckInitialState rangeStdin ccuName tcConfig tcGlobals niceNameGen  tcEnv
    let ilxgenEnv0 = ilxgenEnvInit tcConfig tcImports tcGlobals tcState.tcsCcu 
    let _ =  reportPhase "ilxgenEnvInit" 
    {optEnv    = optEnv0;
     emEnv     = emEnv;
     tcState   = tcState;
     ilxGenEnv = ilxgenEnv0;
     timing    = false;
     viewer    = printVal } 

//----------------------------------------------------------------------------
// interactive state ref - most recent istate 
//----------------------------------------------------------------------------

let istateRef = ref initialInteractiveState
  
// Update the console completion function now we've got an initial type checking state.
// This means completion doesn't work until the initial type checking state has finished loading - fair enough!
begin
#if CLI_AT_LEAST_2_0
    match consoleOpt with 
    | Some console when !readline -> 
        console.Complete <- (fun (s1,s2) -> 
                                completionsForPartialLID !istateRef 
                                  (match s1 with 
                                   | Some s -> s + "." + s2 
                                   | None -> s2) 
                                |> Seq.of_list)
    | _ -> 
#endif
      ()
end

//----------------------------------------------------------------------------
// Reading stdin 
//----------------------------------------------------------------------------

let lexbufFromLineReader (enc:> Encoding) readf = 
    Lexing.from_function (fun s len -> 
        //dprintf0 "Calling ReadLine\n";
        let text = try Some(readf()) with :? EndOfStreamException -> None
        text |> Option.iter (fun t -> syphon (t + "\n"));
        match text with 
        |  Some(null) | None -> 
             if !progress then dprintf0 "End of file from TextReader.ReadLine\n";
             0
        | Some text ->
            let bytes = enc.GetBytes(text + "\n" : string) 
            let nbytes = CompatArray.length bytes 
            if nbytes > len then eprintf "Warning: line too long, ignoring some characters\n";
            let ntrimmed = min len nbytes 
            CompatArray.blit bytes 0 s 0 ntrimmed;
            ntrimmed)

     
#if CLI_AT_MOST_1_1
#else
let TrySetUnhandledExceptionMode() = 
  let i = ref 0
  try 
    Application.SetUnhandledExceptionMode(UnhandledExceptionMode.CatchException) 
    incr i;incr i;incr i;incr i;incr i;incr i;
  with _ -> decr i;decr i;decr i;decr i;()
#endif

//----------------------------------------------------------------------------
// Reading stdin as a lex stream
//----------------------------------------------------------------------------

let stdinLightSyntaxStatus = LightSyntaxStatus (ref tcConfig.light)

let mkStdinTokenizer () =
    let lexbuf = 
#if CLI_AT_MOST_1_1
#else
        match consoleOpt with 
        | Some console when !readline && not (isServer()) -> 
            lexbufFromLineReader Encoding.UTF8 (fun () -> match !consoleFirstLine with Some l -> (consoleFirstLine := None; l) | None -> console.ReadLine()) 
        | _ -> 
#endif
            lexbufFromLineReader Encoding.UTF8 (fun () -> System.Console.In.ReadLine())
            //lexbufFromTextReader Encoding.UTF8 System.Console.In
    Lexhelp.resetLexbufPos stdinMockFilename lexbuf;
    syphonReset();
    let lexargs = mkLexargs ((fun () -> tcConfig.implicitIncludeDir),"stdin","INTERACTIVE"::tcConfig.ifdefs,stdinLightSyntaxStatus) in 
    let skip = true  (* don't report whitespace from lexer *)
    (* A single hardWhite tokenizer must be shared for the entire *)
    (* use of this lexbuf. *)
    let tokenizer = Lexfilter.create stdinLightSyntaxStatus (Lex.token lexargs skip) lexbuf
    tokenizer

//do Console.Out.Encoding <- Encoding.UTF8
//do Console.Error.Encoding <- Encoding.UTF8

//----------------------------------------------------------------------------
// main()
//----------------------------------------------------------------------------
 
let main () = 
  if !progress then dprintf0 "fsi : main()\n";
  let initial exitViaKillThread istate = 
(*       let _,istate = evalRequireDll istate rangeStdin (fsiaux()^".dll")   *)
      let resolver = autoModuleResolver tcConfig tcGlobals tcImports 
(*
      let istate = { istate with tcState = let tcEnv = implicitOpen resolver rangeStdin istate.tcState.tcsTcImplEnv fsiAuxSettingsModulePath in 
                                           { istate.tcState with 
                                                     tcsTcSigEnv = tcEnv;
                                                     tcsTcImplEnv = tcEnv } }
      let _ =  reportPhase "Load fsiaux dll" 
*)
      let istate = 
          let rec consume istate sourceFiles = 
              match sourceFiles with
              | [] -> istate
              | (_,fsx1) :: _ -> 
                  let sourceFiles,rest = list_take_until (fun (_,fsx2) -> fsx1 <> fsx2) sourceFiles 
                  let sourceFiles = map fst sourceFiles 
                  let istate = 
                      if fsx1 
                      then List.fold_left (fun istate sourceFile -> mainThreadProcessInteractiveFile exitViaKillThread istate (sourceFile,rangeStdin) |> fst) istate sourceFiles 
                      else istate |> interactiveCatch (fun istate -> evalLoadFiles istate rangeStdin sourceFiles, Completed) |> fst in 
                  consume istate rest in
           consume istate sourceFiles 
      let _ =  reportPhase "Up until load of command-line files" 
      if nonNil(sourceFiles) then uprintf "\n%s" prompt;
      let _ =  reportPhase "Load of command-line files" 
      istate 

  let istate = initialInteractiveState
  if !interact then 
      // page in the type check env 
      let istate = istate |> interactiveCatch (fun istate ->  evalDefns istate true [],Completed) |> fst
      if !progress then dprintf0 "MAIN: installed CtrlC handler!\n";
      let exitViaKillThread = installCtrlCHandler(mainThread,(if !gui then 400 else 100)) 
      if !progress then dprintf0 "MAIN: got initial state, creating form\n";
      if !gui then 
#if CLI_AT_MOST_1_1
#else
          do (try Application.EnableVisualStyles() with _ -> ())

          Application.add_ThreadException(new ThreadExceptionEventHandler(fun _ args -> fsi.ReportThreadException(args.Exception)));
          if not runningOnMono then (try TrySetUnhandledExceptionMode() with _ -> ());
#endif

          // This is the event loop for winforms
          let evLoop = 
              let mainForm = new DummyForm() 
              mainForm.DoCreateHandle();
              // Set the default thread exception handler
              fsi.ThreadException.Add(fun exn -> workerThreadRunCodeOnWinFormsMainThread mainForm (fun () -> printError true exn));
              let restart = ref false
              { new Microsoft.FSharp.Compiler.Interactive.IEventLoop with
                   member x.Run() =  
                       restart := false;
                       if !progress then dprintf0 "MAIN: Calling Application.Run...\n";
                       Application.Run()
                       if !progress then dprintf0 "MAIN: Returned from Application.Run...\n";
                       !restart
                   member x.Invoke(f) : 'a =   workerThreadRunCodeOnWinFormsMainThread mainForm f  

                   member x.ScheduleRestart()  =   restart := true; Application.Exit() } 
          fsi.EventLoop <- evLoop;
                                      
      let istate = initial exitViaKillThread istate
      if !progress then dprintf0 "creating stdinReaderThread\n";
      let tokenizer = ref (mkStdinTokenizer())
      let stdinReaderThread = 
        istateRef := istate;
        let cont = ref Completed 
        new Thread(new ThreadStart(fun () ->
            try 
               try 
                  if !progress then dprintf0 "READER: stdin thread started...\n";

                  // Delay until we've peeked the input or read the entire first line
                  WaitHandle.WaitAll(Compatibility.CompatArray.of_array  [| (consoleReaderStartupDone :> WaitHandle) |]) |> ignore;
                  
                  if !progress then dprintf0 "READER: stdin thread got first line...\n";

                  // The main stdin loop, running on the stdinReaderThread.
                  // 
                  // The function 'processAndRunOneInteractionFromLexbuf' is blocking: it reads stdin 
                  // until one or more real chunks of input have been received. 
                  //
                  // We run the actual computations for each action on the main GUI thread by using
                  // mainForm.Invoke to pipe a message back through the form's main event loop. (The message 
                  // is a delegate to execute on the main Thread)
                  //
                  while (!cont = CompletedWithReportedError or !cont = Completed or !cont = CtrlC) do
                      if (!cont = CtrlC) then 
                          tokenizer := mkStdinTokenizer();
                      let istate',cont' = 
                          let runCodeOnMainThread f istate = 
                              try fsi.EventLoop.Invoke (fun () -> f istate) 
                              with _ -> (istate,Completed)
                              
                          processAndRunOneInteractionFromLexbuf exitViaKillThread runCodeOnMainThread !istateRef !tokenizer   
                      istateRef := istate'; 
                      cont := cont';
                      if !progress then dprintf1 "READER: cont = %O\n" !cont;
                  done ;
                  if !progress then dprintf0 "\n- READER: Exiting stdinReaderThread\n";  
                with e -> errorRecoveryPointMulti(e);
            finally 
                (if !progress then dprintf0 "\n- READER: Exiting process because of failure/exit on  stdinReaderThread\n";  
                 exit 1)
        ))
      // stdinReaderThread.IsBackground <- true; 
      if !progress then dprintf0 "MAIN: starting stdin thread...\n";
      stdinReaderThread.Start();


      let rec runLoop() = 
          if !progress then dprintf0 "GUI thread runLoop\n";
          let restart = 
              try 
                // BLOCKING POINT: The GUI Thread spends most (all) of its time this event loop
                if !progress then dprintf0 "MAIN:  entering event loop...\n";
                fsi.EventLoop.Run()
              with
              |  :? ThreadAbortException ->
                // If this TAE handler kicks it's almost certainly too late to save the
                // state of the process - the state of the message loop may have been corrupted 
                uprintf "\n- Unexpected ThreadAbortException (Ctrl-C) during event handling: Trying to restart...\n";  
                (try Thread.ResetAbort() with _ -> ());
                true
                // Try again, just case we can restart
              | e -> 
                errorRecoveryPointMulti e;
                true
                // Try again, just case we can restart
          if !progress then dprintf0 "MAIN:  exited event loop...\n";
          if restart then runLoop() 

      runLoop();
      ()
  else // not interact
      let istate = initial false istate
      exit (min !errors 1)

//----------------------------------------------------------------------------
// Server mode:
//----------------------------------------------------------------------------

(*
#if NOTAILCALLS // NOTE: !NOTAILCALLS <--> support Mono
#else 
*)
let spawn f =
    let th = new Thread(new ThreadStart(f))
    th.IsBackground <- true;
    th.Start()
  
let spawnServer() =   
    //Printf.printf "Spawning fsi server on channel '%s'" !fsiServerName;
    spawn (fun () ->
             let server =
                 {new Server.Shared.FSharpInteractiveServer()
                  with Interrupt() = //printf "FSI-SERVER: received CTRL-C request...\n";
                                     !ctrlEventActions |> List.iter (fun act -> act()) 
                  and  Completions(prefix) = completionsForPartialLID !istateRef prefix  |> List.to_array }
             Server.Shared.startServer !fsiServerName server)
  
do if isServer() then spawnServer() 
(*#endif*)
   
//----------------------------------------------------------------------------
// STAThread
// Mark the main thread as STAThread since it is a GUI thread
//----------------------------------------------------------------------------

[<STAThread()>]    
do main()

//----------------------------------------------------------------------------
// Misc
// The Ctrl-C exception handler that we've passed to native code has
// to be explicitly kept alive.
//----------------------------------------------------------------------------

do GC.KeepAlive(ctrlEventHandlers)

