// (c) Microsoft Corporation 2005-2007. 
#light

module Microsoft.FSharp.Collections.Array

open Microsoft.FSharp.Collections
open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Core.Operators

let length (arr: 'a array) =  Microsoft.FSharp.Primitives.Basics.Array.length arr
let get (arr: 'a array) (n:int) =  Microsoft.FSharp.Primitives.Basics.Array.get arr n
let set (arr: 'a array) (n:int) (x:'a) =  Microsoft.FSharp.Primitives.Basics.Array.set arr n x
let zero_create (n:int) = Microsoft.FSharp.Primitives.Basics.Array.zero_create n
let create  (n:int) (x:'a) = Microsoft.FSharp.Primitives.Basics.Array.make n x
let init (n:int) (f: int -> 'a) =  Microsoft.FSharp.Primitives.Basics.Array.init n f

let make (n:int) (x:'a) = create n x

let make_matrix (n:int) (m:int) (x:'a) = 
    let arr = (zero_create n : 'a array array) 
    for i = 0 to n - 1 do 
        let inner = (zero_create m : 'a array) 
        for j = 0 to m - 1 do 
            set inner j x
        done;
        set arr i inner
    done;
    arr

let create_matrix n m x = make_matrix n m x

let blit arr1 start1 arr2 start2 len = Microsoft.FSharp.Primitives.Basics.Array.blit arr1 start1 arr2 start2 len

let rec concat_add_lengths (arrs:'a array list) acc =
    match arrs with 
    | [] -> acc
    | h::t -> concat_add_lengths t (acc + length h)

let rec concat_blit (arrs:'a array list) i tgt =
    match arrs with 
    | [] -> ()
    | h::t -> 
        let len = length h 
        blit h 0 tgt i len; 
        concat_blit t (i+len) tgt

let concat (arrs:'a array list) = 
    let res = zero_create (concat_add_lengths arrs 0) 
    concat_blit arrs 0 res ;
    res

let concat_seq arrs = concat (Seq.to_list arrs)
  
let append arr1 arr2 = concat [arr1; arr2]

let sub (arr:'a array) (start:int) (len:int) = Microsoft.FSharp.Primitives.Basics.Array.sub arr start len

let fill (arr:'a array) (start:int) (len:int) (x:'a) =
    for i = start to start + len - 1 do 
        set arr i x
    done

let copy arr = Microsoft.FSharp.Primitives.Basics.Array.copy arr

let to_list (arr:'a array) = Microsoft.FSharp.Primitives.Basics.Array.to_list arr
let of_list (l:'a list) = Microsoft.FSharp.Primitives.Basics.Array.of_list l

let iter f arr = 
    let len = length arr 
    for i = 0 to len - 1 do 
        f arr.(i)
    done

let map f arr = Microsoft.FSharp.Primitives.Basics.Array.map f arr

let iter2 f arr1 arr2 = 
    let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
    let len1 = length arr1 
    let len2 = length arr2 
    if len1 <> len2 then invalid_arg "Array.iter2";
    for i = 0 to len1 - 1 do 
        f.Invoke(arr1.[i], arr2.[i])

let map2 f (arr1: 'a array) (arr2: 'b array) = 
    let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
    let len1 = arr1.Length 
    let len2 = arr2.Length 
    if len1 <> len2 then invalid_arg "Array.map2";
    let res = zero_create len1 
    for i = 0 to len1 - 1 do 
        res.(i) <- f.Invoke(arr1.(i), arr2.(i))
    res

let iteri f arr =
    let len = length arr 
    for i = 0 to len - 1 do 
      f i arr.(i)
    done

let mapi f arr =
    let len = length arr 
    let res = zero_create len 
    for i = 0 to len - 1 do 
        set res i  (f i (get arr i))
    done;
    res

let exists (f: 'a -> bool) (arr:'a array) =
    let len = length arr 
    let mutable res = false 
    let mutable i = 0 
    while not res && i < len do 
        if f (get arr i) then res <- true;
        i <- i + 1
    done;
    res

let for_all (f: 'a -> bool) (arr:'a array) =
    let len = length arr 
    let mutable res = true 
    let mutable i = 0 
    while res && i < len do 
        if not (f (get arr i)) then res <- false;
        i <- i + 1
    done;
    res

#if CLI_AT_MOST_1_1
// note: slower implementations when using .NET 1.x since ResizeArray is not available
let filter f arr = arr |> to_list |> List.filter f |> of_list
let partition f arr = let l1,l2 = arr |> to_list |> List.partition f in of_list l1, of_list l2
let choose f arr = arr |> to_list |> List.choose f |> of_list 
#else
let choose f arr = 
    let res = new ResizeArray<_>() 
    for i = 0 to length arr - 1 do 
        match f arr.(i) with 
        | None -> ()
        | Some b -> res.Add(b)
    done;
    res.ToArray()

let filter f arr = 
    let res = new ResizeArray<_>() 
    for i = 0 to length arr - 1 do 
        let x = arr.(i) 
        if f x then res.Add(x)
    done;
    res.ToArray()

let partition f arr = 
    let res1 = new ResizeArray<_>() 
    let res2 = new ResizeArray<_>() 
    for i = 0 to length arr - 1 do 
        let x = arr.(i) 
        if f x then res1.Add(x) else res2.Add(x)
    done;
    res1.ToArray(), res2.ToArray()
#endif

let rec findA f arr lim i = 
    if i >= lim then not_found() else 
    let x = arr.(i) 
    if f x then x else findA f arr lim (i+1)

let find f arr = findA f arr (length arr) 0 


let rec tryfindA f arr lim i = 
    if i >= lim then None else 
    let x = arr.(i) 
    if f x then Some x else tryfindA f arr lim (i+1)

let tryfind f arr = tryfindA f arr (length arr) 0 
  
let rec firstA f arr lim i = 
    if i >= lim then None else 
    let x = arr.(i) 
    match f x with | None -> firstA f arr lim (i+1) | res -> res

let first f arr = firstA f arr (length arr) 0 

let combine arr1 arr2 = 
    let len1 = length arr1 
    let len2 = length arr2 
    if len1 <> len2 then invalid_arg "Array2.combine";
    let res = zero_create len1 
    for i = 0 to len1 - 1 do 
        res.(i) <- (arr1.(i),arr2.(i))
    done;
    res

let split arr = 
    let len = length arr 
    let res1 = zero_create len 
    let res2 = zero_create len 
    for i = 0 to len - 1 do 
        let x,y = arr.(i) 
        res1.(i) <- x;
        res2.(i) <- y;
    done;
    res1,res2

let rev arr = 
    let len = length arr 
    let res = zero_create len 
    for i = 0 to len - 1 do 
        res.((len - i) - 1) <- arr.(i)
    done;
    res

let fold_left f acc arr = Microsoft.FSharp.Primitives.Basics.Array.fold_left f acc arr
let fold_right f arr acc = Microsoft.FSharp.Primitives.Basics.Array.fold_right f arr acc

let fold_sub_left f acc arr start fin = 
    let mutable res = acc 
    for i = start to fin do
      res <- f res arr.(i) 
    done;
    res

let fold_sub_right f arr start fin acc = 
    let mutable res = acc 
    for i = fin downto start do
      res <- f arr.(i) res
    done;
    res

let scan_sub_right f arr start fin initState = 
    let mutable state = initState 
    let res = create (2+fin-start) initState 
    for i = fin downto start do
      state <- f arr.(i) state;
      res.[i - start] <- state
    done;
    res

let scan_sub_left f  initState arr start fin = 
    let mutable state = initState 
    let res = create (2+fin-start) initState 
    for i = start to fin do
      state <- f state arr.(i);
      res.[i - start+1] <- state
    done;
    res

let scan_left f acc arr = 
    let arrn = length arr
    scan_sub_left f acc arr 0 (arrn - 1)

let scan_right f arr acc = 
    let arrn = length arr
    scan_sub_right f arr 0 (arrn - 1) acc

let scan1_left f arr = 
    let arrn = length arr
    if arrn = 0 then invalid_arg "Array.scan1_left"
    else scan_sub_left f arr.[0] arr 1 (arrn - 1)

let scan1_right f arr  = 
    let arrn = length arr
    if arrn = 0 then invalid_arg "Array.scan1_right"
    else scan_sub_right f arr 0 (arrn - 2) arr.[arrn - 1]

let reduce_left f arr = 
    let arrn = length arr
    if arrn = 0 then invalid_arg "Array.fold1_left"
    else fold_sub_left f arr.[0] arr 1 (arrn - 1) 

let reduce_right f arr = 
    let arrn = length arr
    if arrn = 0 then invalid_arg "Array.fold1_right"
    else fold_sub_right f arr 0 (arrn - 2) arr.[arrn - 1]

let fold1_left f arr = reduce_left f arr
let fold1_right f arr = reduce_right f arr

let sort (f: 'a -> 'a -> int) (arr : 'a array) =
  let len = length arr 
  if len < 2 then () 
  elif len = 2 then 
      let c = f arr.(0) arr.(1) 
      if c > 0 then
          let tmp = arr.(0) 
          arr.(0) <- arr.(1); 
          arr.(1) <- tmp
  else 
#if CLI_AT_MOST_1_1
        System.Array.Sort((arr :> System.Array), 
                          { new System.Collections.IComparer
                              with Compare(a,b) = f (unbox a) (unbox b) } )
#else
        System.Array.Sort(arr, ComparisonIdentity.FromFunction(f))
#endif

let of_seq ie = Seq.to_array ie
let to_seq a = Seq.of_array a


#if CLI_AT_MOST_1_1
#else

open System.Collections.Generic

let to_IEnumerable a = Seq.of_array a
let of_IEnumerable ie = Seq.to_array ie

let to_ICollection (a : 'a[]) = (a :> ICollection<'a>) 
let of_ICollection (ic :> ICollection<'a>) =
    let c = ic.Count 
    let res = zero_create c 
    ic.CopyTo(res, 0);
    res

let to_List l = new List<_>(to_seq l)
let of_List (l :  List<'a>) = of_ICollection ( l :> ICollection<'a>)

let to_ResizeArray l = new List<_>(to_seq l)
let of_ResizeArray (l :  List<'a>) = of_ICollection ( l :> ICollection<'a>)

#endif

let inline pin (arr: 'a []) (f : 'a nativeptr -> 'b) = 
    Microsoft.FSharp.Primitives.Basics.NativeOps.pinAny (box arr) (fun _ -> f (Microsoft.FSharp.Primitives.Basics.Array.geta arr 0))
let inline pin_unscoped (arr: 'a []) : 'a nativeptr * _ = 
    let gch = Microsoft.FSharp.Primitives.Basics.NativeOps.pinUnscoped (box arr) 
    Microsoft.FSharp.Primitives.Basics.Array.geta arr 0, gch
  
let find_index f arr = 
    let rec go n = if n >= length arr then not_found() elif f arr.(n) then n else go (n+1)
    go 0

let find_indexi f arr = 
    let rec go n = if n >= length arr then not_found() elif f n arr.(n) then n else go (n+1)
    go 0 

let tryfind_index f arr = 
    let rec go n = if n >= length arr then None elif f arr.(n) then Some n else go (n+1)
    go 0 

let tryfind_indexi f arr = 
    let rec go n = if n >= length arr then None elif f n arr.(n) then Some n else go (n+1)
    go 0 

let permute (p:Permutation) arr = 
    arr |> mapi (fun i _ -> arr.[p.[i]])

let sumByInt f arr = let mutable acc = 0 in for i = 0 to length arr - 1 do acc <- acc + f arr.[i] done; acc
let sumByFloat f arr = let mutable acc = 0.0 in for i = 0 to length arr - 1 do acc <- acc + f arr.[i] done; acc
let sumByFloat32 f arr = let mutable acc = 0.0f in for i = 0 to length arr - 1 do acc <- acc + f arr.[i] done; acc
let sumByInt64 f arr = let mutable acc = 0L in for i = 0 to length arr - 1 do acc <- acc + f arr.[i] done; acc
