// (c) Microsoft Corporation 2005-2007. 

#light

namespace Microsoft.FSharp.Control

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Control
open Microsoft.FSharp.Collections

[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
/// Basic operations on IEvents.  
module IEvent = 

    /// Create an IEvent with no initial listeners. Two items are returned: 
    /// a function to invoke (trigger) the event, and the event that clients 
    /// can plug listeners into.
    val create<'a> : unit -> ('a -> unit) * IEvent<'a>

    /// Create an IEvent with no initial listeners and at most one active listener. Adding extra
    /// listeners causes subsequent listeners to be dropped. Two items are returned: 
    /// a function to invoke (trigger) the event, and the event that clients 
    /// can plug listeners into.
    val create1<'a> : unit -> ('a -> unit) * IEvent<'a>

    /// Create an event that is to be published as a standard .NET event using  .NET
    /// delegates and metadata, and which will be recognized as an event by 
    /// any language that supports the CLI Common Language Specification. Two items
    /// are returned: a function to invoke (trigger) the event, and the event itself.
    /// The event value should be published as a public instance property in order for it to be 
    /// published as a standard .NET event.
    ///
    /// When firing the event the "sender" of the event must be specified, e.g. the form
    /// or control firing the event. This parameter is not normally of use for F# events,
    /// but is used by some other .NET event consumers. The sender is passed to 
    /// the handler implementation as type 'object'
    val create_HandlerEvent<'sender,'a> : unit -> ('sender * 'a -> unit) * IHandlerEvent<'a>

    /// Fire the output event when either of the input events fire
    val merge: #IEvent<'a> -> #IEvent<'a> -> IEvent<'a>

    // 'create_DelegateEvent' is used by CodeDOM
    
    /// Create an event that is to be published as standard .NET event using .NET
    /// delegates and metadata, and which will be recognized as an event by 
    /// any language that supports the CLI Common Language Specification. Two items
    /// are returned: a function to invoke (trigger) the event, and the event itself.
    /// The event value should be published as a public instance property in order for it to be 
    /// published as a standard .NET event.
    ///
    /// This function creates event of any .NET EventHandler type (inherited from System.Delegate)
    /// and will be exported as event of that type. Function used to invoke the event accepts
    /// arguments as a list of objects and the event is invoked dynamically. When possible
    /// use 'create_HandlerEvent'. 
    #if CLI_AT_MOST_1_1
    #else
    [<Experimental("It is recommended to use 'create_HandlerEvent' when possible")>]
    val create_DelegateEvent : unit -> (obj list -> unit) * IPrimitiveDelegateEvent<#System.Delegate>
    #endif
    
    /// Return a new event that passes values transformed by the given function
    val map: ('a -> 'b) -> #IEvent<'a> -> IEvent<'b>

    /// Return a new event that listens to the original event and triggers the resulting
    /// event only when the argument to the event passes the given function
    val filter: ('a -> bool) -> #IEvent<'a> -> IEvent<'a>

    /// Return a new event that listens to the original event and triggers the 
    /// first resulting event if the application of the predicate to the event arguments
    /// returned true, and the second event if it returned false
    val partition: ('a -> bool) -> #IEvent<'a> -> (IEvent<'a> * IEvent<'a>)

    /// Return a new event that listens to the original event and triggers the 
    /// first resulting event if the application of the function to the event arguments
    /// returned a Choice2_1, and the second event if it returns a Choice2_2
    val split: ('a -> Choice<'b,'c>) -> #IEvent<'a> -> (IEvent<'b> * IEvent<'c>)

    /// Return a new event which fires on a selection of messages from the original event.
    /// The selection function takes an original message to an optional new message.
    val choose: ('a -> 'b option) -> #IEvent<'a> -> IEvent<'b>

    /// Return a new event consisting of the results of applying the given accumulating function
    /// to successive values triggered on the input event.  An item of internal state
    /// records the current value of the state parameter.  The internal state is not locked during the
    /// execution of the accumulation function, so care should be taken that the 
    /// input IEvent not triggered by multiple threads simultaneously.
    val fold : ('b -> 'a -> 'b) -> 'b -> #IEvent<'a> -> IEvent<'b> 

    /// Return a new event consisting of the observed results of applying the given accumulating function
    /// to successive values triggered on the input event.  The observed results are the
    /// second items in the pairs returned.  An item of internal state
    /// records the current value of the state parameter.  The internal state is not locked during the
    /// execution of the accumulation function, so care should be taken that the 
    /// input IEvent not triggered by multiple threads simultaneously.
    val fold_and_emit : ('b -> 'a -> 'b * 'c) -> 'b -> #IEvent<'a> -> IEvent<'c> 

    /// Run the given function each time the given event is triggered.
    val listen : ('a -> unit) -> #IEvent<'a> -> unit

    /// Return a new event that triggers on the second and subsequent triggerings of the input event.
    /// The Nth triggering of the input event passes the arguments from the N-1th and Nth triggering as
    /// a pair. The argument passed to the N-1th triggering is held in hidden internal state until the 
    /// Nth triggering occurs.
    ///
    /// You should ensure that the contents of the values being sent down the event are
    /// not mutable. Note that many EventArgs types are mutable, e.g. MouseEventArgs, and
    /// each firing of an event using this argument type may reuse the same physical 
    /// argument obejct with different values. In this case you should extract the necessary
    /// information from the argument before using this combinator.    
    val pairwise: #IEvent<'a> -> IEvent<'a * 'a>

    [<System.Obsolete("Consider using IEvent.create_HandlerEvent instead")>]
    val createCompat : unit -> ('a -> unit) * IHandlerEvent<'a>

    [<System.Obsolete("Consider using IEvent.create_HandlerEvent instead")>]
    val create_public<'sender,'a> : unit -> ('sender * 'a -> unit) * IHandlerEvent<'a>
