(* (c) Microsoft Corporation. All rights reserved *)
(*F# 
module Microsoft.FSharp.Compiler.Lib
open Microsoft.Research.AbstractIL
open Microsoft.Research.AbstractIL.Internal 
F#*)


let debug = false (* is this the developer-debug build? *)
let verbose = false
let progress = ref false (* let _ = if !progress then printf dprint_endline "warning: progress is on by default in build" *)

(*------------------------------------------------------------------------
 * Misc
 *-----------------------------------------------------------------------*)

let notlazy v = Lazy.lazy_from_val v
let nonNil x = match x with [] -> false | _ -> true
let isNil x = match x with [] -> true | _ -> false
let (=!=) x y = (x == y)

(*-------------------------------------------------------------------------
!* Library: bits
 *------------------------------------------------------------------------*)

let (!!!) = Int32.of_int  
let (???) = Int32.to_int  
(*IF-OCAML*)
let (&&&) = Int32.logand
let (|||) = Int32.logor
let (<<<) = Int32.shift_left
let (lsr) = Int32.shift_right_logical
let (asr) = Int32.shift_right
(*ENDIF-OCAML*)
let (~~~) = Int32.lognot
let ( *** )  = Int32.mul
let ( +++ )  = Int32.add
let ( %%% )  = Int32.rem

let b0 n = Int32.to_int (n &&& 0xFFl)  
let b1 n = Int32.to_int ((n lsr 8) &&& 0xFFl) 
let b2 n = Int32.to_int ((n lsr 16) &&& 0xFFl) 
let b3 n = Int32.to_int ((n lsr 24) &&& 0xFFl) 

let ( >>>> ) x y = Int64.shift_right_logical x y 
let ( ~~~~ ) x = Int64.lognot x
let ( |||| ) x y = Int64.logor x y 
let ( &&&& ) x y = Int64.logand x y 
let ( !!!! ) x = Int64.of_int x 
let ( ???? ) x = Int64.to_int x 
let ( <<<< ) x y = Int64.shift_left x y 
let ( **** ) x y = Int64.mul x y 

let rec pown32 n = if n = 0 then 0l else (pown32 (n-1) |||  (1l <<<  (n-1)))
let rec pown64 n = if n = 0 then 0L else (pown64 (n-1) |||| (1L <<<< (n-1)))
let mask32 m n = (pown32 n) <<< m
let mask64 m n = (pown64 n) <<<< m

(*-------------------------------------------------------------------------
!* Library: files
 *------------------------------------------------------------------------*)

(* note, this is not a great function to rely on when the compiler is hosted in-process, e.g. in Visual Studio *)
(* since Visual Studio may have different getcwd() *)
let fullpath nm = if Filename.is_relative nm then Filename.concat (Sys.getcwd()) nm else nm 

(*-------------------------------------------------------------------------
!* Library: options
 *------------------------------------------------------------------------*)
 
let map_acc_option f s opt = 
  match opt with None -> None,s | Some x -> let x',s' = f s x in Some x',s'
let option_otherwise opt dflt = match opt with None -> dflt | Some x -> x
let option_orelse opt dflt = match opt with None -> dflt() | Some x -> opt
let option_else opt dflt = match opt with None -> dflt() | Some x -> x
let (+??) opt dflt = option_orelse opt dflt
let (+?) opt dflt = option_else opt dflt

(*IF-OCAML*)
module Option = struct
  let bind f opt = match opt with None -> None | Some x -> f x
  let map f opt = match opt with None -> None | Some x -> Some (f x)
  let exists f opt = match opt with None -> false | Some x -> f x
  let fold_right f opt acc = match opt with None -> acc | Some x -> f x acc
  let iter f opt = match opt with None -> () | Some x -> f x
  let iter2 f opt1 opt2 = 
    match opt1,opt2 with 
    | None,None -> () 
    | Some x,Some y -> f x y
    | _ -> failwith "Option.iter2"
  let to_list = function Some x -> [x] | None -> []
end
(*ENDIF-OCAML*)

type ('a,'b) choice1of2= Choice1 of 'a | Choice2 of 'b

let the = function None -> failwith "the"  | Some x -> x
let isSome = function Some _ -> true | None -> false           
let isNone = function None -> true | Some _ -> false

let fold_option collect z = function None -> z | Some x -> collect z x

let fmapOption f z l = 
    match l with 
    | None   -> z,None
    | Some x -> let z,x = f z x in
                z,Some x

(*-------------------------------------------------------------------------
!* Library: lists
 *------------------------------------------------------------------------*)

let rec last l = 
    match l with
    | [] -> invalid_arg "last" 
    | [h] -> h
    | h::t -> last t

let frontAndBack inputList = 
    let rec loop acc l = 
        match l with
        | [] -> invalid_arg "frontAndBack" 
        | [h] -> List.rev acc,h
        | h::t -> loop  (h::acc) t in
    loop [] inputList

let headAndTail l =
    match l with 
    | [] -> failwith "headAndTail"
    | h::t -> h,t

(* enforce evaluation order on map *)
(* note: must be tail recursive *)
(*IF-OCAML*)
(* fold right with no stack overflow *)
let rec list_fold_right f l acc = 
  match l with 
    [] -> acc
  | [h] -> f h acc
  | [h1;h2] -> f h1 (f h2 acc)
  | [h1;h2;h3] -> f h1 (f h2 (f h3 acc))
  | [h1;h2;h3;h4] -> f h1 (f h2 (f h3 (f h4 acc)))
  | _ -> 
      (* It is faster to allocate and iterate an array than to create all those *)
      (* highly nested stacks.  It also means we won't get stack overflows here. *)
      (* I have not tested whether N>=5 is the right point *)
      (* at which to start doing this. *)
      let arr = Array.of_list l in 
      let res = ref acc in 
      for i = Array.length arr - 1 downto 0 do
        res := f arr.(i) !res
      done;
      !res
let rec list_map_aux f l acc = 
  match l with
  | [] -> List.rev acc
  | (h::t) -> list_map_aux f t (f h :: acc)

let rec list_map f l = 
  match l with
  | [] -> []
  | [h] -> [f h]
  | _ -> list_map_aux f l []
(* note: must be tail recursive *)
let rec list_mapi_aux f l n acc = 
  match l with [] -> List.rev acc | (h::t) -> list_mapi_aux f t (n+1) (f n h :: acc)
let list_mapi f l =  
  match l with [] -> [] | [h] -> [f 0 h] | _ -> list_mapi_aux f l 0 []
let rec list_iteri_aux f l n = 
  match l with [] -> () | (h::t) -> f n h; list_iteri_aux f t (n+1)
let list_iteri f l =  list_iteri_aux f l 0

let (|>) x f = f x
let (>>) f g x = g(f x)
let (<<) f g x = f(g x)

(*ENDIF-OCAML*)
(*F#
let list_map f l = List.map f l
let list_mapi f l = List.mapi f l
let list_iteri f l = List.iteri f l
let list_fold_right f l acc = List.fold_right f l acc
F#*)

let rec mapi2a n f l1 l2 = 
  match l1,l2 with
    [],[] -> [] 
  | (h1::t1), (h2::t2) -> let x = f n h1 h2 in x :: mapi2a (n+1) f t1 t2
  | _ -> invalid_arg "map2"
let list_mapi2 f l1 l2 = mapi2a 0 f l1 l2


let replicate n x =   
    let rec go m acc = 
        if m >= n then acc else go(m+1) (x::acc) in 
    go 0 []

let mapNth n f xs =
  let rec mn i = function
    | []    -> []
    | x::xs -> if i=n then f x::xs else x::mn (i+1) xs
  in
  mn 0 xs


let fold1 f = function
 | []    -> failwith "fold1: given empty list"
 | x::xs -> List.fold_left f x xs

let foldOn p f z x = f z (p x)
let maxOn f x1 x2 = if f x1 > f x2 then x1 else x2
let maxOnL f xs = List.fold_left (foldOn f max) 0 xs

(* warning: not tail recursive *)
let rec list_take_until p l = 
  match l with
  | []    -> [],[]
  | x::xs -> if p x then [],l else let a,b = list_take_until p xs in x::a,b

let rec until p l = match l with [] -> [] | h::t -> if p h then [] else h :: until p t 

let list_take_while pred xs = list_take_until (pred >> not) xs


(*-------------------------------------------------------------------------
!* Library: orderings
 *------------------------------------------------------------------------*)

let string_ord (a:string) (b:string) = compare a b
let int_ord (a:int) (b:int) = compare a b
let int32_ord (a:int32) (b:int32) = compare a b

let pair_ord (compare1,compare2) (a1,a2) (aa1,aa2) =
  let res1 = compare1 a1 aa1 in
  if res1 <> 0 then res1 else compare2 a2 aa2

let proj_ord f a1 a2 = compare (f a1)  (f a2)


(*-------------------------------------------------------------------------
!* Names used to minimize differences with other versions of this codebase.
 *------------------------------------------------------------------------*)
 
(*IF-OCAML*)
module Map = Pmap
(*ENDIF-OCAML*)

(*-------------------------------------------------------------------------
!* Library: Name maps
 *------------------------------------------------------------------------*)

type 'a namemap = (string,'a) Map.t
type nameset = string Zset.t
type 'a namemmap = 'a list namemap


let pmap_of_list l = list_fold_right (fun (x,y) acc -> Map.add x y acc) l Map.empty
let pmap_to_list l = Map.fold (fun x y acc -> (x,y)::acc) l []

let nameset_of_list l : nameset = list_fold_right Zset.add l (Zset.empty string_ord)

module Namemap = struct

    let empty = Map.empty
    let domain m = Map.fold (fun x _ acc -> Zset.add x acc) m (Zset.empty string_ord)
    let domainL m = Zset.elements (domain m)
    let range m = List.rev (Map.fold (fun _ x sofar -> x :: sofar) m [])
    let fold f (m:'a namemap) z = Map.fold f m z
    let forall f m = Map.fold (fun x y sofar -> sofar && f x y) m true
    let exists f m = Map.fold (fun x y sofar -> sofar or f x y) m false
    let of_keyed_list f l = list_fold_right (fun x acc -> Map.add (f x) x acc) l Map.empty
    let of_list l : 'a namemap = pmap_of_list l
    let to_list (l: 'a namemap) = pmap_to_list l
    let layer (m1 : 'a namemap) m2 = Map.fold Map.add m1 m2

    (* not a very useful function - only called in one place - should be changed *)
    let layer_additive addf m1 m2 = 
      Map.fold (fun x y sofar -> Map.add x (addf (Pmap.tryfind_multi x sofar) y) sofar) m1 m2

    let union unionf m1 m2 = 
      Map.fold
        (fun x1 y1 sofar -> 
          Map.add 
            x1 
            (match Map.tryfind x1 sofar with 
              | Some res -> (unionf y1 res) 
              | None -> y1) 
            sofar)
        m1 m2

    (* For every entry in m2 find an entry in m1 and fold *)
    let subfold2 errf f m1 m2 acc =
      Map.fold (fun n x2 acc -> try f n (Map.find n m1) x2 acc with Not_found -> errf n x2) m2 acc
    let suball2 errf p m1 m2 = subfold2 errf (fun _ x1 x2 acc -> p x1 x2 & acc) m1 m2 true
    let map_acc f s (l: 'a namemap) = 
      Map.fold (fun x y (l',s') -> let y',s'' = f s' x y in Map.add x y' l',s'') l (Map.empty,s)
    let map_acc_range f s (l: 'a namemap) = 
      Map.fold (fun x y (l',s') -> let y',s'' = f s' y in Map.add x y' l',s'') l (Map.empty,s)
    let fold_range f (l: 'a namemap) acc = Map.fold (fun _ y acc -> f y acc) l acc
    let filter_range f (l: 'a namemap) = Map.fold (fun x y acc -> if f y then Map.add x y acc else acc) l Map.empty
    let map_filter f (l: 'a namemap) = Map.fold (fun x y acc -> match f y with None -> acc | Some y' -> Map.add x y' acc) l Map.empty
    let map f (l : 'a namemap) = Map.map f l
    let iter f (l : 'a namemap) = Map.iter (fun k v -> f v) l
    let iteri f (l : 'a namemap) = Map.iter f l
    let mapi f (l : 'a namemap) = Map.mapi f l
    let partition f (l : 'a namemap) = Map.filter (fun _ x-> f x) l, Map.filter (fun _ x -> not (f x)) l

    let mem v (m: 'a namemap) = Map.mem v m
    let find v (m: 'a namemap) = Map.find v m
    let tryfind v (m: 'a namemap) = Map.tryfind v m 
    let add v x (m: 'a namemap) = Map.add v x m
    let is_empty (m: 'a namemap) = (Map.is_empty  m)

    let exists_in_range p m =  Map.fold (fun _ y acc -> acc or p y) m false 
    let tryfind_in_range p m = Map.fold (fun _ y acc -> match acc with None -> if p y then Some y else None | _ -> acc) m None 
    let find_in_range p m = match (tryfind_in_range p m) with  Some r -> r | None -> raise Not_found

    let exists_in_range_multi f (m: 'a namemmap) = exists (fun _ l -> List.exists f l) m
    let find_multi v (m: 'a namemmap) = if mem v m then find v m else []
    let add_multi v x (m: 'a namemmap) = add v (x :: find_multi v m) m
    let range_multi (m: 'a namemmap) = List.concat (range m)
    let map_multi f (m: 'a namemmap) = map (List.map f) m 
    let empty_multi : 'a namemmap = Map.empty
end



(*-------------------------------------------------------------------------
!* Library 
 *------------------------------------------------------------------------*)

(*IF-OCAML*) module Ihashtbl = Hashtbl.Make(struct type t = int let equal (x:int) (y:int) = (x = y) let hash (x:int) = x end) (*ENDIF-OCAML*)
(*F# 
module Ihashtbl = struct 
    let hash (x:int) = x 
    let eq (x:int) (y:int) = (x = y) 
    type 'a t = Collections.HashTable<int,'a> 
    let create n : 'a t = Collections.HashTable.Create(Collections.HashIdentity.Structural,n)
    let add (t: 'a t) x y = t.Add(x,y)
    let remove (t: 'a t) x = t.Remove(x)
    let clear (t: 'a t) = t.Clear()
    let find (t: 'a t) x = t.Find(x)
    let mem (t: 'a t) x = t.Contains(x)
    let iter f (t: 'a t) = t.Iterate(f)
end 
F#*)

module Imap = struct
    let int_ord (x:int) (y:int) = compare x y
    let empty () = Zmap.empty int_ord

    type 'a t = (int,'a) Zmap.t
    let add k v (t:'a t) = Zmap.add k v t
    let find k (t:'a t) = Zmap.find k t
    let tryfind k (t:'a t) = Zmap.tryfind k t
    let remove  k (t:'a t) = Zmap.remove k t
    let mem     k (t:'a t)  = Zmap.mem k t
    let iter    f (t:'a t)  = Zmap.iter f t
    let map     f (t:'a t)  = Zmap.map f t 
    let fold     f (t:'a t)  z = Zmap.fold f t z
end

(*-------------------------------------------------------------------------
!* Library 
 *------------------------------------------------------------------------*)
    
let rec chop_at_aux n l = 
    if n <= 0 then ([],l) else 
    match l with 
    | [] -> failwith "chop_at: overchop"
    | (h::t) -> let a,b = chop_at_aux (n-1) t in  h::a, b

let chop_at n l = if n = List.length l then (l,[]) else chop_at_aux n l (* avoids allocation unless necessary *)

let rec frontAcc acc n l = 
    match l with
    | []    -> List.rev acc
    | x::xs -> if n<=0 then List.rev acc else frontAcc (x::acc) (n-1) xs

let front n l = if n = List.length l then l else frontAcc [] n l

let take n xs = front n xs

let rec drop n l = 
    match l with 
    | []    -> []
    | x::xs -> if n=0 then l else drop (n-1) xs

let cons x y = x :: y

(*-------------------------------------------------------------------------
!* Library: strings
 *------------------------------------------------------------------------*)

let string_eq (x:string) (y:string) = (x = y)
  
let try_drop_prefix s t = 
    let lens = String.length s in 
    let lent = String.length t in 
    if (lens >= lent && (String.sub s 0 lent = t)) then  
        Some(String.sub s lent (lens - lent) ) 
    else 
        None

let try_drop_suffix s t = 
    let lens = String.length s in 
    let lent = String.length t in 
    if (lens >= lent && (String.sub s (lens-lent) lent = t)) then 
        Some (String.sub s 0 (lens - lent))
    else
        None

let has_prefix s t = isSome (try_drop_prefix s t)
let drop_prefix s t = match (try_drop_prefix s t) with Some(res) -> res | None -> failwith "drop_prefix"

let has_suffix s t = isSome (try_drop_suffix s t)
let drop_suffix s t = match (try_drop_suffix s t) with Some(res) -> res | None -> failwith "drop_suffix"

let string_is_all_lower s = ((String.lowercase s) = s)    
let underscore_lowercase s =
  (* When generating HTML documentation,
   * some servers/filesystem are case insensitive.
   * This leads to collisions on type names, e.g. complex and Complex.
   *------
   * This function does partial disambiguation, by prefixing lowercase strings with _.
   *)
  if string_is_all_lower s then "_"^s else s
  


(*-------------------------------------------------------------------------
!* Library: generalized association lists
 *------------------------------------------------------------------------*)

let rec gen_assoc f x l = 
  match l with 
  | [] -> raise Not_found
  | (x',y)::t -> if f x x' then y else gen_assoc f x t

let rec gen_mem_assoc f x l = 
  match l with 
  | [] -> false
  | (x',y)::t -> f x x' or gen_mem_assoc f x t


(*-------------------------------------------------------------------------
!* Library: lists as generalized sets
 *------------------------------------------------------------------------*)

let rec gen_exists f l = 
  match l with 
  | [] -> false
  | x'::t -> f x' or gen_exists f t

(* NOTE: O(n)! *)
let rec gen_mem f x l = 
  match l with 
  | [] -> false
  | x'::t -> f x x' or gen_mem f x t

(* NOTE: O(n)! *)
let gen_insert f x l = if gen_mem f x l then l else x::l
let gen_union_favour_right f l1 l2 = 
    if l2 = [] then l1 
    else if l1 = [] then l2 
    else list_fold_right (gen_insert f) l1 l2 (* nb. fold_right to preserve natural orders *)

(* NOTE: O(n)! *)
let rec gen_index_aux eq x l n =
    match l with
    | [] -> raise Not_found
    | (h::t) -> if eq h x then n else gen_index_aux eq x t (n+1)

let gen_index eq x l = gen_index_aux eq x l 0

let rec gen_remove f x l = 
    match l with 
    | (h::t) -> if f x h then t else h:: gen_remove f x t
    | [] -> []

(* NOTE: quadratic! *)
let rec gen_subtract f l1 l2 = 
  match l2 with 
  | (h::t) -> gen_subtract f (gen_remove (fun y2 y1 ->  f y1 y2) h l1) t
  | [] -> l1

let gen_subset_of f l1 l2 = List.for_all (fun x1 -> gen_mem f x1 l2) l1
(* nb. preserve orders here: f must be applied to elements of l1 then elements of l2*)
let gen_superset_of f l1 l2 = List.for_all (fun x2 -> gen_mem (fun y2 y1 ->  f y1 y2) x2 l1) l2
let gen_set_eq f l1 l2 = gen_subset_of f l1 l2 & gen_superset_of f l1 l2

let gen_union_favour_left f l1 l2 = 
  if l2 = [] then l1 
  else if l1 = [] then l2 
  else l1 @ (gen_subtract f l2 l1)


(* NOTE: not tail recursive! *)
let rec gen_intersect f l1 l2 = 
  match l2 with 
  | (h::t) -> if gen_mem f h l1 then h::gen_intersect f l1 t else gen_intersect f l1 t
  | [] -> []

(* NOTE: quadratic! *)
let gen_setify f l = list_fold_right (gen_insert f) l []

(*-------------------------------------------------------------------------
!* Library: list ops
 *------------------------------------------------------------------------*)


let chooseList select =
    let rec ch acc l = 
        match l with 
        | [] -> List.rev acc
        | x::xs -> 
            match select x with
            | Some sx -> ch (sx::acc) xs
            | None -> ch acc xs in
    fun l -> ch [] l

let chooseListFromArray select xs =
  let res = ref [] in 
  for i = 0 to Array.length xs - 1 do
    match  select xs.(i) with
    | Some sx -> res := sx :: !res
    | None -> ()
  done;
  List.rev !res


let count pred xs = List.fold_left (fun n x -> if pred x then n+1 else n) 0 xs
let rec first pred = function [] -> None | x::xs -> if pred x then Some x else first pred xs

let rec repeatA n x acc = if n <= 0 then acc else repeatA (n-1) x (x::acc)
let repeat n x = repeatA n x []

(* WARNING: not tail-recursive *)
let mapHT fhead ftail = function
  | []    -> []
  | [x]   -> [fhead x]
  | x::xs -> fhead x :: List.map ftail xs

(*-------------------------------------------------------------------------
!* Library: listq
 *------------------------------------------------------------------------*)

let rec listq_remove x l = 
  match l with 
    (h::t) -> if x == h then t else h:: listq_remove x t
  | [] -> []
let rec listq_subtract l1 l2 = 
  match l2 with 
    (h::t) -> listq_subtract (listq_remove h l1) t
  | [] -> l1

(*-------------------------------------------------------------------------
!* Library: accumulating functions
 *------------------------------------------------------------------------*)
 
(* must be tail recursive *)
let map_acc_list f s l = 
  let l,s' = List.fold_left (fun (l',s') x -> let x',s'' = f s' x in x'::l',s'') ([],s) l in 
  List.rev l,s'

let map_concat_acc_list f s l = 
  let l, s = map_acc_list f s l in 
  List.concat l, s

let mapi_concat f xs = List.concat (list_mapi f xs)

let map_acc_array f s l = 
   let acc = ref s in 
   let res = ref [] in
   for i = 0 to Array.length l - 1 do
     let h',s' = f !acc l.(i) in 
     res := h' :: !res;
     acc := s'
   done;
   Array.of_list (List.rev !res), !acc

let rec choose select = function
  | []    -> None
  | x::xs -> match select x with
             | None -> choose select xs          
             | Some res -> Some res

let rec tryfind pred = function
  | []    -> None
  | x::xs -> if pred x then Some x else tryfind pred xs

(* note: must be tail-recursive *)
let rec fmapA f z l acc =
  match l with
  | []    -> z,List.rev acc
  | x::xs -> let z,x = f z x in
             fmapA f z xs (x::acc)
             
(* note: must be tail-recursive *)
let fmap f z l = fmapA f z l []

let firstPos pred xs =
  let len = Array.length xs in 
  let rec fp i = 
    if i >= len then raise Not_found
    else if pred xs.(i) then i
    else fp (i+1) in
  fp 0 

let rec foralli p n m = n > m || (p n && foralli p (n+1) m)
let rec iteri f n m = if n > m then () else ( f n; iteri f (n+1) m)
let string_forall p s = foralli (fun i -> p s.[i]) 0 (String.length s - 1)
let string_foralli p s = foralli (fun i -> p i s.[i]) 0 (String.length s - 1)
let string_iter p s = iteri (fun i -> p s.[i]) 0 (String.length s - 1)

(*-------------------------------------------------------------------------
!* Library: pairs
 *------------------------------------------------------------------------*)

let pair_map f1 f2 (a,b) = (f1 a, f2 b)
let map_fst f (a,b) = (f a, b)
let map_snd f (a,b) = (a, f b)
let map_acc_fst f s (x,y) =  let x',s = f s x in  (x',y),s
let map_acc_snd f s (x,y) =  let y',s = f s y in  (x,y'),s
let pair a b = a,b      

let p13 (x,y,z) = x
let p23 (x,y,z) = y
let p33 (x,y,z) = z
let rec split3 = function [] -> [],[],[] | (a1,a2,a3) :: t -> let b1,b2,b3 = split3 t in (a1::b1),(a2::b2),(a3::b3)
let rec split4 = function [] -> [],[],[],[] | (a1,a2,a3,a4) :: t -> let b1,b2,b3,b4 = split4 t in (a1::b1),(a2::b2),(a3::b3),(a4::b4)
let fmap1'2 f z (a1,a2)       = let z,a1 = f z a1 in z,(a1,a2)
let fmap2'2 f z (a1,a2)       = let z,a2 = f z a2 in z,(a1,a2)
let fmap3'3 f z (a1,a2,a3)     = let z,a3 = f z a3 in z,(a1,a2,a3)
let fmap4'4 f z (a1,a2,a3,a4)   = let z,a4 = f z a4 in z,(a1,a2,a3,a4)
let fmap5'5 f z (a1,a2,a3,a4,a5) = let z,a5 = f z a5 in z,(a1,a2,a3,a4,a5)
let fmap6'6 f z (a1,a2,a3,a4,a5,a6) = let z,a6 = f z a6 in z,(a1,a2,a3,a4,a5,a6)

let map1'2 f (a1,a2)       = (f a1,a2)
let map2'2 f (a1,a2)       = (a1,f a2)
let map1'3 f (a1,a2,a3)     = (f a1,a2,a3)
let map2'3 f (a1,a2,a3)     = (a1,f a2,a3)
let map3'3 f (a1,a2,a3)     = (a1,a2,f a3)
let map3'4 f (a1,a2,a3,a4)     = (a1,a2,f a3,a4)
let map4'4 f (a1,a2,a3,a4)   = (a1,a2,a3,f a4)
let map5'5 f (a1,a2,a3,a4,a5) = (a1,a2,a3,a4,f a5)
let map6'6 f (a1,a2,a3,a4,a5,a6) = (a1,a2,a3,a4,a5,f a6)
let foldr'2 (f1,f2)    (a1,a2)    acc      = f1 a1 (f2 a2 acc)
let foldr'3 (f1,f2,f3) (a1,a2,a3) acc      = f1 a1 (f2 a2 (f3 a3 acc))
let foldl'2 (f1,f2)    acc (a1,a2)         = f2 (f1 acc a1) a2
let foldl1'2 f1    acc (a1,a2)         = f1 acc a1
let foldl'3 (f1,f2,f3) acc (a1,a2,a3)      = f3 (f2 (f1 acc a1) a2) a3
let map'2 (f1,f2)    (a1,a2)     = (f1 a1, f2 a2)
let map'3 (f1,f2,f3) (a1,a2,a3)  = (f1 a1, f2 a2, f3 a3)




let splitChooseList select =
    let rec ch acc1 acc2 l = 
        match l with 
        | [] -> List.rev acc1,List.rev acc2
        | x::xs -> 
            match select x with
            | Choice1 sx -> ch (sx::acc1) acc2 xs
            | Choice2 sx -> ch acc1 (sx::acc2) xs in
    fun l -> ch [] [] l

(*---------------------------------------------------------------------------
!* Orders
 *------------------------------------------------------------------------- *)

type 'a order = 'a -> 'a -> int

(* order ops *)    
let bool_order : bool order = Pervasives.compare

let rec listOrder eltOrder xs ys =
    match xs,ys with
    | [],[]       ->  0
    | [],ys       -> -1
    | xs,[]       ->  1
    | x::xs,y::ys -> let cxy = eltOrder x y in
                     if cxy=0 then listOrder eltOrder xs ys else cxy

let tup2Order (xOrder,yOrder) (x,y) (xx,yy) =
    let c = xOrder x xx in
    if c=0 then yOrder y yy else c (* lexiographic *)

let orderOn p pxOrder x xx = pxOrder (p x) (p xx)

(*---------------------------------------------------------------------------
!* Zmap rebinds
 *------------------------------------------------------------------------- *)

let zmap_filter pred mp = Zmap.fold (fun k v nmp -> if pred k v then nmp else Zmap.remove k nmp) mp mp
let setsM kvs mp = List.fold_left (fun mp (k,v) -> Zmap.add k v mp) mp kvs
let forceM  k   mp           = match Zmap.tryfind k mp with Some x -> x | None -> failwith "forceM: lookup failed"
let hasMemM x a = Zmap.mem x a
let memM    a x = Zmap.mem x a
let lengthS a = List.length (Zset.elements a)  
let listM order xs = Zmap.of_list    (Zmap.empty order) xs

let zmap_map key f mp =
  match f (Zmap.tryfind key mp) with
  | Some fx -> Zmap.add key fx mp       
  | None    -> Zmap.remove key mp

let zmap_addL kxs mp = List.fold_left (fun mp (k,x) -> Zmap.add k x mp) mp kxs

(*---------------------------------------------------------------------------
!* Zset
 *------------------------------------------------------------------------- *)

let hasMemS a x = Zset.mem x a
let memS    x a = Zset.mem x a
let listS order xs = Zset.addL   xs (Zset.empty order)

let noRepeats xOrder xs =
  let s = listS xOrder xs in (* build set *)
  Zset.elements s            (* get elements... no repeats *)

(* CLEANUP NOTE: move to library or Zset *)
let rec fixpoint f (s as s0) =
   let s = f s in
   if Zset.equal s s0 then s0           (* fixed *)
                      else fixpoint f s (* iterate *)



(*---------------------------------------------------------------------------
!* Misc
 *------------------------------------------------------------------------- *)

let rec lengthl l = match l with [] -> 0 | h::t -> List.length h + lengthl t
let push_ref x l = l := x :: ! l

let chooseL select =
    let rec ch l acc = 
        match l with 
        | []    -> List.rev acc
        | x::xs -> match select x with
                   | Some sx -> ch xs (sx::acc) 
                   | None    -> ch xs acc in
    fun l -> ch l []

let mapConcat f xs = List.concat (List.map f xs)
let mapiConcat f xs = List.concat (list_mapi f xs)

let equalOn f x y = (f x) = (f y)

let rec upto (n,m) = if n<=m then n::upto (n+1,m) else []

(* NOTE: not tail recursive! *)
let splitAt n xs =
    if n<0 then failwith "splitAt: -ve" else
    let rec split = function
      | 0,xs    -> [],xs
      | n,x::xs -> let front,back = split (n-1,xs) in
                   x::front,back
      | n,[]    -> failwith "splitAt: not enough elts in list" in
    split (n,xs)


let singletonList x = [x]

let sumL xs = List.fold_left (+) 0 xs

(*---------------------------------------------------------------------------
!* Buffer printing utilities
 *------------------------------------------------------------------------- *)

let mapConcat2 f xs ys = List.concat (List.map2 f xs ys)

let bufs f = 
    let buf = Buffer.create 100 in f buf; Buffer.contents buf

let buff os f x = 
    let buf = Buffer.create 100 in f buf x; Buffer.output_buffer os buf

let output_to_file f outfile x =
    let os = open_out outfile in 
    try 
      let res = f os x  in 
      close_out os;
      res
    with e -> 
      (try close_out os with _ -> ());
      (*F# rethrow(); F#*) raise e

(*---------------------------------------------------------------------------
!* Imperative Graphs 
 *------------------------------------------------------------------------- *)

type ('id,'data) node = { nodeId: 'id; nodeData: 'data; mutable nodeNeighbours: ('id,'data) node list }

type ('id,'data) graph = { id: ('data -> 'id);
                           ord: 'id order;
                           nodes: ('id,'data) node list;
                           edges: ('id * 'id) list;
                           getNodeData: ('id -> 'data) }

let mk_graph (id,ord) nodeData edges =
    let nodemap = List.map (fun d -> id d, { nodeId = id d; nodeData=d; nodeNeighbours=[] }) nodeData in 
    let tab = Zmap.of_list (Zmap.empty ord) nodemap  in 
    let getNode nodeId = Zmap.find nodeId tab in 
    let getNodeData nodeId = (getNode nodeId).nodeData in 
    let nodes = List.map snd nodemap in 
    List.iter (fun node -> node.nodeNeighbours <- List.map (snd >> getNode) (List.filter (fun (x,y) -> ord x node.nodeId = 0) edges)) nodes;
    {id=id; ord = ord; nodes=nodes;edges=edges;getNodeData=getNodeData}

let iter_cycles f g = 
    let rec trace path node = 
      if List.exists (g.id >> (=) node.nodeId) path then f (List.rev path)
      else List.iter (trace (node.nodeData::path)) node.nodeNeighbours in 
    List.iter (fun node -> trace [] node) g.nodes 

let dfs g = 
    let grey = ref (Zset.empty g.ord) in 
    let time = ref 0 in 
    let forest = ref [] in 
    let backEdges = ref [] in 
    let discoveryTimes = ref (Zmap.empty g.ord) in 
    let finishingTimes = ref (Zmap.empty g.ord) in 
    g.nodes |> List.iter (fun n ->  
      (* build a dfsTree for each node in turn *)
      let treeEdges = ref [] in 
      let rec visit n1 = 
        incr time;
        grey := Zset.add n1.nodeId !grey;
        discoveryTimes := Zmap.add n1.nodeId !time !discoveryTimes;
        n1.nodeNeighbours |> List.iter (fun n2 ->
          if not (Zset.mem n2.nodeId !grey) then begin
            treeEdges := (n1.nodeId,n2.nodeId) :: !treeEdges;
            visit(n2)
          end else begin
            backEdges := (n1.nodeId,n2.nodeId) :: !backEdges
          end);
        incr time;
        finishingTimes := Zmap.add n1.nodeId !time !finishingTimes;
        () in 
      if not (Zset.mem n.nodeId !grey) then begin 
        visit(n);
        forest := (n.nodeId,!treeEdges) :: !forest
      end);
    !forest, !backEdges,  (fun n -> Zmap.find n !discoveryTimes), (fun n -> Zmap.find n !finishingTimes)
 

(* Present strongly connected components, in dependency order *)
(* Each node is assumed to have a self-edge *)
let topsort_strongly_connected_components g = 
    let forest, backEdges, discoveryTimes, finishingTimes = dfs g in 
    let nodeIds = List.map (fun n -> n.nodeId) g.nodes in 
    let nodesInDecreasingFinishingOrder = 
      List.sort (fun n1 n2 -> -(compare (finishingTimes n1) (finishingTimes n2))) nodeIds in 
    let gT = mk_graph (g.id,g.ord) (List.map g.getNodeData nodesInDecreasingFinishingOrder) (List.map (fun (x,y) -> (y,x)) g.edges) in
    let forest, backEdges, discoveryTimes, finishingTimes = dfs gT in 
    let scc (root,tree) = Zset.add root (List.fold_right (fun (n1,n2) acc -> Zset.add n1 (Zset.add n2 acc)) tree (Zset.empty g.ord)) in 
    let sccs = List.rev (List.map scc forest) in
    List.map (Zset.elements >> List.map g.getNodeData) sccs


(*
let g1 = mk_graph (=) [1;2;3] [(1,2);(2,3);(3,1)]
let g2 = mk_graph (=) [1;2;3] [(1,2);(2,3)]
let g3 = mk_graph (=) [1;2;3] [(1,1);(2,2)]
let g4 = mk_graph (=) [1;2;3] [(1,1);(2,1)]
let g5 = mk_graph (=) [1;2;3] [(3,2);(2,1)]
let g6 = mk_graph (=) [1;2;3] []
let g7 = mk_graph (=) [1;2;3] [(1,2);(2,1);(3,3)]
let g8 = mk_graph (=) [1;2;3] [(3,2);(2,3);(1,1)]


open Printf

let p sccs =  List.iter (fun l -> printf "scc: "; List.iter (fun i -> printf "%d;" i) l; printf "\n") sccs

do p (topsort_strongly_connected_components g1);;
do p (topsort_strongly_connected_components g2);;
do p (topsort_strongly_connected_components g3);;
do p (topsort_strongly_connected_components g4);;
do p (topsort_strongly_connected_components g5);;
do p (topsort_strongly_connected_components g6);;
do p (topsort_strongly_connected_components g7);;
do p (topsort_strongly_connected_components g8);;

*)


let array_forall pred arr =
    let rec forallSoFar i = 
        (i = Array.length arr) or (pred arr.(i) && forallSoFar (i+1)) in
    forallSoFar 0

let array_exists pred arr =
    let rec noWitnessYet i =
        (i <> Array.length arr) && (pred arr.(i) or noWitnessYet (i+1)) in
    noWitnessYet 0  
    
  
(*---------------------------------------------------------------------------
!* In some cases we play games where we use 'null' as a more efficient representation
 * in F#. The functions below are used to give initial values to mutable fields.
 * This is an unsafe trick, as it relies on the fact that the type of values
 * being placed into the slot never utilizes "null" as a representation. To be used with
 * with care.
 *------------------------------------------------------------------------- *)

type 'a nonnull_slot = (*IF-OCAML*) 'a option (*ENDIF-OCAML*) (*F# 'a F#*)
let nullable_slot_empty() : 'a nonnull_slot  = (*IF-OCAML*) None (*ENDIF-OCAML*) (*F# Unchecked.defaultof<'a> F#*)
let nullable_slot_full(x) = 
    (*IF-OCAML*) Some x (*ENDIF-OCAML*) 
    (*F# //assert(match box x with null -> false | _ -> true);
          x F#*)


(*---------------------------------------------------------------------------
!* Caches, mainly for free variables
 *------------------------------------------------------------------------- *)

type 'a cache = { mutable cacheVal: 'a nonnull_slot; }
let new_cache() = { cacheVal = nullable_slot_empty() }

(*IF-OCAML*)
let cached cache resf = 
  match cache.cacheVal with 
  | None -> (let res = resf() in cache.cacheVal <- Some res; res) 
  | Some x -> x
(*ENDIF-OCAML*)

(*F#
let inline cached cache resf = 
  match box cache.cacheVal with 
  | null -> (let res = resf() in  cache.cacheVal <- nullable_slot_full res; res)
  | _ -> cache.cacheVal
F#*)

let cacheOptRef cache f = 
    match !cache with 
    | Some cache -> Obj.obj cache
    | None -> 
       let res = f() in 
       cache := Some (Obj.repr res);
       res 



(*---------------------------------------------------------------------------
!* memoize tables (all entries cached, never collected)
 *------------------------------------------------------------------------- *)
    
let memoize f = 
    let t = Hashtbl.create 10 in 
    fun x -> 
      if Hashtbl.mem t x then Hashtbl.find t x 
      else let res = f x in Hashtbl.add t x res; res 

(*---------------------------------------------------------------------------
!* osgn = Observably shared graph node.
 *
 * osgn's are shared mutable noted internal to a larger data structure.
 * They are used only to ensure sharing is preserved through pickling.  
 * Unlinked nodes should only appear during the first phase of unpickling.
 *------------------------------------------------------------------------- *)

(*IF-OCAML*)
type 'a osgn_target = 
  | Linked of 'a 
  | Unlinked
and 'a osgn = { mutable osgnTarget: 'a osgn_target }

let deref_osgn x = match x.osgnTarget with Linked x -> x |  Unlinked -> failwith "deref_osgn"
let new_osgn x = { osgnTarget = Linked x }
let new_unlinked_osgn () = { osgnTarget = Unlinked }
let link_osgn x tg = x.osgnTarget <- Linked tg
let osgn_is_linked x = match x.osgnTarget with Linked x -> true |  Unlinked -> false
(*ENDIF-OCAML*)
(*F#
// This more efficient F# implementation takes advantage of null pointers
// to represent the lack of a target.  De-referencing through an unlinked node
// will raise a catchable null pointer exception with a good stack trace (which overall 
// is better then raising a 'failure' exception as used above).  In any 
// case failure to derefence an OSGN node represents a critical failure in 
// the pickling logic and there are safety checks made in the pickler to ensure all
// OSGN nodes are relinked.
type 'a osgn = { mutable osgnTarget: 'a }

let deref_osgn x = x.osgnTarget 
let new_osgn x = { osgnTarget = x }
let new_unlinked_osgn () = { osgnTarget = nullable_slot_empty() }
let link_osgn x tg = x.osgnTarget <- nullable_slot_full(tg)
let osgn_is_linked x = let tg = x.osgnTarget in match box tg with null -> false | _ -> true 
F#*)


let readBinaryFile f = 
    let is = open_in_bin f in 
    let n = in_channel_length is in 
    let res = Bytes.really_input is n in 
    close_in is;
    res 

let writeBinaryFile f bytes = 
    let os = open_out_bin f in 
    let res = Bytes.output os bytes in 
    close_out os

let (|<) f x = f x

(* There is a bug in .NET Framework v2.0.52727 DD#153959 that very occasionally hits F# code. *)
(* It is related to recursive class loading in multi-assembly NGEN scenarios. The bug has been fixed but *)
(* not yet deployed. *)
(* The bug manifests itself as an ExecutionEngine failure or fast-fail process exit which comes *)
(* and goes depending on whether components are NGEN'd or not, e.g. 'ngen install FSharp.COmpiler.dll' *)
(* One workaround for the bug is to break NGEN loading and fixups into smaller fragments. Roughly speaking, the NGEN *)
(* loading process works by doing delayed fixups of references in NGEN code. This happens on a per-method *)
(* basis. For example, one manifestation is that a "print" before calling a method like Lexfilter.create gets *)
(* displayed but the corresponding "print" in the body of that function doesn't get displayed. In between, the NGEN *)
(* image loader is performing a whole bunch of fixups of the NGEN code for the body of that method, and also for *)
(* bodies of methods referred to by that method. That second bit is very important: the fixup causing the crash may *)
(* be a couple of steps down the dependency chain. *)
(* *)
(* One way to break work into smaller chunks is to put delays in the call chains, i.e. insert extra stack frames. That's *)
(* what the function 'delayInsertedToWorkaroundKnownNgenBug' is for. If you get this problem, try inserting  *)
(*    delayInsertedToWorkaroundKnownNgenBug "Delay1" (fun () -> ...) *)
(* at the top of the function that doesn't seem to be being called correctly. This will help you isolate out the problem *)
(* and may make the problem go away altogher. Enable the 'print' commands in that function too. *)

let delayInsertedToWorkaroundKnownNgenBug s f = 
    (* Some random code to prevent inlining of this function *)
    let res = ref 10 in
    for i = 0 to 2 do 
       res := !res + String.length s;
    done;
    if verbose then Printf.printf "------------------------executing NGEN bug delay '%s', calling 'f' --------------\n" s;
    let res = f() in
    if verbose then Printf.printf "------------------------exiting NGEN bug delay '%s' --------------\n" s;
    res
    