(* (c) Microsoft Corporation. All rights reserved *)
(* -------------------------------------------------------------------- 
 * Erase abstractions over types to abstractions over 
 * representations. (IL-Lty to IL-Lrep).  See Word documentation
 * for examples & description of soundness.
 * -------------------------------------------------------------------- *)

(*F# 
module Microsoft.Research.AbstractIL.Extensions.ILX.Pp_erase
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
module Ilmorph = Microsoft.Research.AbstractIL.Morphs  
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics  
module Ilx = Microsoft.Research.AbstractIL.Extensions.ILX.Types   
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter  
module Il = Microsoft.Research.AbstractIL.IL   
module Illib = Microsoft.Research.AbstractIL.Internal.Library 
F#*)  

open Illib
open Ildiag
open Nums
open Il
open Ilx
open Ilmorph
open List
open Ilprint

let represent_with_casts () = true
type cenv = { ilg : mscorlib_refs }

(* -------------------------------------------------------------------- 
 * Reduce type variables to type-representation variables.
 * -------------------------------------------------------------------- *)

let rec conv_typ cenv typ = 
  match typ with 
  | Type_fptr t -> Type_fptr (conv_callsig cenv t)
  | Type_ptr t -> Type_ptr (conv_typ cenv t)
  | Type_byref t -> Type_byref (conv_typ cenv t)
  | Type_boxed cr -> Type_boxed (conv_tspec cenv cr)
  | Type_value ir -> Type_value (conv_tspec cenv ir)
  | Type_array(shape,ty) -> Type_array (shape,conv_typ cenv ty)
  | Type_other(e) when is_ilx_ext_typ e -> 
      begin match dest_ilx_ext_typ e with 
      | EType_erasable_array(shape,ty) -> Type_array(shape,conv_typ_as_typar cenv ty)
      end
  | Type_tyvar v -> typ (* REVIEW: genparams *)
  | _ -> typ
and conv_tspec cenv tspec = {tspec with tspecInst=conv_inst cenv tspec.tspecInst}
and conv_inst cenv inst = List.map (conv_typ_as_typar cenv) inst
and conv_typ_as_typar cenv ty =
  match ty with 
  | Type_boxed cr -> cenv.ilg.typ_Object
  | Type_value vtr -> cenv.ilg.typ_Object 
  | Type_array (shape,ty) ->  cenv.ilg.typ_Object
  | Type_tyvar v -> Type_tyvar v 
  | Type_other _ -> cenv.ilg.typ_Object
  | Type_ptr _ -> cenv.ilg.typ_Object  (* boxed as an unsigned int *)
  | Type_fptr _ -> cenv.ilg.typ_Object  (* boxed as an unsigned int *)
  | _ -> dprint_string "*** Error: not allowed as type parameter: "; output_typ stderr ty; prerr_newline(); failwith "internal error"
and conv_callsig cenv csig = mk_callsig(callconv_of_callsig csig,map (conv_typ cenv) (args_of_callsig csig),conv_typ cenv (ret_of_callsig csig))
and conv_genparam cenv gf = {gf with gpConstraints=List.map (conv_typ cenv) gf.gpConstraints}
and conv_genparams cenv l = List.map (conv_genparam cenv) l

(* -------------------------------------------------------------------- 
 * Modify method signatures, identities etc. based on type
 * token passing and type erasure.
 * -------------------------------------------------------------------- *)

let conv_mref cenv x =
  {x with mrefArgs=map (conv_typ cenv) x.mrefArgs; mrefReturn=conv_typ cenv x.mrefReturn }

let conv_mspec cenv x =
  let mref,typ,minst = dest_mspec x in 
  mk_mref_mspec_in_typ(conv_mref cenv mref,conv_typ cenv typ, conv_inst cenv minst)

let conv_fref cenv x = {x with frefType=conv_typ cenv x.frefType}

let conv_fspec cenv x =
  { fspecFieldRef= conv_fref cenv x.fspecFieldRef;
    fspecEnclosingType = conv_typ cenv x.fspecEnclosingType }

let conv_ospec cenv (OverridesSpec(mref,typ)) =
  OverridesSpec(conv_mref cenv mref, conv_typ cenv typ)

let conv_castclass cenv to_typ = 
   [ I_castclass (conv_typ cenv to_typ)]

let conv_isinst cenv to_typ = 
   [ I_isinst (conv_typ cenv to_typ)]

(* -------------------------------------------------------------------- 
 * This section deals with calls and loads.
 *
 * Calls, loads and stores all transform data from 
 * type-representation (actual) to type-variable-representation (formal)
 * and back again.  We generate a number of helper functions to help
 * out with this.  The mutable environments below are shared by all the
 * functions in a single pass of some lexical block.
 *   wrappers - used to wrap method code pointers passed across a 
 *              polymorphic boundary.  Never unwrap.
 * -------------------------------------------------------------------- *)

let internal_gen_count = ref 0
let new_name n = (incr internal_gen_count; n^"$"^string_of_int !internal_gen_count)

let mref_passes_tyvars csig = 
  List.exists (fun ty -> is_tyvar_ty ty) (args_of_callsig csig) or is_tyvar_ty (ret_of_callsig csig)

type 'a wrappers = { mutable wrappers: 'a list }

let is_array_or_boxed typ = 
  match typ with 
  | Type_array _ | Type_boxed _ -> true
  | Type_other(e) when is_ilx_ext_typ e -> 
      begin match dest_ilx_ext_typ e with 
      | EType_erasable_array _ -> true
      end
  | _ -> false

(* Generally used to coerce a value after a load of a field of variable *)
(* type, or a return value of variable type. *)
(* In all of the following, "formal" must be a _representation_ type *)
let coerce_tyvar_to_typ cenv typ =
  match typ with 
  | typ when typ_is_Object cenv.ilg typ  ->  []
  | _ when is_array_or_boxed typ -> 
      if not (represent_with_casts()) then [] 
      else [ I_castclass (conv_typ cenv typ) ]
  | Type_value vtr -> [ I_unbox_any (conv_typ cenv typ) ]
  | Type_ptr _ -> [ I_unbox_any (Type_value cenv.ilg.tspec_IntPtr) ] 
  | Type_tyvar v -> []
  | _ -> failwith "*** Error: type not allowed as type parameter (B2)"

let coerce_tyvar_to_typa cenv typ =
  match typ with 
  | _ when is_array_or_boxed typ -> []
  | Type_value vtr -> [ I_unbox (conv_typ cenv typ) ]
  | Type_ptr _ -> [ I_unbox (Type_value cenv.ilg.tspec_IntPtr) ] 
  | Type_tyvar v -> []
  | _ -> failwith "*** Error: type not allowed as type parameter (C)"

let fptr_msg = "ILXASM Error: ILXASM limits the use of function code pointers whose signatures include type variables to a very restricted set of circumstances, e.g. passing them as the last argument to a method.  You cannot return them from a method or load them from a field, except in polymorphic code.  They are only really to be used internally for implementing closures.@."

let coerce_tyrepr_to_type cenv tyrep typ =
  match tyrep with 
  | Type_tyvar v -> coerce_tyvar_to_typ cenv typ 
  | Type_fptr mref when mref_passes_tyvars mref -> failwith fptr_msg
  | _ -> [] 

let coerce_tyrepr_to_typea cenv tyrep typ =
  match tyrep with 
  | Type_tyvar v -> coerce_tyvar_to_typa cenv typ 
  | Type_fptr mref when mref_passes_tyvars mref -> failwith fptr_msg
  | _ -> [] 

let coerce_typ_to_tyvar cenv typ =
  match typ with 
  | _ when is_array_or_boxed typ -> []
  | Type_value vtr -> [I_box (conv_typ cenv (Type_value vtr)) ]
  | Type_ptr _ -> [ I_arith (AI_conv DT_I8); I_box (Type_value cenv.ilg.tspec_Int64) ]
  | Type_tyvar _ -> []
  | _ -> failwith "type not allowed as type parameter (D)"

let coerce_type_to_tyrepr cenv typ tyrep =
  match tyrep with 
  | Type_tyvar v -> coerce_typ_to_tyvar cenv typ 
  | _ ->  [] 

let repchange_typ_to_tyrep cenv typ tyrep = 
  coerce_type_to_tyrepr cenv typ tyrep <> [] 

let repchange_tyrep_to_typ cenv tyrep typ = 
  coerce_tyrepr_to_type cenv tyrep typ <> [] 

(* Once again, the tyrep here must be a representation type. *)
(* Must take care to get the contra/co directions right. *)
let repchange_typ_to_tyrep_callsig cenv csig  csigrep = 
    if (args_of_callsig csigrep |> length) <> (args_of_callsig csig |> length) then 
        dprintf2 "#(args_of_callsig csigrep) = %d, #(args_of_callsig csig) = %d\n" (args_of_callsig csigrep |> length) (args_of_callsig csig |> length);
    List.exists2 (repchange_tyrep_to_typ cenv) (args_of_callsig csigrep) (args_of_callsig csig) or
    repchange_typ_to_tyrep cenv (ret_of_callsig csig)  (ret_of_callsig csigrep)

(* nb. We don't allow any fptrs to be passed tyrep to typ *)
(* nb. We only allow fptrs to be passed typ to tyrep when *)
(* pointing to a known method *)

(* may need to allocate some tmps to swap around on the stack *)
let coerce_typ_stack_to_tyrep_stack cenv tmps typs tyreps  = 
try
  let rec pop_n_convert l1 l2 = match l1,l2 with 
     [],[] -> []
   | (a::rest1),(f::rest2) -> 
       if exists2 (repchange_typ_to_tyrep cenv) rest1 rest2 then
	 begin 
	   let locn = alloc_tmp tmps (mk_local (conv_typ cenv a)) in
	 (* pop off as typ, read back as tyrep *)
	   [ I_stloc locn ] @ 
	   pop_n_convert rest1 rest2 @ 
	   [ I_ldloc locn ] @
	   coerce_type_to_tyrepr cenv a f 
	 end
       else 
	 (* simply do in place on stack, ignore rest *)
	 coerce_type_to_tyrepr cenv a f
   | _,_ -> failwith "coerce_typ_stack_to_tyrep_stack cenv: tyreps/typs don't match" in
  pop_n_convert (List.rev typs) (List.rev tyreps) 
  with e -> 
    dprint_endline "*** Error: pp_erase.ml: Internal error in coerce_typ_stack_to_tyrep_stack cenv"; 
    (*F# rethrow(); F#*) raise e

(* -------------------------------------------------------------------- 
 * Wrap a call to a function so we can pass a method pointer across 
 * a representation boundary....
 *
 * Load up each argument and convert it if needed.  Convert the 
 * return if needed.
 * -------------------------------------------------------------------- *)

let mk_wrapper_def cenv ((outer_gparams, mr1,_,formal_fptr_rep), wrapper_mspec) =
  let argtyreps = args_of_callsig formal_fptr_rep in 
  let rtyrep = ret_of_callsig formal_fptr_rep in 
  let argtys = args_of_callsig (actual_callsig_of_mspec mr1) in 
  let rty = ret_of_callsig (actual_callsig_of_mspec mr1) in 
  if List.length argtys <> List.length argtyreps then failwith "huh?";
  let is_instance = not (is_static_callconv (callconv_of_mspec mr1)) in 
  let ret_fixup = coerce_type_to_tyrepr cenv rty rtyrep in 
  mk_static_mdef 
    (gparams_of_inst (minst_of_mspec wrapper_mspec),
     name_of_mspec wrapper_mspec,
     MemAccess_public,
     map mk_unnamed_param (formal_args_of_mspec wrapper_mspec),
     mk_return (formal_ret_of_mspec wrapper_mspec),
     mk_impl
       (false,
	[], (* locals *)
	List.length argtys + (if is_instance then 1 else 0), (* maxstack *)
	nonbranching_instrs_to_code
	  begin 
	    (if is_instance then [ ldarg_0 ] else []) @
            List.concat
	      (list_mapi2 
		 (fun i tyrep typ -> 
		   [ I_ldarg (int_to_u16 (i + (if is_instance then 1 else 0))) ] @
		   coerce_tyrepr_to_type cenv tyrep typ) 
		 argtyreps
		 argtys) @
	    [ I_call((if ret_fixup = [] then Tailcall else Normalcall), 
                     conv_mspec cenv mr1,
                     None) ] @
	    ret_fixup
	  end,
	None))  (* @todo: add DebuggerDoNotStop attribute (or whatever it's called) *)


(* Then make a wrapper that transacts appropriate representation types. *)
let mk_wrapper_ref cenv uniq (outer_gparams,mr1,formal_fptr_inst, formal_fptr_rep) = 
  try
  (* Compute the method signature that the "apply" method must satisfy. *)
  (* This will be in terms of the type variables in the method in which the *)
  (* "ldftn" will be executed on the wrapper. *)
  (* This is very subtle... convert the callsig first, then instantiate *)
  let csig = inst_callsig (conv_inst cenv formal_fptr_inst) formal_fptr_rep in 
  let cc = callconv_of_callsig csig in 
  let wrapper_argtys = args_of_callsig csig in
  let wrapper_rty = ret_of_callsig csig in 
  mk_toplevel_static_mspec ScopeRef_local
    (new_name ("__"^uniq^"_ppwrap"),
     (if is_static_callconv (callconv_of_mspec mr1) then [] else [enclosing_typ_of_mspec mr1 ]) @ wrapper_argtys, 
     wrapper_rty, 
     conv_inst cenv (generalize_gparams outer_gparams))
    with e -> 
      dprint_endline "*** Error: pp_erase.ml: Error while building wrapper reference"; 
      (*F# rethrow(); F#*) raise e

let alloc_wrapper cenv  uniq wrappersE info =
  try  List.assoc info wrappersE.wrappers 
  with Not_found ->
    let mspec = mk_wrapper_ref cenv  uniq info in
    wrappersE.wrappers <- (info,mspec)::wrappersE.wrappers;
    mspec
    
(* -------------------------------------------------------------------- 
 * Modify instructions.  Can generate new temporary variables, recorded
 * in "tmps".  [conv_instr cenv] itself needs some aspects from the context
 * in which the instruction occurs within a method/class: 
 *  -- it needs the type variables that are in scope so, 
 *     if necessary, it can generate a new helper function for 
 *     the "ldftn_then_calli" instruction and pass the correct number
 *     of types to this function.
 *  -- it needs the current scope reference for the same purpose
 * -------------------------------------------------------------------- *)

(* may need to allocate some tmps to swap around on the stack *)
let conv_call cenv tmps tl mk mr =
    let precall = 
        let caller_typ_stack = actual_args_of_mspec mr in 
        let callee_tyrep_stack =  map (conv_typ cenv) (formal_args_of_mspec mr) in
        coerce_typ_stack_to_tyrep_stack cenv tmps caller_typ_stack  callee_tyrep_stack in 
    let postcall = 
        let caller_typ = actual_ret_of_mspec mr in 
        let callee_tyrep =  conv_typ cenv (formal_ret_of_mspec mr) in
        coerce_tyrepr_to_type cenv callee_tyrep caller_typ in
    let call = 
        let do_tailcall = and_tailness tl (List.length postcall = 0) in 
        let i1 =  mk do_tailcall (conv_mspec cenv mr) in
        [ i1 ] @ postcall @
        (if tl = Tailcall && do_tailcall <> Tailcall then [I_ret] else []) in
    (precall@call)

let conv_ldfld cenv mk fr =
    let i1 = mk (conv_fspec cenv fr) in
    let caller_typ = actual_typ_of_fspec fr in 
    let callee_tyrep_rep = conv_typ cenv (formal_typ_of_fspec fr) in 
    i1 ::  coerce_tyrepr_to_type cenv callee_tyrep_rep caller_typ 

(* Loading the address of something that has been boxed requires a special case: *)
(* we load the object and then get the internal address of the thing boxed inside it. *)
let conv_ldflda cenv fspec =
    let typ = actual_typ_of_fspec fspec in 
    let tyrep = conv_typ cenv (formal_typ_of_fspec fspec) in 
    let fspec'= conv_fspec cenv fspec in 
    if repchange_typ_to_tyrep cenv typ tyrep then 
        mk_normal_ldfld fspec' :: coerce_tyrepr_to_typea cenv tyrep typ 
    else [ I_ldflda fspec' ]

let conv_stfld cenv mk fr =
  let store = 
    let i1 = mk (conv_fspec cenv fr) in
    let caller_typ = actual_typ_of_fspec fr in 
    let callee_tyrep_rep = conv_typ cenv (formal_typ_of_fspec fr) in 
    coerce_type_to_tyrepr cenv caller_typ callee_tyrep_rep @ [ i1 ]  in 
   store

let conv_erasable_ldelem cenv (shape,typ) =
  let tyrep = Type_tyvar u16_zero in 
  let i = I_ldelem_any(shape,conv_typ_as_typar cenv typ) in 
  i :: coerce_tyrepr_to_type cenv tyrep typ 

let conv_erasable_stelem cenv (shape,typ) =
  let tyrep = Type_tyvar u16_zero in 
  let i = [ I_stelem_any (shape,conv_typ_as_typar cenv typ) ] in 
  coerce_type_to_tyrepr cenv typ tyrep @ i

let conv_instr cenv (wrappersE,tmps) uniq gparams instr = 
  match instr with 
  | I_call (a,mr,b) -> 
      conv_call cenv tmps a (fun tl x -> I_call (tl,x,b)) mr
  | I_other e when is_ilx_ext_instr e -> 
      begin match (dest_ilx_ext_instr e) with 
        EI_ld_instance_ftn_then_newobj (mspec1,formal_fptr,(mspec2,varargs2)) ->
          let formal_fptr_rep = conv_callsig cenv formal_fptr  in 
          let actual_fptr_rep = inst_callsig (conv_inst cenv (active_inst_of_mspec mspec2)) formal_fptr_rep in 
          let mspec1' = 
	    mk_instance_mspec_in_boxed_tspec 
	      (tspec_of_typ (conv_typ cenv (enclosing_typ_of_mspec mspec1)),
	       name_of_mspec mspec1,
	       args_of_callsig actual_fptr_rep, 
	       ret_of_callsig actual_fptr_rep, 
	       conv_inst cenv (minst_of_mspec mspec1)) in 
	  
          let ldftn = [I_ldftn mspec1'] in 
          let call = conv_call cenv  tmps Normalcall (fun _ x -> I_newobj (x,varargs2))  mspec2  in 
          ldftn @ call
                    
      | EI_ldftn_then_call (mspec1,(tailness2,mspec2,varargs2)) -> 
          begin 
            let last_arg_ty =
	      let fargs = formal_args_of_mspec mspec2 in 
	      List.nth fargs (List.length fargs - 1)   in 
	    match last_arg_ty with 
	      Type_fptr formal_fptr -> 
	        let formal_fptr_rep = conv_callsig cenv formal_fptr in
	        let mspec1' = 
	          if repchange_typ_to_tyrep_callsig cenv  (actual_callsig_of_mspec mspec1)  formal_fptr_rep
	          then 
		    alloc_wrapper cenv uniq wrappersE 
		      (gparams,
		       mspec1,active_inst_of_mspec mspec2, formal_fptr_rep)
	          else conv_mspec cenv mspec1 in 
	        let ldftn = [I_ldftn mspec1'] in 
	        let call = conv_call cenv tmps tailness2 (fun tl x -> I_call (tl,x,varargs2)) mspec2 in
	        ldftn @ call
	    | _ -> failwith "INTERNAL ILXASM ERROR: EI_ldftn_then_call: function does not expect a code pointer as its last argument"
          end
      | EI_ldelem_any_erasable (shape, typ) -> conv_erasable_ldelem cenv (shape,typ)
      | EI_stelem_any_erasable (shape,typ) -> conv_erasable_stelem cenv (shape,typ)
      | EI_newarr_erasable (shape,typ) ->   [I_newarr (shape,conv_typ_as_typar cenv typ)]
      | _ ->  [instr]
      end
        
  | I_ldftn mr ->   [I_ldftn (conv_mspec cenv mr)]
  | I_ldvirtftn mr ->   [I_ldvirtftn (conv_mspec cenv mr)]

  (* REVIEW: calli for polymorphic methods *)
  | I_calli (a,mref,b) ->  [I_calli (a,conv_callsig cenv mref,b) ]
  | I_callvirt (a,mr,b) ->   (conv_call cenv tmps a (fun tl x -> I_callvirt (tl,x,b)) mr)
  | I_callconstraint (a,ty,mr,b) ->   (conv_call cenv tmps a (fun tl x -> I_callconstraint (tl,conv_typ cenv ty,x,b)) mr)
  | I_newobj (mr,b) ->  (conv_call cenv tmps Normalcall (fun _ x -> I_newobj (x,b)) mr )
  | I_ldelem_any (shape,typ) ->   [I_ldelem_any (shape,conv_typ cenv typ)]
  | I_stelem_any (shape,typ) ->   [I_stelem_any (shape,conv_typ cenv typ)]
  | I_newarr (shape,typ) ->   [I_newarr (shape,conv_typ cenv typ)]
  | I_initobj typ ->   [I_initobj (conv_typ cenv typ)]
  | I_box typ ->   [I_box (conv_typ cenv typ)]
  | I_unbox typ ->   [I_unbox (conv_typ cenv typ)]
  | I_unbox_any typ ->   [I_unbox_any (conv_typ cenv typ)]
  | I_ldobj (a,b,typ) ->   [I_ldobj (a,b,conv_typ cenv typ)]
  | I_stobj (a,b,typ) ->   [I_stobj (a,b,conv_typ cenv typ)]
  | I_sizeof typ ->   [I_sizeof (conv_typ cenv typ)]
  | I_ldelema (ro,shape, typ) ->   [I_ldelema (ro,shape,conv_typ cenv typ)] 
  | I_ldfld (a,b,fr) -> conv_ldfld cenv (fun x -> I_ldfld (a,b,x)) fr
  | I_ldsfld (a,fr) ->  conv_ldfld cenv (fun x -> I_ldsfld (a,x)) fr
  | I_ldflda fr ->   conv_ldflda cenv fr
  | I_stfld (a,b,fr) -> conv_stfld cenv (fun x -> I_stfld (a,b,x)) fr
  | I_stsfld (a,fr) -> conv_stfld cenv (fun x -> I_stsfld (a,x)) fr
  | I_castclass typ -> conv_castclass cenv typ
  | I_isinst typ -> conv_isinst cenv typ
  | I_ldtoken tok -> 
      begin match tok with 
	Token_type typ ->  [ I_ldtoken (Token_type (conv_typ cenv typ))]
      | Token_method mr ->  [I_ldtoken (Token_method (conv_mspec cenv mr))]
      | Token_field fr ->  [I_ldtoken (Token_field (conv_fspec cenv fr))]
      end
  | _ ->  [instr]

       
let conv_ilmbody cenv wrappersE uniq gparams il = 
  (* Allocate a new temporary variables environment.  *)
  (* This will be side-effected as temporarites get added. *)
  let tmps = new_tmps (List.length il.ilLocals) in  
  let newcode = 
    topcode_instr2instrs 
      (conv_instr cenv (wrappersE,tmps) uniq gparams) 
      il.ilCode in 
  {il with ilCode=newcode;
           ilLocals = locals_typ2typ (conv_typ cenv) il.ilLocals @ 
                      get_tmps tmps }


let dest_callsig csig = (args_of_callsig csig, ret_of_callsig csig)

let mk_virtual_wrapper cenv uniq tdefs_ctxt md ospec =
  let formal_overriden_callsig = formal_callsig_of_ospec ospec in 
  let actual_overriden_callsigrep = actual_callsig_of_ospec (conv_ospec cenv ospec) in 
  let (actual_argtys,actual_rty) = dest_callsig (callsig_of_mdef md) in 
  let (actual_overriden_argtyreps,actual_overriden_rtyrep) = dest_callsig (actual_overriden_callsigrep )in 
  let (formal_overriden_argtyreps,formal_overriden_rtyrep) = dest_callsig (conv_callsig cenv formal_overriden_callsig) in 
  if List.length actual_argtys <> List.length formal_overriden_argtyreps then failwith "mk_virtual_wrapper cenv: huh?";
  let ret_fixup = coerce_type_to_tyrepr cenv actual_rty formal_overriden_rtyrep in 
  
  mk_normal_virtual_mdef 
    (md.mdName,
     MemAccess_public,
     map mk_unnamed_param actual_overriden_argtyreps,
     mk_return actual_overriden_rtyrep,
     mk_impl
       (false,
	[], (* locals *)
	List.length actual_argtys + 1, (* maxstack *)
	nonbranching_instrs_to_code
	  begin 
	    (* Load the "this" pointer. *)
            [ ldarg_0 ] @
	    (* Load and convert the arguments as needed. *)
            List.concat
	      (list_mapi2 
		 (fun i tyrep typ -> 
		   [ I_ldarg (int_to_u16 (i+1)) ] @
		   coerce_tyrepr_to_type cenv tyrep typ)
		 formal_overriden_argtyreps
		 actual_argtys) @
	    (* Make the call. *)
	    [ I_call ((if ret_fixup = [] then Tailcall else Normalcall), 
                      (conv_mspec cenv (mk_mspec_to_mdef (Type_boxed (generalize_nested_tdef ScopeRef_local tdefs_ctxt), md, mk_empty_gactuals))),
                      None) ] @
	    (* Convert the return value as needed. *)
	    ret_fixup
	  end,
	None))    (* @todo: add DebuggerDoNotStop attribute (or whatever it's called) *)

let gparams_of_nested_tdef ((enc : type_def list),td) = gparams_of_tdef td

(* Virtual method definitions where the calling convention has changed due *)
(* to polymorphic inheritance require special attention.  We first convert *)
(* the old method definition, then insert a new method definition that *)
(* forwards if necessary. *)
let conv_mdef cenv wrappersE uniq tdefs_ctxt md =
  let gparams = 
    let cgparams = gparams_of_nested_tdef tdefs_ctxt in 
    let mgparams = md.mdGenericParams in 
    (cgparams@mgparams) in
  let md' = 
    mdef_typ2typ_ilmbody2ilmbody 
      ((fun mdef_ctxt ty -> conv_typ cenv ty),
       (fun mdef_ctxt ilmbody -> conv_ilmbody cenv wrappersE uniq gparams ilmbody)) md  in

  match md.mdKind with 
  | MethodKind_virtual ({virtOverrides=Some ospec} as vinfo) ->
      let formal_overriden_callsig = conv_callsig cenv (formal_callsig_of_ospec ospec) in 
      let overriden_callsig = actual_callsig_of_ospec (conv_ospec cenv ospec) in 
      let repchange = 
        try repchange_typ_to_tyrep_callsig cenv (callsig_of_mdef md) formal_overriden_callsig 
        with e -> 
            dprintf2 "*** Error: pp_erase.ml: Internal error in repchange_typ_to_tyrep_callsig cenv, td = %s, md = %s" (snd tdefs_ctxt).tdName md.mdName; 
            (*F# rethrow(); F#*) raise e in 

(*
      dprint_endline (name_of_tdef (snd tdefs_ctxt)^"::"^ md.mdName^" - rep change: "^if repchange then "yes" else "no"); 
      (dprint_string "md callsig: "; output_callsig stderr (callsig_of_mdef md); dprint_endline ""; dprint_string "mp formal callsig: "; output_callsig stderr formal_overriden_callsig; dprint_endline ""; dprint_string "mp actual callsig: "; output_callsig stderr overriden_callsig; dprint_endline "");
*)
      if repchange then begin
	
	    if md.mdGenericParams <> [] then failwith "conv_mdef cenv: cannot currently convert polymorphic virtual methods that have a representation change because of the scheme of generics being used";
    	
	    let virt_md' =  mk_virtual_wrapper cenv uniq tdefs_ctxt md ospec in 
	    [ {md' with mdKind=MethodKind_nonvirtual}; virt_md' ]
          end else if callsig_of_mdef md' <> overriden_callsig then begin
	    (* dprint_endline "callsigs still differ, but no rep. change..."; *)
	    [ {md' with mdKind= MethodKind_virtual {vinfo with virtOverrides=None};
                        mdReturn={md'.mdReturn with returnType=ret_of_callsig overriden_callsig}; 
                        mdParams= map2 (fun p1 p2 -> {p1 with paramType=p2}) md'.mdParams (args_of_callsig overriden_callsig) } ]
      end else
	    [{md' with mdKind= MethodKind_virtual {vinfo with virtOverrides=None} }]

  | _ -> 
      [ md' ]

let conv_mdefs cenv wrappersE uniq tdefs_ctxt = 
  mdefs_mdef2mdefs (conv_mdef cenv wrappersE uniq tdefs_ctxt)

let conv_module ilg uniq modul =
  (* Allocate a new environment to generate box/unbox helper functions.  *)
  (* This will be side-effected as helper functions are needed. *)
  let wrappersE = { wrappers=[] } in 
  let cenv = { ilg=ilg } in 
  (* Use a general morphing function to do what we need *)
  let modul' = 
    module_typ2typ_ilmbody2ilmbody_mdefs2mdefs 
      ((fun mod_ctxt tdefs_ctxt mdef_ctxt ty -> conv_typ cenv ty),
       (* The next one is only used to convert closure bodies, which *)
       (* should have disappeared by now. *)
       (fun mod_ctxt tdefs_ctxt mdef_ctxt ilmbody -> ilmbody),
       (* The next one does the job on all the method definition tables *)
       (* inside classes etc. *)
       (fun mod_ctxt tdefs_ctxt mdefs ->
	 conv_mdefs cenv wrappersE uniq tdefs_ctxt mdefs))
      modul in 
  let wrapper_mdefs = map (mk_wrapper_def cenv) wrappersE.wrappers in 
  List.fold_right (add_toplevel_mdef ilg) wrapper_mdefs modul'

        