/* Copyright (C) 2011 SpringSource
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.grails.datastore.mapping.core;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.grails.datastore.mapping.transactions.SessionHolder;
import org.springframework.transaction.support.TransactionSynchronizationManager;

public abstract class AbstractAttributeStoringSession implements Session {

    protected Map<Integer, Map<String, Object>> attributes = new ConcurrentHashMap<Integer, Map<String, Object>>();
    private boolean connected = true;

    public void setAttribute(Object entity, String attributeName, Object value) {
        if (entity == null) {
            return;
        }

        int id = System.identityHashCode(entity);
        Map<String, Object> attrs = attributes.get(id);
        if (attrs == null) {
            attrs = new ConcurrentHashMap<String, Object>();
            attributes.put(id, attrs);
        }

        if (attributeName != null && value != null) {
            attrs.put(attributeName, value);
        }
    }

    public Object getAttribute(Object entity, String attributeName) {
        if (entity == null) {
            return null;
        }

        final Map<String, Object> attrs = attributes.get(System.identityHashCode(entity));
        if (attrs == null || attributeName == null) {
            return null;
        }

        return attrs.get(attributeName);
    }

    /**
     * Performs clear up. Subclasses should always call into this super
     * implementation.
     */
    public void disconnect() {
        connected = false;
        try {
            clear();
            attributes.clear();
        }
        finally {
            SessionHolder sessionHolder = (SessionHolder)TransactionSynchronizationManager.getResource(getDatastore());
            if (sessionHolder != null) {
                sessionHolder.removeSession(this);
                if (sessionHolder.isEmpty()) {
                    try {
                        TransactionSynchronizationManager.unbindResource(getDatastore());
                    } catch (IllegalStateException e) {
                        // ignore session disconnected by a another thread
                    }
                }
            }
        }
    }

    public boolean isConnected() {
        return connected;
    }
}
