/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.IO;
using System.Collections;
using System.Text;
using System.Net;
using System.Net.Sockets;

using Tls=Mono.Security.Protocol.Tls;
using Mono.Security.Protocol.Tls;

namespace PostgreSql.Data.NPgClient
{
	internal delegate void SslConnectionCallback();

	internal class PgDbClient
	{
		#region Events

		public event PgClientMessageEventHandler		InfoMessage;
		public event PgClientNotificationEventHandler	Notification;
		public event SslConnectionCallback				SslConnection;

		#endregion

		#region Static Fields

		private static PgTypeCollection		types;
		private static PgCharSetCollection	charSets;

		#endregion

		#region Fields

		private int					handle;
		private int					secretKey;
		private Hashtable			parameterStatus;
		private Socket				socket;
		private NetworkStream		networkStream;
		private	SslClientStream		sslStream;
		private BinaryReader		receive;
		private BinaryWriter		send;		
		private PgResponsePacket	buffer;
		private PgConnectionParams	settings;
		private char				transactionStatus;		

		#endregion

		#region Static Properties

		public static PgTypeCollection Types
		{
			get { return types; }
			set { types = value; }
		}

		public static PgCharSetCollection CharSets
		{
			get { return charSets; }
			set { charSets = value; }
		}

		#endregion

		#region Properties

		public int Handle
		{
			get { return handle; }
			set { handle = value; }
		}

		public int SecretKey
		{
			get { return secretKey; }
			set { secretKey = value; }
		}

		public Hashtable ParameterStatus
		{
			get { return parameterStatus; }
			set { parameterStatus = value; }
		}

		public BinaryReader Receive
		{
			get { return receive; } 
		}

		public BinaryWriter Send
		{
			get { return send; } 
		}

		public PgConnectionParams Settings
		{
			get { return settings; }
			set { settings = value; }
		}

		public SslClientStream SslClientStream
		{
			get { return sslStream; }
		}

		public SslConnectionCallback SslConnectionDelegate
		{
			get { return this.SslConnection; }
			set { this.SslConnection = value; }
		}

		#endregion

		#region Constructors

		public PgDbClient()
		{
			this.parameterStatus	= new Hashtable();
			this.buffer				= new PgResponsePacket();
		}

		public PgDbClient(PgConnectionParams settings) : this()
		{
			this.settings = settings;
		}

		#endregion

		#region Database Methods

		public void Connect()
		{
			try
			{								
				PgDbClient.InitializeTypes();
				PgDbClient.InitializeCharSets();

				this.initializeSocket();
				
				lock (this)
				{
					if (this.settings.SSL)
					{
						// Send SSL request message
						this.SSLRequest();

						if (this.settings.SSL)
						{
							this.sslStream = new SslClientStream(
								this.networkStream,
								this.settings.ServerName,
								true,
								Tls.SecurityProtocolType.Tls|
								Tls.SecurityProtocolType.Ssl3);

							this.receive	= new BinaryReader(this.sslStream);
							this.send		= new BinaryWriter(this.sslStream);

							if (this.SslConnection != null)
							{
								this.SslConnection();
							}
						}
					}

					// Send Startup message
					PgOutputPacket packet = new PgOutputPacket(this.settings.Encoding);
					
					packet.WriteInt(PgCodes.PROTOCOL_VERSION3);
					packet.WriteString("user");
					packet.WriteString(this.settings.UserName);					
					if (settings.Database != null && 
						this.settings.Database.Length > 0)
					{
						packet.WriteString("database");
						packet.WriteString(this.settings.Database);
					}
					packet.Write((byte)0);	// Terminator

					// Handshake protocol will be negotiated here if the connection is using SSL/TLS
					this.SendSimplePacket(packet);

					PgResponsePacket response = new PgResponsePacket();

					// Check if the is ready for Query
					while (response.Message != PgBackendCodes.READY_FOR_QUERY)
					{
						response = this.ReceiveResponsePacket();
						this.processResponsePacket(response);
					}
				}
			}
			catch (IOException ex)
			{
				this.detach();
				throw new PgClientException(ex.Message);
			}
			catch (PgClientException ex)
			{
				this.detach();
				throw ex;
			}
		}

		public void Disconnect()
		{
			try
			{			
				// Send packet to the server
				PgOutputPacket packet = new PgOutputPacket();				
				this.SendPacket(packet, PgFrontEndCodes.TERMINATE);

				this.detach();
			}
			catch (IOException ex)
			{
				throw new PgClientException(ex.Message);
			}
			catch (PgClientException ex)
			{
				throw ex;
			}
		}

		#endregion

		#region Send Methods

		internal void SendPacket(PgOutputPacket packet, char type)
		{
			try
			{
				this.send.Write(packet.GetPacketBytes(type));
			}
			catch (IOException ex)
			{
				throw ex;
			}
		}

		internal void SendSimplePacket(PgOutputPacket packet)
		{
			try
			{
				this.send.Write(packet.GetSimplePacketBytes());
			}
			catch (IOException ex)
			{
				throw ex;
			}
		}

		#endregion

		#region Response Methods

		public PgResponsePacket ReceiveResponsePacket()
		{
			PgResponsePacket responsePacket = null;

			lock (this)
			{
				responsePacket = this.receiveStandardPacket();

				switch (responsePacket.Message)
				{
					case PgBackendCodes.ERROR_RESPONSE:
					{
						// Read the error message and trow the exception
						PgClientException ex = processErrorPacket(responsePacket);

						// Perform a sync
						Sync();

						throw ex;
					}

					case PgBackendCodes.NOTICE_RESPONSE:
					{
						// Read the notice message and raise an InfoMessage event
						PgClientException ex = processErrorPacket(responsePacket);

						this.InfoMessage(this, new PgClientMessageEventArgs(ex));
					}
					break;

					case PgBackendCodes.NOTIFICATION_RESPONSE:
					{
						processNotificationResponse(responsePacket);
					}
					break;

					default:
						break;
				}
			}

			return responsePacket;
		}

		private PgResponsePacket receiveStandardPacket()
		{
			PgResponsePacket responsePacket = null;

			try
			{
				char	type	= this.receive.ReadChar();			
				int		length	= IPAddress.HostToNetworkOrder(this.receive.ReadInt32()) - 4;

				// Read the message data
				byte[]	buffer		= new byte[length];
				int		received	= 0;
				while (received < length)
				{
					received += this.receive.Read(buffer, received, length - received);
				}
				responsePacket			= new PgResponsePacket(type, buffer);				
				responsePacket.Encoding = Settings.Encoding;
			}
			catch (IOException ex)
			{
				throw ex;
			}

			return responsePacket;
		}

		private void processResponsePacket(PgResponsePacket packet)
		{
			switch (packet.Message)
			{
				case PgBackendCodes.AUTHENTICATION:
					processAuthPacket(packet);
					break;

				case PgBackendCodes.PARAMETER_STATUS:
					processParameterStatus(packet);
					break;
				
				case PgBackendCodes.READY_FOR_QUERY:
					transactionStatus = packet.ReadChar();
					break;

				case PgBackendCodes.BACKEND_KEY_DATA:
					// BackendKeyData					
					Handle		= packet.ReadInt();
					SecretKey	= packet.ReadInt();
					break;
			}
		}

		private void processParameterStatus(PgResponsePacket packet)
		{
			string parameterName	= packet.ReadNullString();
			string parameterValue	= packet.ReadNullString();

			parameterStatus.Add(parameterName, parameterValue);

			switch (parameterName)
			{
				case "client_encoding":
					settings.Encoding = charSets[parameterValue].Encoding;
					break;
			}
		}

		private void processAuthPacket(PgResponsePacket packet)
		{
			// Authentication response
			int authType = packet.ReadInt();

			PgOutputPacket outPacket = new PgOutputPacket(settings.Encoding);

			switch (authType)
			{
				case PgCodes.AUTH_OK:
					// Authentication successful
					return;
						
				case PgCodes.AUTH_KERBEROS_V4:
					// Kerberos V4 authentication is required
					break;

				case PgCodes.AUTH_KERBEROS_V5:
					// Kerberos V5 authentication is required
					break;

				case PgCodes.AUTH_CLEARTEXT_PASSWORD:
					// Cleartext password is required
					outPacket.WriteString(settings.UserPassword);
					break;

				case PgCodes.AUTH_CRYPT_PASSWORD:
					// crypt()-encrypted password is required
					break;

				case PgCodes.AUTH_MD5_PASSWORD:
					// MD5-encrypted password is required
					
					// First read salt to use when encrypting the password
					byte[] salt = packet.ReadBytes(4);

					// Second calculate md5 of password + user
					string userHash = MD5Authentication.GetMD5Hash(
									settings.Encoding.GetBytes(settings.UserName), 
									settings.UserPassword);

					// Third calculate real MD5 hash
					string hash = MD5Authentication.GetMD5Hash(salt, userHash);

					// Finally write the md5 hash to the packet
					outPacket.WriteString(PgCodes.MD5_PREFIX + hash);
					break;

				case PgCodes.AUTH_SCM_CREDENTIAL:
					// SCM credentials message is required
					break;
			}

			// Send the packet to the server
			this.SendPacket(outPacket, PgFrontEndCodes.PASSWORD_MESSAGE);
		}

		private PgClientException processErrorPacket(PgResponsePacket packet)
		{
			char			type	= ' ';
			PgClientError	error	= new PgClientError();

			while (type != PgErrorCodes.END)
			{
				type = packet.ReadChar();
				switch (type)
				{
					case PgErrorCodes.SEVERITY:
						error.Severity = packet.ReadNullString();
						break;

					case PgErrorCodes.CODE:
						error.Code = packet.ReadNullString();
						break;

					case PgErrorCodes.MESSAGE:
						error.Message = packet.ReadNullString();
						break;

					case PgErrorCodes.DETAIL:
						error.Detail = packet.ReadNullString();
						break;

					case PgErrorCodes.HINT:
						error.Hint = packet.ReadNullString();
						break;

					case PgErrorCodes.POSITION:
						error.Position = packet.ReadNullString();
						break;

					case PgErrorCodes.WHERE:
						error.Where = packet.ReadNullString();
						break;

					case PgErrorCodes.FILE:
						error.File = packet.ReadNullString();
						break;

					case PgErrorCodes.LINE:
						error.Line = Convert.ToInt32(packet.ReadNullString());
						break;

					case PgErrorCodes.ROUTINE:
						error.Routine = packet.ReadNullString();
						break;
				}
			}

			PgClientException exception = new PgClientException(error.Message);

			exception.Errors.Add(error);

			return exception;
		}

		private void processNotificationResponse(PgResponsePacket packet)
		{
			int		processID	= packet.ReadInt();
			string	condition	= packet.ReadNullString();
			string	additional	= packet.ReadNullString();

			// Raise an event as an InfoMessage
			Notification(this, new PgClientNotificationEventArgs(processID, condition, additional));
		}

		#endregion

		#region Transaction Methods

		public void BeginTransaction(IsolationLevel isolationLevel)
		{
			string sql = "START TRANSACTION ISOLATION LEVEL ";

			switch (isolationLevel)
			{
				case IsolationLevel.ReadCommitted:
					sql += "READ COMMITTED";
					break;

				case IsolationLevel.ReadUncommitted:
					throw new NotSupportedException("Read uncommitted transaction isolation is not supported");
					
				case IsolationLevel.RepeatableRead:
					throw new NotSupportedException("Repeatable read transaction isolation is not supported");
					
				case IsolationLevel.Serializable:
					sql += "SERIALIZABLE";
					break;
			}

			PgStatement stmt = CreateStatement(sql);
			stmt.Query();

			if (stmt.Tag != "START TRANSACTION")
			{
				throw new PgClientException("A transaction is currently active. Parallel transactions are not supported.");
			}

			transactionStatus = stmt.TransactionStatus;
		}

		public void CommitTransaction()
		{
			PgStatement stmt = CreateStatement("COMMIT TRANSACTION");		
			stmt.Query();

			if (stmt.Tag != "COMMIT")
			{
				throw new PgClientException("There are no transaction for commit.");
			}

			transactionStatus = stmt.TransactionStatus;
		}

		public void RollbackTransction()
		{
			PgStatement stmt = CreateStatement("ROLLBACK TRANSACTION");
			stmt.Query();

			if (stmt.Tag != "ROLLBACK")
			{
				throw new PgClientException("There are no transaction for rollback.");
			}

			transactionStatus = stmt.TransactionStatus;
		}

		#endregion

		#region Client Methods

		public void Flush()
		{
			lock (this)
			{
				try
				{
					PgOutputPacket packet = new PgOutputPacket(Settings.Encoding);

					// Send packet to the server
					this.SendPacket(packet, PgFrontEndCodes.FLUSH);
				}
				catch (Exception ex)
				{
					throw ex;
				}
			}
		}

		public void Sync()
		{
			lock (this)
			{
				try
				{
					PgOutputPacket packet = new PgOutputPacket(Settings.Encoding);

					// Send packet to the server
					this.SendPacket(packet, PgFrontEndCodes.SYNC);

					// Receive response
					PgResponsePacket response = new PgResponsePacket();
					while (response.Message != PgBackendCodes.READY_FOR_QUERY)
					{
						response = ReceiveResponsePacket();
						processResponsePacket(response);
					}
				}
				catch (Exception ex)
				{
					PgResponsePacket response = new PgResponsePacket();
					while (response.Message != PgBackendCodes.READY_FOR_QUERY)
					{
						response = ReceiveResponsePacket();
						processResponsePacket(response);
					}

					throw ex;
				}
			}
		}

		public void CancelRequest()
		{
			lock (this)
			{
				try
				{
					PgOutputPacket packet = new PgOutputPacket();

					packet.WriteInt(16);
					packet.WriteInt(PgCodes.CANCEL_REQUEST);
					packet.WriteInt(Handle);
					packet.WriteInt(SecretKey);

					// Send packet to the server
					this.SendSimplePacket(packet);
				}
				catch (Exception ex)
				{
					throw ex;
				}
			}
		}

		public void SSLRequest()
		{
			lock (this)
			{
				this.settings.SSL = false;

				try
				{
					PgOutputPacket packet = new PgOutputPacket();

					packet.WriteInt(PgCodes.SSL_REQUEST);

					// Send packet to the server
					this.SendSimplePacket(packet);

					// Receive server response
					char sslSupport = Convert.ToChar(this.networkStream.ReadByte());

					switch (sslSupport)
					{
						case 'S':
							this.settings.SSL = true;
							break;

						default:
							this.settings.SSL = false;
							break;
					}
				}
				catch (Exception ex)
				{
					throw ex;
				}
			}
		}

		#endregion

		#region Static Methods

		public static void InitializeTypes()
		{
			if (types != null)
			{
				return;
			}

			types = new PgTypeCollection();

			types.Add(16	, "bool"		, PgDataType.Boolean	, 0, 1, 1);
			types.Add(17	, "bytea"		, PgDataType.Binary		, 0, 1, 0);
			types.Add(18	, "char"		, PgDataType.Char		, 0, 0, 0);
			types.Add(19	, "name"		, PgDataType.VarChar	, 0, 0, 0);
			types.Add(20	, "int8"		, PgDataType.Int8		, 0, 1, 8);
			types.Add(21	, "int2"		, PgDataType.Int2		, 0, 1, 2);
			types.Add(22	, "int2vector"	, PgDataType.Vector		, 21, 1, 2);
			types.Add(23	, "int4"		, PgDataType.Int4		, 0, 1, 4);
			types.Add(24	, "regproc"		, PgDataType.VarChar	, 0, 0, 0);
			types.Add(25	, "text"		, PgDataType.VarChar	, 0, 0, 0);
			types.Add(26	, "oid"			, PgDataType.Int4		, 0, 1, 4);
			types.Add(30	, "oidvector"	, PgDataType.Vector		, 26, 1, 4);			
			types.Add(600	, "point"		, PgDataType.Point		, 701, 1, 16);
			types.Add(601	, "lseg"		, PgDataType.LSeg		, 600, 1, 32);
			types.Add(602	, "path"		, PgDataType.Path		, 0, 1, 16);
			types.Add(603	, "box"			, PgDataType.Box		, 600, 1, 32);
			types.Add(604	, "polygon"		, PgDataType.Polygon	, 0, 1, 16);
			types.Add(628	, "line"		, PgDataType.Line		, 701, 1, 32);
			types.Add(629	, "_line"		, PgDataType.Array		, 628, 1, 32);
			types.Add(718	, "circle"		, PgDataType.Circle		, 0, 1, 24);
			types.Add(719	, "_circle"		, PgDataType.Array		, 718, 1, 24);
			types.Add(700	, "float4"		, PgDataType.Float		, 0, 1, 4);
			types.Add(701	, "float8"		, PgDataType.Double		, 0, 1, 8);
			types.Add(705	, "unknown"		, PgDataType.Binary		, 0, 1, 0);
			types.Add(790	, "money"		, PgDataType.Currency	, 0, 1, 4);
			types.Add(829	, "macaddr"		, PgDataType.VarChar	, 0, 0, 6);
			types.Add(869	, "inet"		, PgDataType.VarChar	, 0, 0, 0);			
			types.Add(1000	, "_bool"		, PgDataType.Array		, 16, 1, 1);
			types.Add(1005	, "_int2"		, PgDataType.Array		, 21, 1, 2);
			types.Add(1007	, "_int4"		, PgDataType.Array		, 23, 1, 4);
			types.Add(1009	, "_text"		, PgDataType.Array		, 25, 1, 0);
			types.Add(1016	, "_int8"		, PgDataType.Array		, 20, 1, 8);
			types.Add(1017	, "_point"		, PgDataType.Array		, 600, 1, 16);
			types.Add(1018	, "_lseg"		, PgDataType.Array		, 601, 1, 32);
			types.Add(1019	, "_path"		, PgDataType.Array		, 602, 1, -1);
			types.Add(1020	, "_box"		, PgDataType.Array		, 603, 1, 32);
			types.Add(1021	, "_float4"		, PgDataType.Array		, 700, 1, 4);
			types.Add(1027	, "_polygon"	, PgDataType.Array		, 604, 1, 16);
			types.Add(1033	, "aclitem"		, PgDataType.VarChar	, 0, 0, 12);
			types.Add(1034	, "_aclitem"	, PgDataType.Array		, 1033, 0, 0);
			types.Add(1042	, "bpchar"		, PgDataType.VarChar	, 0, 0, 0);
			types.Add(1043	, "varchar"		, PgDataType.VarChar	, 0, 0, 0);
			types.Add(1082	, "date"		, PgDataType.Date		, 0, 1, 4);
			types.Add(1083	, "time"		, PgDataType.Time		, 0, 1, 8);
			types.Add(1114	, "timestamp"	, PgDataType.Timestamp	, 0, 1, 8);
			types.Add(1184	, "timestamptz"	, PgDataType.TimestampWithTZ, 0, 1, 8);
			types.Add(1186	, "interval"	, PgDataType.Interval	, 0, 1, 12);
			types.Add(1266	, "timetz"		, PgDataType.TimeWithTZ	, 0, 1, 12);
			types.Add(1560	, "bit"			, PgDataType.Byte		, 0, 0, 1);
			types.Add(1562	, "varbit"		, PgDataType.Byte		, 0, 1, 0);
			types.Add(1700	, "numeric"		, PgDataType.Decimal	, 0, 0, 8);
			types.Add(1790	, "refcursor"	, PgDataType.VarChar	, 0, 0, 0);
			types.Add(2277	, "anyarray"	, PgDataType.Array		, 0, 1, 8);
		}

		public static void InitializeCharSets()
		{
			if (charSets != null)
			{
				return;
			}

			charSets = new PgCharSetCollection();
			
			PgDbClient.addCharset("SQL_ASCII"	, "ascii");			// ASCII
			PgDbClient.addCharset("EUC_JP"		, "euc-jp");		// Japanese EUC
			PgDbClient.addCharset("EUC_CN"		, "euc-cn");		// Chinese EUC
			PgDbClient.addCharset("UNICODE"		, "UTF-8");			// Unicode (UTF-8)			
			PgDbClient.addCharset("LATIN1"		, "iso-8859-1");	// ISO 8859-1/ECMA 94 (Latin alphabet no.1)			
			PgDbClient.addCharset("LATIN2"		, "iso-8859-2");	// ISO 8859-2/ECMA 94 (Latin alphabet no.2)			
			PgDbClient.addCharset("LATIN4"		, 1257);			// ISO 8859-4/ECMA 94 (Latin alphabet no.4)
			PgDbClient.addCharset("ISO_8859_7"	, 1253);			// ISO 8859-7/ECMA 118 (Latin/Greek)			
			PgDbClient.addCharset("LATIN9"		, "iso-8859-15");	// ISO 8859-15 (Latin alphabet no.9)
			PgDbClient.addCharset("KOI8"		, "koi8-r");		// KOI8-R(U)
			PgDbClient.addCharset("WIN"			, "windows-1251");	// Windows CP1251			
			PgDbClient.addCharset("WIN1256"		, "windows-1256");	// Windows CP1256 (Arabic)			
			PgDbClient.addCharset("WIN1256"		, "windows-1256");	// Windows CP1256 (Arabic)			
			PgDbClient.addCharset("WIN1256"		, "windows-1258");	// TCVN-5712/Windows CP1258 (Vietnamese)			
			PgDbClient.addCharset("WIN1256"		, "windows-874");	// Windows CP874 (Thai)
		}

		private static void addCharset(string charset, string systemCharset)
		{
			try
			{
				charSets.Add(charset, systemCharset);
			}
			catch (Exception)
			{
			}
		}

		private static void addCharset(string charset, int cp)
		{
			try
			{
				charSets.Add(charset, cp);
			}
			catch (Exception)
			{
			}
		}

		#endregion

		#region Methods

		public void SendInfoMessage(PgClientException exception) 
		{
			if (InfoMessage != null)
			{
				InfoMessage(this, new PgClientMessageEventArgs(exception));
			}
		}

		public PgStatement CreateStatement()
		{
			return new PgStatement(this);
		}

		public PgStatement CreateStatement(string stmtText)
		{
			return new PgStatement(this, stmtText);
		}

		public PgStatement CreateStatement(string parseName, string portalName)
		{
			return new PgStatement(this, parseName, portalName);
		}

		public PgStatement CreateStatement(string parseName, string portalName, string stmtText)
		{
			return new PgStatement(this, parseName, portalName, stmtText);
		}

		#endregion

		#region Private Methods

		private void initializeSocket()
		{
			IPAddress hostadd = Dns.Resolve(settings.ServerName).AddressList[0];
			IPEndPoint EPhost = new IPEndPoint(hostadd, settings.ServerPort);

			this.socket = new Socket(AddressFamily.InterNetwork,
				SocketType.Stream,
				ProtocolType.IP);

			// Set Receive Buffer size.
			socket.SetSocketOption(SocketOptionLevel.Socket,
				SocketOptionName.ReceiveBuffer, settings.PacketSize);

			// Set Send Buffer size.
			socket.SetSocketOption(SocketOptionLevel.Socket,
				SocketOptionName.SendBuffer, settings.PacketSize);

			// Make the socket to connect to the Server
			this.socket.Connect(EPhost);					
			this.networkStream = new NetworkStream(socket, true);

			// Create objects for read & write
			this.receive	= new BinaryReader(this.networkStream);
			this.send		= new BinaryWriter(this.networkStream);
		}

		private void detach()
		{
			// Close streams
			if (sslStream != null)
			{
				sslStream.Close();
			}
			if (networkStream != null)
			{
				networkStream.Close();
			}
			if (socket != null)
			{
				socket.Close();
			}
		}

		#endregion
	}
}
