/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.springframework.config.java.annotation.valuesource.EnvironmentValueSource;
import org.springframework.config.java.annotation.valuesource.PropertiesValueSource;
import org.springframework.config.java.valuesource.ValueResolver;


/**
 * Annotation used to assign externally-supplied values to fields and/or methods.
 * If a field has a default value, that default value will be used if no property
 * can be found.  If an ExternalValue-annotated method is non-abstract, its
 * method body will be executed if no matching external value can be found.
 *
 * <h3>use on fields</h3>
 * <pre>
 * &#64;Configuration
 * &#64;PropertiesValueSource(locations="some.properties")
 * public class Config {
 *     &#64;ExternalValue("value.1") String value1;
 *     &#64;ExternalValue("value.2") String value2;
 *
 *     public &#64;Bean Foo foo() {
 *         return new Foo(value1, value2);
 *     }
 * }
 * </pre>
 *
 * <h3>use on methods</h3>
 * <pre>
 * &#64;Configuration
 * &#64;PropertiesValueSource(locations="some.properties")
 * public abstract class Config {
 *     abstract &#64;ExternalValue("value.1") String value1();
 *     abstract &#64;ExternalValue("value.2") String value2();
 *
 *     public &#64;Bean Foo foo() {
 *         return new Foo(value1(), value2());
 *     }
 * }
 * </pre>
 *
 * @see ValueResolver
 * @see PropertiesValueSource
 * @see EnvironmentValueSource
 *
 * @author  Chris Beams
 * @author  Rod Johnson
 */
@Retention(RetentionPolicy.RUNTIME)
@Target({ ElementType.METHOD, ElementType.FIELD })
@Inherited
@Documented
public @interface ExternalValue {

    /**
     * Indicates the property name to be looked up against any
     * registered {@link ValueResolver} beans. If left empty,
     * value will default to the name of the annotated field
     * or method.
     */
    String value() default "";

}
