/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.context;

import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;

import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;

import org.springframework.core.io.ResourceLoader;
import org.springframework.core.type.filter.AnnotationTypeFilter;


/**
 * Component provider that detects and returns candidates classes that are both:
 *
 * <ol>
 *   <li>Annotated with {@link Configuration @Configuration}</li>
 *   <li>NOT a nested/inner configuration class.</li>
 * </ol>
 *
 * <p>Expressly eliminating the matching of nested configurations is important because of the
 * semantics of treating declaring configuration classes as parent application contexts. If all
 * configuration classes were to be treated as equals during classpath scanning, the nested classes
 * would get picked up and risk inadvertently overriding their declaring classes' bean definitions.
 * Eliminating them from scanning selection process requires callers to explicitly reference nested
 * classes. Doing this allows guarantees the preservation of the context hierarchy/nested class
 * semantics.</p>
 *
 * <p>Note that when {@link Configuration @Configuration} classes are supplied explicitly, the
 * {@link Configuration} annotation is optional. In the case of using this provider, classes must be
 * annotated. This is essentially a performance limitation, as JavaConfig would be required to
 * introspect every class in the base pacakge(s) for {@link Bean} methods, etc otherwise.</p>
 *
 * @author  Chris Beams
 */
class ClassPathScanningConfigurationProvider extends ClassPathScanningCandidateComponentProvider {

    /**
     * Create a configuration provider that uses <var>resourceLoader</var> to scan for candidates.
     */
    public ClassPathScanningConfigurationProvider(ResourceLoader resourceLoader) {
        super(false);
        this.addIncludeFilter(new AnnotationTypeFilter(Configuration.class));
        this.addExcludeFilter(new NestedClassTypeFilter());
        this.setResourceLoader(resourceLoader);
    }

}
