/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package test.feature.autowiring;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertThat;

import javax.annotation.Resource;

import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;
import org.springframework.config.java.annotation.Import;
import org.springframework.config.java.context.JavaConfigApplicationContext;
import org.springframework.config.java.plugin.context.AnnotationDrivenConfig;

import test.common.beans.TestBean;

/**
 * Feature test that proves {@link Configuration} classes can
 * declare {@link Autowired} fields and have them behave as expected.
 * <p/>
 * This test is important because it suggests {@code @ExternalBean} may
 * not be necessary for JavaConfig.  {@code @Autowired} and {@code @Resource}
 * more than provide the same functionality, and arguably in cleaner fashion
 * given that they do not require/encourage an abstract method.
 * <p/>
 * In {@code @ExternalBean}'s defense, however, it does provide the option of
 * defaulting to the method body (if present) in case the bean has not been
 * externally provided. I'm not sure that feature is 'worth it', however.
 * <p/>
 * One other consideration is that using {@code @Autowired}/{@code @Resource}
 * requires the use of {@code @AnnotationDrivenConfig}, whereas
 * {@code @ExternalBean} does not.
 * 
 * @see SJC-196
 * @see http://forum.springframework.org/showpost.php?p=200866 (post #8)
 *
 * @author Chris Beams
 */
public class ConfigurationClassAutowiringTests {

    @Configuration
    static class NameProvidingConfig {
        public @Bean String aName() {
            return "dave";
        }
    }
    
    
    // -------------------------------------------------------------------------
    
    public @Test void testAutowiredConfigurationClass() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(AutowiredConfig.class);
        assertThat(ctx.getBean(TestBean.class).getName(), equalTo("dave"));
    }

    @Configuration
    @AnnotationDrivenConfig
    @Import(NameProvidingConfig.class)
    static class AutowiredConfig {
        @Autowired String name;
        
        public @Bean TestBean beanThatNeedsName() {
            return new TestBean(name);
        }
    }
    
    
    // -------------------------------------------------------------------------
    
    public @Test void testResourceConfigurationClass() {
        JavaConfigApplicationContext ctx = new JavaConfigApplicationContext(ResourceConfig.class);
        assertThat(ctx.getBean(TestBean.class).getName(), equalTo("dave"));
    }

    @Configuration
    @AnnotationDrivenConfig
    @Import(NameProvidingConfig.class)
    static class ResourceConfig {
        @Resource(name="aName") String name;
        
        public @Bean TestBean beanThatNeedsName() {
            return new TestBean(name);
        }
    }
    
}
