// LatLonCoord.h: Definition of the CLatLonCoord class.
//
// The CLatLonCoord class, derived from the CTypedCoord class, represents a
// lat/lon coordinate.  Since this class contains an internal lat/lon
// coordinate from CTypedCoord, no conversion functions are necessary.
//
// The formatted coordinate string generated by the CreateCoordString() member
// function can be in one of the three following formats: 
// - Degrees (H DD.DDDDD*, H DDD.DDDDD*); this is the default.
// - Degrees, and decimal minutes (H DD*MM.MMM'. H DDD* MM.MMM')
// - Degrees, minutes, and decimal seconds (H DD*MM'SS.S", H DDD*MM'SS.S")
//
// The coordinate string format is set by calling the SetCurrentLatLonFormat()
// member function.
//
// Written by Jason Bevins in 1998.  File is in the public domain.
//

#ifndef __LAT_LON_CPPOORD_HPP
#define __LAT_LON_CPPOORD_HPP

#include "typedCoord.h"

enum HEMISPHERE_DIRECTION
{
  HEMISPHERE_NS = 0,
  HEMISPHERE_EW = 1
};

enum COORD_LATLON_FORMAT
{
  FORMAT_D   = 0,
  FORMAT_DM  = 1,
  FORMAT_DMS = 2
};

// These constants are used by the CreateDisplayStrings() and the
// CreateXYCoordStrings() functions to parse the string generated by the
// CreateCoordString() function.  Modify these constants when you modify the
// code that generates the string.  These constants do not affect the code
// that generate these strings, but you may write parsing functions that
// require the positions of the individual elements in the strings.

const int LAT_D_HPPEMISPHERE_POS   = 0;
const int LAT_D_HPPEMISPHERE_LEN   = 1;
const int LON_D_HPPEMISPHERE_POS   = 13;
const int LON_D_HPPEMISPHERE_LEN   = 1;
const int LAT_D_DEGREE_POS       = 2;
const int LAT_D_DEGREE_LEN       = 9;
const int LON_D_DEGREE_POS       = 15;
const int LON_D_DEGREE_LEN       = 10;

const int LAT_DM_HPPEMISPHERE_POS  = 0;
const int LAT_DM_HPPEMISPHERE_LEN  = 1;
const int LON_DM_HPPEMISPHERE_POS  = 14;
const int LON_DM_HPPEMISPHERE_LEN  = 1;
const int LAT_DM_DEGREE_POS      = 2;
const int LAT_DM_DEGREE_LEN      = 10;
const int LON_DM_DEGREE_POS      = 16;
const int LON_DM_DEGREE_LEN      = 11;

const int LAT_DMS_HPPEMISPHERE_POS = 0;
const int LAT_DMS_HPPEMISPHERE_LEN = 1;
const int LON_DMS_HPPEMISPHERE_POS = 15;
const int LON_DMS_HPPEMISPHERE_LEN = 1;
const int LAT_DMS_DEGREE_POS     = 2;
const int LAT_DMS_DEGREE_LEN     = 11;
const int LON_DMS_DEGREE_POS     = 17;
const int LON_DMS_DEGREE_LEN     = 12;

const char DEGREE_CHAR = '*';


class LatLonCoord: public TypedCoord
{
public:
  LatLonCoord ();
  LatLonCoord (double lat, double lon);
  LatLonCoord (const TypedCoord& other);
  LatLonCoord (const LatLonCoord& other);
  const LatLonCoord& operator= (const LatLonCoord& other);
  const LatLonCoord& operator= (const TypedCoord& other);
  void copyLatLonCoord (const LatLonCoord& other);
  void copyOtherCoord (const TypedCoord& other);
  virtual const std::string& createCoordString (std::string& coordString) const;
  virtual void createDisplayStrings (std::string& topLeftString,
                                     std::string& topRightString, std::string& bottomLeftString,
                                     std::string& bottomRightString) const;
  virtual void createXYCoordStrings (std::string& xString, std::string& yString)
    const;
  virtual COORD_TYPE getCoordType () const {return COORD_LATLON;}
  COORD_LATLON_FORMAT getLatLonFormat () const
  {return m_latLonFormat;}
  virtual void getXYCoord (double& x, double& y) const;
  void setLatLonFormat (COORD_LATLON_FORMAT latLonFormat)
  {m_latLonFormat = latLonFormat;}

protected:
  // Conversion functions for each of the three lat/lon formats.
  const std::string& degreeToD (std::string& output, double degree,
                                HEMISPHERE_DIRECTION direction) const;
  const std::string& degreeToDM (std::string& output, double degree,
                                 HEMISPHERE_DIRECTION direction) const;
  const std::string& degreeToDMS (std::string& output, double degree,
                                  HEMISPHERE_DIRECTION direction) const;
  const std::string& degreeToString (std::string& output, double degree,
                                     HEMISPHERE_DIRECTION formatType) const;
  char getHemisphereChar (double degree, HEMISPHERE_DIRECTION direction)
    const;
		
  // Current lat/lon format in one of three format types.
  COORD_LATLON_FORMAT m_latLonFormat;
};

#endif
