import { GraphQL } from "../graphql";
import { OperationType, KOGITO_BUSINESS_KEY, } from "@kie-tools/runtime-tools-shared-gateway-api/dist/types";
import { buildWorkflowListWhereArgument } from "./QueryUtils";
import axios from "axios";
import { v4 as uuidv4 } from "uuid";
export const getWorkflowInstances = async (offset, limit, filters, sortBy, client) => {
    return new Promise((resolve, reject) => {
        client
            .query({
            query: GraphQL.GetProcessInstancesDocument,
            variables: {
                where: buildWorkflowListWhereArgument(filters),
                offset: offset,
                limit: limit,
                orderBy: sortBy,
            },
            fetchPolicy: "network-only",
            errorPolicy: "all",
        })
            .then((value) => {
            resolve(value.data.ProcessInstances);
        })
            .catch((reason) => {
            reject({ errorMessage: JSON.stringify(reason) });
        });
    });
};
export const getChildWorkflowInstances = async (rootWorkflowInstanceId, client) => {
    return new Promise((resolve, reject) => {
        client
            .query({
            query: GraphQL.GetChildInstancesDocument,
            variables: {
                rootProcessInstanceId: rootWorkflowInstanceId,
            },
        })
            .then((value) => {
            resolve(value.data.ProcessInstances);
        })
            .catch((reason) => reject(reason));
    });
};
export const performMultipleCancel = async (jobsToBeActioned, client) => {
    const multipleCancel = [];
    for (const job of jobsToBeActioned) {
        multipleCancel.push(new Promise((resolve, reject) => {
            client
                .mutate({
                mutation: GraphQL.JobCancelDocument,
                variables: {
                    jobId: job.id,
                },
                fetchPolicy: "no-cache",
            })
                .then((value) => {
                resolve({ successJob: job });
            })
                .catch((reason) => {
                job.errorMessage = JSON.stringify(reason.message);
                reject({ failedJob: job });
            });
        }));
    }
    return Promise.all(multipleCancel.map((mc) => mc.catch((error) => error))).then((result) => {
        return Promise.resolve(result);
    });
};
export const jobCancel = async (job, client) => {
    let modalTitle;
    let modalContent;
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.JobCancelDocument,
            variables: {
                jobId: job.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            modalTitle = "success";
            modalContent = `The job: ${job.id} is canceled successfully`;
            resolve({ modalTitle, modalContent });
        })
            .catch((reason) => {
            modalTitle = "failure";
            modalContent = `The job: ${job.id} failed to cancel. Error message: ${reason.message}`;
            reject({ modalTitle, modalContent });
        });
    });
};
export const handleJobReschedule = async (job, repeatInterval, repeatLimit, scheduleDate, client) => {
    let modalTitle;
    let modalContent;
    let parameter = {};
    if (repeatInterval === null && repeatLimit === null) {
        parameter = {
            expirationTime: new Date(scheduleDate),
        };
    }
    else {
        parameter = {
            expirationTime: new Date(scheduleDate),
            repeatInterval,
            repeatLimit,
        };
    }
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.HandleJobRescheduleDocument,
            variables: {
                jobId: job.id,
                data: JSON.stringify(parameter),
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            modalTitle = "success";
            modalContent = `Reschedule of job: ${job.id} is successful`;
            resolve({ modalTitle, modalContent });
        })
            .catch((reason) => {
            modalTitle = "failure";
            modalContent = `Reschedule of job ${job.id} failed. Message: ${reason.message}`;
            reject({ modalTitle, modalContent });
        });
    });
};
export const handleWorkflowSkip = async (workflowInstance, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.AbortProcessInstanceDocument,
            variables: {
                processId: workflowInstance.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve(value.data);
        })
            .catch((reason) => reject(reason));
    });
};
export const handleWorkflowRetry = async (workflowInstance, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.RetryProcessInstanceDocument,
            variables: {
                processId: workflowInstance.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve(value.data);
        })
            .catch((reason) => reject(reason));
    });
};
export const handleWorkflowAbort = async (workflowInstance, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.AbortProcessInstanceDocument,
            variables: {
                processId: workflowInstance.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve(value.data);
        })
            .catch((reason) => reject(reason));
    });
};
export const handleWorkflowMultipleAction = async (workflowInstances, operationType, client) => {
    return new Promise(async (resolve, reject) => {
        let operation;
        const successWorkflowInstances = [];
        const failedWorkflowInstances = [];
        switch (operationType) {
            case OperationType.ABORT:
                operation = handleWorkflowAbort;
                break;
            case OperationType.SKIP:
                operation = handleWorkflowSkip;
                break;
            case OperationType.RETRY:
                operation = handleWorkflowRetry;
                break;
        }
        for (const workflowInstance of workflowInstances) {
            await operation(workflowInstance, client)
                .then(() => {
                successWorkflowInstances.push(workflowInstance);
            })
                .catch((error) => {
                workflowInstance.errorMessage = error.message;
                failedWorkflowInstances.push(workflowInstance);
            });
        }
        resolve({ successWorkflowInstances, failedWorkflowInstances });
    });
};
export const handleNodeTrigger = async (workflowInstance, node, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.HandleNodeTriggerDocument,
            variables: {
                processId: workflowInstance.id,
                nodeId: node.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve(value.data);
        })
            .catch((reason) => reject(reason));
    });
};
export const handleWorkflowVariableUpdate = async (workflowInstance, updatedJson, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.HandleProcessVariableUpdateDocument,
            variables: {
                processId: workflowInstance.id,
                processInstanceVariables: JSON.stringify(updatedJson),
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve(JSON.parse(value.data.ProcessInstanceUpdateVariables));
        })
            .catch((reason) => reject(reason));
    });
};
export const handleNodeInstanceCancel = async (workflowInstance, node, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.HandleNodeInstanceCancelDocument,
            variables: {
                processId: workflowInstance.id,
                nodeInstanceId: node.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve();
        })
            .catch((reason) => reject(JSON.stringify(reason.message)));
    });
};
export const handleNodeInstanceRetrigger = async (workflowInstance, node, client) => {
    return new Promise((resolve, reject) => {
        client
            .mutate({
            mutation: GraphQL.HandleNodeInstanceRetriggerDocument,
            variables: {
                processId: workflowInstance.id,
                nodeInstanceId: node.id,
            },
            fetchPolicy: "no-cache",
        })
            .then((value) => {
            resolve();
        })
            .catch((reason) => reject(JSON.stringify(reason.message)));
    });
};
export const getWorkflowDetails = async (id, client) => {
    return new Promise((resolve, reject) => {
        client
            .query({
            query: GraphQL.GetProcessInstanceByIdDocument,
            variables: {
                id,
            },
            fetchPolicy: "network-only",
        })
            .then((value) => {
            resolve(value.data.ProcessInstances[0]);
        })
            .catch((error) => {
            reject(error["graphQLErrors"][0]["message"]);
        });
    });
};
export const getJobs = async (id, client) => {
    return client
        .query({
        query: GraphQL.GetJobsByProcessInstanceIdDocument,
        variables: {
            processInstanceId: id,
        },
        fetchPolicy: "network-only",
    })
        .then((value) => {
        return value.data.Jobs;
    })
        .catch((error) => {
        return error;
    });
};
export const getTriggerableNodes = async (workflowInstance, client) => {
    return client
        .query({
        query: GraphQL.GetProcessDefinitionNodesDocument,
        variables: {
            processId: workflowInstance.id,
        },
        fetchPolicy: "no-cache",
    })
        .then((value) => {
        return value.data.ProcessInstances[0].nodeDefinitions;
    })
        .catch((reason) => {
        return reason;
    });
};
export const getJobsWithFilters = async (offset, limit, filters, orderBy, client) => {
    try {
        const response = await client.query({
            query: GraphQL.GetJobsWithFiltersDocument,
            variables: {
                values: filters,
                offset: offset,
                limit: limit,
                orderBy,
            },
            fetchPolicy: "network-only",
        });
        return Promise.resolve(response.data.Jobs);
    }
    catch (error) {
        return Promise.reject(error);
    }
};
const doTriggerCloudEvent = (event, baseUrl, proxyEndpoint) => {
    var _a;
    const cloudEvent = {
        ...event.headers.extensions,
        specversion: "1.0",
        id: uuidv4(),
        source: (_a = event.headers.source) !== null && _a !== void 0 ? _a : "",
        type: event.headers.type,
        data: event.data ? JSON.parse(event.data) : {},
    };
    if (baseUrl.endsWith("/")) {
        baseUrl = baseUrl.slice(0, baseUrl.length - 1);
    }
    const url = `${baseUrl}${event.endpoint.startsWith("/") ? "" : "/"}${event.endpoint}`;
    return axios.request({
        url: proxyEndpoint || url,
        method: event.method,
        data: cloudEvent.data,
        headers: {
            ...(proxyEndpoint ? { "Target-Url": url } : {}),
            "ce-type": cloudEvent.type,
            "ce-source": cloudEvent.source,
            "ce-id": cloudEvent.id,
            "ce-specversion": cloudEvent.specversion,
            ...event.headers.extensions,
        },
    });
};
export const triggerStartCloudEvent = (event, baseUrl, proxyEndpoint) => {
    if (!event.headers.extensions[KOGITO_BUSINESS_KEY]) {
        event.headers.extensions[KOGITO_BUSINESS_KEY] = String(Math.floor(Math.random() * 100000));
    }
    return new Promise((resolve, reject) => {
        doTriggerCloudEvent(event, baseUrl, proxyEndpoint)
            .then((response) => resolve(event.headers.extensions[KOGITO_BUSINESS_KEY]))
            .catch((error) => reject(error));
    });
};
export const triggerCloudEvent = (event, baseUrl, proxyEndpoint) => {
    return doTriggerCloudEvent(event, baseUrl, proxyEndpoint);
};
export const getWorkflowDefinitions = (client) => {
    return new Promise((resolve, reject) => {
        client
            .query({
            query: GraphQL.GetProcessDefinitionsDocument,
            fetchPolicy: "network-only",
            errorPolicy: "all",
        })
            .then((value) => {
            const workflowDefinitions = value.data.ProcessDefinitions;
            resolve(value.data.ProcessDefinitions.map((item) => {
                return {
                    workflowName: item.id,
                    endpoint: item.endpoint,
                    serviceUrl: item.serviceUrl,
                };
            }));
        })
            .catch((reason) => {
            reject({ errorMessage: JSON.stringify(reason) });
        });
    });
};
export const getWorkflowSchema = (workflowDefinitionData) => {
    return new Promise((resolve, reject) => {
        axios
            .get(`${workflowDefinitionData.endpoint}/schema`)
            .then((response) => {
            if (response.status === 200) {
                resolve(response.data);
            }
        })
            .catch((error) => {
            reject(error);
        });
    });
};
export const startWorkflowInstance = (formData, businessKey, workflowDefinitionData, proxyEndpoint) => {
    return new Promise((resolve, reject) => {
        const requestURL = `${workflowDefinitionData.endpoint}${businessKey.length > 0 ? `?businessKey=${businessKey}` : ""}`;
        axios
            .post(proxyEndpoint, formData, {
            headers: {
                "Target-Url": requestURL,
                "Content-Type": "application/json",
            },
        })
            .then((response) => {
            resolve(response.data.id);
        })
            .catch((error) => reject(error));
    });
};
export const startWorkflowRest = (data, endpoint, businessKey, proxyEndpoint) => {
    const requestURL = `${endpoint}${businessKey.length > 0 ? `?businessKey=${businessKey}` : ""}`;
    return new Promise((resolve, reject) => {
        axios
            .post(proxyEndpoint || endpoint, { workflowdata: data }, {
            headers: {
                ...(proxyEndpoint ? { "Target-Url": requestURL } : {}),
            },
        })
            .then((response) => {
            resolve(response.data);
        })
            .catch((err) => reject(err));
    });
};
export const getCustomDashboard = (customDashboardFilter) => {
    return new Promise((resolve, reject) => {
        axios
            .get("/customDashboard/list", {
            params: {
                names: customDashboardFilter.join(";"),
            },
        })
            .then((result) => {
            resolve(result.data);
        })
            .catch((error) => reject(error));
    });
};
export const getCustomDashboardContent = (name) => {
    return new Promise((resolve, reject) => {
        axios
            .get(`/customDashboard/${name}`)
            .then((result) => {
            resolve(result.data);
        })
            .catch((error) => reject(error));
    });
};
export const getCustomWorkflowSchemaFromApi = async (api, workflowName) => {
    var _a, _b;
    let schema = {};
    try {
        const schemaFromRequestBody = api.paths["/" + workflowName].post.requestBody.content["application/json"].schema;
        if (schemaFromRequestBody.type) {
            schema = {
                type: schemaFromRequestBody.type,
                properties: schemaFromRequestBody.properties,
            };
        }
        else {
            schema = api.components.schemas[workflowName + "_input"];
        }
    }
    catch (e) {
        console.log(e);
        schema = api.components.schemas[workflowName + "_input"];
    }
    if (schema) {
        const { [workflowName + "_input"]: _, ...schemas } = (_b = (_a = api.components) === null || _a === void 0 ? void 0 : _a.schemas) !== null && _b !== void 0 ? _b : {};
        schema["components"] = { schemas };
    }
    return schema !== null && schema !== void 0 ? schema : null;
};
export const getCustomWorkflowSchema = async (baseUrl, openApiPath, workflowName, proxyEndpoint) => {
    return new Promise((resolve, reject) => {
        const url = `${baseUrl}/${openApiPath}`;
        axios
            .request({
            url: proxyEndpoint || url,
            headers: {
                ...(proxyEndpoint ? { "Target-Url": url } : {}),
            },
        })
            .then(async (response) => {
            resolve(await getCustomWorkflowSchemaFromApi(response.data, workflowName));
        })
            .catch((err) => reject(err));
    });
};
export const getForms = (formFilter) => {
    return new Promise((resolve, reject) => {
        axios
            .get("/forms/list", {
            params: {
                names: formFilter.join(";"),
            },
        })
            .then((result) => {
            resolve(result.data);
        })
            .catch((error) => reject(error));
    });
};
export const getFormContent = (formName) => {
    return new Promise((resolve, reject) => {
        axios
            .get(`/forms/${formName}`)
            .then((result) => {
            resolve(result.data);
        })
            .catch((error) => reject(error));
    });
};
export const saveFormContent = (formName, content) => {
    return new Promise((resolve, reject) => {
        axios
            .post(`/forms/${formName}`, content)
            .then((result) => {
            resolve();
        })
            .catch((error) => reject(error));
    });
};
export async function verifyDataIndex(dataIndexUrl) {
    if (!dataIndexUrl) {
        return false;
    }
    try {
        const response = await fetch(dataIndexUrl, {
            headers: {
                "Content-Type": "application/json",
            },
            method: "POST",
            body: '{"query":""}',
        });
        return response.status === 200;
    }
    catch (e) {
        return false;
    }
}
//# sourceMappingURL=apis.js.map