import { applyPatches, produce } from "immer";
import { cloneDeep, get, set } from "lodash";
export class HistoryService {
    constructor(listeners) {
        this.listeners = listeners;
        this.pending = new Array();
        this.history = {
            changes: [],
            index: 0,
        };
        this.batch = (state, path, recipe, validate) => {
            this.pending.push({ state: state, path: path, recipe: recipe, validate: validate });
        };
        this.commit = (state) => {
            if (state === undefined) {
                return;
            }
            const newState = this.mutate(state, null, (draft) => {
                this.pending.forEach((be) => {
                    const segment = be.path === null ? draft : get(draft, be.path.path);
                    be.recipe(segment);
                });
            });
            this.pending.forEach((be) => {
                if (be.validate !== undefined) {
                    be.validate(newState);
                }
            });
            this.listeners.forEach((listener) => listener(`Command${this.history.index}`));
            this.pending = new Array();
            return newState;
        };
        this.mutate = (state, path, recipe) => {
            if (this.history.index < this.history.changes.length) {
                this.history.changes = this.history.changes.slice(0, this.history.index);
            }
            const newState = produce(state, recipe, (patches, inversePatches) => {
                this.history.changes.push({ path: path, change: patches, reverse: inversePatches });
                this.history.index = this.history.changes.length;
            });
            return newState;
        };
        this.undo = (state) => {
            if (this.history.index > 0) {
                const change = this.history.changes[--this.history.index];
                return this.apply(state, change.path, change.reverse);
            }
            return state;
        };
        this.redo = (state) => {
            if (this.history.index < this.history.changes.length) {
                const change = this.history.changes[this.history.index++];
                return this.apply(state, change.path, change.change);
            }
            return state;
        };
        this.index = () => {
            return this.history.index;
        };
        this.changes = () => {
            return this.history.changes;
        };
        this.apply = (state, path, patch) => {
            if (path === null) {
                return applyPatches(state, patch);
            }
            const branch = get(state, path.path);
            const branchUndone = applyPatches(branch, patch);
            const newState = cloneDeep(state);
            return set(newState, path.path, branchUndone);
        };
    }
}
//# sourceMappingURL=HistoryProvider.js.map