/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.commons.memory;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.LongUnaryOperator;
import org.apache.iotdb.commons.memory.AtomicLongMemoryBlock;
import org.apache.iotdb.commons.memory.IMemoryBlock;
import org.apache.iotdb.commons.memory.MemoryBlockType;
import org.apache.iotdb.commons.memory.MemoryException;
import org.apache.iotdb.commons.utils.TestOnly;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemoryManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(MemoryManager.class);
    private static final int MEMORY_ALLOCATE_MAX_RETRIES = 3;
    private static final long MEMORY_ALLOCATE_RETRY_INTERVAL_IN_MS = 1000L;
    private static final long MEMORY_ALLOCATE_MIN_SIZE_IN_BYTES = 32L;
    private final String name;
    private final boolean enabled;
    private volatile long totalMemorySizeInBytes;
    private volatile long allocatedMemorySizeInBytes = 0L;
    private final MemoryManager parentMemoryManager;
    private final Map<String, MemoryManager> children = new ConcurrentHashMap<String, MemoryManager>();
    private final Map<String, IMemoryBlock> allocatedMemoryBlocks = new ConcurrentHashMap<String, IMemoryBlock>();

    @TestOnly
    public MemoryManager(long totalMemorySizeInBytes) {
        this.name = "Test";
        this.parentMemoryManager = null;
        this.totalMemorySizeInBytes = totalMemorySizeInBytes;
        this.enabled = false;
    }

    MemoryManager(String name, MemoryManager parentMemoryManager, long totalMemorySizeInBytes) {
        this.name = name;
        this.parentMemoryManager = parentMemoryManager;
        this.totalMemorySizeInBytes = totalMemorySizeInBytes;
        this.enabled = false;
    }

    private MemoryManager(String name, MemoryManager parentMemoryManager, long totalMemorySizeInBytes, boolean enabled) {
        this.name = name;
        this.parentMemoryManager = parentMemoryManager;
        this.totalMemorySizeInBytes = totalMemorySizeInBytes;
        this.enabled = enabled;
    }

    public synchronized IMemoryBlock exactAllocate(String name, long sizeInBytes, MemoryBlockType type) {
        if (!this.enabled) {
            return this.getOrRegisterMemoryBlock(name, sizeInBytes, type);
        }
        for (int i = 0; i < 3; ++i) {
            if (this.totalMemorySizeInBytes - this.allocatedMemorySizeInBytes >= sizeInBytes) {
                return this.getOrRegisterMemoryBlock(name, sizeInBytes, type);
            }
            try {
                this.wait(1000L);
                continue;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.warn("exactAllocate: interrupted while waiting for available memory", (Throwable)e);
            }
        }
        throw new MemoryException(String.format("exactAllocate: failed to allocate memory after %d retries, total memory size %d bytes, used memory size %d bytes, requested memory size %d bytes", 3, this.totalMemorySizeInBytes, this.allocatedMemorySizeInBytes, sizeInBytes));
    }

    public synchronized IMemoryBlock exactAllocate(String name, MemoryBlockType memoryBlockType) {
        return this.exactAllocate(name, this.totalMemorySizeInBytes - this.allocatedMemorySizeInBytes, memoryBlockType);
    }

    public synchronized IMemoryBlock exactAllocateIfSufficient(String name, long sizeInBytes, float maxRatio, MemoryBlockType memoryBlockType) {
        if (maxRatio < 0.0f || maxRatio > 1.0f) {
            return null;
        }
        if (!this.enabled) {
            return this.getOrRegisterMemoryBlock(name, sizeInBytes, memoryBlockType);
        }
        if (this.totalMemorySizeInBytes - this.allocatedMemorySizeInBytes >= sizeInBytes && (float)this.allocatedMemorySizeInBytes / (float)this.totalMemorySizeInBytes < maxRatio) {
            return this.exactAllocate(name, sizeInBytes, memoryBlockType);
        }
        LOGGER.debug("exactAllocateIfSufficient: failed to allocate memory, total memory size {} bytes, used memory size {} bytes, requested memory size {} bytes, used threshold {}", new Object[]{this.totalMemorySizeInBytes, this.allocatedMemorySizeInBytes, sizeInBytes, Float.valueOf(maxRatio)});
        return null;
    }

    public synchronized IMemoryBlock tryAllocate(String name, long sizeInBytes, LongUnaryOperator customAllocateStrategy, MemoryBlockType type) {
        if (!this.enabled) {
            return this.getOrRegisterMemoryBlock(name, sizeInBytes, type);
        }
        if (this.totalMemorySizeInBytes - this.allocatedMemorySizeInBytes >= sizeInBytes) {
            return this.getOrRegisterMemoryBlock(name, sizeInBytes, type);
        }
        long sizeToAllocateInBytes = sizeInBytes;
        while (sizeToAllocateInBytes >= 32L) {
            if (this.totalMemorySizeInBytes - this.allocatedMemorySizeInBytes >= sizeToAllocateInBytes) {
                LOGGER.debug("tryAllocate: allocated memory, total memory size {} bytes, used memory size {} bytes, original requested memory size {} bytes, actual requested memory size {} bytes", new Object[]{this.totalMemorySizeInBytes, this.allocatedMemorySizeInBytes, sizeInBytes, sizeToAllocateInBytes});
                return this.getOrRegisterMemoryBlock(name, sizeToAllocateInBytes, type);
            }
            sizeToAllocateInBytes = Math.max(customAllocateStrategy.applyAsLong(sizeToAllocateInBytes), 32L);
        }
        LOGGER.warn("tryAllocate: failed to allocate memory, total memory size {} bytes, used memory size {} bytes, requested memory size {} bytes", new Object[]{this.totalMemorySizeInBytes, this.allocatedMemorySizeInBytes, sizeInBytes});
        return null;
    }

    private IMemoryBlock getOrRegisterMemoryBlock(String name, long sizeInBytes, MemoryBlockType type) {
        if (sizeInBytes < 0L) {
            throw new MemoryException(String.format("register memory block %s failed: sizeInBytes should be non-negative", name));
        }
        return this.allocatedMemoryBlocks.compute(name, (blockName, block) -> {
            if (block != null) {
                if (block.getTotalMemorySizeInBytes() != sizeInBytes) {
                    LOGGER.warn("getOrRegisterMemoryBlock failed: memory block {} already exists, it's size is {}, requested size is {}", new Object[]{blockName, block.getTotalMemorySizeInBytes(), sizeInBytes});
                }
                return block;
            }
            this.allocatedMemorySizeInBytes += sizeInBytes;
            return new AtomicLongMemoryBlock(name, this, sizeInBytes, type);
        });
    }

    public synchronized void release(IMemoryBlock block) {
        if (block == null || block.isReleased()) {
            return;
        }
        this.releaseWithOutNotify(block);
        this.notifyAll();
    }

    public synchronized void releaseWithOutNotify(IMemoryBlock block) {
        if (block == null || block.isReleased()) {
            return;
        }
        block.markAsReleased();
        this.allocatedMemorySizeInBytes -= block.getTotalMemorySizeInBytes();
        this.allocatedMemoryBlocks.remove(block.getName());
        try {
            block.close();
        }
        catch (Exception e) {
            LOGGER.error("releaseWithOutNotify: failed to close memory block {}", (Object)block, (Object)e);
        }
    }

    public synchronized MemoryManager getOrCreateMemoryManager(String name, long sizeInBytes, boolean enabled) {
        return this.children.compute(name, (managerName, manager) -> {
            if (sizeInBytes < 0L) {
                LOGGER.warn("getOrCreateMemoryManager {}: sizeInBytes should be positive", (Object)name);
                return null;
            }
            if (manager != null) {
                LOGGER.debug("getMemoryManager: memory manager {} already exists, it's size is {}, enabled is {}", new Object[]{managerName, manager.getTotalMemorySizeInBytes(), manager.isEnable()});
                return manager;
            }
            if (this.enabled && sizeInBytes + this.allocatedMemorySizeInBytes > this.totalMemorySizeInBytes) {
                LOGGER.warn("getOrCreateMemoryManager failed: total memory size {} bytes is less than allocated memory size {} bytes", (Object)sizeInBytes, (Object)this.allocatedMemorySizeInBytes);
                return null;
            }
            this.allocatedMemorySizeInBytes += sizeInBytes;
            return new MemoryManager(name, this, sizeInBytes, enabled);
        });
    }

    public synchronized MemoryManager getOrCreateMemoryManager(String name, long totalMemorySizeInBytes) {
        return this.getOrCreateMemoryManager(name, totalMemorySizeInBytes, false);
    }

    private void reAllocateMemoryAccordingToRatio(double ratio) {
        this.totalMemorySizeInBytes = (long)((double)this.totalMemorySizeInBytes * ratio);
        for (IMemoryBlock iMemoryBlock : this.allocatedMemoryBlocks.values()) {
            iMemoryBlock.setTotalMemorySizeInBytes((long)((double)iMemoryBlock.getTotalMemorySizeInBytes() * ratio));
        }
        for (Map.Entry entry : this.children.entrySet()) {
            ((MemoryManager)entry.getValue()).reAllocateMemoryAccordingToRatio(ratio);
        }
    }

    public MemoryManager getMemoryManager(String ... names) {
        return this.getMemoryManager(0, names);
    }

    private MemoryManager getMemoryManager(int index, String ... names) {
        if (index >= names.length) {
            return this;
        }
        MemoryManager memoryManager = this.children.get(names[index]);
        if (memoryManager != null) {
            return memoryManager.getMemoryManager(index + 1, names);
        }
        return null;
    }

    public void releaseChildMemoryManager(String memoryManagerName) {
        MemoryManager memoryManager = this.children.remove(memoryManagerName);
        if (memoryManager != null) {
            memoryManager.clearAll();
            this.allocatedMemorySizeInBytes -= memoryManager.getTotalMemorySizeInBytes();
        }
    }

    public synchronized void clearAll() {
        for (MemoryManager child : this.children.values()) {
            child.clearAll();
        }
        this.children.clear();
        for (IMemoryBlock block : this.allocatedMemoryBlocks.values()) {
            if (block == null || block.isReleased()) continue;
            block.markAsReleased();
            this.allocatedMemorySizeInBytes -= block.getTotalMemorySizeInBytes();
            try {
                block.close();
            }
            catch (Exception e) {
                LOGGER.error("releaseWithOutNotify: failed to close memory block", (Throwable)e);
            }
        }
        this.allocatedMemoryBlocks.clear();
    }

    public String getName() {
        return this.name;
    }

    public boolean isEnable() {
        return this.enabled;
    }

    public long getTotalMemorySizeInBytes() {
        return this.totalMemorySizeInBytes;
    }

    public void setTotalMemorySizeInBytes(long totalMemorySizeInBytes) {
        this.totalMemorySizeInBytes = totalMemorySizeInBytes;
    }

    public void setTotalMemorySizeInBytesWithReload(long totalMemorySizeInBytes) {
        this.reAllocateMemoryAccordingToRatio((double)totalMemorySizeInBytes / (double)this.totalMemorySizeInBytes);
    }

    public long getAvailableMemorySizeInBytes() {
        return this.totalMemorySizeInBytes - this.allocatedMemorySizeInBytes;
    }

    public long getAllocatedMemorySizeInBytes() {
        return this.allocatedMemorySizeInBytes;
    }

    public long getUsedMemorySizeInBytes() {
        long memorySize = this.allocatedMemoryBlocks.values().stream().mapToLong(IMemoryBlock::getUsedMemoryInBytes).sum();
        for (MemoryManager child : this.children.values()) {
            memorySize += child.getUsedMemorySizeInBytes();
        }
        return memorySize;
    }

    public String toString() {
        return "MemoryManager{name=" + this.name + ", enabled=" + this.enabled + ", totalMemorySizeInBytes=" + this.totalMemorySizeInBytes + ", allocatedMemorySizeInBytes=" + this.allocatedMemorySizeInBytes + '}';
    }

    public void print() {
        this.print(0);
    }

    private void print(int indent) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < indent; ++i) {
            sb.append("  ");
        }
        sb.append(this);
        LOGGER.info(sb.toString());
        for (IMemoryBlock block : this.allocatedMemoryBlocks.values()) {
            block.print(indent + 2);
        }
        for (MemoryManager child : this.children.values()) {
            child.print(indent + 1);
        }
    }
}

