# Bombardier QSeries 100, 200 and 300 Electrical System
# Bea Wolf (D-ECHO) 2019-2021
# Reference(s): http://www.smartcockpit.com/docs/Dash8-200-300-Electrical.pdf http://www.smartcockpit.com/docs/Dash8-200-300-APU.pdf

#based on turboprop engine electrical system by Syd Adams

# Register Busses so they can be called
var busses = {
	left_essential: nil,
	right_essential: nil,
	left_main: nil,
	right_main: nil,
	left_secondary: nil,
	right_secondary: nil,
	left_variable_freq: nil,
	right_variable_freq: nil,
	left_fixed_freq: [ nil, nil ],
	right_fixed_freq: [ nil, nil ],
};

var internal_starter = check_or_create("/controls/engines/internal-engine-starter", 0, "INT");
var fdm_starter = [
	props.globals.getNode("/controls/engines/engine[0]/starter", 1 ),
	props.globals.getNode("/controls/engines/engine[1]/starter", 1 ),
];

var output	=	props.globals.getNode("/systems/electrical/outputs", 1);
var breakers	=	props.globals.getNode("/controls/electric/circuit-breakers", 1);

var lessential_lmain = breakers.getNode("lessential-lmain",1);
var lessential_auxbatt = breakers.getNode("lessential-auxbatt",1);

var lmain_lessential = breakers.getNode("lmain-lessential",1);

var tru_out = [
	output.initNode( "TRU[0]", 0.0, "DOUBLE" ),
	output.initNode( "TRU[1]", 0.0, "DOUBLE" ),
];

var autostart = props.globals.getNode("/controls/electric/autostart", 1);

var ext_power_avail = check_or_create( "/services/ext-pwr/enable", 0, "BOOL");


var switches = {
	apu:		check_or_create("/controls/APU/master",				0,	"BOOL"),
	battery_master:	check_or_create("/controls/electric/battery-master-switch",	0,	"BOOL"),
	aux_battery:	check_or_create("/controls/electric/aux-battery",		0,	"BOOL"),
	main_battery:	check_or_create("/controls/electric/main-battery",		0,	"BOOL"),
	dc_external:	check_or_create("/controls/electric/dc-external-power",		0,	"BOOL"),
	ac_external:	check_or_create("/controls/electric/ac-external-power",		0,	"BOOL"),
	vhf_com_stby:	check_or_create("/instrumentation/vhf-com-stby/power-knob",	0,	"INT"),
	tru: [
		check_or_create("/controls/electric/TRU-switch[0]",	0,	"BOOL"),
		check_or_create("/controls/electric/TRU-switch[1]",	0,	"BOOL"),
	],
	lighting:	{
		navigation:	check_or_create("/controls/lighting/nav-lights",		0,	"BOOL"),
		anticoll:	check_or_create("/controls/lighting/anti-coll-lights",		0,	"INT" ),	# 1 = white (strobe); 0 = off; -1 = red (beacon)
		landing_app:	check_or_create("/controls/lighting/landing-lights-app",	0,	"BOOL"),
		landing_flr:	check_or_create("/controls/lighting/landing-lights-flr",	0,	"BOOL"),
		taxi:		check_or_create("/controls/lighting/taxi-light",		0,	"BOOL"),
		logo:		check_or_create("/controls/lighting/logo-lights",		0,	"BOOL"),
		dome:		check_or_create("/controls/lighting/dome-lights",		0,	"BOOL"),
		pilot_flood:	check_or_create("/controls/lighting/panel/plt-flt",		0.0,	"DOUBLE"),
		copilot_flood:	check_or_create("/controls/lighting/panel/cop-flt",		0.0,	"DOUBLE"),
		cabin:		check_or_create("/controls/lighting/cabin",			0,	"BOOL"),
		panel:	{
			overhead:	check_or_create("/controls/lighting/panel/overhead",		0.0,	"DOUBLE"),
			glareshield:	check_or_create("/controls/lighting/panel/glareshield",		0.0,	"DOUBLE"),
			engine:		check_or_create("/controls/lighting/panel/engine",		0.0,	"DOUBLE"),
			center:		check_or_create("/controls/lighting/panel/center",		0.0,	"DOUBLE"),
		},
	},
	antiice:	{
		pitot_heat: [
			check_or_create("/controls/anti-ice/pitot-heat[0]",	0,	"BOOL"),
			check_or_create("/controls/anti-ice/pitot-heat[1]",	0,	"BOOL"),
		],
		prop_deice: [
			check_or_create("/systems/anti-ice/prop-left",		0,	"BOOL"),
			check_or_create("/systems/anti-ice/prop-right",		0,	"BOOL"),
		],
	},
	cabin:	{
		seatbelts:	check_or_create("/controls/cabin/seatbelts",	0,	"BOOL"),
	},
};

var serviceable = {
	pfd: [
		check_or_create( "/instrumentation/pfd[0]/serviceable",	1,	"BOOL"),
		check_or_create( "/instrumentation/pfd[1]/serviceable",	1,	"BOOL"),
	],
	mfd: [
		check_or_create( "/instrumentation/mfd[0]/serviceable",	1,	"BOOL"),
		check_or_create( "/instrumentation/mfd[1]/serviceable",	1,	"BOOL"),
	],
	nav: [
		check_or_create( "/instrumentation/nav[0]/serviceable",		1,	"BOOL"),
		check_or_create( "/instrumentation/nav[1]/serviceable",		1,	"BOOL"),
	],
	gps: [
		check_or_create( "/instrumentation/gps[0]/serviceable",		1,	"BOOL"),
		check_or_create( "/instrumentation/gps[1]/serviceable",		1,	"BOOL"),
	],
	advsy: [
		check_or_create( "/instrumentation/advsy[0]/serviceable",	1,	"BOOL"),
		check_or_create( "/instrumentation/advsy[1]/serviceable",	1,	"BOOL"),
	],
	fms: [
		check_or_create( "/instrumentation/fms[0]/serviceable",	1,	"BOOL"),
		check_or_create( "/instrumentation/fms[1]/serviceable",	1,	"BOOL"),
	],
	adf: [
		check_or_create( "/instrumentation/adf[0]/serviceable",		1,	"BOOL"),
		check_or_create( "/instrumentation/adf[1]/serviceable",		1,	"BOOL"),
	],
	dme: [
		check_or_create( "/instrumentation/dme[0]/serviceable",		1,	"BOOL"),
		check_or_create( "/instrumentation/dme[1]/serviceable",		1,	"BOOL"),
	],
	comm: [
		check_or_create( "/instrumentation/comm[0]/serviceable",	1,	"BOOL"),
		check_or_create( "/instrumentation/comm[1]/serviceable",	1,	"BOOL"),
		check_or_create( "/instrumentation/comm[2]/serviceable",	1,	"BOOL"),
	],
	transponder: [
		check_or_create( "/instrumentation/transponder[0]/serviceable",		1,	"BOOL"),
		check_or_create( "/instrumentation/transponder[1]/serviceable",		1,	"BOOL"),
	],
	elt:	check_or_create( "/instrumentation/elt/serviceable",		1,	"BOOL"),
	transponder: [
		check_or_create( "/instrumentation/transponder[0]/inputs/serviceable",	1,	"BOOL"),
		check_or_create( "/instrumentation/transponder[1]/inputs/serviceable",	1,	"BOOL"),
	],
	tcas:	check_or_create( "/instrumentation/tcas/serviceable",		1,	"BOOL"),
	mk_viii:	check_or_create( "/instrumentation/mk-viii/serviceable",	1,	"BOOL"),
	gear_horn:	check_or_create( "/instrumentation/gear-horn/serviceable",	1,	"BOOL"),
	to_horn:	check_or_create( "/instrumentation/to-horn/serviceable",	1,	"BOOL"),
	ovspd_horn:	check_or_create( "/instrumentation/ovspd-horn/serviceable",	1,	"BOOL"),
	vsi:	check_or_create( "/instrumentation/vertical-speed-indicator/serviceable",	1,	"BOOL"),
	esid:	check_or_create( "/instrumentation/esid/serviceable",		1,	"BOOL"),
	hyd_qty: [
		check_or_create( "/instrumentation/hydraulic/hyd-qty[0]/serviceable",	1,	"BOOL"),
		check_or_create( "/instrumentation/hydraulic/hyd-qty[1]/serviceable",	1,	"BOOL"),
	],
	stby_hyd_press: [
		check_or_create("/instrumentation/hydraulic/stby-hyd-press[0]/serviceable",	1,	"BOOL"),
		check_or_create("/instrumentation/hydraulic/stby-hyd-press[1]/serviceable",	1,	"BOOL"),
	],
};

setprop("/systems/electrical/main_battery/serviceable", 1);
setprop("/systems/electrical/aux_battery/serviceable",  1);
setprop("/systems/electrical/stby_battery/serviceable", 1);

var main_battery = Battery.new("main", "/systems/electrical/main_battery/serviceable", 24, 30, 40, 1.0, 7.0);
var aux_battery  = Battery.new("aux",  "/systems/electrical/aux_battery/serviceable",  24, 30, 40, 1.0, 7.0);
var stby_battery = Battery.new("stby", "/systems/electrical/stby_battery/serviceable", 24, 30, 40, 1.0, 7.0);
var dcgen1 = Alternator.new(0, "controls/electric/engine[0]/dc-generator", "/engines/engine[0]/n2", 30.0, 28.0,  60.0);
var dcgen2 = Alternator.new(1, "controls/electric/engine[1]/dc-generator", "/engines/engine[1]/n2", 30.0, 28.0,  60.0);
var acgen1 = Alternator.new(2, "controls/electric/engine[0]/ac-generator", "/engines/engine[0]/n2", 30.0, 115.0, 60.0);
var acgen2 = Alternator.new(3, "controls/electric/engine[1]/ac-generator", "/engines/engine[1]/n2", 30.0, 115.0, 60.0);
var apu_gen = Alternator.new(4, "controls/APU/generator", "/engines/APU/rpm", 50.0, 28.0, 60.0);

#####################################
setlistener("/sim/signals/fdm-initialized", func {
	init_switches();
	print("Electrical System ... ok");
	
});

init_switches = func() {
	
	setprop("/controls/electric/dc-bus-select", 0);
	
	lessential_lmain.setBoolValue(1);
	lessential_auxbatt.setBoolValue(1);
	
	lmain_lessential.setBoolValue(1);

}

var feed_from_rmain = 0;

busses.left_essential = bus.new( 
	"lessential", 
	"DC",
	func( bv ) {
		var power_source = "none";
		
		if( switches.battery_master.getBoolValue() == 1 ){
			var stby_battery_volts = stby_battery.get_output_volts();
			var aux_battery_volts = aux_battery.get_output_volts();
			if( stby_battery_volts > me.bus_volts ){
				if( aux_battery_volts > stby_battery_volts and lessential_auxbatt.getBoolValue() ){
					me.bus_volts = aux_battery_volts;
					power_source = "aux_battery";
				} else {
					me.bus_volts = stby_battery_volts;
					power_source = "stby_battery";
				}
			}
		}
		if( bv > me.bus_volts and lessential_lmain.getBoolValue() and lmain_lessential.getBoolValue() ){
			power_source = "lmain_bus";
			me.bus_volts = bv;
		}
		me.bus_volts *= me.serviceable.getBoolValue();
		
		var load = me.update_consumers();
		load += busses.left_fixed_freq[0].on_update( me.bus_volts );
		
		var ignition_switch0=getprop("/controls/engines/engine[0]/ignition-switch");
		if(ignition_switch0 and me.bus_volts>=20){
			setprop("/controls/engines/engine[0]/ignition", 1);
			load+=5;
		}else{
			setprop("/controls/engines/engine[0]/ignition", 0);
		}
		
		
		me.volts.setDoubleValue( me.bus_volts );   
		
		if(power_source == "aux_battery"){
			aux_battery.apply_load( load, delta_sec.getDoubleValue() );
			return 0;
		}else if(power_source == "stby_battery"){
			stby_battery.apply_load( load, delta_sec.getDoubleValue() );
			return 0;
		}else if(power_source == "lmain_bus"){
			return load;
		}else{
			return 0;
		}
	},
	[	
		consumer.new( "pitot-heat[0]", switches.antiice.pitot_heat[0], 0.1 ),
		consumer.new( "pass-warn", switches.cabin.seatbelts, 0.1 ),
		consumer_int.new( "vhf-com-stby", switches.vhf_com_stby, 0.1, 0, 1 ),
				
	#	Performance Improvement Test: Set in Systems/electrical-400.xml instead of here
	#	consumer.new( "stby-comm", serviceable.comm[2], 0.1 ),
	#	consumer.new( "mfd[0]", serviceable.mfd[0], 0.1 ),
	#	consumer.new( "advsy[0]", serviceable.advsy[0], 0.1 ),
	],
);
busses.right_essential = bus.new( 
	"ressential",  
	"DC",
	func( bv ) {
		var power_source = "none";
	
		if( switches.battery_master.getBoolValue() ){
			var main_battery_volts = main_battery.get_output_volts();
			var stby_battery_volts = stby_battery.get_output_volts();
			if(main_battery_volts > me.bus_volts){
				if( stby_battery_volts > main_battery_volts ){
					me.bus_volts = stby_battery_volts;
					power_source = "stby_battery";
				}else{
					me.bus_volts = main_battery_volts;
					power_source = "main_battery";
				}
			}
		}
		if( bv > me.bus_volts ){
			power_source = "rmain_bus";
			me.bus_volts = bv;
		}
	
		me.bus_volts *= me.serviceable.getBoolValue();
	
		#Add R Essential Bus Users here
		var load = me.update_consumers();
		
		var ignition_switch1=getprop("/controls/engines/engine[1]/ignition-switch");
		if(ignition_switch1 and me.bus_volts>=20){
			setprop("/controls/engines/engine[1]/ignition", 1);
			load+=5;
		}else{
			setprop("/controls/engines/engine[1]/ignition", 0);
		}
		
		load += busses.right_fixed_freq[0].on_update( me.bus_volts );
		
		me.volts.setDoubleValue( me.bus_volts ); 
		
		if(power_source == "stby_battery"){
			stby_battery.apply_load( load, delta_sec.getDoubleValue() );
			return 0;
		}else if(power_source == "main_battery"){
			main_battery.apply_load( load, delta_sec.getDoubleValue() );
			return 0;
		}else if(power_source == "rmain_bus"){
			return load;
		}else{
			return 0;
		}  
	},
	[	
		consumer.new( "nav-lights", switches.lighting.navigation, 0.1 ),
		
	#	Performance Improvement Test: Set in Systems/electrical-400.xml instead of here
	#	consumer.new( "stby-att", nil, 0.1 ),
	],
);
	
	
busses.left_main = bus.new( 
	"lmain",  
	"DC",
	func( dt ) {
		
		var aux_battery_volts = aux_battery.get_output_volts();
		var dcgen1_volts = dcgen1.get_output_volts();
		var external_volts = 28.0 * ext_power_avail.getBoolValue();
		var rmain_bus_volts = busses.right_main.bus_volts;
		
		load = 0.0;
		me.bus_volts = 0.0;
		power_source = "none";
		
		if( switches.aux_battery.getBoolValue() ){
			me.bus_volts = aux_battery_volts;
			power_source="aux_battery";
		}
		
		if(dcgen1_volts > me.bus_volts){
			me.bus_volts = dcgen1_volts;
			power_source = "dcgen1";
		}
		if ( switches.dc_external.getBoolValue() and ( external_volts > me.bus_volts) ) {
			power_source = "external";
			me.bus_volts = external_volts;
		}
		if( autostart.getBoolValue() and ( me.bus_volts < 25 ) ){
			power_source="temporary_autostart";
			me.bus_volts = 28;
		}
		if ( rmain_bus_volts > (me.bus_volts+1) ){ #Main Bus Connector
			power_source="rmain_bus";
			me.bus_volts=rmain_bus_volts;
			feed_from_rmain = 1;
		}
		
		me.bus_volts *= me.serviceable.getBoolValue();
		
		var load = me.update_consumers();
		
		#Engine 1 Starter
		if( internal_starter.getIntValue() == 1){
			if( me.bus_volts>15 ){
				fdm_starter[0].setBoolValue( 1 );
			} else {
				fdm_starter[0].setBoolValue( 1 );
			}
			load += 5;
		}
		
		load += busses.left_essential.on_update( me.bus_volts );
		load += busses.left_secondary.on_update( me.bus_volts );
		
		var ammeter = 0.0;
		
		if ( power_source == "aux_battery" ) {
			aux_battery.apply_load( load, dt );
			ammeter = -load;
		} elsif ( me.bus_volts > aux_battery_volts ) {
			aux_battery.apply_load( -aux_battery.charge_amps, dt );
			ammeter = aux_battery.charge_amps;
		}
		if ( power_source == "dcgen1" ){
			dcgen1.apply_load( load );
		} else {
			dcgen1.apply_load( 0.0 );
		}
		
		
		me.ammeter_ave = 0.8 * me.ammeter_ave + 0.2 * ammeter;
		
		me.volts.setDoubleValue( me.bus_volts );   
		me.amps.setDoubleValue( me.ammeter_ave );
	},
	[	
		consumer.new( "landing-lights-app", switches.lighting.landing_app, 0.1 ),
		consumer.new( "taxi-light", switches.lighting.taxi, 0.1 ),
		consumer.new( "logo-lights", switches.lighting.logo, 0.1 ),
		consumer.new( "overhead", switches.lighting.panel.overhead, 0.1 ),
		consumer.new( "glareshield", switches.lighting.panel.glareshield, 0.1 ),
		consumer.new( "plt-flt", switches.lighting.pilot_flood, 0.1 ),
		    
	#	Performance Improvement Test: Set in Systems/electrical-400.xml instead of here
	#	consumer.new( "adf[0]", serviceable.adf[0], 0.1 ),
	#	consumer.new( "dme[0]", serviceable.dme[0], 0.1 ),
	#	consumer.new( "comm[0]", serviceable.comm[0], 0.1 ),
	#	consumer.new( "nav[0]", serviceable.nav[0], 0.1 ),
	#	consumer.new( "gps[0]", serviceable.gps[0], 0.1 ),
	#	consumer.new( "transponder[0]", serviceable.transponder[0], 0.1 ),
	#	consumer.new( "elt", serviceable.elt, 0.1 ),
	#	consumer.new( "tcas", serviceable.tcas, 0.1 ),
	#	consumer.new( "mk-viii", serviceable.mk_viii, 0.1 ),
	#	consumer.new( "gear-horn", serviceable.gear_horn, 0.1 ),
	#	consumer.new( "to-horn", serviceable.to_horn, 0.1 ),
	#	consumer.new( "ovspd-horn", serviceable.ovspd_horn, 0.1 ),
	#	consumer.new( "vertical-speed-indicator", serviceable.vsi, 0.1 ),
	#	consumer.new( "pfd[0]", serviceable.pfd[0], 0.1 ),
	#	consumer.new( "fms[0]", serviceable.fms[0], 0.1 ),
	],
);

busses.right_main = bus.new( 
	"rmain",  
	"DC",
	func( dt ) {
		var main_battery_volts = main_battery.get_output_volts();
		var dcgen2_volts = dcgen2.get_output_volts();
		var external_volts = 28.0 * ext_power_avail.getBoolValue();
		var lmain_bus_volts = busses.left_main.bus_volts;
		
		load = 0.0;
		me.bus_volts = 0.0;
		power_source = "none";
		
		if ( switches.main_battery.getValue() == 1 ){
			me.bus_volts = main_battery_volts;
			power_source="main_battery";
		}
		
		if ( dcgen2_volts > me.bus_volts ){
			me.bus_volts = dcgen2_volts;
			power_source = "dcgen2";
		}
		if ( lmain_bus_volts > (me.bus_volts+1) and !feed_from_rmain ){ #Main Bus Connector
			power_source="lmain_bus";
			bus_volts=lmain_bus_volts;
		}
		if ( switches.dc_external.getBoolValue() and ( external_volts > me.bus_volts) ) {
			me.bus_volts = external_volts;
			power_source = "external";
		}
		if ( autostart.getValue()==1 and (me.bus_volts < 25 ) ){
			me.bus_volts = 28;
			power_source="temporary_autostart";
		}
		if(apu_gen.get_output_volts() > me.bus_volts){
			power_source="APU";
			me.bus_volts = apu_gen.get_output_volts();
		}
		
		me.bus_volts *= me.serviceable.getBoolValue();
		
		var load = me.update_consumers();
		
		#Engine 2 Starter
		if( internal_starter.getIntValue() == -1){
			if( me.bus_volts>15 ){
				fdm_starter[1].setBoolValue( 1 );
			} else {
				fdm_starter[1].setBoolValue( 1 );
			}
			load += 5;
		}
		
		load += busses.right_essential.on_update( me.bus_volts );
		load += busses.right_secondary.on_update( me.bus_volts );
		
		var ammeter = 0.0;
		
		if ( power_source == "main_battery" ) {
			aux_battery.apply_load( load, dt );
			ammeter = -load;
		} elsif ( me.bus_volts > main_battery_volts ) {
			aux_battery.apply_load( -main_battery.charge_amps, dt );
			ammeter = main_battery.charge_amps;
		}
		if ( power_source == "APU"){
			apu_gen.apply_load( load );
		} else {
			apu_gen.apply_load( 0.0 );
		}
		
		if(power_source == "dcgen2") {
			dcgen2.apply_load( load );
		} else {
			dcgen2.apply_load( 0.0 );
		}
		
		me.ammeter_ave = 0.8 * me.ammeter_ave + 0.2 * ammeter;
		
		me.amps.setDoubleValue( me.ammeter_ave );
		me.volts.setDoubleValue( me.bus_volts );   
	},
	[	
		consumer_int.new( "anti-coll/strobe", switches.lighting.anticoll, 0.1, 1, 0 ),
		consumer_int.new( "anti-coll/beacon", switches.lighting.anticoll, 0.1, -1, 0 ),
		consumer.new( "pitot-heat[1]", switches.antiice.pitot_heat[1], 0.1 ),
		consumer.new( "cabin-lights", switches.lighting.cabin, 0.1 ),
		consumer.new( "engineinstr", switches.lighting.panel.engine, 0.1 ),
		consumer.new( "center-panel", switches.lighting.panel.center, 0.1 ),
		consumer.new( "cop-flt", switches.lighting.copilot_flood, 0.1 ),
		consumer.new( "APU", switches.apu, 0.1 ),
		    
	#	Performance Improvement Test: Set in Systems/electrical-400.xml instead of here
	#	consumer.new( "adf[1]", serviceable.adf[1], 0.1 ),
	#	consumer.new( "dme[1]", serviceable.dme[1], 0.1 ),
	#	consumer.new( "comm[1]", serviceable.comm[1], 0.1 ),
	#	consumer.new( "gps[1]", serviceable.gps[1], 0.1 ),
	#	consumer.new( "transponder[1]", serviceable.transponder[1], 0.1 ),
	#	consumer.new( "nav[1]", serviceable.nav[1], 0.1 ),
	#	consumer.new( "pfd[1]", serviceable.pfd[1], 0.1 ),
	#	consumer.new( "mfd[1]", serviceable.mfd[1], 0.1 ),
	#	consumer.new( "esid", serviceable.esid, 0.1 ),
	#	consumer.new( "fms[1]", serviceable.fms[1], 0.1 ),
	],
);


busses.left_secondary = bus.new( 
	"lsecondary",  
	"DC",
	func( bv ) {
		
		me.bus_volts = 0.0;
		var power_source = "nil";
		var load = 0.0;
		
		var tru_volts = tru_out[0].getDoubleValue();
		
		if(tru_volts > 72){
			me.bus_volts = tru_volts / 4;
			power_source="LTRU";
		} else if( bv > 25 ) {		# Batteries don't power the secondary busses
			me.bus_volts = bv;
			power_source = "lmain_bus";
		}
		
		me.bus_volts *= me.serviceable.getBoolValue();
		
		var load = me.update_consumers();
		
		me.volts.setDoubleValue( me.bus_volts );   
		
		if(power_source == "lmain_bus"){
			return load;
		}else{
			return 0;
		}
	},
	[	
		consumer.new( "dome-lights", switches.lighting.dome, 0.1 ),
					  
	#	Performance Improvement Test: Set in Systems/electrical-400.xml instead of here
	#	consumer.new( "stby-hyd-pump[0]", nil, 0.1 ),
	#	consumer.new( "stby-hyd-press[0]", serviceable.stby_hyd_press[0], 0.1 ),
	],
);

busses.right_secondary = bus.new( 
	"rsecondary",  
	"DC",
	func( bv ) {
		
		me.bus_volts = 0.0;
		var power_source = "nil";
		var load = 0.0;
		
		var tru_volts = tru_out[1].getDoubleValue();
		
		if(tru_volts > 72){
			me.bus_volts = tru_volts / 4;
			power_source="RTRU";
		} else if( bv > 25 ) {		# Batteries don't power the secondary busses
			me.bus_volts = bv;
			power_source = "rmain_bus";
		}
		
		
		me.bus_volts *= me.serviceable.getBoolValue();
		var load = me.update_consumers();
		
		me.volts.setDoubleValue( me.bus_volts );   
		
		if(power_source == "rmain_bus"){
			return load;
		}else{
			return 0;
		}
	},
	[	
		consumer.new( "landing-lights-flr", switches.lighting.landing_flr, 0.1 ),
		
	#	Performance Improvement Test: Set in Systems/electrical-400.xml instead of here
	#	consumer.new( "stby-hyd-pump[1]", nil, 0.1 ),
	#	consumer.new( "stby-hyd-press[1]", serviceable.stby_hyd_press[1], 0.1 ),
	],
);

#########################
#	AC System	#
#########################

#	Variable Frequency Busses
busses.left_variable_freq = bus.new( 
	"lvarfreq",  
	"AC",
	func() {
		
		# Possible power sources: AC gens and AC EXT
		
		var acgen1_volts = acgen1.get_output_volts();
		var external_volts = 115.0 * ext_power_avail.getBoolValue();
		
		load = 0.0;
		power_source = "none";
		
		if(acgen1_volts > me.bus_volts){
			me.bus_volts = acgen1_volts;
			power_source = "acgen1";
		}
		if( switches.ac_external.getBoolValue() and ( external_volts > me.bus_volts) ){
			power_source = "external";
			me.bus_volts = external_volts;
		}
		if( autostart.getValue()==1 and (me.bus_volts < 25 ) ){
			power_source="temporary_autostart";
			me.bus_volts = 115;
		}
		
		me.bus_volts *= me.serviceable.getBoolValue();
		
		var load = me.update_consumers();
		
		me.volts.setDoubleValue( me.bus_volts ); 
		
		if ( power_source == "acgen1" ) {
			acgen1.apply_load(load);
			return 0.0;
		} else {
			return load;
		}
	},
	[	
		consumer.new( "l-prop-deice[0]", switches.antiice.prop_deice[0], 0.1 ),
		consumer.new( "l-prop-deice[1]", switches.antiice.prop_deice[0], 0.1 ),
		consumer.new( "TRU[0]",		 switches.tru[0], 0.1 ),
	],
);

busses.right_variable_freq = bus.new( 
	"rvarfreq", 
	"AC",
	func() {
		
		# Possible power sources: AC gens and AC EXT
		
		var acgen2_volts = acgen2.get_output_volts();
		var external_volts = 115.0 * ext_power_avail.getBoolValue();
		
		load = 0.0;
		power_source = "none";
		
		if(acgen2_volts > me.bus_volts){
			me.bus_volts = acgen2_volts;
			power_source = "acgen2";
		}
		if( switches.ac_external.getBoolValue() and ( external_volts > me.bus_volts) ){
			power_source = "external";
			me.bus_volts = external_volts;
		}
		if(autostart.getValue()==1 and (me.bus_volts < 25 ) ){
			power_source="temporary_autostart";
			me.bus_volts = 115;
		}
		
		me.bus_volts *= me.serviceable.getBoolValue();
		
		var load = me.update_consumers();
		
		me.volts.setDoubleValue( me.bus_volts );
		
		if ( power_source == "acgen2" ) {
			acgen2.apply_load(load);
			return 0.0;
		} else {
			return load;
		}
	},
	[	
		consumer.new( "r-prop-deice[0]", switches.antiice.prop_deice[1], 0.1 ),
		consumer.new( "r-prop-deice[1]", switches.antiice.prop_deice[1], 0.1 ),
		consumer.new( "TRU[1]",		 switches.tru[1], 0.1 ),
	],
);

#	Fixed Frequency Busses
busses.left_fixed_freq[0] = bus.new(
	"lff1",
	"AC",
	func( bv ) {
		me.bus_volts = bv * 4.107;
		var load = 0.0;
		var power_source = "lessential_bus";
		
		
		me.bus_volts *= me.serviceable.getBoolValue();
		
		var load = me.update_consumers();
		load + busses.left_fixed_freq[1].on_update( me.bus_volts );
		
		me.volts.setDoubleValue( me.bus_volts );
		
		return load;
	},
	[],
);
busses.left_fixed_freq[1] = bus.new(
	"lff2",
	"AC",
	func( bv ){
		me.bus_volts = bv / 4.423;
		var load = 0.0;
		var power_source = "lff1_bus";
		
		me.bus_volts *= me.serviceable.getBoolValue();
		
		var load = me.update_consumers();
		
		me.volts.setDoubleValue( me.bus_volts );
		
		return load;
	},
	[
		consumer.new( "hyd-qty[0]",	serviceable.hyd_qty[0], 0.1 ),
	],
);
busses.right_fixed_freq[0] = bus.new(
	"rff1",
	"AC",
	func( bv ) {
		me.bus_volts = bv * 4.107;
		var load = 0.0;
		var power_source = "ressential_bus";
		
		
		me.bus_volts *= me.serviceable.getBoolValue();
		
		var load = me.update_consumers();
		load + busses.right_fixed_freq[1].on_update( me.bus_volts );
		
		me.volts.setDoubleValue( me.bus_volts );
		
		return load;
	},
	[
		consumer.new( "hyd-qty[1]",	serviceable.hyd_qty[0], 0.1 ),
	],
);
busses.right_fixed_freq[1] = bus.new(
	"rff2",
	"AC",
	func( bv ){
		me.bus_volts = bv / 4.423;
		var load = 0.0;
		var power_source = "rff1_bus";
		
		me.bus_volts *= me.serviceable.getBoolValue();
		
		var load = me.update_consumers();
		
		me.volts.setDoubleValue( me.bus_volts );
		
		return load;
	},
	[],
);


update_electrical = func {
	var dt = delta_sec.getDoubleValue();
	busses.left_main.on_update( dt );
	busses.right_main.on_update( dt );
	busses.left_variable_freq.on_update();
	busses.right_variable_freq.on_update();
}

var electrical_updater = maketimer( 0.0, update_electrical );
electrical_updater.simulatedTime = 1;
electrical_updater.start();
